% This script creates maps of the temporal class distribution

function Temporal_Distribution_Figures(rows_data, columns_data, number_classes, class_names, year_list, number_years, subdomains_class_proportion_cell, subdomains_delta_proportion_cell, subdomains_relative_delta_class_cell, Bounds, Include_water)
    %%% Inputs %%%
        % If the proportion of the class is below this percentage, the results are omitted from the map
        proportion_threshold = 0.5;  % [%]
        
        % Colorbar settings
        cb_rounding                 = 5;    % The percentage to which the colorbar steps are rounded (i.e. 23% -> 25%)
        number_cb_steps             = 10;   % The maximum number of steps for the colorbar
        
        cb_steps_relative_change    = 05;   % The steps in percentage, relative change
        cb_min_relative_change      = -50;  % The minimum level of change displayed in the colorbar, relative change
        cb_max_relative_change      = 50;   % The maximum level of change displayed in the colorbar, relative change
        
        cb_steps_absolute_change    = 0.5;  % The steps in percentage, relative change
        cb_min_absolute_change      = -05;  % The minimum level of change displayed in the colorbar, relative change
        cb_max_absolute_change      = 05;   % The maximum level of change displayed in the colorbar, relative change
    
        % Outline thickness around the plot
        outline_thickness   = 10;
        
        % Border thickness used to make the lines appear more clearly in the plots
        border_thickness    = 10;           
        
        % Coarsening factor used to create the figures and save memory
        figure_coarsening_factor = 2;
            
    %%% Retrieve coarsened subdomain indices %%%
        if figure_coarsening_factor ~= 1
            Resolution_downscaling = 'On';
            [subdomain_indices_cell, ~, number_subdomains] = Subdomain_Indices_Reader_P(Resolution_downscaling, figure_coarsening_factor, rows_data, columns_data);
            
            rows_data = ceil(rows_data / figure_coarsening_factor);
            columns_data = ceil(columns_data / figure_coarsening_factor);
        end
        
    %%% Determine the borders %%%    
        border_cell = cell(1, number_subdomains);
            
        for n = 1:number_subdomains
            subdomain_indices = subdomain_indices_cell{n};
            
            matrix = zeros(rows_data, columns_data);
            
            matrix(subdomain_indices) = 1;
            
            B = bwboundaries(matrix, 8, 'holes');
            
            border_indices = cell(1, length(B));
            
            for k = 1:length(B)
                % Single-sided border width
                border_width = round((border_thickness - 1) / 2);
                
                boundary = B{k};
    
                r = boundary(:, 1)';
                c = boundary(:, 2)';

                for w = 1:border_width
                    c_e = min(columns_data, c + w);
                    c_w = max(1, c - w);
                    
                    r_n = max(1, r - w);
                    r_s = min(rows_data, r + w);
                    
                    r = [r, r_n, r_s, r, r];
                    c = [c, c, c, c_e, c_w];
                end
                    
                ind = sub2ind([rows_data, columns_data], r, c);
                
                border_indices{k} = ind;
            end
            
            border_cell{n} = horzcat(border_indices{:});
        end

        % Merge the results into one
        European_borders_list = horzcat(border_cell{:});        
        European_borders_list = unique(European_borders_list);
        
    %%% Indices that are outside the domain %%%
        if strcmp(Bounds, 'Yes')
            MODIS_Coordinates_file = load('MODIS_Coordinates.mat');
            latitude_list = MODIS_Coordinates_file.latitude_list;
            longitude_matrix = MODIS_Coordinates_file.longitude_matrix;

            if strcmp(Resolution_downscaling, 'On')
                latitude_list = latitude_list(1 : figure_coarsening_factor : end);
                longitude_matrix = longitude_matrix(1 : figure_coarsening_factor : end, 1 : figure_coarsening_factor : end);
            end

            lat_N = 71.5;
            lat_S = 34.5;
            long_W = -11;
            long_E = 32;

            matrix_OoB = zeros(rows_data, columns_data);

            % Latitude
            Latitude_OoB = latitude_list > lat_N | latitude_list < lat_S;
            matrix_OoB(Latitude_OoB) = 1;

            % Longitude
            Longitude_OoB = longitude_matrix > long_E | longitude_matrix < long_W;
            matrix_OoB(Longitude_OoB) = 1;

            ind_OoB = matrix_OoB == 1;
        elseif strcmp(Bounds, 'No')
            ind_OoB = [];
        end
        
        % Free memory
        clear matrix_OoB & Longitude_OoB & longitude_matrix & Latitude_OoB & latitude_list & MODIS_Coordinates_file
        
        % Light grey background for data that is outside the bounds
        OoB_background = 0.75 * ones(rows_data + 2*outline_thickness, columns_data + 2*outline_thickness, 3);
        OoB_alpha = zeros(rows_data, columns_data);
        OoB_alpha(ind_OoB) = 1;
        OoB_alpha(European_borders_list) = 0;
        OoB_alpha = padarray(OoB_alpha, [outline_thickness, outline_thickness], 0);
        
    %%% Plots that show the change in map format %%%     
        % The water bodies class is removed, if desired
        if strcmp(Include_water, 'No')
            for c = 1 : number_classes
                class_name = class_names{c};
                
                if strcmp(class_name, 'Water bodies')
                    WB_ind = c;
                end
            end
            
            number_classes = number_classes - 1;
            class_names(WB_ind) = [];
        end
        
        year_first = num2str(year_list(1));
        year_last = num2str(year_list(end));
   
        % Black background for the 'padding'
        black_background = zeros(rows_data + 2*outline_thickness, columns_data + 2*outline_thickness, 3);

        black_alpha = zeros(rows_data, columns_data);
        black_alpha = padarray(black_alpha, [outline_thickness, outline_thickness], 1);

        %--% Relative change in the class distribution %--%
        for c = 1:number_classes
            class_name = class_names{c};
            class_file_name = strrep(class_name, '/', '-');

            % Image data
            delta_class_image_matrix = zeros(rows_data, columns_data);

            for n = 1:number_subdomains
                % Country information
                subdomain_indices = subdomain_indices_cell{n};

                % Frequency of this class, first year is used
                class_proportion_first = subdomains_class_proportion_cell{n}(1, c);

                % Relative change w.r.t class
                delta_class = subdomains_relative_delta_class_cell{n}(number_years - 1, c);

                % Append the results                
                if class_proportion_first > proportion_threshold
                    delta_class_image_matrix(subdomain_indices) = delta_class;
                else
                    delta_class_image_matrix(subdomain_indices) = NaN;
                end
            end

            % Grey background in case the frequency threshold is not met
            NaN_ind = isnan(delta_class_image_matrix);
            Sea_ind = delta_class_image_matrix == 0;

            land_background = 0.65 * ones(rows_data + 2 * outline_thickness, columns_data + 2 * outline_thickness, 3);

            land_alpha = ones(rows_data, columns_data);
            land_alpha(Sea_ind) = 0;
            land_alpha = padarray(land_alpha, [outline_thickness, outline_thickness], 0);

            % White background for sea pixels
            sea_background = ones(rows_data + 2 * outline_thickness, columns_data + 2 * outline_thickness, 3);

            sea_alpha = zeros(rows_data, columns_data);
            sea_alpha(Sea_ind) = 1;
            sea_alpha = padarray(sea_alpha, [outline_thickness, outline_thickness], 0);

            % Black background for borders
            border_background = zeros(rows_data + 2 * outline_thickness, columns_data + 2 * outline_thickness, 3);

            border_alpha = zeros(rows_data, columns_data);
            border_alpha(European_borders_list) = 1;
            border_alpha = padarray(border_alpha, [outline_thickness, outline_thickness], 0);

            % The data itself is transparent anywhere it is NaN, 0 or where borders are located
            data_alpha = ones(rows_data, columns_data);
            data_alpha(Sea_ind) = 0;
            data_alpha(NaN_ind) = 0;
            data_alpha(European_borders_list) = 0;
            data_alpha(ind_OoB) = 0;
            data_alpha = padarray(data_alpha, [outline_thickness, outline_thickness], 0);

            delta_class_image_matrix = padarray(delta_class_image_matrix, [outline_thickness, outline_thickness], NaN);
            
            % Delta class                
            figure(1)

            % Set the size and white background color
            set(gcf, 'Units', 'Normalized', 'Position', [0 0 1.0 1.0]);
            set(gcf, 'color', [1, 1, 1]);

            % Coloured image backgrounds
            hold on
            image(sea_background, 'AlphaData', sea_alpha)
            image(land_background, 'AlphaData', land_alpha)
            image(OoB_background, 'AlphaData', OoB_alpha);
            image(border_background, 'AlphaData', border_alpha)
            image(black_background, 'AlphaData', black_alpha)
            set(gca,'YDir','reverse');
            axis off;

            % Image itself
            h = imshow(delta_class_image_matrix);
            set(h, 'AlphaData', data_alpha);

            % Define colormap that go from red to white to blue     
            cb_ticks = cb_min_relative_change : cb_steps_relative_change : cb_max_relative_change;

            cmap_redblue = cbrewer('div', 'RdBu', length(cb_ticks) - 1);
            colormap(cmap_redblue)

            % Colorbar
            cb = colorbar;
            caxis manual
            shading interp
            caxis([cb_min_relative_change, cb_max_relative_change])    

            set(cb, 'Ticks', cb_ticks);
            
            cb_ticklabels = string(cb_ticks);
            cb_ticklabels{1} = ['< ', cb_ticklabels{1}];
            cb_ticklabels{end} = ['> ', cb_ticklabels{end}];
            
            set(cb, 'TickLabels', cb_ticklabels);

            colortitlehandle = get(cb, 'Title');
            titlestring = ['\Delta', ' ', class_name, ' [%]'];
            set(colortitlehandle, 'String', titlestring);
            cb.FontSize = 25;
                
            % Save the figure
            figure_name = sprintf('Relative_Delta_Class_%s_%s', class_file_name, year_last);

            export_fig([figure_name, '.png']);
        
            close(1)
            
            % Free memory
            clear delta_class_image_matrix
                            
            % Image data
            delta_proportion_image_matrix = zeros(rows_data, columns_data);

            for n = 1:number_subdomains
                % Country information
                subdomain_indices = subdomain_indices_cell{n};

                % Frequency of this class, first year is used
                class_proportion_first = subdomains_class_proportion_cell{n}(1, c);

                % Absolute change in surface area
                delta_proportion = subdomains_delta_proportion_cell{n}(number_years - 1, c);

                % Append the results                
                if class_proportion_first > proportion_threshold
                    delta_proportion_image_matrix(subdomain_indices) = delta_proportion;
                else
                    delta_proportion_image_matrix(subdomain_indices) = NaN;
                end
            end
                
            delta_proportion_image_matrix = padarray(delta_proportion_image_matrix, [outline_thickness, outline_thickness], NaN);
            
            % Change in proportion              
            figure(2)

            % Set the size and white background color
            set(gcf, 'Units', 'Normalized', 'Position', [0 0 1.0 1.0]);
            set(gcf, 'color', [1, 1, 1]);

            % Coloured image backgrounds
            hold on
            image(sea_background, 'AlphaData', sea_alpha)
            image(land_background, 'AlphaData', land_alpha)
            image(OoB_background, 'AlphaData', OoB_alpha);
            image(border_background, 'AlphaData', border_alpha)
            image(black_background, 'AlphaData', black_alpha)
            set(gca,'YDir','reverse');
            axis off;

            % Image itself
            h = imshow(delta_proportion_image_matrix);
            set(h, 'AlphaData', data_alpha);
            
            cb_ticks = cb_min_absolute_change : cb_steps_absolute_change : cb_max_absolute_change;
            cmap_redblue = cbrewer('div', 'RdBu', length(cb_ticks) - 1);
            colormap(cmap_redblue)

            % Colorbar
            cb = colorbar;
            caxis manual
            shading interp
            caxis([cb_min_absolute_change, cb_max_absolute_change])    

            set(cb, 'Ticks', cb_ticks);
            
            cb_ticklabels = string(cb_ticks);
            cb_ticklabels{1} = ['< ', cb_ticklabels{1}];
            cb_ticklabels{end} = ['> ', cb_ticklabels{end}];
            set(cb, 'TickLabels', cb_ticklabels);
           
            colortitlehandle = get(cb, 'Title');
            titlestring = ['\Delta', ' ', class_name, ' [%]'];
            set(colortitlehandle, 'String', titlestring);
            cb.FontSize = 25;

            % Save the figure
            figure_name = sprintf('Proportion_Delta_Class_%s_%s', class_name, year_last);

            export_fig([figure_name, '.png']);
            
            close(2)
            
            % Free memory
            clear delta_proportion_image_matrix
            
            % Progress
            fprintf('Figures of the relative change in proportion have been created for %g%% of the classes \n', c / number_classes * 100);
        end
        
    	%--% The distribution in map format of the first and last years %--%           
        for c = 1:number_classes
            class_name = class_names{c};
            class_file_name = strrep(class_name, '/', '-');
            
            % First year
            distribution_first_matrix = zeros(rows_data, columns_data);
            
            for n = 1:number_subdomains
                % Country information
                subdomain_indices = subdomain_indices_cell{n};
                
                % Distribution
                class_frequency_first = subdomains_class_proportion_cell{n}(1, c);
                
                % Append the results
                distribution_first_matrix(subdomain_indices) = class_frequency_first;
            end
            
            % The data itself is transparent for sea indices and for the borders
            data_alpha = ones(rows_data, columns_data);
            data_alpha(Sea_ind) = 0;
            data_alpha(European_borders_list) = 0;
            data_alpha(ind_OoB) = 0;
            data_alpha = padarray(data_alpha, [outline_thickness, outline_thickness], 0);
            
            % Plot the results    
            distribution_first_matrix = padarray(distribution_first_matrix, [outline_thickness, outline_thickness], NaN);

            % First year
            figure(1)

            % Set the size and white background color
            set(gcf, 'Units', 'Normalized', 'Position', [0 0 1.0 1.0])
            set(gcf, 'color', [1, 1, 1])

            hold on
            image(sea_background, 'AlphaData', sea_alpha)
            image(border_background, 'AlphaData', border_alpha)
            image(OoB_background, 'AlphaData', OoB_alpha);
            image(black_background, 'AlphaData', black_alpha)
            set(gca,'YDir','reverse');
            axis off;

            % Image itself
            h = imshow(distribution_first_matrix);
            set(h, 'AlphaData', data_alpha);

            % Colormaps from white to various colors, depending on class      
            cb_limit = min(100, ceil(max(max(distribution_first_matrix)) / cb_rounding) * cb_rounding); 
            cb_limit = max(cb_limit, 1);    % To ensure that the limits are not [0, 0]

            % The number of bins is selected
            n_c = 0;
            
            if mod(cb_limit, number_cb_steps) == 0
                n_c = cb_limit / 10;
            end
            if mod(cb_limit, 5) == 0 & cb_limit / 5 < number_cb_steps + 1
                n_c = cb_limit / 5;
            end
            if mod(cb_limit, 2) == 0 & cb_limit / 2 < number_cb_steps + 1
                n_c = cb_limit / 2;
            end
            
            n_c = max(n_c, 3);  % There is a minimum of 3 colours
            
            cmap_red = cbrewer('seq', 'Reds', n_c);          % Artificial
            cmap_orange = cbrewer('seq', 'Oranges', n_c);    % Agricultural
            
            cmap_green = cbrewer('seq', 'Greens', 2*n_c);      
            cmap_darkgreen = cmap_green(n_c + 1 : 2*n_c, :); % Forests
            cmap_lightgreen = cmap_green(1 : n_c, :);        % Shrubs & Herbaceous vegetation
                                                             
            cmap_grey = cbrewer('seq', 'Greys', n_c);        % Non-vegetated
            cmap_purple = cbrewer('seq', 'RdPu', n_c);       % Wetlands
            cmap_blue = cbrewer('seq', 'Blues', n_c);        % Water bodies

            colormap_cell = {cmap_red, cmap_orange, cmap_darkgreen, cmap_lightgreen, cmap_grey, cmap_purple, cmap_blue};

            % As there can be fewer colormaps than classes, the modulo is used
            color = mod(c - 1, length(colormap_cell)) + 1;

            colormap(colormap_cell{color})

            cb = colorbar;
            caxis manual
            shading interp
            caxis([0 cb_limit])  

            set(cb, 'Ticks', linspace(0, cb_limit, n_c + 1));

            colortitlehandle = get(cb, 'Title');
            titlestring = sprintf('%s Proportion [%%]', class_name);
            set(colortitlehandle, 'String', titlestring);
            cb.FontSize = 25;

            % Save the figure
            figure_name = sprintf('Class_Distribution_%s_%s', year_first, class_name);

            export_fig([figure_name, '.png']);

            close(1)
                
            % Free memory
            clear distribution_first_matrix
            
            % Last year
            distribution_last_matrix = zeros(rows_data, columns_data);
            
            for n = 1:number_subdomains
                % Country information
                subdomain_indices = subdomain_indices_cell{n};
                
                % Distribution
                class_frequency_last = subdomains_class_proportion_cell{n}(number_years, c);
                
                % Append the results
                distribution_last_matrix(subdomain_indices) = class_frequency_last;
            end
            
            distribution_last_matrix = padarray(distribution_last_matrix, [outline_thickness, outline_thickness], NaN);

            figure(2)

            % Set the size and white background color
            set(gcf, 'Units', 'Normalized', 'Position', [0 0 1.0 1.0])
            set(gcf, 'color', [1, 1, 1])

            hold on
            image(sea_background, 'AlphaData', sea_alpha)
            image(border_background, 'AlphaData', border_alpha)
            image(OoB_background, 'AlphaData', OoB_alpha);
            image(black_background, 'AlphaData', black_alpha)
            set(gca,'YDir','reverse');
            axis off;

            % Image itself
            h = imshow(distribution_last_matrix);
            set(h, 'AlphaData', data_alpha);

            % Colormaps from white to various colors, depending on class      
            cb_limit = min(100, ceil(max(max(distribution_last_matrix)) / cb_rounding) * cb_rounding);
            cb_limit = max(cb_limit, 1);    % To ensure that the limits are not [0, 0]

            % The number of bins is selected
            if mod(cb_limit, number_cb_steps) == 0
                n_c = cb_limit / 10;
            end
            if mod(cb_limit, 5) == 0 & cb_limit / 5 < number_cb_steps + 1
                n_c = cb_limit / 5;
            end
            if mod(cb_limit, 2) == 0 & cb_limit / 2 < number_cb_steps + 1
                n_c = cb_limit / 2;
            end
            
            if ~exist('n_c', 'Var')
                n_c = 3;
            end
            
            n_c = max(n_c, 3);  % There is a minimum of 3 colours

            cmap_red = cbrewer('seq', 'Reds', n_c);          % Artificial
            cmap_orange = cbrewer('seq', 'Oranges', n_c);    % Agricultural
            
            cmap_green = cbrewer('seq', 'Greens', 2*n_c);      
            cmap_darkgreen = cmap_green(n_c + 1 : 2*n_c, :); % Forests
            cmap_lightgreen = cmap_green(1 : n_c, :);        % Shrubs & Herbaceous vegetation
                                                             
            cmap_grey = cbrewer('seq', 'Greys', n_c);        % Non-vegetated
            cmap_purple = cbrewer('seq', 'RdPu', n_c);       % Wetlands
            cmap_blue = cbrewer('seq', 'Blues', n_c);        % Water bodies

            colormap_cell = {cmap_red, cmap_orange, cmap_darkgreen, cmap_lightgreen, cmap_grey, cmap_purple, cmap_blue};

            colormap(colormap_cell{color})

            cb = colorbar;
            caxis manual
            shading interp
            caxis([0 cb_limit])    

            set(cb, 'Ticks', linspace(0, cb_limit, n_c + 1));

            colortitlehandle = get(cb, 'Title');
            titlestring = sprintf('%s Proportion [%%]', class_name);
            set(colortitlehandle, 'String', titlestring);
            cb.FontSize = 25;

            % Save the figure
            figure_name = sprintf('Class_Distribution_%s_%s', year_last, class_file_name);

            export_fig([figure_name, '.png']);

            close(2)
            
            % Free memory
            clear distribution_last_matrix
            
            % Progress
            fprintf('Figures of the proportion have been created for %g%% of the classes \n', c / number_classes * 100);
        end
        
    %%% Move the figures %%%
        if ~exist('Class_Distribution_Maps', 'dir')
            mkdir('Class_Distribution_Maps')
        end
        
        movefile('./*.png', 'Class_Distribution_Maps');

        disp('The maps have been moved to Class_Distribution_Maps');
end