% This script determines the evolution of land cover, both for the entire domain and the given subdomains

function [year_list, number_years, subdomains_class_proportion_cell, subdomains_delta_proportion_cell, subdomains_relative_delta_class_cell] = Temporal_Class_Distribution(Resolution_downscaling, coarsening_factor, rows_data, columns_data, number_files, class_data_cell, number_classes, class_names, class_values, RGB_codes, subdomain_indices_cell, subdomain_code_cell, number_subdomains, R, Bounds, Include_water)
    %%% Determine the annual classification data of Europe as well as the subdomains, expressed in surface area %%%
        % Determine which indices fall outside the bounds of the domain
        if strcmp(Bounds, 'Yes')
            MODIS_Coordinates_file = load('MODIS_Coordinates.mat');
            latitude_list = MODIS_Coordinates_file.latitude_list;
            longitude_matrix = MODIS_Coordinates_file.longitude_matrix;

            if strcmp(Resolution_downscaling, 'On')
                latitude_list = latitude_list(1 : coarsening_factor : end);
                longitude_matrix = longitude_matrix(1 : coarsening_factor : end, 1 : coarsening_factor : end);
                
                rows_data = ceil(rows_data / coarsening_factor);
                columns_data = ceil(columns_data / coarsening_factor);
            end

            lat_N = 71.5;
            lat_S = 34.5;
            long_W = -11;
            long_E = 32;

            matrix_OoB = zeros(rows_data, columns_data);

            % Latitude
            Latitude_OoB = latitude_list > lat_N | latitude_list < lat_S;
            matrix_OoB(Latitude_OoB) = 1;

            % Longitude
            Longitude_OoB = longitude_matrix > long_E | longitude_matrix < long_W;
            matrix_OoB(Longitude_OoB) = 1;

            ind_OoB = find(matrix_OoB == 1);
        elseif strcmp(Bounds, 'No')
            ind_OoB = [];
        end
        
        % Free memory
        clear matrix_OoB & Longitude_OoB & longitude_matrix & Latitude_OoB & latitude_list & MODIS_Coordinates_file
        
        % The surface area per pixel
        resolution = R.CellExtentInWorldX;                          % Equal in x and y, in metres
        
        if strcmp(Resolution_downscaling, 'On')
            pixel_area = (coarsening_factor * resolution)^2 / 1e6;      % Area in km2
        elseif strcmp(Resolution_downscaling, 'Off')
            pixel_area = resolution^2 / 1e6;
        end
        
        % The water bodies class is removed, if desired
        if strcmp(Include_water, 'No')
            for c = 1 : number_classes
                class_name = class_names{c};
                
                if strcmp(class_name, 'Water bodies')
                    WB_ind = c;
                end
            end
            
            number_classes = number_classes - 1;
            class_values(WB_ind) = [];
            class_names(WB_ind) = [];
        end
    
        % The annual land cover data of the entire data set
        number_years = length(class_data_cell);

        domain_class_area_matrix = zeros(number_years, number_classes);
        
        for y = 1:number_years            
            % This year's land cover data
            class_data = class_data_cell{y};
            
            class_data(ind_OoB) = [];
            
            for c = 1:number_classes
                % Indices of this class
                class_value = class_values(c);
                class_indices = find(class_data == class_value);
                
                % Surface area occupied by this class
                class_surface_area = pixel_area * length(class_indices);
    
                domain_class_area_matrix(y, c) = class_surface_area;
            end
        end
        
        % The annual surface area of the classes in the subdomains        
        subdomains_class_area_cell = cell(1, number_subdomains);
        
        for n = 1:number_subdomains
            subdomain_indices = subdomain_indices_cell{n};
            
            % The indices outside of the domain are ignored
            subdomain_indices = setdiff(subdomain_indices, ind_OoB);
            
            annual_class_area_matrix = zeros(number_years, number_classes);
            
            for y = 1:number_years
                class_data = class_data_cell{y};
                
                % Land cover and surface area of this subdomain in this year
                subdomain_class_data = class_data(subdomain_indices);
                
                for c = 1:number_classes
                    class_value = class_values(c);
                    class_indices = find(subdomain_class_data == class_value);
                    
                    % Surface area of this class within the subdomain, for this year
                    class_surface_area = pixel_area * length(class_indices);
                    
                    annual_class_area_matrix(y, c) = class_surface_area;
                end
            end
            
            subdomains_class_area_cell{n} = annual_class_area_matrix;
            
            % Progress
            fprintf('The surface area of the classes has been determined for %g%% of the domains \n', n / number_subdomains * 100);
        end
        
        % Free memory
        clear class_data_cell & class_data & subdomain_indices_cell & subdomain_indices & subdomain_class_data & ind_OoB
        
    %%% Difference between the first year and succeeding years %%%    
        % List of years the classification data covers
        file_list = dir('*LC*.tif');
        LC_first = file_list(1).name;
        digit_ind = isstrprop(LC_first, 'digit');
        year_first = LC_first(digit_ind);

        LC_last = file_list(number_files).name;
        digit_ind = isstrprop(LC_last, 'digit');
        year_last = LC_last(digit_ind);

        year_list = str2double(year_first) : str2double(year_last);
        
        year_first = num2str(year_list(1));
        year_last = num2str(year_list(end));
        
        % Entire domain
        domain_relative_delta_class_matrix = zeros(number_years - 1, number_classes);       % Relative change in land cover surface area [%]
        domain_absolute_delta_class_matrix = zeros(number_years - 1, number_classes);       % Absolute change in land cover surface area [km2]
        
        domain_class_proportion_matrix = zeros(number_years, number_classes);               % The proportion of each class [%]
        domain_delta_proportion_matrix = zeros(number_years - 1, number_classes);           % The change in proportion of each class [%]
        
        for c = 1:number_classes
            class_area_TS = domain_class_area_matrix(:, c);
            class_area_first_year = class_area_TS(1);
            
            domain_absolute_delta_class_matrix(:, c) = class_area_TS(2:number_years) - class_area_first_year;                                   % The absolute change in land cover area
            domain_relative_delta_class_matrix(:, c) = (class_area_TS(2:number_years) - class_area_first_year)/class_area_first_year * 100;     % The relative change in land cover area
            
            class_proportion_list = class_area_TS./sum(domain_class_area_matrix, 2) * 100;                          % Proportion of area covered by the class
            domain_class_proportion_matrix(:, c) = class_proportion_list;
            
            delta_proportion_list = class_proportion_list - class_proportion_list(1);
            domain_delta_proportion_matrix(:, c) = delta_proportion_list(2 : number_years);                         % The difference in proportion w.r.t first year
        end
        
        disp('The relative and absolute differences in land cover have been determined for the entire domain');
    
        % Subdomains
        subdomains_relative_delta_class_cell = cell(1, number_subdomains);      % Relative change in land cover surface area [%]
        subdomains_absolute_delta_class_cell = cell(1, number_subdomains);      % Absolute change in land cover surface area [km2]
    
        subdomains_surface_area_class_cell = cell(1, number_subdomains);        % Land cover surface area [km2]
        
        subdomains_class_proportion_cell = cell(1, number_subdomains);          % The proportion of each class [%]
        subdomains_delta_proportion_cell = cell(1, number_subdomains);          % The change in proportion of each class [%]
        
        for n = 1:number_subdomains
            % Surface area information of this subdomain
            subdomains_class_area = subdomains_class_area_cell{n};
            
            relative_delta_class_matrix = zeros(number_years, number_classes);
            absolute_delta_class_matrix = zeros(number_years, number_classes);
            
            surface_area_class_matrix = zeros(number_years, number_classes);
            
            class_proportion_matrix = zeros(number_years, number_classes);
            delta_proportion_matrix = zeros(number_years, number_classes);
            
            for y = 1 : number_years
                first_year_class_area = subdomains_class_area(1, :);
                annual_class_area = subdomains_class_area(y, :);
                                
                for c = 1:number_classes
                    first_year_area = first_year_class_area(c);
                    class_area = annual_class_area(c);
                    
                    % Surface area
                    surface_area_class_matrix(y, c) = class_area;
                    
                    % Relative change
                    relative_delta_class_area = (class_area - first_year_area) / first_year_area * 100;      
                    relative_delta_class_matrix(y, c) = relative_delta_class_area;
                    
                    % Land cover proportion
                    class_proportion = class_area / sum(annual_class_area) * 100;
                    class_proportion_matrix(y, c) = class_proportion;
                    
                    % Absolute change
                    absolute_delta_class_area = class_area - first_year_area;
                    absolute_delta_class_matrix(y, c) = absolute_delta_class_area;
                    
                    % Change in land cover proportion
                    delta_proportion = class_proportion - class_proportion_matrix(1, c);
                    delta_proportion_matrix(y, c) = delta_proportion;
                end
            end
            
            % The delta of the first years is removed as it is meaningless
            relative_delta_class_matrix(1, :) = [];
            absolute_delta_class_matrix(1, :) = [];
            delta_proportion_matrix(1, :) = [];
            
            % Append the results
            subdomains_relative_delta_class_cell{n} = relative_delta_class_matrix;
            subdomains_absolute_delta_class_cell{n} = absolute_delta_class_matrix;
            subdomains_surface_area_class_cell{n} = surface_area_class_matrix;
            subdomains_class_proportion_cell{n} = class_proportion_matrix;
            subdomains_delta_proportion_cell{n} = delta_proportion_matrix;
            
            % Progress
            fprintf('The relative and absolute differences in land cover have been determined for %g%% of the domains \n', n / number_subdomains * 100);
        end
        
        % Save the results in .mat files
        save('Europe_Relative_Delta.mat', 'domain_relative_delta_class_matrix');
        save('Europe_Absolute_Delta.mat', 'domain_absolute_delta_class_matrix');
        save('Subdomains_Relative_Delta.mat', 'subdomains_relative_delta_class_cell');
        save('Subdomains_Absolute_Delta.mat', 'subdomains_absolute_delta_class_cell');
        
        disp('The changes in land cover (relative and absolute) of the subdomains and entire domain have been saved as .mat files.');
        
        save('Europe_Class_Proportions.mat', 'domain_class_proportion_matrix');
        save('Europe_Delta_Proportions.mat', 'domain_delta_proportion_matrix');
        save('Subdomains_Class_Proportions.mat', 'subdomains_class_proportion_cell');
        save('Subdomains_Delta_Proportions.mat', 'subdomains_delta_proportion_cell');
        save('Subdomains_Surface_Area_Classes.mat', 'subdomains_surface_area_class_cell');
        
        disp('The proportions of classes and changes therein of the subdomains and entire domain have been saved as .mat files');
                        
        % Save the results for the entire domain in .xls files
        domain_relative_delta_table = array2table(domain_relative_delta_class_matrix, 'VariableNames', class_names, 'RowNames', string(year_list(2 : end)));
        writetable(domain_relative_delta_table, 'Europe_Relative_Delta.xls');
        domain_absolute_delta_table = array2table(domain_absolute_delta_class_matrix, 'VariableNames', class_names, 'RowNames', string(year_list(2 : end)));
        writetable(domain_absolute_delta_table, 'Europe_Absolute_Delta.xls');
        domain_delta_proportion_table = array2table(domain_delta_proportion_matrix, 'VariableNames', class_names, 'RowNames', string(year_list(2 : end)));
        writetable(domain_delta_proportion_table, 'Europe_Delta_Proportions.xls');
        
        domain_class_proportion_table = array2table(domain_class_proportion_matrix, 'VariableNames', class_names, 'RowNames', string(year_list));
        writetable(domain_class_proportion_table, 'Europe_Class_Proportions.xls');
        
    %%% Theil-Sen estimator, applied on the class proportion time series %%%
        %--% Entire domain %--%
        domain_relative_delta_TS_cell = cell(1, number_classes);
        domain_proportion_TS_cell = cell(1, number_classes);
        
        for c = 1:number_classes
            relative_delta_TS = domain_relative_delta_class_matrix(:, c)';
            domain_relative_delta_TS_cell{c} = relative_delta_TS;
            
            proportion_TS = domain_class_proportion_matrix(:, c)';
            domain_proportion_TS_cell{c} = proportion_TS;
        end
        
        % Relative delta
        [domain_relative_delta_TS_slopes_cell, ~] = Theil_Sen_Estimator_P(domain_relative_delta_TS_cell, 1:number_years - 1, [], 1, number_classes, number_years - 1);
        domain_relative_delta_TS_slopes_list = horzcat(domain_relative_delta_TS_slopes_cell{:});
        
        % Class proportion
        [domain_proportion_TS_slopes_cell, domain_proportion_TS_levels_cell] = Theil_Sen_Estimator_P(domain_proportion_TS_cell, 1:number_years, [], 1, number_classes, number_years);
        domain_proportion_TS_slopes_list = horzcat(domain_proportion_TS_slopes_cell{:});
        domain_proportion_TS_levels_list = horzcat(domain_proportion_TS_levels_cell{:});
        
        disp('The Theil-Sen estimator has been applied to the data of the entire domain');
        
        %--% Subdomains %--%
        subdomains_relative_delta_TS_cell = cell(1, number_subdomains);
        subdomains_relative_delta_TS_slopes_cell = cell(1, number_subdomains);
        subdomains_relative_delta_TS_levels_cell = cell(1, number_subdomains);
        
        subdomains_proportion_TS_cell = cell(1, number_subdomains);
        subdomains_proportion_TS_slopes_cell = cell(1, number_subdomains);
        subdomains_proportion_TS_levels_cell = cell(1, number_subdomains);

        for n = 1:number_subdomains
            class_relative_delta_matrix = subdomains_relative_delta_class_cell{n};
            class_proportion_matrix = subdomains_class_proportion_cell{n};
            
            subdomain_relative_delta_TS_cell = cell(1, number_classes);
            subdomain_proportion_TS_cell = cell(1, number_classes);
            
            for c = 1:number_classes
                class_relative_delta_TS = class_relative_delta_matrix(:, c)';
                subdomain_relative_delta_TS_cell{c} = class_relative_delta_TS;
                
                class_proportion_TS = class_proportion_matrix(:, c)';
                subdomain_proportion_TS_cell{c} = class_proportion_TS;
            end
            
            subdomains_relative_delta_TS_cell{n} = subdomain_relative_delta_TS_cell;
            subdomains_proportion_TS_cell{n} = subdomain_proportion_TS_cell;
            
            % Relative delta
            [Theil_Sen_slopes_cell, Theil_Sen_levels_cell] = Theil_Sen_Estimator_P(subdomain_relative_delta_TS_cell, 1:number_years - 1, [], 1, number_classes, number_years - 1);
            subdomain_relative_delta_TS_slopes_list = horzcat(Theil_Sen_slopes_cell{:});
            subdomain_relative_delta_TS_levels_list = horzcat(Theil_Sen_levels_cell{:});
            
            subdomains_relative_delta_TS_slopes_cell{n} = subdomain_relative_delta_TS_slopes_list;
            subdomains_relative_delta_TS_levels_cell{n} = subdomain_relative_delta_TS_levels_list;
            
            % Proportion
            [Theil_Sen_slopes_cell, Theil_Sen_levels_cell] = Theil_Sen_Estimator_P(subdomain_proportion_TS_cell, 1:number_years, [], 1, number_classes, number_years);
            subdomain_proportion_TS_slopes_list = horzcat(Theil_Sen_slopes_cell{:});
            subdomain_proportion_TS_levels_list = horzcat(Theil_Sen_levels_cell{:});
            
            subdomains_proportion_TS_slopes_cell{n} = subdomain_proportion_TS_slopes_list;
            subdomains_proportion_TS_levels_cell{n} = subdomain_proportion_TS_levels_list;
            
            % Progress
            fprintf('The Theil-Sen estimator has been applied to %g%% of the domain \n', n / number_subdomains * 100);
        end
        
    %%% Mann-Kendall test, applied on the class proportion time series %%%
        %--% Entire domain %--%
        [~, domain_relative_delta_p_cell] = Mann_Kendall_Test_P(domain_relative_delta_TS_cell, [], 1, number_classes);
        domain_relative_delta_p_list = horzcat(domain_relative_delta_p_cell{:});
        
        [~, domain_proportion_p_cell] = Mann_Kendall_Test_P(domain_proportion_TS_cell, [], 1, number_classes);
        domain_proportion_p_list = horzcat(domain_proportion_p_cell{:});
        
        disp('The Mann-Kendall test has been applied to the entire domain');
        
        %--% Subdomains %--%
        subdomains_relative_delta_p_cell = cell(1, number_subdomains);
        subdomains_proportion_p_cell = cell(1, number_subdomains);
        
        for n = 1:number_subdomains
            subdomain_relative_delta_TS_cell = subdomains_relative_delta_TS_cell{n};
            subdomain_proportion_TS_cell = subdomains_proportion_TS_cell{n};
            
            [~, p_cell] = Mann_Kendall_Test_P(subdomain_relative_delta_TS_cell, [], 1, number_classes);
            subdomain_relative_delta_p_list = horzcat(p_cell{:});
            
            subdomains_relative_delta_p_cell{n} = subdomain_relative_delta_p_list;
            
            [~, p_cell] = Mann_Kendall_Test_P(subdomain_proportion_TS_cell, [], 1, number_classes);
            subdomain_proportion_p_list = horzcat(p_cell{:});
            
            subdomains_proportion_p_cell{n} = subdomain_proportion_p_list;
            
            % Progress
            fprintf('The Mann-Kendall test has been applied to %g%% of the domain \n', n / number_subdomains * 100);
        end
        
    %%% Save the land cover information in table format %%%             
        %--% Entire domain %--%
        % Lists containing the required information
        domain_relative_delta_class_list = domain_relative_delta_class_matrix(number_years - 1, :); 
        domain_absolute_delta_class_list = domain_absolute_delta_class_matrix(number_years - 1, :);
        
        domain_class_proportion_first_list = domain_class_proportion_matrix(1, :);
        domain_class_proportion_last_list = domain_class_proportion_matrix(number_years, :);
        
        % Create the tables        
        domain_table_data = [domain_class_proportion_first_list', domain_class_proportion_last_list', domain_relative_delta_class_list', domain_absolute_delta_class_list', domain_relative_delta_TS_slopes_list', domain_relative_delta_p_list', domain_proportion_TS_slopes_list', domain_proportion_p_list'];
        Variable_Names = {sprintf('Proportion %s', year_first), sprintf('Proportion %s', year_last), 'Rel. Change [%]', 'Abs. Change [km2]', 'Rate of Change (rel.)[%/yr]', 'Stat. Sign. (rel.) [-]', 'Rate of Change (abs.) [%/yr]', 'Stat. Sign. (abs.) [%/yr]'};
        
        domain_class_table = array2table(domain_table_data, 'VariableNames', Variable_Names, 'RowNames', class_names);

        format bank         % 2 digits after period
        
        disp('----------------------------------------------')
        disp('Table showing the class properties of the entire domain')
        disp(domain_class_table)
        save('Europe_Class_Table.mat', 'domain_class_table');
        writetable(domain_class_table, 'Europe_Class_Table.xls');
        disp('----------------------------------------------')
    
        %--% Subdomains %--%
        % The full names of the subdomains can be given
        if number_subdomains == 39
            subdomain_name_cell = {'Albania', 'Austria', 'Belgium', 'Bulgaria', 'Bosnia and H.', 'Belarus', 'Switzerland', 'Czech Rep.', 'Germany', 'Denmark', 'Spain', 'Estonia', 'Finland', 'France', 'Great Britain', 'Greece', 'Croatia', 'Hungary', 'Ireland', 'Italy', 'Lithuania', 'Luxembourg', 'Latvia', 'Moldova', 'Macedonia', 'Montenegro', 'Netherlands', 'Norway', 'Poland', 'Portugal', 'Romania', 'Russia', 'Serbia', 'Slovakia', 'Slovenia', 'Sweden', 'Turkey', 'Ukraine', 'Kosovo'};
        else
            subdomain_name_cell = subdomain_code_cell;
        end

        [subdomain_name_cell, subdomain_order] = sort(subdomain_name_cell);
        
        % Matrices containing only the information to be put into the tables
        subdomains_relative_delta_class_matrix = zeros(number_subdomains, number_classes);
        subdomains_absolute_delta_class_matrix = zeros(number_subdomains, number_classes);
        
        subdomains_surface_area_first_matrix = zeros(number_subdomains, number_classes);
        subdomains_surface_area_last_matrix = zeros(number_subdomains, number_classes);
        
        subdomains_proportion_first_matrix = zeros(number_subdomains, number_classes);
        subdomains_proportion_last_matrix = zeros(number_subdomains, number_classes);
        subdomains_delta_proportion_matrix = zeros(number_subdomains, number_classes);
        
        TS_relative_delta_slopes_matrix = zeros(number_subdomains, number_classes);
        MK_relative_delta_p_matrix = zeros(number_subdomains, number_classes);
        
        TS_proportion_slopes_matrix = zeros(number_subdomains, number_classes);
        MK_proportion_p_matrix = zeros(number_subdomains, number_classes);
        
        for n = 1:number_subdomains  
            subdomain_ind = subdomain_order(n);
            
            relative_delta_class_matrix = subdomains_relative_delta_class_cell{subdomain_ind};
            absolute_delta_class_matrix = subdomains_absolute_delta_class_cell{subdomain_ind};
            
            subdomains_relative_delta_class_matrix(n, :) = relative_delta_class_matrix(number_years - 1, :);
            subdomains_absolute_delta_class_matrix(n, :) = absolute_delta_class_matrix(number_years - 1, :);
            
            surface_area_matrix = subdomains_surface_area_class_cell{subdomain_ind};
            
            surface_area_first_list = surface_area_matrix(1, :);
            surface_area_last_list = surface_area_matrix(number_years, :);
            
            subdomains_surface_area_first_matrix(n, :) = round(surface_area_first_list);    % Rounded to avoid unnecessary significant digits
            subdomains_surface_area_last_matrix(n, :) = round(surface_area_last_list);  
            
            class_proportions_matrix = subdomains_class_proportion_cell{subdomain_ind};
            class_proportions_first_year = class_proportions_matrix(1, :);
            class_proportions_last_year = class_proportions_matrix(number_years, :);
            
            subdomains_proportion_first_matrix(n, :) = class_proportions_first_year;
            subdomains_proportion_last_matrix(n, :) = class_proportions_last_year;
            
            subdomains_delta_proportion_matrix(n, :) = class_proportions_last_year - class_proportions_first_year;
            
            TS_relative_delta_slopes_matrix(n, :) = subdomains_relative_delta_TS_slopes_cell{subdomain_ind};
            MK_relative_delta_p_matrix(n, :) = subdomains_relative_delta_p_cell{subdomain_ind};
                        
            TS_proportion_slopes_matrix(n, :) = subdomains_proportion_TS_slopes_cell{subdomain_ind};
            MK_proportion_p_matrix(n, :) = subdomains_proportion_p_cell{subdomain_ind};
        end
    
        % Create the tables
        relative_delta_class_table = table(subdomain_name_cell', 'VariableNames', {'Subdomain'});
        absolute_delta_class_table = table(subdomain_name_cell', 'VariableNames', {'Subdomain'});
        
        class_surface_area_first_table = table(subdomain_name_cell', 'VariableNames', {'Subdomain'});
        class_surface_area_last_table = table(subdomain_name_cell', 'VariableNames', {'Subdomain'});
        
        proportion_class_table_first = table(subdomain_name_cell', 'VariableNames', {'Subdomain'});
        proportion_class_table_last = table(subdomain_name_cell', 'VariableNames', {'Subdomain'});
        delta_proportion_table = table(subdomain_name_cell', 'VariableNames', {'Subdomain'});
        
        TS_relative_delta_slopes_table = table(subdomain_name_cell', 'VariableNames', {'Subdomain'});
        MK_relative_delta_p_table = table(subdomain_name_cell', 'VariableNames', {'Subdomain'});
        
        TS_proportion_slopes_table = table(subdomain_name_cell', 'VariableNames', {'Subdomain'});
        MK_proportion_p_table = table(subdomain_name_cell', 'VariableNames', {'Subdomain'});
        
        for c = 1:number_classes
            relative_delta_table_column = table(subdomains_relative_delta_class_matrix(:, c), 'VariableNames', class_names(c));
            absolute_delta_table_column = table(subdomains_absolute_delta_class_matrix(:, c), 'VariableNames', class_names(c));

            relative_delta_class_table = [relative_delta_class_table, relative_delta_table_column];
            absolute_delta_class_table = [absolute_delta_class_table, absolute_delta_table_column];
            
            class_surface_area_first_column = table(subdomains_surface_area_first_matrix(:, c), 'VariableNames', class_names(c));
            class_surface_area_last_column = table(subdomains_surface_area_last_matrix(:, c), 'VariableNames', class_names(c));
            
            class_surface_area_first_table = [class_surface_area_first_table,  class_surface_area_first_column];
            class_surface_area_last_table = [class_surface_area_last_table,  class_surface_area_last_column];
            
            proportion_first_table_column = table(subdomains_proportion_first_matrix(:, c), 'VariableNames', class_names(c));
            proportion_table_last_column = table(subdomains_proportion_last_matrix(:, c), 'VariableNames', class_names(c));
            delta_proportion_table_column = table(subdomains_delta_proportion_matrix(:, c), 'VariableNames', class_names(c));
            
            proportion_class_table_first = [proportion_class_table_first, proportion_first_table_column];
            proportion_class_table_last = [proportion_class_table_last, proportion_table_last_column];
            delta_proportion_table = [delta_proportion_table, delta_proportion_table_column];
            
            TS_relative_delta_table_column = table(TS_relative_delta_slopes_matrix(:, c), 'VariableNames', class_names(c));
            MK_relative_delta_p_table_column = table(MK_relative_delta_p_matrix(:, c), 'VariableNames', class_names(c));
            
            TS_relative_delta_slopes_table = [TS_relative_delta_slopes_table, TS_relative_delta_table_column];
            MK_relative_delta_p_table = [MK_relative_delta_p_table, MK_relative_delta_p_table_column];
            
            TS_proportion_table_column = table(TS_proportion_slopes_matrix(:, c), 'VariableNames', class_names(c));
            MK_proportion_p_table_column = table(MK_proportion_p_matrix(:, c), 'VariableNames', class_names(c));
            
            TS_proportion_slopes_table = [TS_proportion_slopes_table, TS_proportion_table_column];
            MK_proportion_p_table = [MK_proportion_p_table, MK_proportion_p_table_column];
        end
               
        disp('----------------------------------------------')
        fprintf('The relative percentage difference in land cover between %s and %s for each class and for each subdomain \n', year_first, year_last);
        disp(relative_delta_class_table);
        writetable(relative_delta_class_table, 'Subdomains_Relative_Delta_Table.xls');
        
        disp('----------------------------------------------')
        fprintf('The difference in land cover surface area (km2) between %s and %s for each class and for each subdomain \n', year_first, year_last);
        disp(absolute_delta_class_table);
        writetable(absolute_delta_class_table, 'Subdomains_Absolute_Delta_Table.xls');
        
        format long        % No digits after period
        
        disp('----------------------------------------------')
        fprintf('Surface area for each class and for each subdomain in year %s \n', year_first);
        disp(class_surface_area_first_table);
        writetable(class_surface_area_first_table, sprintf('Subdomains_Surface_Area_%s_Table.xls', year_first));
        
        disp('----------------------------------------------')
        fprintf('Surface area for each class and for each subdomain in year %s \n', year_last);
        disp(class_surface_area_last_table);
        writetable(class_surface_area_last_table, sprintf('Subdomains_Surface_Area_%s_Table.xls', year_last));
        
        format bank         % 2 digits after period
        
        disp('----------------------------------------------')
        fprintf('Class distribution in percent of the year %s \n', year_first);
        disp(proportion_class_table_first)
        writetable(proportion_class_table_first, sprintf('Subdomains_Class_Proportion_%s_Table.xls', year_first));
        
        disp('----------------------------------------------')
        fprintf('Class distribution in percent of the year %s \n', year_last);
        disp(proportion_class_table_last)
        writetable(proportion_class_table_last, sprintf('Subdomains_Class_Proportion_%s_Table.xls', year_last));
        
        disp('----------------------------------------------')
        fprintf('Percentage difference in class distributions between year %s and year %s \n', year_first, year_last);
        disp(delta_proportion_table)
        writetable(delta_proportion_table, sprintf('Subdomains_Delta_Proportion_%s_Table.xls', year_last));
        
        disp('----------------------------------------------')
        
        disp('----------------------------------------------')
        fprintf('Theil-Sen rate of change (rel.) [percent/yr] between year %s and year %s \n', year_first, year_last);
        disp(TS_relative_delta_slopes_table)
        writetable(TS_relative_delta_slopes_table, 'Subdomains_TS_relative_delta_Slopes_Table.xls');
        
        disp('----------------------------------------------')
        
        disp('----------------------------------------------')
        fprintf('The statistical significance (Mann-Kendall p-value) of any trend in class distribution between year %s and year %s \n', year_first, year_last);
        disp(MK_relative_delta_p_table)
        writetable(MK_relative_delta_p_table, 'Subdomains_MK_relative_delta_p_Table.xls');
        
        disp('----------------------------------------------')
         
        fprintf('Theil-Sen rate of change (abs.) [percent/yr] between year %s and year %s \n', year_first, year_last);
        disp(TS_proportion_slopes_table)
        writetable(TS_proportion_slopes_table, 'Subdomains_TS_proportion_Slopes_Table.xls');
        
        disp('----------------------------------------------')
        
        disp('----------------------------------------------')
        fprintf('The statistical significance (Mann-Kendall p-value) of any trend in class distribution between year %s and year %s \n', year_first, year_last);
        disp(MK_proportion_p_table)
        writetable(MK_proportion_p_table, 'Subdomains_MK_proportion_p_Table.xls');
        
        disp('----------------------------------------------')
        
    %%% Plot the temporal evolution of classes, per subdomain %%%
        for n = 1:number_subdomains
            subdomain_code = subdomain_code_cell{n};
            
            class_distribution_matrix = subdomains_class_proportion_cell{n};
            
            % Plot for the individual time series
            for c = 1:number_classes
                class_name = class_names{c};
                class_file_name = strrep(class_name, '/', '-');
                
                class_distribution_list = class_distribution_matrix(:, c);
                Theil_Sen_slope = subdomains_proportion_TS_slopes_cell{n}(c);
                Theil_Sen_level = subdomains_proportion_TS_levels_cell{n}(c);
                
                Theil_Sen_trend = (1:number_years) * Theil_Sen_slope + Theil_Sen_level;
                
                Theil_Sen_relative_slope = subdomains_relative_delta_TS_slopes_cell{n}(c);
                
                figure(1)
                % Set the size and white background color
                set(gcf, 'Units', 'Normalized', 'Position', [0, 0, 1, 1])
                set(gcf, 'color', [1, 1, 1])
                                                
                hold on                
                plot(year_list, class_distribution_list, 'LineWidth', 2.5, 'DisplayName', sprintf('%s class proportion', class_name));
            
                plot(year_list, Theil_Sen_trend, 'LineWidth', 2.5, 'DisplayName', sprintf('Trend: %.3g [%%/yr]', Theil_Sen_relative_slope));
                
                grid on
                legend('show')
                hold off
                
                ylabel('Proportion [%]')
                xlabel('Time [year]')
                
                list_amp = max(class_distribution_list) - min(class_distribution_list);
                
                if list_amp < 1e-6      % To ensure that the plot can be created
                    list_amp = 0.1;
                end
                
                % y tick resolution options
                y_resolution_list = [10, 5, 2, 1, 5e-1, 1e-1, 5e-2, 1e-2, 5e-3, 1e-3];

                % The desired number of y ticks, the closest value that is greater than this number is selected
                number_y_ticks = 4;
                
                number_y_steps = list_amp ./ y_resolution_list;
                
                diff_y_steps = abs(number_y_steps - number_y_ticks);
                y_resolution_ind = diff_y_steps == min(diff_y_steps);
                y_resolution = y_resolution_list(y_resolution_ind);

                y_min = min(class_distribution_list) - 0.5 * list_amp;
                y_min = y_resolution * floor(y_min / y_resolution);
                y_min = max(0, y_min);
                y_max = max(class_distribution_list) + 0.5 * list_amp;
                y_max = y_resolution * ceil(y_max / y_resolution);
                y_max = min(100, y_max);

                y_ticks = y_min : y_resolution : y_max;
                yticks(y_ticks);
                ylim([y_min, y_max])      
                
                x_ticks = year_list(1) : 3 : year_list(end);
                xticks(x_ticks);
                xlim([year_list(1), year_list(end)])
                
                set(gca, 'FontSize', 35);
                set(gca, 'LineWidth', 2.5);
                
                % Save the figure
                figure_name = sprintf('%s_Class_Distr_%s.png', subdomain_code, class_file_name);
                
                export_fig(figure_name);
                
                close(1)
            end
            
            % Plot showing all the time series
            figure(2)
            
            % Set the size and white background color
            set(gcf, 'Units', 'Normalized', 'Position', [0, 0, 1, 1])
            set(gcf, 'color', [1, 1, 1])
                            
            hold on            
            for c = 1:number_classes
                class_name = class_names{c};
                
                class_distribution_list = class_distribution_matrix(:, c);
                plot(year_list, class_distribution_list, 'color', RGB_codes(c, :), 'DisplayName', class_name, 'LineWidth', 2)
            end
            
            xlabel('Time [year]')
            ylabel('Proportion [%]')
            xlim([year_list(1), year_list(end)])

            legend('show', 'location', 'eastoutside')
            grid on
            
            hold off
            
            set(gca, 'FontSize', 25);
            set(gca, 'LineWidth', 2);

            % Save the figure
            figure_name = sprintf('%s_Classes_Distribution.png', subdomain_code);

            export_fig(figure_name);

            close(2)
        end
                
    %%% Plots for the full data set %%%
        % Individual plots
        for c = 1:number_classes
            class_name = class_names{c};
            class_file_name = strrep(class_name, '/', '-');
            
            Theil_Sen_slope = domain_proportion_TS_slopes_list(c);
            Theil_Sen_level = domain_proportion_TS_levels_list(c);
            
            Theil_Sen_trend = (1:number_years) * Theil_Sen_slope + Theil_Sen_level;
            
            Theil_Sen_relative_slope = domain_relative_delta_TS_slopes_list(c);
            
            figure(1)
            
            % Set the size and white background color
            set(gcf, 'Units', 'Normalized', 'Position', [0, 0, 1, 1])
            set(gcf, 'color', [1, 1, 1])
                        
            class_distribution_list = domain_class_proportion_matrix(:, c);
            
            hold on            
            plot(year_list, class_distribution_list, 'LineWidth', 2.5, 'DisplayName', sprintf('%s class proportion', class_name));
            
            plot(year_list, Theil_Sen_trend, 'LineWidth', 2.5, 'DisplayName', sprintf('Trend: %.3g [%%/yr]', Theil_Sen_relative_slope));
            
            grid on
            legend('show')

            hold off
            
            list_amp = max(class_distribution_list) - min(class_distribution_list);
            
            if list_amp < 1e-6      % To ensure that the plot can be created
                list_amp = 0.1;
            end
            
            xlabel('Time [year]')
            ylabel('Proportion [%]')
            
            % y tick resolution options
            y_resolution_list = [10, 5, 2, 1, 5e-1, 1e-1, 5e-2, 1e-2, 5e-3, 1e-3];

            % The desired number of y ticks, the closest value that is greater than this number is selected
            number_y_ticks = 4;

            number_y_steps = list_amp ./ y_resolution_list;

            diff_y_steps = abs(number_y_steps - number_y_ticks);
            y_resolution_ind = diff_y_steps == min(diff_y_steps);
            y_resolution = y_resolution_list(y_resolution_ind);

            y_min = min(class_distribution_list) - 0.5 * list_amp;
            y_min = y_resolution * floor(y_min / y_resolution);
            y_min = max(0, y_min);
            y_max = max(class_distribution_list) + 0.5 * list_amp;
            y_max = y_resolution * ceil(y_max / y_resolution);
            y_max = min(100, y_max);

            y_ticks = y_min : y_resolution : y_max;
            yticks(y_ticks);
            ylim([y_min, y_max])      
            
            x_ticks = year_list(1) : 3 : year_list(end);
            xticks(x_ticks);
            xlim([year_list(1), year_list(end)])

            set(gca, 'FontSize', 35);
            set(gca, 'LineWidth', 2.5);
                
            % Save the figure
            figure_name = sprintf('Europe_Class_Distr_%s', class_file_name);
            
            export_fig([figure_name, '.png']);
            export_fig([figure_name, '.fig']);

            close(1)
        end
        
        % Full plot
        figure(2)
        
        % Set the size and white background color
        set(gcf, 'Units', 'Normalized', 'Position', [0, 0, 1, 1])
        set(gcf, 'color', [1, 1, 1])
                    
        hold on
        for c = 1:number_classes
            class_name = class_names{c};
            
            class_distribution_list = domain_class_proportion_matrix(:, c);
            plot(year_list, class_distribution_list, 'DisplayName', class_name, 'LineWidth', 2);
        end
        
        xlabel('Time [year]')
        ylabel('Proportion [%]')
        xlim([year_list(1), year_list(end)])
        
        legend('show', 'location', 'eastoutside')
        grid on
        
        hold off

        set(gca, 'FontSize', 25);
        set(gca, 'LineWidth', 2);

        % Save the figure
        figure_name = sprintf('Europe_Classes_Distribution');

        export_fig([figure_name, '.png']);
        export_fig([figure_name, '.fig']);

        close(2)
        
    %%% Move results %%%
        % Move the .mat files to their folder
        movefile('Europe*.mat', './Class_Distribution_Data');
        movefile('Subdomains*.mat', './Class_Distribution_Data');
        
        % Move all of the created tables
        movefile('*Subdomains*.xls', './Class_Distribution_Tables');
        movefile('*Europe*.xls', './Class_Distribution_Tables');
        
        % Move the files that were created of the subdomains
        for n = 1 : number_subdomains
            subdomain_code = subdomain_code_cell{n};
            
            if ~exist(subdomain_code, 'dir')
                mkdir(subdomain_code);
            end
            
            movefile(sprintf('%s*.png', subdomain_code), subdomain_code);
        end
        
        % Remaining figures are also moved
        movefile('*.fig', './Class_Distribution_Figures');
        movefile('*.png', './Class_Distribution_Figures');
end