% This script merges the classification data a posteriori

clear all
close all
clc

%%% Inputs %%%
    % Pre-processing inputs, Resolution downscaling should always be 'Off' if the native resolution is desired.
    Resolution_downscaling  = 'Off';            % [On/Off]
    coarsening_factor 		= 10;               % Integer value

    % The partitions of the clusters can also be optimised
    Cluster_Optimisation    = 'Off';             % [On/Off]
    
    % Whether the smoothed land cover should be merged
    Smoothing               = 'Off';             % [On/Off]
    
    % The Corine colormap is prettier, but less distinguishable
    Color_Choice            = 'Colorbrewer';         % [Custom, Corine, Colorbrewer]   
    
    % The source of the input data
    Data_source             = 'MODIS';          % [Copernicus, MODIS], note that not all data types are available for each source
    
    % The desired input data [Yes/No], note that this is only used to determine the expected size
    NDVI_Data               = 'Yes';        % The pre-processed NDVI data
    NDVI_Component_Data     = 'No';        % The decomposed NDVI data
    LAI_Data                = 'No';        % The leaf area index data
    FAPAR_Data              = 'No';        % The fraction of absorbed photosynthetically active radiation data
    Fcover_Data             = 'No';        % The fraction of vegetation cover data
    
    Data_choices = {NDVI_Data, NDVI_Component_Data, LAI_Data, FAPAR_Data, Fcover_Data};
    
%%% Data information %%%
    % Land cover data file list
    LC_file_list = dir('LC*.tif');
    
    % The number of chunks in x and y
    LC_file_last = LC_file_list(end).name;
    LC_file_last = erase(LC_file_last, '.tif');
    LC_file_last_parts = strsplit(LC_file_last, '_');
    LC_chunk_part = LC_file_last_parts{end};
    LC_chunk_part = erase(LC_chunk_part, 'x');
    LC_chunk_parts = strsplit(LC_chunk_part, 'y');
    
    number_chunks_x = str2double(LC_chunk_parts{1});
    number_chunks_y = str2double(LC_chunk_parts{2});

    % Determine the size of the data and divide it into chunks
    [rows_chunks, columns_chunks, rows_data, columns_data, R_CLC, zero_string] = Smart_Clusterer(Cluster_Optimisation, Data_choices, Data_source, number_chunks_x, number_chunks_y);
    
    % The years that the data covers
    LC_file_first = LC_file_list(1).name;
    LC_file_first_parts = strsplit(LC_file_first, '_');
    
    LC_first_year = str2double(LC_file_first_parts{2});
    LC_last_year = str2double(LC_file_last_parts{2});
    
    classified_year_list = LC_first_year : LC_last_year;
    number_classified_years = length(classified_year_list);
    
    % Class information    
    CLC_year_list = [2000, 2006, 2012, 2018];
    
    for c = 1 : length(CLC_year_list)
        CLC_year = CLC_year_list(c);
        
        try
            [~, number_classes, class_names, class_values, RGB_codes] = Corine_Class_Data(Resolution_downscaling, coarsening_factor, CLC_year, 'On', Color_Choice, [], [], [], []);
        catch
            continue
        end
    end
    
%%% The data is merged %%%    
    tic;
    
    % The non-smoothed land cover is merged
    classified_year_list = string(classified_year_list);

    file_format = 'geotiff';
    Plot = 'On';
        
    Data_Merger_P('LC', Data_source, R_CLC, Resolution_downscaling, coarsening_factor, number_chunks_x, number_chunks_y, rows_chunks, columns_chunks, zero_string, file_format, number_classified_years, classified_year_list, Plot, RGB_codes, class_values);

    % The smoothed land cover is merged
    if strcmp(Smoothing, 'On')
        Data_Merger_P('SLC', Data_source, R_CLC, Resolution_downscaling, coarsening_factor, number_chunks_x, number_chunks_y, rows_chunks, columns_chunks, zero_string, file_format, number_classified_years, classified_year_list, Plot, RGB_codes, class_values);
    end

    % The confusion matrices are merged
    Confusion_Matrix_Merger(number_chunks_x, number_chunks_y, class_names, number_classes, zero_string);
    
    t_merging = toc;
    
    fprintf('   Merging the data took %f seconds \n', t_merging);
    
    
    