% This script creates a pretty image of the LC data, including the borders of the domains 
% For the Corine data, it also indicates which data is missing

clear all
close all
clc

%%% Inputs %%%
    % Coarsening of the input land cover data
    Resolution_downscaling = 'Off';  % [On, Off]
    coarsening_factor = 02;
    
    %--% Input data %--%
    % Corine
    Corine_data     = 'Yes';    % [Yes, No]
    CLC_year_list   = [2018];     % Years for which the CLC data is assessed
    
    % Copernicus
    Copernicus_data = 'No';     % [Yes, No]
    
    % Classified land cover data
    DTC_data        = 'Yes';    % [Yes, No]
    Smoothed        = 'No';    % [Yes, No]
    
    % NDVI data
    NDVI_data       = 'No';     % [Yes, No], note that if it is off, land indices outside of Corine cannot be detected
    
    %--% Plot settings %--%
    % Corine or colorbrewer colour map
    Color_choice        = 'Custom';         % [Corine, Custom, Colorbrewer]

    % Outline thickness around the plot
    outline_thickness   = 05;

    % Determine the subdomain borders
    plot_borders        = 'No';            % [Yes, No] Whether or not borders are included in the figure
    border_thickness    = 10;                % Border thickness used to make the lines appear more clearly in the plots
    
%%% Data %%%
    % Corine LC data
    if strcmp(Corine_data, 'Yes')
        Corine_class_data_cell = cell(1, length(CLC_year_list));

        for c = 1:length(CLC_year_list)
            [class_data, number_classes, class_names, class_values, RGB_codes] = Corine_Class_Data(Resolution_downscaling, coarsening_factor, CLC_year_list(c), 'Off', Color_choice, [], [], [], []);

            Corine_class_data_cell{c} = class_data;
        end
        
        [rows_data, columns_data] = size(class_data);
    end
    
    % Copernicus data
    if strcmp(Copernicus_data, 'Yes')
        [Copernicus_class_data, number_classes, class_names, class_values, RGB_codes] = Copernicus_LC_Reader(Resolution_downscaling, coarsening_factor, Color_choice, [], [], [], []);
        
        [rows_data, columns_data] = size(Copernicus_class_data);
    end
    
    % Classified land cover data
    if strcmp(DTC_data, 'Yes')
        [Classified_data_cell, number_classes, class_names, class_values, RGB_codes] = DTC_Class_Data(Resolution_downscaling, coarsening_factor, Smoothed, Color_choice, [], [], [], []);
        
        [rows_data, columns_data] = size(Classified_data_cell{1});
    end
    
    % NDVI data
    if strcmp(NDVI_data, 'Yes')
        NDVI_file_list = dir('*.tiff');

        if ~isempty(NDVI_file_list)
            NDVI_IMG_data = geotiffread(NDVI_file_list(1).name);
            NDVI_IMG_data = double(NDVI_IMG_data);

            if strcmp(Resolution_downscaling, 'On')
                NDVI_IMG_data = NDVI_IMG_data(1 : coarsening_factor : end, 1 : coarsening_factor : end);
            end
        end
        
        [rows_data, columns_data] = size(NDVI_IMG_data);
    end
    
%%% Image backgrounds %%%
    % National borders
    if strcmp(plot_borders, 'Yes')
        % Subdomain indices
        if strcmp(DTC_data, 'Yes')
            [LC_data_cell_o, ~, ~, ~, ~] = DTC_Class_Data('Off', coarsening_factor, Smoothed, Color_choice, [], [], [], []);
            [rows_original, columns_original] = size(LC_data_cell_o{1});
        elseif strcmp(Corine_data, 'Yes')
            [Corine_IMG_data_o, ~, ~, ~, ~] = Corine_Class_Data('Off', coarsening_factor, CLC_year_list(1), 'Off', Color_choice, [], [], [], []);
            [rows_original, columns_original] = size(Corine_IMG_data_o);
        elseif strcmp(NDVI_data, 'Yes')
            NDVI_IMG_data_o = geotiffread(NDVI_file_list(1).name);
            [rows_original, columns_original] = size(NDVI_IMG_data_o);
        end

        [subdomain_indices_cell, ~, number_subdomains] = Subdomain_Indices_Reader_P(Resolution_downscaling, coarsening_factor, rows_original, columns_original);
        
        border_cell = cell(1, number_subdomains);

        % parallel loop
        for n = 1:number_subdomains
            subdomain_indices = subdomain_indices_cell{n};
            
            % Ensure horizontal shape
            subdomain_indices = subdomain_indices(:)';
            subdomain_indices_cell{n} = subdomain_indices;

            matrix = zeros(rows_data, columns_data);

            matrix(subdomain_indices) = 1;

            B = bwboundaries(matrix, 8, 'holes');

            border_indices = cell(1, length(B));

            for k = 1:length(B)
                % Single-sided border width
                border_width = round((border_thickness - 1) / 2);

                boundary = B{k};

                r = boundary(:, 1)';
                c = boundary(:, 2)';

                for w = 1:border_width
                    c_e = min(columns_data, c + w);
                    c_w = max(1, c - w);

                    r_n = max(1, r - w);
                    r_s = min(rows_data, r + w);

                    r = [r, r_n, r_s, r, r];
                    c = [c, c, c, c_e, c_w];
                end

                ind = sub2ind([rows_data, columns_data], r, c);

                border_indices{k} = ind;
            end
            
            border_cell{n} = horzcat(border_indices{:});
        end
        
        % Merge the results into one
        European_borders_list = horzcat(border_cell{:});
        European_borders_list = unique(European_borders_list);

        Land_indices = horzcat(subdomain_indices_cell{:});
        
        % Sea indices, are the opposite of land indices
        Sea_indices = 1 : rows_data * columns_data;
        Sea_indices = setdiff(Sea_indices, Land_indices);
    else
        European_borders_list = [];
        Land_indices = [];
        Sea_indices = [];
    end

    % Determine which indices are missing in Corine, yet are given by the NDVI data
    if strcmp(NDVI_data, 'Yes')
        if max(max(NDVI_IMG_data)) < 256                % Copernicus data
            Data_source = 'Copernicus';

            NDVI_land_ind = find(NDVI_IMG_data ~= 254);
        else                                        % MODIS data
            Data_source = 'MODIS';

            NDVI_land_ind = find(NDVI_IMG_data ~= -3000);   
        end
        
        if strcmp(Corine_data, 'Yes')
            Missing_ind = find(class_data == 999);
            Missing_CLC_ind = intersect(NDVI_land_ind, Missing_ind);
        else
            Missing_CLC_ind = [];
        end
    else
        Missing_CLC_ind = [];
    end
    
    % Determine the indices that are out of bounds, if MODIS data is used
    if strcmp(NDVI_data, 'Yes')
        if strcmp(Data_source, 'MODIS')
            MODIS_Coordinates_file = load('MODIS_Coordinates.mat');
            latitude_list = MODIS_Coordinates_file.latitude_list;
            longitude_matrix = MODIS_Coordinates_file.longitude_matrix;

            if strcmp(Resolution_downscaling, 'On')
                latitude_list = latitude_list(1 : coarsening_factor : end);
                longitude_matrix = longitude_matrix(1 : coarsening_factor : end, 1 : coarsening_factor : end);
            end

            lat_N = 71.5;
            lat_S = 34.5;
            long_W = -11;
            long_E = 32;

            matrix_OoB = zeros(rows_data, columns_data);

            % Latitude
            Latitude_OoB = find(latitude_list > lat_N | latitude_list < lat_S);
            matrix_OoB(Latitude_OoB) = 1;

            % Longitude
            Longitude_OoB = find(longitude_matrix > long_E | longitude_matrix < long_W);
            matrix_OoB(Longitude_OoB) = 1;

            ind_OoB = find(matrix_OoB == 1);
        end
    else
        ind_OoB = [];
    end
    
    % Light grey background for data that is outside the bounds
    OoB_background = 0.75 * ones(rows_data + 2*outline_thickness, columns_data + 2*outline_thickness, 3);
    OoB_alpha = zeros(rows_data, columns_data);
    OoB_alpha(ind_OoB) = 1;
    OoB_alpha(European_borders_list) = 0;
    OoB_alpha = padarray(OoB_alpha, [outline_thickness, outline_thickness], 0);
    
    % Dark grey background for NDVI land data that is missing from Corine
    missing_CLC_background = 0.35 * ones(rows_data + 2*outline_thickness, columns_data + 2*outline_thickness, 3);
    missing_alpha = zeros(rows_data, columns_data);
    missing_alpha(Missing_CLC_ind) = 1;
    missing_alpha = padarray(missing_alpha, [outline_thickness, outline_thickness], 0);
    
    % White background for the sea indices
    sea_background = ones(rows_data + 2*outline_thickness, columns_data + 2*outline_thickness, 3);
    sea_alpha = zeros(rows_data, columns_data);
    sea_alpha(Sea_indices) = 1;
    sea_alpha = padarray(sea_alpha, [outline_thickness, outline_thickness], 0);
    
    % Black for the borders
    border_background = zeros(rows_data + 2*outline_thickness, columns_data + 2*outline_thickness, 3);
    border_alpha = zeros(rows_data, columns_data);
    border_alpha(European_borders_list) = 1;
    border_alpha = padarray(border_alpha, [outline_thickness, outline_thickness], 0);
    
%%% Corine images %%%
    if strcmp(Corine_data, 'Yes')
        % The data is invisible where it is missing or borders need to appear
        data_alpha = ones(rows_data, columns_data);
        data_alpha(Missing_CLC_ind) = 0;
        data_alpha(ind_OoB) = 0;
        data_alpha(Sea_indices) = 0;
        data_alpha(European_borders_list) = 0;
        data_alpha = padarray(data_alpha, [outline_thickness, outline_thickness], 1);

        for c = 1:length(CLC_year_list)
            class_data = Corine_class_data_cell{c};
            CLC_year = CLC_year_list(c);

            % To make the plot easier to read, specify colours for each class
            class_image = zeros(rows_data + 2*outline_thickness, columns_data + 2*outline_thickness, 3);

            % RGB colour are given to the image
            for m = 1:3
                M = zeros(rows_data, columns_data);
                lst = unique(class_data);
                for i = 1:length(lst)
                    lst_value = lst(i);
                    ind = find(class_values == lst_value);

                    if ~isempty(ind)
                        M(class_data == lst(i)) = RGB_codes(ind, m);
                    end
                end

                % Add black borders around the image
                M = padarray(M, [outline_thickness, outline_thickness], 0);

                class_image(:, :, m) = M;
            end    

            figure(1)        

            % Set the size and white background color
            set(gcf, 'Units', 'Normalized', 'Position', [0 0 1.0 1.0]);
            set(gcf, 'color', [1, 1, 1]);

            % Backgrounds
            hold on
            image(sea_background, 'AlphaData', sea_alpha);
            image(OoB_background, 'AlphaData', OoB_alpha);
            image(missing_CLC_background, 'AlphaData', missing_alpha);
            image(border_background, 'AlphaData', border_alpha);
            set(gca, 'YDir', 'reverse');
            axis off;

            h = imshow(class_image);
            set(h, 'AlphaData', data_alpha);

            CLC_figure_name = sprintf('CLC_%s_Pretty.png', num2str(CLC_year));

            export_fig(CLC_figure_name)

            close(1)
        end
    end
    
%%% Copernicus land cover data %%%
    if strcmp(Copernicus_data, 'Yes')
        % The data is invisible where it is missing or borders need to appear
        data_alpha = ones(rows_data, columns_data);
        data_alpha(European_borders_list) = 0;
        data_alpha(ind_OoB) = 0;
        data_alpha = padarray(data_alpha, [outline_thickness, outline_thickness], 1);

        % To make the plot easier to read, specify colours for each class
        class_image = zeros(rows_data + 2*outline_thickness, columns_data + 2*outline_thickness, 3);

        % RGB colour are given to the image
        for m = 1:3
            M = zeros(rows_data, columns_data);
            lst = unique(Copernicus_class_data);
            for i = 1:length(lst)
                lst_value = lst(i);
                ind = class_values == lst_value;

                M(Copernicus_class_data == lst(i)) = RGB_codes(ind, m);
            end

            % Add black borders around the image
            M = padarray(M, [outline_thickness, outline_thickness], 0);

            class_image(:, :, m) = M;
        end    

        figure(1)        

        % Backgrounds
        hold on
        image(border_background, 'AlphaData', border_alpha);
        image(OoB_background, 'AlphaData' , OoB_alpha);
        set(gca, 'YDir', 'reverse');
        axis off;

        h = imshow(class_image);
        set(h, 'AlphaData', data_alpha);

        % Set the size and white background color
        set(gcf, 'Units', 'Normalized', 'Position', [0 0 1.0 1.0]);
        set(gcf, 'color', [1, 1, 1]);

        Copernicus_figure_name = 'Copernicus_2015_Pretty.png';
        export_fig(Copernicus_figure_name)

        close(1)
    end
    
 %%% Classified land cover images %%%
    if strcmp(DTC_data, 'Yes')
        % The years the land cover data covers
        if strcmp(Smoothed, 'Yes')
            LC_file_list = dir('SLC*.tif');
        else
            LC_file_list = dir('LC*.tif');
        end

        number_years = length(LC_file_list);

        year_list = cell(1, number_years);

        for y = 1:number_years
            LC_file_name = LC_file_list(y).name;

            year_ind = isstrprop(LC_file_name, 'digit');
            year = LC_file_name(year_ind);

            year_list{y} = year;
        end

        % The data is invisible where it is missing or borders need to appear
        data_alpha = ones(rows_data, columns_data);
        data_alpha(European_borders_list) = 0;
        data_alpha(ind_OoB) = 0;
        data_alpha = padarray(data_alpha, [outline_thickness, outline_thickness], 1);

        for y = 1:length(year_list)
            class_data = Classified_data_cell{y};

            year = year_list{y};

            % To make the plot easier to read, specify colours for each class
            class_image = zeros(rows_data + 2*outline_thickness, columns_data + 2*outline_thickness, 3);

            % RGB colour are given to the image
            for m = 1:3
                M = zeros(rows_data, columns_data);
                lst = unique(class_data);
                for i = 1:length(lst)
                    lst_value = lst(i);
                    
                    if lst_value ~= 999
                        ind = class_values == lst_value;

                        M(class_data == lst(i)) = RGB_codes(ind, m);
                    else    % Missing data is given the white colour
                        M(class_data == lst_value) = 1;
                    end
                end

                % Add black borders around the image
                M = padarray(M, [outline_thickness, outline_thickness], 0);

                class_image(:, :, m) = M;
            end    

            figure(2)        

            % Backgrounds
            hold on
            image(border_background, 'AlphaData', border_alpha);
            image(OoB_background, 'AlphaData' , OoB_alpha);
            set(gca, 'YDir', 'reverse');
            axis off;

            h = imshow(class_image);
            set(h, 'AlphaData', data_alpha);

            % Set the size and white background color
            set(gcf, 'Units', 'Normalized', 'Position', [0 0 1.0 1.0]);
            set(gcf, 'color', [1, 1, 1]);

            if strcmp(Smoothed, 'Yes')
                Classified_figure_name = sprintf('SLC_%s_Pretty.png', year);
            elseif strcmp(Smoothed, 'No')
                Classified_figure_name = sprintf('LC_%s_Pretty.png', year);
            end

            export_fig(Classified_figure_name)

            close(2)
        end
    end
    
%%% NDVI image %%%
    if strcmp(NDVI_data, 'Yes')
        cmap_brown_blue = cbrewer('div', 'BrBG', 2000);
        cmap_brown = cmap_brown_blue(1:1000, :);
        cmap_green = cbrewer('seq', 'Greens', 1000);

        cmap_greenbrown = [cmap_brown; [1, 1, 1]; cmap_green];

        if strcmp(Data_source, 'Copernicus')
            NDVI_IMG_data = NDVI_IMG_data / 250 - 0.08;
        elseif strcmp(Data_source, 'MODIS')
            NDVI_IMG_data = NDVI_IMG_data / 1e4;
        end

        % Add black borders around the image
        NDVI_IMG_data = padarray(NDVI_IMG_data, [outline_thickness, outline_thickness], NaN);

        data_alpha = ones(rows_data, columns_data);
        data_alpha(ind_OoB) = 0;
        data_alpha(Sea_indices) = 0;
        data_alpha(European_borders_list) = 0;
        data_alpha = padarray(data_alpha, [outline_thickness, outline_thickness], 1);

        figure(3)    

        % Backgrounds
        hold on
        image(sea_background, 'AlphaData', sea_alpha);
        image(border_background, 'AlphaData', border_alpha);
        image(OoB_background, 'AlphaData', OoB_alpha);
        set(gca, 'YDir', 'reverse');
        axis off;

        h = imshow(NDVI_IMG_data);
        set(h, 'AlphaData', data_alpha);

        set(gca, 'Units', 'Normalized', 'Position', [0.0, 0.05, 0.9, 0.85])

        shading interp
        caxis manual

        caxis([0, 1])

        colormap(cmap_greenbrown);

        cb = colorbar;
        colortitlehandle = get(cb, 'Title');
        titlestring = 'NDVI [-]';
        set(colortitlehandle, 'String', titlestring);
        cb.Title.FontSize = 30;
        cb.FontSize = 25;
        cb.Ruler.Exponent = 0;

        % Set the size and white background color
        set(gcf, 'Units', 'Normalized', 'Position', [0 0 1.0 1.0])
        set(gcf, 'color', [1, 1, 1])
        
        % Save the figure
        NDVI_file_name = NDVI_file_list(1).name;
        NDVI_file_name = NDVI_file_name(1 : end - 5);
        NDVI_file_name = [NDVI_file_name, '.png'];
        export_fig(NDVI_file_name);

        close(3)
    end

    