% This script creates a legend based on the given grouped classes

clear all
close all
clc

%%% Inputs %%% 
    % The type of colormap used
    Color_Choice            = 'Custom'; % [Custom, Corine]
    
    % If aggregated classes were used, only the first colour of each group is used in the legend
    Aggregated_Colors       = 'No';    % [Yes, No]

    % Note that aspect ratios differ between x and y
    box_width       = 0.03;     % Normalised width/height of the boxes
    box_height      = 0.03;     
    
    distance_boxes_horizontal   = 0.04;     % Normalised distance between the boxes
    distance_boxes_vertical     = 0.04;
    
    distance_boxes_text         = 0.05;     % Distance between the boxes and the legend text
    text_fontsize               = 20;       % Fontsize of the legend text

%%% Colour codes and class values %%%
    class_values_urban = [111, 112, 121, 122, 123, 124, 131, 132, 133, 141, 142];
    class_values_agricultural = [211, 212, 213, 221, 222, 223, 231, 241, 242, 243, 244];
    class_values_forest = [311, 312, 313, 321, 322, 323, 324, 331, 332, 333, 334, 335];
    class_values_wetlands = [411, 412, 421, 422, 423];
    class_values_waterbodies = [511, 512, 521, 522, 523];
    class_values_nodata = 999;

    class_values_total = [class_values_urban, class_values_agricultural, class_values_forest, class_values_wetlands, class_values_waterbodies, class_values_nodata];
        
    % Corine
    if strcmp(Color_Choice, 'Corine')
        RGB_codes_total =   [230, 0, 77; 255, 0, 0; 204, 77, 242; 204, 0, 0; 230, 204, 204; 230, 204, 230; 166, 0, 204; 166, 77, 0; 255, 77, 255; 255, 166, 255; 255, 230, 255;  ...
                             255, 255, 168; 255, 255, 0; 230, 230, 0; 230, 128, 0; 242, 166, 77; 230, 166, 0; 230, 230, 77; 255, 230, 166; 255, 230, 77; 230, 204, 77; 242, 204, 166;
                             128, 255, 0; 0, 166, 0; 77, 255, 0; 204, 242, 77; 166, 255, 128; 166, 230, 77; 166, 242, 0; 230, 230, 230; 204, 204, 204; 204, 255, 204; 0, 0, 0; 166, 230, 204;
                             166, 166, 255; 77, 77, 255; 204, 204, 255; 230, 230, 255; 166, 166, 230;
                             0, 204, 242; 128, 242, 230; 0, 255, 166; 166, 255, 230; 230, 242, 255;
                             255, 255, 255] / 255; 
                         
    % Custom
    elseif strcmp(Color_Choice, 'Custom')
        RGB_codes_total =   [230, 176, 170; 217, 136, 128; 205, 97, 85; 192, 57, 43; 169, 50, 38; 146, 43, 33;  ...         % Reds: Urban fabric, commerical, industrial
                             171, 178, 185; 86, 101, 115; 28, 40, 51; ...                                                   % Blacks: Mining, dumping, constructions
                             212, 239, 223; 125, 206, 160; ...                                                              % Light greens: artificial vegetation
                             253, 235, 208; 252, 243, 207; 247, 220, 111; 241, 196, 15; 183, 149, 11; 154, 125, 10; ...     % Yellows: agriculture
                             248, 196, 113; 243, 156, 18; 230, 126, 34; 235, 152, 78; 175, 96, 26; ...                      % Oranges: pastures and complex agriculture
                             34, 153, 84; 20, 90, 50; ...                                                                   % Dark greens: forests
                             167, 217, 206; 115, 198, 182; 22, 160, 133; 19, 141, 117; 14, 102, 85; ...                     % Pastel greens: shrubs & herbaceous vegetaton
                             98, 101, 103; 144, 148, 151; 189, 195, 199; 215, 219, 221; 242, 243, 244; ...                  % Greys: non-vegetated
                             215, 189, 226; 195, 155, 211; 175, 122, 197; 136, 78, 160; 99, 57, 116; ...                    % Purples: wetlands
                             52, 152, 219; 46, 134, 193; 40, 116, 166; 33, 97, 140; 27, 79, 114; ...                        % Blues: water bodies
                             255, 255, 255] / 255;                                                                          % White: missing
    end
                            
%%% Retrieve the grouped classes %%%
    grouped_classes_list = dir('Grouped_Classes*.xls');
    grouped_classes_file = grouped_classes_list(1).name;

    % Read the file
    grouped_classes_data = readtable(grouped_classes_file);

    % Retrieve the grouped classes
    [number_groups, ~] = size(grouped_classes_data);

    grouped_class_values = cell(1, number_groups);
    class_names = cell(1, number_groups);

    for c = 1:number_groups
        % The name of this group of classes
        group_name = grouped_classes_data{c, 1};
        class_names{c} = group_name{1};

        % The values of the classes within this group
        group_class_values = grouped_classes_data{c, 2 : end};
        group_class_values(isnan(group_class_values)) = [];

        grouped_class_values{c} = group_class_values;
    end
    
%%% Create a legend containing the colours of each of the subclasses %%%
    % Determine the x-location of the class names
    number_classes_list = cellfun(@length, grouped_class_values);
    
    % The distance in x depends on whether aggregated colours are used or not
    if strcmp(Aggregated_Colors, 'No')
        class_names_x = max(number_classes_list) * distance_boxes_horizontal + distance_boxes_text;
    elseif strcmp(Aggregated_Colors, 'Yes')
        class_names_x = distance_boxes_horizontal + distance_boxes_text;
    end

    f = figure(1);

    % Set the size and white background color
    set(gcf, 'Units', 'Normalized', 'Position', [0 0 1 1])
    set(gcf, 'color', [1, 1, 1])

    % Set the axes for the true/predicted labels
    a = axes('Parent', f, 'Units', 'Normalized', 'Position', [0, 0, 1, 1], 'Visible', 'off', 'XLim', [0, 1], 'YLim', [0, 1], 'NextPlot', 'add');
    
    text_length_list = zeros(1, number_groups);
    
    for g = 1 : number_groups
        % This group's name as well as classes
        class_name = class_names{g};
        
        % The missing class is not of interest
        if strcmp(class_name, 'Missing')
            continue
        end
        
        class_group = grouped_class_values{g};
        number_classes = number_classes_list(g);
        
        y = 1 - distance_boxes_vertical * g;
        
        % Place the class name
        t = text(class_names_x, y, class_name, 'FontSize', text_fontsize, 'FontName', 'Ariel');
        text_length_list(g) = t.Extent(3);
                
        for c = 1 : number_classes
            class = class_group(c);
            
            % If aggregated colours are used, only the first colour is given
            if strcmp(Aggregated_Colors, 'Yes') & c > 1
                continue
            end
            
            % For the water bodies class, only the first entry is of interest
            if strcmp(class_name, 'Water bodies') & c > 1
                continue
            end
            
            % Determine the colour for this class
            class_ind = find(class_values_total == class);
            RGB_code = RGB_codes_total(class_ind, :);
            
            x_box = distance_boxes_horizontal * c;
            y_box = y - 0.5 * box_height;
            
            % Place the boxes
            rectangle('Position', [x_box, y_box, box_width, box_height], 'FaceColor', RGB_code, 'EdgeColor', 'k', 'LineWidth', 1);
        end
    end

    % Place lines above and below
    text_length = max(text_length_list);
    
    for g = 1 : number_groups
        class_name = class_names{g};
        
        % The missing class is not of interest
        if strcmp(class_name, 'Missing')
            continue
        end
        
        y = 1 - distance_boxes_vertical * g;

        y_top = y + 0.5 * distance_boxes_vertical;
        line([distance_boxes_horizontal, class_names_x + text_length], [y_top, y_top], 'color', 'k');
        y_bot = y - 0.5 * distance_boxes_vertical;
        line([distance_boxes_horizontal, class_names_x + text_length], [y_bot, y_bot], 'color', 'k');
    end
        
    export_fig('Legend.png');
    export_fig('Legend.fig');
    
    
        