% This script analyses the land cover smoothing s.t. suitable smoothing parameters can be determined

clear all
close all
clc

%%% Inputs %%%
    % The desired resolution
    Resolution_downscaling  = 'Off';             % [On, Off]
    coarsening_factor       = 1;
    
    % The colormap choice
    Color_choice            = 'Colorbrewer';     % [Custom, Corine, Colorbrewer]
    
    % Additionally, the size of the data set that is smoothed at any given time can be reduced by dividing it into chunks in x and y
    number_chunks_x = 16;
    number_chunks_y = 16;
    
%%% Start the parallel pool %%%
    try
        number_cores = 8;
        parpool('local', number_cores);
    catch
        disp('Parallel pool was already running, or could not be started for other reasons.')
    end
    
    % Determine the size of the data and divide it into chunks
    [rows_data, columns_data, number_years, R_LC] = Size_Reader('LC', 'geotiff');
    
    % The years covered by this data
    LC_file_list = dir('LC*.tif');
    LC_first = LC_file_list(1).name;
    LC_last = LC_file_list(end).name;
    
    year_ind = isstrprop(LC_first, 'digit');
    year_first = str2double(LC_first(year_ind));
    year_last = str2double(LC_last(year_ind));
    
    year_list = year_first : year_last;
    
    % Partitions
    rows_chunks = linspace(0, rows_data, number_chunks_y + 1);
    rows_chunks = round(rows_chunks);
    columns_chunks = linspace(0, columns_data, number_chunks_x + 1);
    columns_chunks = round(columns_chunks);
    
    % To identify the chunks, strings are used
    number_digits = ceil(log10(max(number_chunks_x, number_chunks_y) + 1));     % The number of digits is used to append 0 at the start of the numbers, for clarity    
    zero_string = sprintf('%%0%g.f', number_digits);                            % The string used to append these zeroes
        
%%% Land cover smoothing %%%
    for c_x = 1 : number_chunks_x
        % Bounds of this chunk
        column_W = columns_chunks(c_x) + 1;
        column_E = columns_chunks(c_x + 1);
                
        for c_y = 1 : number_chunks_y
            row_N = rows_chunks(c_y) + 1;
            row_S = rows_chunks(c_y + 1);
                        
            % String to identify this chunk, and the data produced with it        
            s_x = num2str(c_x, zero_string);
            s_y = num2str(c_y, zero_string);

            chunk_string = sprintf('x%sy%s', s_x, s_y);
            
            % Retrieve the classified data
            tic;
            [LC_data_chunk_cell, number_classes, class_names, class_values, RGB_codes] = DTC_Class_Data(Resolution_downscaling, coarsening_factor, 'No', Color_choice, row_N, row_S, column_W, column_E);
            t_data = toc;
            
            fprintf('   Reading the data of chunk %s took %g seconds \n', chunk_string, t_data);
            
            % Size of the data
            [rows_chunk, columns_chunk] = size(LC_data_chunk_cell{1});
    
            % Smooth the data
            tic;
            Temporal_Classification_Smoothing_P(LC_data_chunk_cell, R_LC, 'MODIS', rows_chunk, columns_chunk, year_list, number_years, chunk_string);
            t_smooth = toc;
            
            fprintf('   Smoothing the data of chunk %s took %g seconds \n', chunk_string, t_smooth);
        end
    end
    
%%% Merge the smoothed data %%%
    year_list = string(year_list);
    Data_Merger_P('SLC', 'MODIS', R_LC, Resolution_downscaling, coarsening_factor, number_chunks_x, number_chunks_y, rows_chunks, columns_chunks, zero_string, 'geotiff', number_years, year_list, 'Off', RGB_codes, class_values);
        
%%% Retrieve the merged smoothed data %%%
    [SLC_data_cell, ~, ~, ~, ~] = DTC_Class_Data(Resolution_downscaling, coarsening_factor, 'Yes', Color_choice, [], [], [], []);
        
%%% Retrieve the regular land cover data %%%
    [LC_data_cell, ~, ~, ~, ~] = DTC_Class_Data(Resolution_downscaling, coarsening_factor, 'No', Color_choice, [], [], [], []);
    
%%% Retrieve the Corine data %%%
    CLC_year_list = [2000, 2006, 2012, 2018];
    number_CLC_years = length(CLC_year_list);
    CLC_data_cell = cell(1, number_CLC_years);
    
    for c = 1:length(CLC_year_list)
        CLC_year = CLC_year_list(c);
        [class_data, ~, ~, ~, ~] = Corine_Class_Data(Resolution_downscaling, coarsening_factor, CLC_year, 'On', 'Colorbrewer', [], [], [], []);
        CLC_data_cell{c} = class_data;
    end
    
%%% Similarity between subsequent years %%%
    % Corine
    Corine_similarity_list = zeros(1, number_CLC_years - 1);
%     LC_CLC_similarity_list = zeros(1, number_CLC_years - 1);
    SLC_CLC_similarity_list = zeros(1, number_CLC_years - 1);
    
    for c = 1:number_CLC_years - 1
        CLC_data = CLC_data_cell{c};
        CLC_data_next = CLC_data_cell{c + 1};
        
        similarity = CLC_data == CLC_data_next;
        similarity_percentage = length(find(similarity == 1)) / (rows_data * columns_data) * 100;
        Corine_similarity_list(c) = similarity_percentage;
        
        CLC_year = CLC_year_list(c);
        year_diff = abs(year_list - CLC_year);  % As the year 2000 doesn't exist for MODIS data
        year_ind = find(year_diff == min(year_diff));
        CLC_year_next = CLC_year_list(c + 1);
        year_diff = abs(year_list - CLC_year_next); 
        year_ind_next = find(year_diff == min(year_diff));
        
%         LC_data = LC_data_cell{year_ind};
%         LC_data_next = LC_data_cell{year_ind_next};
%         
%         similarity = LC_data == LC_data_next;
%         similarity_percentage = length(find(similarity == 1)) / (rows_data * columns_data) * 100;
%         LC_CLC_similarity_list(c) = similarity_percentage;
        
        SLC_data = SLC_data_cell{year_ind};
        SLC_data_next = SLC_data_cell{year_ind_next};
        
        similarity = SLC_data == SLC_data_next;
        similarity_percentage = length(find(similarity == 1)) / (rows_data * columns_data) * 100;
        SLC_CLC_similarity_list(c) = similarity_percentage;
    end
    
    disp('Similarity between Corine data sets:')
    disp(Corine_similarity_list);
%     disp('Similarity between classified land cover for the same years:')
%     disp(LC_CLC_similarity_list);
    disp('Similarity between smoothed classified land cover for the same years:')
    disp(SLC_CLC_similarity_list);
    disp('----')
    
    % Classified data
%     LC_similarity_list = zeros(1, number_years - 1);
    SLC_similarity_list = zeros(1, number_years - 1);
    
    for y = 1:number_years - 1
%         LC_data = LC_data_cell{y};
%         LC_data_next = LC_data_cell{y + 1};
%         
%         similarity = LC_data == LC_data_next;
%         similarity_percentage = length(find(similarity == 1)) / (rows_data * columns_data) * 100;
%         LC_similarity_list(y) = similarity_percentage;
        
        SLC_data = SLC_data_cell{y};
        SLC_data_next = SLC_data_cell{y + 1};
        
        similarity = SLC_data == SLC_data_next;
        similarity_percentage = length(find(similarity == 1)) / (rows_data * columns_data) * 100;
        SLC_similarity_list(y) = similarity_percentage;
    end

%     disp('Similarity between classified land cover for all years:')
%     disp(LC_similarity_list);
    disp('Similarity between smoothed classified land cover for all years:')
    disp(SLC_similarity_list);
    
    
    
    
    
    