% This script is used to analyse the feature space, in order to diagnose issues with classification

clear all
close all
clc

%%% Required files %%%
    % TYPE              NUMBER              NAME
    % ---------------------------------------------------------------------------------------------------------------
    
    %                     Input data, any combination of the following
    % NDVI data:        number_files        NDVI_YYYYMMDD.tiff
    % NDVI comp. data:  6                   NDVI_Trend*.mat, NDVI_Season*.mat, NDVI_T_UB*.mat, NDVI_T_LB*.mat, NDVI_S_UB*.mat, NDVI_S_LB*.mat
    % LAI data:         number_files        LAI_YYYYMMDD.tiff
    % FAPAR data:       number_files        FAPAR_YYYYMMDD.tiff
    % Fcover data:      number_files        Fcover_YYYYMMDD.tiff
    
%%% Inputs %%%
    % Pre-processing inputs, Resolution downscaling should always be 'Off' if the native resolution is desired.
    Resolution_downscaling  = 'On';            % [On/Off]
    coarsening_factor 		= 10;               % Integer value
    
    % Whether or not textural feature data is created
    Textural_features       = 'Yes';            % [Yes/No], based on the first feature data entry

    % The source of the input data
    Data_source             = 'Landsat';          % [Copernicus, MODIS, Landsat], note that not all data types are available for each source
    
    % The desired input data [Yes/No]
    NDVI_Data               = 'No';        % The pre-processed NDVI data
    NDVI_Component_Data     = 'No';        % The decomposed NDVI data
    LAI_Data                = 'No';        % The leaf area index data
    FAPAR_Data              = 'No';        % The fraction of absorbed photosynthetically active radiation data
    Fcover_Data             = 'No';        % The fraction of vegetation cover data
    
    % Spectral data (Landsat)
    Blue_band               = 'No';        % Blue
    Green_band              = 'Yes';        % Green
    Red_band                = 'No';        % Red
    NIR_band                = 'No';        % Near infrared
    SWIR_band               = 'No';        % Shortwave infrared
    SWIR2_band              = 'No';        % Shortwave infrared (better at night)
    LGTH_band               = 'No';        % Low gain thermal
    HGTH_band               = 'No';        % High gain thermal
    Panchr_band             = 'No';        % Panchromatic band

    Spectral_Data = {Blue_band, Green_band, Red_band, NIR_band, SWIR_band, SWIR2_band, LGTH_band, HGTH_band, Panchr_band};      % All the spectral data choices

    Data_choices = {NDVI_Data, NDVI_Component_Data, LAI_Data, FAPAR_Data, Fcover_Data, Spectral_Data};                          % All data choices
            
%%% Start the parallel pool %%%
    try
        number_cores = 8;
        parpool('local', number_cores, 'IdleTimeout', 500);
    catch
        disp('Parallel pool was already running, or could not be started for other reasons.')
    end
    
%%% Read the data %%%
    tic

    % NDVI time series
    if strcmp(NDVI_Data, 'Yes')
        if strcmp(Data_source, 'Copernicus')
            [NDVI_IMG_cell, rows_data, columns_data, number_files, ~, number_years, year_list] = NDVI_Data_Reader_P(Resolution_downscaling, coarsening_factor, [], [], [], []);
        elseif strcmp(Data_source, 'MODIS')
            [NDVI_IMG_cell, rows_data, columns_data, number_files, ~, number_years, year_list] = MODIS_NDVI_Data_Reader_P(Resolution_downscaling, coarsening_factor, [], [], [], []);
        end

        NDVI_TS_cell = Data_Packer(NDVI_IMG_cell, number_files, columns_data, rows_data, 'Unpack');
        clear NDVI_IMG_cell
    else
        NDVI_TS_cell = {};
    end

    % NDVI trend and seasonal component data
    if strcmp(NDVI_Component_Data, 'Yes')
        if strcmp(Data_source, 'Copernicus')
            Data_prefix = 'NDVI';
        elseif strcmp(Data_source, 'MODIS')
            Data_prefix = 'MOD13Q1';
        end

        [NDVI_Trend_cell, NDVI_Season_cell, rows_data, columns_data, number_files, ~, number_years, year_list] = Decomposition_Data_Reader(Data_prefix, Resolution_downscaling, coarsening_factor, [], [], [], []);
    else
        NDVI_Trend_cell = {};
        NDVI_Season_cell = {};
    end

    % LAI data
    if strcmp(LAI_Data, 'Yes') & strcmp(Data_source, 'Copernicus')
        [LAI_IMG_cell, rows_data, columns_data, number_files, ~, number_years, year_list] = LAI_Data_Reader_P(Resolution_downscaling, coarsening_factor, [], [], [], []);
        LAI_TS_cell = Data_Packer(LAI_IMG_cell, number_files, columns_data, rows_data, 'Unpack');
        clear LAI_IMG_cell
    else
        LAI_TS_cell = {};
    end

    % FAPAR data
    if strcmp(FAPAR_Data, 'Yes') & strcmp(Data_source, 'Copernicus')
        [FAPAR_IMG_cell, rows_data, columns_data, number_files, ~, number_years, year_list] = FAPAR_Data_Reader_P(Resolution_downscaling, coarsening_factor, [], [], [], []);
        FAPAR_TS_cell = Data_Packer(FAPAR_IMG_cell, number_files, columns_data, rows_data, 'Unpack');
        clear FAPAR_IMG_cell
    else
        FAPAR_TS_cell = {};
    end

    % Fcover data
    if strcmp(Fcover_Data, 'Yes') & strcmp(Data_source, 'Copernicus')
        [Fcover_IMG_cell, rows_data, columns_data, number_files, ~, number_years, year_list] = Fcover_Data_Reader_P(Resolution_downscaling, coarsening_factor, [], [], [], []);
        Fcover_TS_cell = Data_Packer(Fcover_IMG_cell, number_files, columns_data, rows_data, 'Unpack');
        clear Fcover_IMG_cell
    else
        Fcover_TS_cell = {};
    end
    
    % Spectral data
    number_bands = length(Spectral_Data);
    Spectral_Fourier_Coefficients_cell = cell(1, number_bands);

    if strcmp(Data_source, 'Landsat')    
        Landsat_bands = {'B1', 'B2', 'B3', 'B4', 'B5', 'B7', 'B6-VCID-1', 'B6-VCID-2', 'B8'};

        % Retrieve each band's Fourier series
        for b = 1 : number_bands
            if strcmp(Spectral_Data{b}, 'Yes')
                Landsat_band = Landsat_bands{b};

                [Landsat_Fourier_Coefficients_cell, Fourier_series_length, Landsat_last_time, rows_data, columns_data, ~, ~, number_years, year_list] = Landsat_Fourier_Coefficients_Retriever(Landsat_band, Resolution_downscaling, coarsening_factor, [], [], [], []);

                Spectral_Fourier_Coefficients_cell{b} = Landsat_Fourier_Coefficients_cell;
            end
        end
    else
        Fourier_series_length = [];
        Landast_last_time = [];
    end

    t_data = toc;

    fprintf('   Reading the data took %g seconds \n', t_data);
    
%%% Analyse the feature space %%%
    tic;
    Classification_Metrics_Analysis_P(Data_choices, Data_source, NDVI_TS_cell, NDVI_Trend_cell, NDVI_Season_cell, LAI_TS_cell, FAPAR_TS_cell, Fcover_TS_cell, Spectral_Fourier_Coefficients_cell, Fourier_series_length, Landsat_last_time, rows_data, columns_data, year_list, number_years, Textural_features);
    t_features = toc;
    
    fprintf('   Analysing the feature space took %g seconds \n', t_features);
    
    
    
    