% This script analyses the metrics used for classification (the feature space)

% The metrics can be computed in parallel
function Classification_Metrics_Analysis_P(Data_choices, Data_source, NDVI_TS_cell, NDVI_Trend_cell, NDVI_Season_cell, LAI_TS_cell, FAPAR_TS_cell, Fcover_TS_cell, Spectral_Fourier_Coefficients_cell, Fourier_series_length, Landsat_last_time, rows_data, columns_data, year_list, number_years, Textural_features)
    %%% The data for which feature data is created %%%
        NDVI_Data = Data_choices{1};
        NDVI_Component_Data = Data_choices{2};
        LAI_Data = Data_choices{3};
        FAPAR_Data = Data_choices{4};
        Fcover_Data = Data_choices{5};
        
%--% Tonal features %--%
        % The number of samples per year
        if strcmp(Data_source, 'Copernicus')
            number_annual_timestamps = 36;      % 10-daily
        elseif strcmp(Data_source, 'MODIS')
            number_annual_timestamps = 23;      % 16-daily
        elseif strcmp(Data_source, 'Landsat')
            number_annual_timestamps = 365;     % Daily
        end
            
        % NDVI
        if strcmp(NDVI_Data, 'Yes')           
            [NDVI_feature_data_years_cell, ~, NDVI_feature_names] = NDVI_metrics_P(NDVI_TS_cell, 1 : number_years, number_years, number_annual_timestamps);
            clear NDVI_TS_cell
        else
            NDVI_feature_data_years_cell = cell(1, number_years);
            NDVI_feature_names = {};
        end
        % NDVI (decomposed)
        if strcmp(NDVI_Component_Data, 'Yes')
            [NDVI_component_feature_data_years_cell, ~, NDVI_component_feature_names] = NDVI_component_metrics_P(NDVI_Trend_cell, NDVI_Season_cell, 1 : number_years, number_years, number_annual_timestamps);     
            clear NDVI_Trend_cell & NDVI_Season_cell
        else
            NDVI_component_feature_data_years_cell = cell(1, number_years);
            NDVI_component_feature_names = {};
        end
        
        % LAI
        if strcmp(LAI_Data, 'Yes')
            [LAI_feature_data_years_cell, ~, LAI_feature_names] = LAI_metrics_P(LAI_TS_cell, 1 : number_years, number_years, number_annual_timestamps);
            clear LAI_TS_cell
        else
            LAI_feature_data_years_cell = cell(1, number_years);
            LAI_feature_names = {};
        end
        % FAPAR
        if strcmp(FAPAR_Data, 'Yes')
            [FAPAR_feature_data_years_cell, ~, FAPAR_feature_names] = FAPAR_metrics_P(FAPAR_TS_cell, 1 : number_years, number_years, number_annual_timestamps);
            clear FAPAR_TS_cell
        else
            FAPAR_feature_data_years_cell = cell(1, number_years);  
            FAPAR_feature_names = {};
        end
        % Fcover
        if strcmp(Fcover_Data, 'Yes')
            [Fcover_feature_data_years_cell, ~, Fcover_feature_names] = Fcover_metrics_P(Fcover_TS_cell, 1 : number_years, number_years, number_annual_timestamps);
            clear Fcover_TS_cell
        else
            Fcover_feature_data_years_cell = cell(1, number_years);        
            Fcover_feature_names = {};
        end
        
        % Spectral data
        if strcmp(Data_source, 'Landsat')
            [Spectral_feature_data_years_cell, ~, spectral_feature_names] = Spectral_metrics_P(Spectral_Fourier_Coefficients_cell, Fourier_series_length, Landsat_last_time, 1 : number_years, number_years, number_annual_timestamps);
            clear Spectral_Fourier_Coefficients_cell
        else
            Spectral_feature_data_years_cell = cell(1, number_years);      
            spectral_feature_names = {};
        end
                
        %--% The tonal features are merged %--%
        tonal_feature_data_years_cell = cell(1, number_years);
        
        for y = 1 : number_years
            % This year's data is concatenated
            tonal_feature_data_years_cell{y} = horzcat(NDVI_feature_data_years_cell{y}, NDVI_component_feature_data_years_cell{y}, LAI_feature_data_years_cell{y}, FAPAR_feature_data_years_cell{y}, Fcover_feature_data_years_cell{y}, Spectral_feature_data_years_cell{y}); 
        end
        
        % Free memory
        clear NDVI_feature_data_years_cell & NDVI_component_feature_data_years_cell & LAI_feature_data_years_cell & FAPAR_feature_data_years_cell & Fcover_feature_data_years_cell & Spectral_feature_data_years_cell        

        %--% Textural features %--%
        if strcmp(Textural_features, 'Yes')
            % Textural features are computed based on the first tonal feature data
            feature_data_IMG_cell = cell(1, number_years);
            
            for y = 1 : number_years
                feature_data_IMG_cell{y} = tonal_feature_data_years_cell{y}{1};
            end

            [textural_feature_data_years_cell, ~, textural_feature_names] = Textural_Features(feature_data_IMG_cell, number_years, rows_data, columns_data);
        else
            textural_feature_data_years_cell = cell(1, number_years);
            textural_feature_names = {};
        end

        %--% The tonal and textural feature data is merged %--%
        feature_names = [NDVI_feature_names, NDVI_component_feature_names, LAI_feature_names, FAPAR_feature_names, Fcover_feature_names, spectral_feature_names, textural_feature_names];
        
        feature_data_years_cell = cell(1, number_years);
        
        for y = 1 : number_years
            feature_data_years_cell{y} = horzcat(tonal_feature_data_years_cell{y}, textural_feature_data_years_cell{y});
        end
        
        number_features = length(feature_data_years_cell{1});
        
        % Free memory
        clear tonal_feature_data_years_cell & textural_feature_data_years_cell
        
    %%% Plots showing the annual feature data %%%
        % The colormap
        cmap = cbrewer('seq', 'YlOrRd', 1000);
        
        % The number of rows/columns of the subplots
        number_rows = min(2, number_years);
        number_columns = ceil(number_years / 2);
    
        for f = 1 : number_features
            % This feature's name
            feature_name = feature_names{f};
            
            % To create consistent bounds
            ub_list = zeros(1, number_years);
            lb_list = zeros(1, number_years);
            
            for y = 1 : number_years
                feature_data_year = feature_data_years_cell{y}{f};
                             
                % Save the bounds
                ub_list(y) = max(max(feature_data_year));
                lb_list(y) = min(min(feature_data_year));
            end
                            
            % Set consistent bounds for all subplots
            ub = max(ub_list);
            lb = min(lb_list);
            
            figure(f)
            sgtitle(feature_name)
                       
            for y = 1 : number_years
                % This year's data
                year = year_list(y);
                feature_data_year = feature_data_years_cell{y}{f};
                
                subplot(number_rows, number_columns, y);
                imshow(feature_data_year)
                title(num2str(year));
 
                colorbar
                caxis manual
                shading interp
                caxis([lb, ub])
                colormap(cmap);
            end
            
            % Set the size and white background color
            set(gcf, 'Units', 'Normalized', 'Position', [0 0 1 1])
            set(gcf, 'color', [1, 1, 1])
            
            % Save the figure
            figure_name = sprintf('%s_Feature_Data.png', feature_name);
            export_fig(figure_name);
            
            close(f);
        end
end




