
% This script compares the Copernicus land cover to the classified and Corine land cover data

clear all
close all
clc

%%% Inpust %%%
    Resolution_downscaling  = 'Off';    % [On, Off]
    coarsening_factor       = 2;
    
    % Whether a subdomain is analysed, or the entire domain
    Subdomain               = 'Yes';    % [Yes, No]
    
    % Whether plots of the land cover are created or not
    Plots                   = 'No';            % [Yes, No]
    Color_Choice            = 'Custom';         % Colormap for said plots
    
    % Classified land cover data
    DTC_data                = 'Yes';    % [Yes, No]
    Smoothed                = 'Yes';    % [Yes, No]
    
    % Corine data
    Corine_data             = 'Yes';    % [Yes, No]
    CLC_year                = 2018;     % [2000, 2006, 2012, 2018]

%%% Retrieve the data %%%
    % Copernicus data
    [Copernicus_data, number_classes, class_names, class_values, RGB_codes] = Copernicus_LC_Reader(Resolution_downscaling, coarsening_factor, Color_Choice, [], [], [], []);
    
    % Classified land cover data
    if strcmp(DTC_data, 'Yes')
        [class_data_cell, ~, ~, ~, ~] = DTC_Class_Data(Resolution_downscaling, coarsening_factor, Smoothed, Color_Choice, [], [], [], []);
        LC_data = class_data_cell{1};
    end
    
    % Corine data
    if strcmp(Corine_data, 'Yes')
        [Corine_LC_data, ~, ~, ~, ~] = Corine_Class_Data(Resolution_downscaling, coarsening_factor, CLC_year, 'Off', Color_Choice, [], [], [], []);
    end
    
%%% Indices of subdomains can be specified here %%%
    if strcmp(Subdomain, 'Yes')
        indices_files = dir('*IND*.mat');
        number_subdomains = length(indices_files);

        indices_subdomains = [];

        for n = 1:number_subdomains
            disp(indices_files(n).name);
            indices_file = load(indices_files(n).name);
            subdomain_indices = indices_file.national_indices;

            indices_subdomains = [indices_subdomains; subdomain_indices];
        end

        Copernicus_data = Copernicus_data(indices_subdomains);
        
        if strcmp(DTC_data, 'Yes')
            LC_data = LC_data(indices_subdomains);
        end
        
        if strcmp(Corine_data, 'Yes')
            Corine_LC_data = Corine_LC_data(indices_subdomains);  
        end
    end
    
%%% Confusion matrices %%%
    % Indices missing from Copernicus are ignored
    missing_ind_Cop = find(Copernicus_data == 999);
        
    if strcmp(Subdomain, 'No')
        % Additionally, indices outside the bounds are also ignored
        MODIS_Coordinates_file = load('MODIS_Coordinates.mat');
        latitude_list = MODIS_Coordinates_file.latitude_list;
        longitude_matrix = MODIS_Coordinates_file.longitude_matrix;

        if strcmp(Resolution_downscaling, 'On')
            latitude_list = latitude_list(1 : coarsening_factor : end);
            longitude_matrix = longitude_matrix(1 : coarsening_factor : end, 1 : coarsening_factor : end);
        end

        lat_N = 71.5;
        lat_S = 34.5;
        long_W = -11;
        long_E = 32;

        if strcmp(DTC_data, 'Yes')
            matrix_OoB = zeros(size(LC_data));
        elseif strcmp(Corine_data, 'Yes')
            matrix_OoB = zeros(size(Corine_LC_data));
        end

        % Latitude
        Latitude_OoB = find(latitude_list > lat_N | latitude_list < lat_S);
        matrix_OoB(Latitude_OoB) = 1;

        % Longitude
        Longitude_OoB = find(longitude_matrix > long_E | longitude_matrix < long_W);
        matrix_OoB(Longitude_OoB) = 1;

        ind_OoB = find(matrix_OoB == 1);
    elseif strcmp(Subdomain, 'Yes')
        ind_OoB = [];
    end

    Copernicus_subset = Copernicus_data;
    Copernicus_subset([missing_ind_Cop; ind_OoB]) = [];

    % Classified w/ Copernicus
    if strcmp(DTC_data, 'Yes')
        LC_subset = LC_data;
        LC_subset([missing_ind_Cop; ind_OoB]) = [];

        figure(1)
        confusion_matrix = confusionchart(Copernicus_subset(:)', LC_subset(:)');
        confusion_matrix_data = confusion_matrix.NormalizedValues;
        close(1)    

        % Not all classes may exist in the domain
        class_values_present = unique([unique(Copernicus_subset), unique(LC_subset)]);
        number_classes_present = length(class_values_present);

        class_names_present = cell(1, number_classes_present);

        for v = 1:number_classes_present
            class_value = class_values_present(v);
            class_ind = class_values == class_value;

            class_name = class_names{class_ind};
            class_names_present{v} = class_name;
        end

        figure(1)
        % Set the size and white background color
        set(gcf, 'Units', 'Normalized', 'Position', [0 0 1 1]);
        set(gcf, 'color', [1, 1, 1]);

        confusion_matrix = confusionchart(confusion_matrix_data, class_names_present);
        sortClasses(confusion_matrix, class_names_present);
        confusion_matrix.RowSummary = 'row-normalized';
        confusion_matrix.ColumnSummary = 'column-normalized'; 
        confusion_matrix.Title = ''; 

        set(gca, 'FontSize', 25);

        CM_name = 'CM_Copernicus_Classified.png';

        export_fig(CM_name);

        save('CM_Copernicus_DTC.mat', 'confusion_matrix');

        close(1)
    end
    
    if strcmp(Corine_data, 'Yes')
        % Corine w/ Copernicus
        % The missing data from Corine and Copernicus is ignored, as well as the indices that are out of bounds
        missing_ind_Corine = find(Corine_LC_data == 999);

        Corine_subset = Corine_LC_data;
        Corine_subset([missing_ind_Cop; missing_ind_Corine; ind_OoB]) = [];

        Copernicus_subset = Copernicus_data;
        Copernicus_subset([missing_ind_Cop; missing_ind_Corine; ind_OoB]) = [];

        figure(2)
        confusion_matrix = confusionchart(Copernicus_subset(:)', Corine_subset(:)');
        confusion_matrix_data = confusion_matrix.NormalizedValues;
        close(2)    

        % Not all classes may exist in the domain
        class_values_present = unique([unique(Copernicus_subset), unique(Corine_subset)]);
        number_classes_present = length(class_values_present);

        class_names_present = cell(1, number_classes_present);

        for v = 1:number_classes_present
            class_value = class_values_present(v);
            class_ind = class_values == class_value;

            class_name = class_names{class_ind};
            class_names_present{v} = class_name;
        end

        figure(2)
        % Set the size and white background color
        set(gcf, 'Units', 'Normalized', 'Position', [0 0 1 1]);
        set(gcf, 'color', [1, 1, 1]);

        confusion_matrix = confusionchart(confusion_matrix_data, class_names_present);
        sortClasses(confusion_matrix, class_names_present);
        confusion_matrix.RowSummary = 'row-normalized';
        confusion_matrix.ColumnSummary = 'column-normalized'; 
        confusion_matrix.Title = ''; 

        set(gca, 'FontSize', 25);

        CM_name = 'CM_Copernicus_Corine.png';

        export_fig(CM_name);

        save('CM_Copernicus_Corine.mat', 'confusion_matrix');

        close(2)
    end
    
%%% Land cover plots %%%
    if strcmp(Plots, 'On')
        class_values = [class_values, 999];

        % Copernicus
        class_image = cat(3, Copernicus_data, Copernicus_data, Copernicus_data);

        % RGB colour are given to the image
        for m = 1:3
            M = class_image(:, :, m);
            lst = unique(M);
            for i = 1:length(lst)
                lst_value = lst(i);
                ind = class_values == lst_value;

                M(M == lst(i)) = RGB_codes(ind, m);
            end

            class_image(:, :, m) = M;
        end    

        figure(1)        
        imshow(class_image);

        % Set the size and white background color
        set(gcf, 'Units', 'Normalized', 'Position', [0 0 0.9 1]);
        set(gcf, 'color', [1, 1, 1]);

        Copernicus_LC_figure_name = 'Copernicus_LC_2015.png';

        export_fig(Copernicus_LC_figure_name);

        close(1)

        % Classified
        if strcmp(DTC_data, 'Yes')
            class_image = cat(3, LC_data, LC_data, LC_data);

            % RGB colour are given to the image
            for m = 1:3
                M = class_image(:, :, m);
                lst = unique(M);
                for i = 1:length(lst)
                    lst_value = lst(i);
                    ind = class_values == lst_value;

                    M(M == lst(i)) = RGB_codes(ind, m);
                end

                class_image(:, :, m) = M;
            end    

            figure(2)        
            imshow(class_image);

            % Set the size and white background color
            set(gcf, 'Units', 'Normalized', 'Position', [0 0 0.9 1]);
            set(gcf, 'color', [1, 1, 1]);

            if strcmp(Smoothed, 'Yes')
                SLC_figure_name = 'SLC_2015.png';
                export_fig(SLC_figure_name);
            elseif strcmp(Smoothed, 'No')
                LC_figure_name = 'LC_2015.png';
                export_fig(LC_figure_name);
            end

            close(2)
        end

        % Corine
        if strcmp(Corine_data, 'Yes')
            class_image = cat(3, Corine_data, Corine_data, Corine_data);

            % RGB colour are given to the image
            for m = 1:3
                M = class_image(:, :, m);
                lst = unique(M);
                for i = 1:number_classes
                    lst_value = lst(i);
                    ind = class_values == lst_value;

                    M(M == lst(i)) = RGB_codes(ind, m);
                end

                class_image(:, :, m) = M;
            end    

            figure(3)        
            imshow(class_image);

            % Set the size and white background color
            set(gcf, 'Units', 'Normalized', 'Position', [0 0 0.9 1]);
            set(gcf, 'color', [1, 1, 1]);

            Corine_LC_figure_name = 'Corine_LC_2015.png';

            export_fig(Corine_LC_figure_name);

            close(3)
        end
    end
    
