% This cript merges the subsets of the Copernicus data

clear all
close all
clc

%%% Retrieve the Copernicus data %%%
    % Divided into longitudinal parts, increasing longitude
    longitudinal_parts = {'W020', 'E000', 'E020'};
    
    % And latitudinal parts, decreasing latitude
    latitudinal_parts = {'N80', 'N60', 'N40'};
    
    Copernicus_data_cell = cell(length(latitudinal_parts), length(longitudinal_parts));
    
    Long_W_matrix = zeros(length(latitudinal_parts), length(longitudinal_parts));
    Long_E_matrix = zeros(length(latitudinal_parts), length(longitudinal_parts));
    Lat_N_matrix = zeros(length(latitudinal_parts), length(longitudinal_parts));
    Lat_S_matrix = zeros(length(latitudinal_parts), length(longitudinal_parts));

    for x = 1 : length(longitudinal_parts)
        longitude = longitudinal_parts{x};
        
        for y = 1 : length(latitudinal_parts)
            latitude = latitudinal_parts{y};
            
            % Retrieve this file's data and coordinates
            Copernicus_file_list = dir(sprintf('%s%s*LC100*.tif', longitude, latitude));
            Copernicus_file = Copernicus_file_list(1).name;
            
            data = geotiffread_2(Copernicus_file);      % Changed due to a bug in MATLAB
            Copernicus_data_cell{y, x} = data;
            
            info = geotiffinfo_2(Copernicus_file);      % Changed due to a bug in MATLAB
            R = info.SpatialRef;
            
            Long_W = R.LongitudeLimits(1);
            Long_W_matrix(x, y) = Long_W;
            Long_E = R.LongitudeLimits(2);
            Long_E_matrix(x, y) = Long_E;
            Lat_N = R.LatitudeLimits(2);
            Lat_N_matrix(x, y) = Lat_N;
            Lat_S = R.LatitudeLimits(1);
            Lat_S_matrix(x, y) = Lat_S;
        end
    end
    
%%% Merge the data %%%
    % First the latitudinal chunks are merged
    Copernicus_data_longitude_cell = cell(1, length(longitudinal_parts));
    
    for x = 1 : length(longitudinal_parts)
        c_start = (x - 1) * length(latitudinal_parts) + 1;
        c_end = x * length(latitudinal_parts);
        
        Copernicus_cells = Copernicus_data_cell(c_start : c_end);
        
        Copernicus_data_longitude = vertcat(Copernicus_cells{:});
        Copernicus_data_longitude_cell{x} = Copernicus_data_longitude;
    end

    % Merge the longitudinal chunks
    Copernicus_data = horzcat(Copernicus_data_longitude_cell{:});
    
    % Show the merged data in a plot
    figure(1)
    imshow(Copernicus_data);
    caxis([0, 255])
    
%%% Save the merged file %%%
    % Edit the spatial reference
    Long_W = min(min(Long_W_matrix));
    Long_E = max(max(Long_E_matrix));
    Lat_N = max(max(Lat_N_matrix));
    Lat_S = min(min(Lat_S_matrix));
    
    R.LongitudeLimits = [Long_W, Long_E];
    R.LatitudeLimits = [Lat_S, Lat_N];
    
    [rows_data, columns_data] = size(Copernicus_data);
    R.RasterSize = [rows_data, columns_data];
    
    % Save the file
    geotiffwrite('Copernicus_LC_2015.tif', Copernicus_data, R);
    
    
    