% This file determines the textural features based on the mean NDVI

% Note: An annual time series is expected to be the input!
% Note: NDVI mean is expected to be the first output!

% All outputs are normalised, with 0 and 1 being the lowest and highest amount found within the domain

% Suitable for parallel computing
function [feature_data, number_features] = Textural_Features(data_matrix, rows_data, columns_data)
    
    %%% Inputs %%%
        % Number of sections into which the data can be split for parallel processing, fastest is for it to be equal to the number of cores
        number_sections = 32;
      
        window_width    = 25;   % The single-sided window width over which the textural features are computed in both x and y
        number_levels   = 10;   % The number of levels between which is distinguished to create the GLCM
                
    %%% The textural features are determined based on the given data matrix %%%
        % In case the data matrix is smaller than the number of sections, the number of sections is reduced
        number_sections = min(number_sections, rows_data);
        
        % The bounds of the data
        data_LB = min(min(data_matrix));
        data_UB = max(max(data_matrix));
    
        % Loop preparation
        rows = linspace(1, rows_data + 1, number_sections + 1);
        rows = round(rows);

        rows_North = zeros(1, number_sections);
        rows_South = zeros(1, number_sections);
        rows_sections = zeros(1, number_sections);

        section_input_data_matrix = cell(1, number_sections);

        for p = 1:number_sections
            % Specify the dimensions of the section
            row_N = rows(p);
            rows_North(p) = row_N;
            row_S = rows(p + 1) -1;
            rows_South(p) = row_S;

            rows_sections(p) = row_S - row_N + 1;

            % The input data for this section
            section_input_data_matrix{p} = data_matrix(row_N : row_S, :, :);

            % Free memory
            data_matrix(row_N : row_S, :, :) = zeros(rows_sections(p), columns_data);
        end 

        ASM_cell = cell(1, number_sections);                % Angular second moment
        Contrast_cell = cell(1, number_sections);
        Correlation_cell = cell(1, number_sections);
        Homogeneity_cell = cell(1, number_sections);

        % Parallel computing loop
        DQ = parallel.pool.DataQueue;
        tick = 0;
        N = number_sections;
        afterEach(DQ, @ProgressUpdate);
        
        parfor p = 1:number_sections
            % This section's data
            data_matrix = section_input_data_matrix{p};
            rows_section = rows_sections(p);
            
            ASM_matrix_section = zeros(rows_section, columns_data);        
            Contrast_matrix_section = zeros(rows_section, columns_data);
            Correlation_matrix_section = zeros(rows_section, columns_data);
            Homogeneity_matrix_section = zeros(rows_section, columns_data);
            
            for r = 1:rows_section
                % The limits of the assessed area
                r_N = max(1, r - window_width);
                r_S = min(rows_section, r + window_width);

                for c = 1:columns_data
                    c_W = max(1, c - window_width);
                    c_E = min(columns_data, c + window_width);

                    % This area's data
                    data_area = data_matrix(r_N : r_S, c_W : c_E);

                    % The gray-level co-occurence matrix
                    GLCM = graycomatrix(data_area, 'GrayLimits', [data_LB, data_UB], 'NumLevels', number_levels);       

                    % The textural properties
                    GLCM_properties = graycoprops(GLCM, {'Correlation', 'Contrast', 'Energy', 'Homogeneity'});
                    contrast = GLCM_properties.Contrast;
                    correlation = GLCM_properties.Correlation;
                    ASM = GLCM_properties.Energy;
                    homogeneity = GLCM_properties.Homogeneity;

                    Contrast_matrix_section(r, c) = contrast;
                    Correlation_matrix_section(r, c) = correlation;
                    ASM_matrix_section(r, c) = ASM;
                    Homogeneity_matrix_section(r, c) = homogeneity;
                end
            end
            
            % Append the data
            ASM_cell{p} = ASM_matrix_section;
            Contrast_cell{p} = Contrast_matrix_section;
            Correlation_cell{p} = Correlation_matrix_section;
            Homogeneity_cell{p} = Homogeneity_matrix_section;
            
            % Progress update
            send(DQ, p);
        end
        
        % The sections are merged
        ASM_matrix = zeros(rows_data, columns_data);        
        Contrast_matrix = zeros(rows_data, columns_data);
        Correlation_matrix = zeros(rows_data, columns_data);
        Homogeneity_matrix = zeros(rows_data, columns_data);
        
        for p = 1:number_sections
            row_N = rows_North(p);
            row_S = rows_South(p);
            
            ASM_matrix(row_N : row_S, :) = ASM_cell{p};
            Contrast_matrix(row_N : row_S, :) = Contrast_cell{p};
            Correlation_matrix(row_N : row_S, :) = Correlation_cell{p};
            Homogeneity_matrix(row_N : row_S, :) = Homogeneity_cell{p};
        end

        % If the number of unique digits > number of levels (i.e. for water bodies), NaN values can appear
        % These values are substituted
        Correlation_matrix(isnan(Correlation_matrix)) = 1;  % A constant area is assumed to be fully correlated
        
        % The feature data is normalised
        con_LB = min(min(Contrast_matrix));
        con_UB = max(max(Contrast_matrix));
        Contrast_matrix = (Contrast_matrix - con_LB)/(con_UB - con_LB);
        
        corr_LB = min(min(Correlation_matrix));
        corr_UB = max(max(Correlation_matrix));
        Correlation_matrix = (Correlation_matrix - corr_LB)/(corr_UB - corr_LB);
        
        ASM_LB = min(min(ASM_matrix));
        ASM_UB = max(max(ASM_matrix));
        ASM_matrix = (ASM_matrix - ASM_LB)/(ASM_UB - ASM_LB);
        
        hom_LB = min(min(Homogeneity_matrix));
        hom_UB = max(max(Homogeneity_matrix));
        Homogeneity_matrix = (Homogeneity_matrix - hom_LB)/(hom_UB - hom_LB);
        
        % Feature data cell
        feature_data = {Contrast_matrix, Correlation_matrix, ASM_matrix, Homogeneity_matrix};

        number_features = length(feature_data);
        
    % Progress function
    function ProgressUpdate(~)
        tick = tick + 1;
        
        progress = tick / N * 100;
			
        fprintf('   Determining textural metrics progress: %g%% \n', round(progress));

        % Print the current time
        time = datestr(now, 'HH:MM:SS');
        fprintf('       t = %s \n', time);
    end
end