% This file determines the growing season as well as the NDVI metrics

% Note: An annual time series is expected to be the input!
% Note: All outputs are normalised, with 1 being the highest possible theoretical amount

% This file is suited for parallel computing
function [feature_data, number_features] = NDVI_metrics_P(NDVI_TS_cell, rows_data, columns_data, number_annual_timestamps)
    %%% Inputs %%%
        number_partitions = 32;     % The number of partitions into which the data is divided. Optimal is if it is equal to the number of cores.

    %%% Divide the data into the partitions %%%
        partition_rows = linspace(0, rows_data, number_partitions + 1);
        partition_rows = round(partition_rows);
        
        NDVI_TS_partitions_cell = cell(1, number_partitions);
        
        for p = 1 : number_partitions
            % Dimensions of this partition
            row_N = partition_rows(p) + 1;
            row_S = partition_rows(p + 1);
            
            % The data is appended
            NDVI_TS_partitions_cell{p} = NDVI_TS_cell(row_N : row_S, :);
            
            % Free memory
            rows_partition = row_S - row_N + 1;
            NDVI_TS_cell(row_N : row_S, :) = cell(rows_partition, columns_data);
        end
    
    %%% Compute the feature data for each partition %%%
        % Create a handle for the growing season related metric function
        GS_function_handle = @(NDVI_TS) GS_metrics_function(NDVI_TS);
    
        feature_data_partitions_cell = cell(1, number_partitions);
    
        % Parallel computing loop
        DQ = parallel.pool.DataQueue;
        tick = 0;
        N = number_partitions;
        afterEach(DQ, @ProgressUpdate);
    
        parfor p = 1 : number_partitions
            % This partition's data
            NDVI_TS_partition = NDVI_TS_partitions_cell{p};
            
            % Mean, max and minimum NDVI
            NDVI_mean_cell = cellfun(@mean, NDVI_TS_partition, 'UniformOutput', false);
            NDVI_mean_matrix = cell2mat(NDVI_mean_cell);
            NDVI_max_cell = cellfun(@max, NDVI_TS_partition, 'UniformOutput', false);
            NDVI_max_matrix = cell2mat(NDVI_max_cell);
            NDVI_min_cell = cellfun(@min, NDVI_TS_partition, 'UniformOutput', false);
            NDVI_min_matrix = cell2mat(NDVI_min_cell);

            % Growing season related metrics
            [t_max_cell, SoS_cell, EoS_cell, Greenup_rate_cell, Senescence_rate_cell, GPP_cell] = cellfun(GS_function_handle, NDVI_TS_partition, 'UniformOutput', false);
            t_max_matrix = cell2mat(t_max_cell);
            SoS_matrix = cell2mat(SoS_cell);
            EoS_matrix = cell2mat(EoS_cell);
            Greenup_rate_matrix = cell2mat(Greenup_rate_cell);
            Senescence_rate_matrix = cell2mat(Senescence_rate_cell);
            GPP_matrix = cell2mat(GPP_cell);
            
            % Extremely rarely, a time series does not meet the previous two if statements and one of the parameters becomes a NaN value
            % In this case, a simple zero value is given
            NDVI_mean_matrix(isnan(NDVI_mean_matrix)) = 0;
            NDVI_max_matrix(isnan(NDVI_max_matrix)) = 0;
            NDVI_min_matrix(isnan(NDVI_min_matrix)) = 0;

            SoS_matrix(isnan(SoS_matrix)) = 0;
            EoS_matrix(isnan(EoS_matrix)) = 0;
            t_max_matrix(isnan(t_max_matrix)) = 0;

            Greenup_rate_matrix(isnan(Greenup_rate_matrix)) = 0;
            Senescence_rate_matrix(isnan(Senescence_rate_matrix)) = 0;
            GPP_matrix(isnan(GPP_matrix)) = 0;

            % The length of season and NDVI amplitude matrices are also computed
            LoS_matrix = EoS_matrix - SoS_matrix;
            NDVI_amp_matrix = NDVI_max_matrix - NDVI_min_matrix;

            % Append the feature data
            feature_data_partition = {NDVI_mean_matrix, NDVI_max_matrix, NDVI_min_matrix, t_max_matrix, NDVI_amp_matrix, SoS_matrix, EoS_matrix, LoS_matrix, Greenup_rate_matrix, Senescence_rate_matrix, GPP_matrix};
            feature_data_partitions_cell{p} = feature_data_partition;
            
            % Progress update
            if ~isempty(NDVI_TS_partition)
                send(DQ, p);
            end
        end
    
    %%% Merge the partitions' data %%%
        number_features = length(feature_data_partitions_cell{1});
        
        feature_data = cell(1, number_features);
        
        for m = 1 : number_features
            feature_data{m} = zeros(rows_data, columns_data);
        end
        
        for p = 1 : number_partitions
            % Dimensions of this partition
            row_N = partition_rows(p) + 1;
            row_S = partition_rows(p + 1);  
            
            % Feature data of this partition
            feature_data_partition = feature_data_partitions_cell{p};
            
            for m = 1 : number_features
                feature_data{m}(row_N : row_S, :) = feature_data_partition{m};
            end
            
            % Free memory
            feature_data_partitions_cell{p} = [];
        end

    % Function to determine growing season related metrics
    function [t_max, SoS, EoS, Greenup_rate, Senescence_rate, GPP] = GS_metrics_function(NDVI_TS)
        % Check if the time series is a constant pixel (sea), in which case the following code is unecessary
        if max(NDVI_TS) - min(NDVI_TS) < 1e-5
            % SoS and EoS are defined as 0 and 1 respectively
            SoS = 0;
            EoS = 1;

            t_max = 0;

            % Greenup and senescence rates are set to 0, GPP is set to 1 to correspond to the SoS and EoS
            Greenup_rate = 0;
            Senescence_rate = 0;
            GPP = 1;

        % Check whether the pixel contains zero values and is not vegetated, which can occur in the north
        % Taking the yearly average is not sensible, as the time series doesn't show seasonal behaviour
        % Note, for classification purposes it is important that different feature values are given than the sea pixels
        elseif max(NDVI_TS) < 0.10 & min(NDVI_TS) < 1e-5
            % The following growing season values are also specified
            SoS = 0.5;
            EoS= 0.5;

            t_max = 1;

            % Greenup and senescence rates are set to 0.5, the GPP is set to 0 to correspond to the SoS and EoS
            Greenup_rate = 0.5;
            Senescence_rate = 0.5;
            GPP = 0;

        % Otherwise the metrics are computed normally
        else
            % Time at which max. NDVI occurs
            t_max = find(NDVI_TS == max(NDVI_TS), 1);

            % NDVI threshold for the growing season
            NDVI_threshold = 0.5*(max(NDVI_TS) + min(NDVI_TS));

            % Growing season
            NDVI_list_next = NaN(1, number_annual_timestamps);                                             % To create equal length lists
            NDVI_list_next(1 : number_annual_timestamps - 1) = NDVI_TS(2: number_annual_timestamps);       % Such that the difference can be checked more easily

            start_candidates = find(NDVI_TS <= NDVI_threshold & NDVI_list_next >= NDVI_threshold);
            end_candidates = find(NDVI_TS >= NDVI_threshold & NDVI_list_next <= NDVI_threshold);

            % The values closest to t_max are selected, before or after
            start_diff_list = t_max - start_candidates;
            end_diff_list = t_max - end_candidates;

            min_start_list = min(start_diff_list(start_diff_list >= 0));
            max_end_list = max(end_diff_list(end_diff_list <= 0));

            if ~isempty(min_start_list) & ~isempty(max_end_list)
                start_ind = start_diff_list == min_start_list;
                end_ind = end_diff_list == max_end_list;

                SoS = start_candidates(start_ind);
                EoS = end_candidates(end_ind);

                % Compute the rates of greenup and senescence, defined as the slopes between SoS and t_max and t_max and EoS respectively
                Greenup_rate = (NDVI_TS(t_max) - NDVI_TS(SoS))/(t_max - SoS);
                Senescence_rate = (NDVI_TS(t_max) - NDVI_TS(EoS))/(EoS - t_max);      

                % Compute the gross primary productivity, defined as the area between SoS and EoS
                GPP = 0;

                for t = SoS : EoS - 1                                       % Computed using a Riemannn sum
                    GPP = GPP + 0.5*(NDVI_TS(t) + NDVI_TS(t + 1)) * 1;      % Width of 1, as the distance between time stamps is always 1
                end
                
            else
                % If no start of season or end of season is found, the mean of the time series is appended
                SoS = 0.5 * number_annual_timestamps;
                EoS = 0.5 * number_annual_timestamps;

                % Greenup rates and senescence rates now cannot be computed, so are given zero values
                Greenup_rate = 0;
                Senescence_rate = 0;

                % Similarly, GPP also cannot be computed
                GPP = 0;
            end

            % Normalise the values
            t_max = t_max / number_annual_timestamps;

            SoS =  SoS / number_annual_timestamps;
            EoS =  EoS / number_annual_timestamps;

            Senescence_rate = abs(Senescence_rate);     % The absolute value is taken, as otherwise the values vary between -1 and 0
                                                        % For the greenup rate no normalisation is necessary for either, as they already take values between 0 and 1
            % The normalised GPP is appended
            GPP = GPP / number_annual_timestamps;    
        end 
    end

    % Progress function
    function ProgressUpdate(~)
        tick = tick + 1;
        
        progress = round(tick / N * 100);

        fprintf('   Determining NDVI time series metrics progress: %g%% \n', progress);

        % Print the current time
        time = datestr(now, 'HH:MM:SS');
        fprintf('       t = %s \n', time);
    end
end