% This file computes the class separability and variability

clear all
close all
clc

%%% Required files %%%
    % TYPE              NUMBER              NAME
    % ---------------------------------------------------------------------------------------------------------------
    %                     Input data, any combination of the following
    % NDVI data:        number_files        NDVI_YYYYMMDD.tiff
    % NDVI comp. data:  6                   NDVI_Trend*.mat, NDVI_Season*.mat, NDVI_T_UB*.mat, NDVI_T_LB*.mat, NDVI_S_UB*.mat, NDVI_S_LB*.mat
    % LAI data:         number_files        LAI_YYYYMMDD.tiff
    % FAPAR data:       number_files        FAPAR_YYYYMMDD.tiff
    % Fcover data:      number_files        Fcover_YYYYMMDD.tiff
    
    %                     Optional: MODIS Coordinates
    % MODIS coord.:     1                   MODS_Coordinates.mat
    
    %                     Land cover data and classes, any combination of the following
    % Corine LC:        1                   CLC_YYYY_REGION.tif
    % Classified LC:    1                   SLC_YYYY.tif or LC_YYYY.tif
    % Copernicus LC:    1                   Copernicus_LC_2015*.tif
    
    %                     Grouped classes
    % Grouped classes:  1                   Grouped_Classes*.xls

%%% Inputs %%%
    % Number of years of data to be analysed, used to reduce the computational time
    number_years = 1;

    % Pre-processing inputs, Resolution downscaling should always be 'Off' if the native resolution is desired.
    Resolution_downscaling  = 'On';            % [On/Off]
    coarsening_factor 		= 02;               % Integer value
    
    % Land cover data
    Land_Cover_Data         = 'CLC';            % [CLC, SLC, LC, COP], the type of land cover data whose classes are read
    CLC_year                = 2000;             % [2000, 2006, 2012, 2018], the year for which the Corine data is read
    
    % Seperate between data outside and inside the Corine bounds
    % Note that Land_Cover_Data must be SLC or LC, and the Corine data is read as well
    OoCB_separation          = 'No';           % [Yes, No]
    
    % Additionally, if MODIS projection data is used, this option can be toggled to not consider any data that is out of bounds
    MODIS_OoB               = 'No';            % [Yes, No]

    % The desired input data [Yes/No]
    Data_source             = 'Landsat';          % [Copernicus, MODIS], note that not all data types are available for each source
    
    NDVI_Data               = 'No';             % The pre-processed NDVI data
    NDVI_Component_Data     = 'No';             % The decomposed NDVI data
    LAI_Data                = 'No';             % The leaf area index data
    FAPAR_Data              = 'No';             % The fraction of absorbed photosynthetically active radiation data
    Fcover_Data             = 'No';             % The fraction of vegetation cover data
    
    % Spectral data (Landsat)
    Blue_band               = 'Yes';        % Blue
    Green_band              = 'No';        % Green
    Red_band                = 'No';        % Red
    NIR_band                = 'No';        % Near infrared
    SWIR_band               = 'No';        % Shortwave infrared
    SWIR2_band              = 'No';        % Shortwave infrared (better at night)
    LGTH_band               = 'No';        % Low gain thermal
    HGTH_band               = 'No';        % High gain thermal
    Panchr_band             = 'No';        % Panchromatic band

    Spectral_Data = {Blue_band, Green_band, Red_band, NIR_band, SWIR_band, SWIR2_band, LGTH_band, HGTH_band, Panchr_band};      % All the spectral data choices

    Data_choices = {NDVI_Data, NDVI_Component_Data, LAI_Data, FAPAR_Data, Fcover_Data, Spectral_Data};                          % All data choices
    
%%% Initialise the parallel pool %%%
    try
        parpool('local', 8);
    catch
        disp('Parallel pool was already running, or could not be started for other reasons')
    end
    
%%% Retrieve the input data %%%
    tic;

    % NDVI time series
    if strcmp(NDVI_Data, 'Yes')
        if strcmp(Data_source, 'Copernicus')
            [NDVI_IMG_cell, rows_data, columns_data, number_files, ~, ~, ~] = NDVI_Data_Reader_P(Resolution_downscaling, coarsening_factor, [], [], [], []);
        elseif strcmp(Data_source, 'MODIS')
            [NDVI_IMG_cell, rows_data, columns_data, number_files, ~, ~, ~] = MODIS_NDVI_Data_Reader_P(Resolution_downscaling, coarsening_factor, [], [], [], []);
        end

        NDVI_TS_cell = Data_Packer(NDVI_IMG_cell, number_files, columns_data, rows_data, 'Unpack');
        clear NDVI_IMG_cell
    else
        NDVI_TS_cell = {};
    end

    % NDVI trend and seasonal component data
    if strcmp(NDVI_Component_Data, 'Yes')
        if strcmp(Data_source, 'Copernicus')
            Data_prefix = 'NDVI';
        elseif strcmp(Data_source, 'MODIS')
            Data_prefix = 'MOD13Q1';
        end

        [NDVI_Trend_cell, NDVI_Season_cell, rows_data, columns_data, number_files, ~, ~, ~] = Decomposition_Data_Reader(Data_prefix, Resolution_downscaling, coarsening_factor, [], [], [], []);
    else
        NDVI_Trend_cell = {};
        NDVI_Season_cell = {};
    end

    % LAI data
    if strcmp(LAI_Data, 'Yes') & strcmp(Data_source, 'Copernicus')
        [LAI_IMG_cell, rows_data, columns_data, number_files, ~, ~, ~] = LAI_Data_Reader_P(Resolution_downscaling, coarsening_factor, [], [], [], []);
        LAI_TS_cell = Data_Packer(LAI_IMG_cell, number_files, columns_data, rows_data, 'Unpack');
        clear LAI_IMG_cell
    else
        LAI_TS_cell = {};
    end

    % FAPAR data
    if strcmp(FAPAR_Data, 'Yes') & strcmp(Data_source, 'Copernicus')
        [FAPAR_IMG_cell, rows_data, columns_data, number_files, ~, ~, ~] = FAPAR_Data_Reader_P(Resolution_downscaling, coarsening_factor, [], [], [], []);
        FAPAR_TS_cell = Data_Packer(FAPAR_IMG_cell, number_files, columns_data, rows_data, 'Unpack');
        clear FAPAR_IMG_cell
    else
        FAPAR_TS_cell = {};
    end

    % Fcover data
    if strcmp(Fcover_Data, 'Yes') & strcmp(Data_source, 'Copernicus')
        [Fcover_IMG_cell, rows_data, columns_data, number_files, ~, ~, ~] = Fcover_Data_Reader_P(Resolution_downscaling, coarsening_factor, [], [], [], []);
        Fcover_TS_cell = Data_Packer(Fcover_IMG_cell, number_files, columns_data, rows_data, 'Unpack');
        clear Fcover_IMG_cell
    else
        Fcover_TS_cell = {};
    end
    
        % Spectral data
    number_bands = length(Spectral_Data);
    Spectral_Fourier_Coefficients_cell = cell(1, number_bands);

    if strcmp(Data_source, 'Landsat')    
        Landsat_bands = {'B1', 'B2', 'B3', 'B4', 'B5', 'B7', 'B6-VCID-1', 'B6-VCID-2', 'B8'};

        % Retrieve each band's Fourier series
        for b = 1 : number_bands
            if strcmp(Spectral_Data{b}, 'Yes')
                Landsat_band = Landsat_bands{b};

                [Landsat_Fourier_Coefficients_cell, Fourier_series_length, Landsat_last_time, rows_data, columns_data, ~, ~, number_years, year_list] = Landsat_Fourier_Coefficients_Retriever(Landsat_band, Resolution_downscaling, coarsening_factor, [], [], [], []);

                Spectral_Fourier_Coefficients_cell{b} = Landsat_Fourier_Coefficients_cell;
            end
        end
    else
        Fourier_series_length = [];
        Landast_last_time = [];
    end
    
    % Land cover data
    if strcmp(OoCB_separation, 'No')
        if strcmp(Land_Cover_Data, 'CLC')
            [LC_data, number_classes, class_names, class_values, ~] = Corine_Class_Data(Resolution_downscaling, coarsening_factor, CLC_year, 'No', 'Custom', [], [], [], []);
        elseif strcmp(Land_Cover_Data, 'LC')
            [class_data_cell, number_classes, class_names, class_values, ~] = DTC_Class_Data(Resolution_downscaling, coarsening_factor, 'No', 'Custom', [], [], [], []);
            LC_data = class_data_cell{1};
        elseif strcmp(Land_Cover_Data, 'SLC')
            [class_data_cell, number_classes, class_names, class_values, ~] = DTC_Class_Data(Resolution_downscaling, coarsening_factor, 'Yes', 'Custom', [], [], [], []);
            LC_data = class_data_cell{1};    
        elseif strcmp(Land_Cover_Data, 'COP')
            [LC_data, number_classes, class_names, class_values, ~] = Copernicus_LC_Reader(Resolution_downscaling, coarsening_factor, 'Custom', [], [], [], []);
        end
        
    % The land cover is separated between the data inside the Corine data set, and the land cover outside of it
    elseif strcmp(OoCB_separation, 'Yes')
        if strcmp(Land_Cover_Data, 'LC')
            [class_data_cell, number_classes, class_names, class_values, ~] = DTC_Class_Data(Resolution_downscaling, coarsening_factor, 'No', 'Custom', [], [], [], []);
            LC_data = class_data_cell{1};
        elseif strcmp(Land_Cover_Data, 'SLC')
            [class_data_cell, number_classes, class_names, class_values, ~] = DTC_Class_Data(Resolution_downscaling, coarsening_factor, 'Yes', 'Custom', [], [], [], []);
            LC_data = class_data_cell{1};   
        elseif strcmp(Land_Cover_Data, 'COP')
            [LC_data, number_classes, class_names, class_values, ~] = Copernicus_LC_Reader(Resolution_downscaling, coarsening_factor, 'Custom', [], [], [], []);
        end        
        
        % Detect missing pixels
        [Corine_data, ~, ~, ~, ~] = Corine_Class_Data(Resolution_downscaling, coarsening_factor, CLC_year, 'No', 'Custom', [], [], [], []);
        missing_pixels = find(Corine_data == 999);

        % The data in these missing areas, is given the current value + 1000
        LC_data(missing_pixels) = LC_data(missing_pixels) + 1000;
        
        % The new class values are appended
        class_values_missing = class_values + 1000;
        class_values = [class_values, class_values_missing];
                
        % To represent these new classes, M is added before their names
        class_names_missing = cell(1, number_classes);
        
        for c = 1 : number_classes
            class_name = ['M_', class_names{c}];
            class_names_missing{c} = class_name;
        end
        
        class_names = [class_names, class_names_missing];
        
        number_classes = length(class_values);
    end
    
    t_data = toc;
    
    fprintf('   Reading the data took %f seconds \n', t_data);
    
%%% Determining the separability and variability %%%
    tic;
    Class_Separability_Variability_P(Data_choices, Data_source, LC_data, number_classes, class_values, class_names, NDVI_TS_cell, NDVI_Trend_cell, NDVI_Season_cell, LAI_TS_cell, FAPAR_TS_cell, Fcover_TS_cell, Spectral_Fourier_Coefficients_cell, Fourier_series_length, Landsat_last_time, number_years, rows_data, columns_data, MODIS_OoB, Resolution_downscaling, coarsening_factor);
    t_class_properties = toc;
    
    fprintf('   Determining the class separability and variability took %f seconds \n', t_class_properties);
    
    
    
    
    
    
    
    
    