% This script computes the class distribution of the given files

clear all
close all
clc

%%% Inputs %%%
    % Resolution of the input data
    Resolution_downscaling  = 'Off';
    coarsening_factor       = 02;
    
    % Type of land cover data
    LC_type                 = 'LC';        % [CLC, LC, SLC];
    
%%% Retrieve the land cover data %%%
    % Corine data
    if strcmp(LC_type, 'CLC')
        % Cycle through the years for which Corine data is available
        year_list = [2000, 2006, 2012, 2018];
        
        class_data_cell = cell(1, length(year_list));
        
        for c = 1 : length(year_list)
            CLC_year = year_list(c);
            
            try
                [class_data, number_classes, class_names, class_values, ~] = Corine_Class_Data(Resolution_downscaling, coarsening_factor, CLC_year, 'Off', 'Corine', [], [], [], []);
                class_data_cell{c} = class_data;
            catch
                continue
            end
        end
        
        % Remove the files that weren't found
        empty_ind = cellfun(@isempty, class_data_cell);
        class_data_cell(empty_ind) = [];
        year_list(empty_ind) = [];
                
    % Classified land cover data
    elseif strcmp(LC_type, 'LC')
        [class_data_cell, number_classes, class_names, class_values, ~] = DTC_Class_Data(Resolution_downscaling, coarsening_factor, 'No', 'Corine', [], [], [], []);
        
        % The years the land cover data covers
        LC_file_list = dir('LC*.tif');
        year_list = zeros(1, length(class_data_cell));
                
        for y = 1 : length(class_data_cell)
            LC_file = LC_file_list(y).name;
            year_ind = isstrprop(LC_file, 'digit');
            year = LC_file(year_ind);
            
            year_list(y) = str2double(year);
        end
        
    % Smoothed classified land cover data
    elseif strcmp(LC_type, 'SLC')
        [class_data_cell, number_classes, class_names, class_values, ~] = DTC_Class_Data(Resolution_downscaling, coarsening_factor, 'Yes', 'Corine', [], [], [], []);
        
        % The years the land cover data covers
        SLC_file_list = dir('SLC*.tif');
        year_list = zeros(1, length(class_data_cell));
                
        for y = 1 : length(class_data_cell)
            SLC_file = SLC_file_list(y).name;
            year_ind = isstrprop(SLC_file, 'digit');
            year = SLC_file(year_ind);
            
            year_list(y) = str2double(year);
        end
    end
    
    number_years = length(year_list);
    
%%% Compute the class distributions %%%
    class_distribution_matrix = zeros(number_classes, number_years);
    class_samples_matrix = zeros(number_classes, number_years);
    
    for y = 1 : number_years
        % This year's land cover data
        class_data = class_data_cell{y};
        
        number_pixels = numel(class_data);
        
        for c = 1 : number_classes
            % The indices belonging to this class
            class_value = class_values(c);
            class_indices = find(class_data == class_value);
            
            % Number of samples belonging to this class
            number_class_indices = length(class_indices);
            class_samples_matrix(c, y) = number_class_indices;
            
            % The percentage of land cover this class occupies
            percentage_class = number_class_indices / number_pixels * 100;
            class_distribution_matrix(c, y) = percentage_class;
        end
    end
    
%%% Save the class distribution in a table %%%
    class_distribution_table_name = sprintf('%s_Class_Distribution_%s-%s.xls', LC_type, num2str(min(year_list)), num2str(max(year_list)));

    % Delete it if it already exists
    if exist(class_distribution_table_name, 'file')
        delete(class_distribution_table_name);
    end
    
    % Two-digit after the period format
    format bank
    
    % Create the table containing the class distribution and number of samples
    class_distribution_table = table(class_names(:), 'VariableNames', {'Class [-]'});    
    class_samples_table = table(class_names(:), 'VariableNames', {'Class [-]'});
    
    for y = 1 : number_years
        % This year's class distribution and samples
        year = num2str(year_list(y));
        class_distribution_data = class_distribution_matrix(:, y);
        class_samples_data = class_samples_matrix(:, y);
        
        % The distributon
        column_name = sprintf('Class. distr. %s [%%]', year);
        
        table_column = table(class_distribution_data, 'VariableNames', {column_name});
        class_distribution_table = [class_distribution_table, table_column];
        
        % The samples
        column_name = sprintf('Class. samples %s [-]', year);
        
        table_column = table(class_samples_data, 'VariableNames', {column_name});
        class_samples_table = [class_samples_table, table_column];
    end
    
    % Save the table
    writetable(class_distribution_table, class_distribution_table_name, 'FileType', 'spreadsheet');
    cell_location = sprintf('A%s', num2str(number_classes + 3));
    writetable(class_samples_table, class_distribution_table_name, 'FileType', 'spreadsheet', 'Range', cell_location);

    
    