clear all
close all

%% Load variables
% Load variables from fits
load('variablesHorizontal.mat')
load('variablesVertical.mat')

% Define field vectors
B_h = [Bx_h, Bz_h];
B_v = [Bx_v, Bz_v];

% Make hyperfine values row vectors in MHz
hypdeltaf_v = hypdeltaf_v.*1e3;
hypdeltaf_h = hypdeltaf_h.*1e3;
hypdeltafneg_h = hypdeltafneg_h*10^3;
hypdeltafneg_v = hypdeltafneg_v*10^3;
hypdeltafpos_h = hypdeltafpos_h*10^3;
hypdeltafpos_v = hypdeltafpos_v*10^3;

%% Plot data without any fit

figure;
vd = polarplot((angle_v -90)*pi/180, hypdeltaf_v, 'o');
vd.MarkerFaceColor = '#0072BD';
vd.Color = '#0072BD';
vd.MarkerSize = 6;
hold on
hd = polarplot((angle_h -90)*pi/180, hypdeltaf_h, 'o');
hd.MarkerFaceColor = '#EDB120';
hd.Color = '#EDB120';
hd.MarkerSize = 6;

%% From the hyperfine and g values try to fit the tip field needed to account for the tilt

gx = 1.653;
gy = 1.917;
gz = 1.989;
Ax = 68;
Ay = 18;
Az = 19;

phi = 14*pi/180; %(rad) angle between Bpar and x
T = 1;%K temperature
Hk = 100; % = Ps/2K with Ps magnetization saturation and K anisotropy

% coefs = gamma,delta,Ps 
fit_start = [60*pi/180, 20*pi/180, 0.1];
fit_up = [pi, pi, 1];
fit_low = [-pi, -pi, 0.1];

% Fit vertical data
fun_vert0 = @(gamma,delta,Ps,Bz_vec,Bpar_vec) hyperfine_Boltzmann(gx,gy,gz,Ax,Ay,Az,Bz_vec,Bpar_vec,phi,gamma,delta,Hk,Ps,T);
fun_vert = @(coefs, B) fun_vert0(coefs(1),coefs(2),coefs(3),B(2,:),B(1,:));

[fit_v, ~, res_v, ~, ~, ~, jac_v] = ...
    lsqcurvefit(fun_vert, fit_start, B_v', hypdeltaf_v, fit_low, fit_up);
% 95% confidence intervals for fit coefficients
conf_v = nlparci(fit_v,res_v,'jacobian',jac_v);


% Fit horizontal data
fun_hor0 = @(gamma,delta,Ps,Bz_vec,Bpar_vec) hyperfine_Boltzmann(gx,gy,gz,Ax,Ay,Az,Bz_vec,Bpar_vec,pi/2-phi,gamma,delta,Hk,Ps,T);
fun_hor = @(coefs, B) fun_hor0(coefs(1),coefs(2),coefs(3),B(2,:),B(1,:));

[fit_h, ~, res_h, ~, ~, ~, jac_h] = ...
    lsqcurvefit(fun_hor, fit_start, B_h', hypdeltaf_h, fit_low, fit_up);
% 95% confidence intervals for fit coefficients
conf_h = nlparci(fit_h,res_h,'jacobian',jac_h);

%% To have the fits look smooth interpolate the data

angle_fit = -180:1:180;
Bz_v_fit = interp1([-180; angle_v],[Bz_v(end); Bz_v],angle_fit); %repeat last element at the beginning (180=-180) to get full circle
Bx_v_fit = interp1([-180; angle_v],[Bx_v(end); Bx_v],angle_fit);
Bz_h_fit = interp1([-180; angle_h],[Bz_h(end); Bz_h],angle_fit);
Bx_h_fit = interp1([-180; angle_h],[Bx_h(end); Bx_h],angle_fit);

% Define field vectors
B_h_fit = [Bx_h_fit', Bz_h_fit'];
B_v_fit = [Bx_v_fit', Bz_v_fit'];

resfit_v = fun_vert(fit_v, B_v_fit');
resfit_h = fun_hor(fit_h, B_h_fit');

% Get angles that correspond to field values - very close to angle_fit but
% technically can be a bit different
angle_plot_h = atan2(Bx_h_fit,Bz_h_fit) * 180 /pi;
angle_plot_v = atan2(Bx_v_fit,Bz_v_fit) * 180 /pi;

%% Plot data with fits for the tip field

figure;
vd = polarplot((angle_v -90)*pi/180, hypdeltaf_v, 'o');
vd.MarkerFaceColor = '#0072BD';
vd.MarkerSize = 6;
hold on
vf = polarplot((angle_plot_v-90)*pi/180,resfit_v);
vf.Color = '#0072BD';
vf.LineWidth = 1;
hd = polarplot((angle_h -90)*pi/180, hypdeltaf_h, 'o');
hd.MarkerFaceColor = '#EDB120';
hd.MarkerSize = 6;
hf = polarplot((angle_plot_h-90)*pi/180,resfit_h);
hf.Color = '#EDB120';
hf.LineWidth = 1;

%% Functions

function res = FindMinimum(Hm,Hn,Hk,theta)
% Find the minimum of the Stoner Wohlfarth model in 3D (projected in 2D)
% (Hm,Hn) values of the external field along m and n
% Hk = 2K/Ps with K anisotropy and Ps saturation polarization

% theta is a vector among which we find the minimum

% Define the two equations from the Stoner Woglfarth model
eq = Hn .* cos(theta) - Hm .* sin(theta) - 1./2 .* Hk .* sin(2.*theta);
eq2 = Hn .* sin(theta) + Hm .* cos(theta) + Hk .* cos(2.*theta);

% find solutions for eq = 0
% just taking the minimum doesn't ensure that the solution is stable
% so sort to get list of minima and pick the first one that is stable
[~,index_sorted] = sort(abs(eq));
index = find(eq2(index_sorted)>0);
res1 = theta(index_sorted(index(1)));

i = 1;
% to find the second solution go through indexes and pick one that is at
% least 15 degrees apart
while (abs(res1 - theta(index_sorted(index(i)))) <= 1*pi/180) && (i<length(theta))
    i = i+1;
end
% % check that second solution makes sense
% if abs(eq(index_sorted(index(i))))<0.012
%     res2 = theta(index_sorted(index(i)));
% else
%     res2 = NaN;
% end
res2 = theta(index_sorted(index(i)));

res = sort([res1 res2]');
end

function Btip = CalculateBtip(Bz,Bpar,phi,gamma,delta,Hk,Ps,nsol)
%Calculate the tip field vector in (x,y,z) basis
%   Bz : value of the out-of-plane field
%   Bpar : value of the in-plane field
%   phi(rad) : angle between Bpar and x
%   gamma(rad) : angle between easy axis and z
%   delta(rad) : azimutal angle between easy axis and x
%   Hk = Ps/2K with Ps magnetization saturation and K anisotropy
%   Ps : amplitude of tip field
%   nsol : decide if take 1st or 2nd solution

% Write external field in (x,y,z) basis
Bext = [Bpar.*cos(phi); Bpar.*sin(phi); Bz];
% Write vector M in (x,y,z) basis
M = [sin(gamma).*cos(delta); sin(gamma).*sin(delta); cos(gamma) ];

% Calculate Bm
Bm = Bext' * M;
% Calculate the left over vector
Bn_vec = Bext - Bm * M;
% Calculate Bn
Bn = norm(Bn_vec);
% Calculate N (unitary vector)
N = Bn_vec/Bn;

% Now solve the problem in theta
theta = -pi:pi/180:pi;
sols_theta = FindMinimum(Bm,Bn,Hk,theta);
sol_theta = sols_theta(nsol);

% Finally get Btip in (x,y,z) basis
Btip = Ps* (cos(sol_theta)*M + sin(sol_theta)*N);
end

function A = CalculateHyperfineSplitting(gx,gy,gz,Ax,Ay,Az,Bpar_vec,Bz_vec,phi,gamma,delta,Hk,Ps,nsol)
% (gx,gy,gz) g vector
% (Ax,Ay,Az) A vector
% Bpar_vec vector for the different values of Bpar
% Bz_vec vector for the different values of Bz
% phi(rad) angle between Bpar and x
% gamma(rad) angle between easy axis and z
% delta(rad) azimutal angle between easy axis and x
% Hk = Ps/2K with Ps magnetization saturation and K anisotropy
% Ps amplitude of tip field
% nsol decide if take 1st or 2nd solution


% Get tip field in (x,y,z) basis
fun = @(Bz, Bpar) CalculateBtip(Bz,Bpar,phi,gamma,delta,Hk,Ps,nsol);
Btip = arrayfun(fun, Bz_vec, Bpar_vec, 'UniformOutput', false);
Btip = cell2mat(Btip);
% Get external field in (x,y,z) basis
Bext = [Bpar_vec*cos(phi); Bpar_vec*sin(phi); Bz_vec];
% Get total field and cosine directions
Btot_vec = Bext + Btip;
DirCos = Btot_vec ./ sqrt(sum(Btot_vec .* Btot_vec));
% Calculate effective g factor
gvec = DirCos .* [gx; gy; gz]; % row with column [lgu; ngv; mgz]
g = sqrt(sum(gvec.*gvec)); % sum of each column and sqrt to get norm
% Calculate hyperfine splitting
Avec = DirCos .* [gx; gy; gz] .* [Ax; Ay; Az]; % row with column [lgxAx; ngyAy; mgzAz]
A = 1./g .* sqrt(sum(Avec.*Avec)); % sum of each column and sqrt to get norm

end


function pop = CalculatePopulations(Bz,Bpar,phi,gamma,delta,Hk,Ps,T)
%Calculate the tip field vector in (x,y,z) basis
%   Bz : value of the out-of-plane field
%   Bpar : value of the in-plane field
%   phi(rad) : angle between Bpar and x
%   gamma(rad) : angle between easy axis and z
%   delta(rad) : azimutal angle between easy axis and x
%   Hk = Ps/2K with Ps magnetization saturation and K anisotropy
%   Ps : amplitude of tip field
%   T : temperature of the experiment

kB = 1.3807 * 1e-16; %CGS units
V = 4/3*pi *2^3* 1e-24; %CGS units - radius of 2angstroms
kBV = 1e-8 * kB / V;% Boltzmann constant divided by volume, extra factor 1e8 to convert twice T to CGS units

% Write external field in (x,y,z) basis
Bext = [Bpar.*cos(phi); Bpar.*sin(phi); Bz];
% Write vector M in (x,y,z) basis
M = [sin(gamma).*cos(delta); sin(gamma).*sin(delta); cos(gamma) ];

% Calculate Bm
Bm = Bext' * M;
% Calculate the left over vector
Bn_vec = Bext - Bm * M;
% Calculate Bn
Bn = norm(Bn_vec);

% Now solve the problem in theta
theta = -pi:pi/180:pi;
sols = FindMinimum(Bm,Bn,Hk,theta);

if isnan(sols(2))
    % if only one minimum then all population in the ground state
    pop = [1,0];
else
    en = zeros(2,length(Bz));
    for i = [1,2]
        Btip = CalculateBtip(Bz,Bpar,phi,gamma,delta,Hk,Ps,i);
        % Get total field
        Bext = [Bpar*cos(phi); Bpar*sin(phi); Bz];
        Btot = Btip + Bext;
        % Get vector for the easy axis
        M = [sin(gamma).*cos(delta), sin(gamma).*sin(delta), cos(gamma) ];
        % Calculate theta0, angle between magnetic field and easy axis
        theta0 = acos( M*Btot./ sqrt(sum(Btot .* Btot)) );
        % Now calculate the energy
        en(i,:) = -Ps/2 *(Hk.*cos(sols(i)).^2 + 2*sqrt(Bpar.^2+Bz.^2).*cos(sols(i) - theta0));
    end
    pop = exp(-en/(kBV*T))./sum(exp(-en/(kBV*T)));
end

end

function A = hyperfine_Boltzmann(gx,gy,gz,Ax,Ay,Az,Bz_vec,Bpar_vec,phi,gamma,delta,Hk,Ps,T)
%Calculate hyperfine splitting in presence of magnetic tip at finite
%temperature
%   (gx,gy,gz) g vector
%   (Ax,Ay,Az) A vector
%   Bz : value of the out-of-plane field
%   Bpar : value of the in-plane field
%   phi(rad) : angle between Bpar and x
%   gamma(rad) : angle between easy axis and z
%   delta(rad) : azimutal angle between easy axis and x
%   Hk = Ps/2K with Ps magnetization saturation and K anisotropy
%   Ps : amplitude of tip field
%   T : temperature of the experiment

% Calculate populations
fun = @(Bz, Bpar) CalculatePopulations(Bz,Bpar,phi,gamma,delta,Hk,Ps,T);
pop = arrayfun(fun, Bz_vec, Bpar_vec, 'UniformOutput', false);
pop = cell2mat(pop);

% Calculate hyperfine splitting for the two solutions
A1 = CalculateHyperfineSplitting(gx,gy,gz,Ax,Ay,Az,Bpar_vec,Bz_vec,phi,gamma,delta,Hk,Ps,1);
A2 = CalculateHyperfineSplitting(gx,gy,gz,Ax,Ay,Az,Bpar_vec,Bz_vec,phi,gamma,delta,Hk,Ps,2);

% Weight the results
pop(isnan(pop)) = 0;
A = sum(pop.*[A1;A2]);
A = A';
end

function f0 = f0_Boltzmann(gx,gy,gz,Bz_vec,Bpar_vec,phi,gamma,delta,Hk,Ps,T)
%Calculate position of the resonance in presence of magnetic tip at finite
%temperature
%   (gx,gy,gz) g vector
%   Bz : value of the out-of-plane field
%   Bpar : value of the in-plane field
%   phi(rad) : angle between Bpar and x
%   gamma(rad) : angle between easy axis and z
%   delta(rad) : azimutal angle between easy axis and x
%   Hk = Ps/2K with Ps magnetization saturation and K anisotropy
%   Ps : amplitude of tip field
%   T : temperature of the experiment

% Calculate populations
fun = @(Bz, Bpar) CalculatePopulations(Bz,Bpar,phi,gamma,delta,Hk,Ps,T);
pop = arrayfun(fun, Bz_vec, Bpar_vec, 'UniformOutput', false);
pop = cell2mat(pop);

% Calculate position of the resonance for the two solutions
f1 = Calculatef0(gx,gy,gz,Bpar_vec,Bz_vec,phi,gamma,delta,Hk,Ps,1);
f2 = Calculatef0(gx,gy,gz,Bpar_vec,Bz_vec,phi,gamma,delta,Hk,Ps,2);

% Weight the results
f0 = sum(pop.*[f1;f2]);
f0 = f0';

end