clear all
close all

%% Load variables
extension = "";
load(strcat('MabbsC2v', extension,'.mat'))

%% Define space

xspace = 4;
yspace = 4;
zspace = 4;
delta = 0.1;

xvec = -xspace:delta:xspace;
yvec = -yspace:delta:yspace;
zvec = -zspace:delta:zspace;

[X, Y, Z] = meshgrid(xvec, yvec, zvec);
R = sqrt(X.^2 + Y.^2 + Z.^2);

% The origin is always ill defined for the wave function
ix = ceil( length(xvec)/2 );
iy = ceil( length(yvec)/2 );
iz = ceil( length(zvec)/2 );

% get Z values corresponding to the P values
Znuc3d = 8.1414; %effective nuclear charge for 3d electron in titanium 
%http://www.knowledgedoor.com/2/elements_handbook/clementi-raimondi_effective_nuclear_charge_part_4.html#titanium
Znuc4s = 4.8168; %effective nuclear charge for 4s electron in titanium 
Z_sol = ZfromP(P_sol);

%% Define potential

a0 = 0.529177; % Bohr radius (angstrom)

% % from J.Kim et al., PRB 104, 174408 (2021)
% charges = -1*[0.75, 0.75, -0.73, -0.73, 0.46]; % charge unit is -e
% xpos = 1/a0*[0, 0, -1.67, 1.67, 0]; % length unit is bohr radius
% ypos = 1/a0*[-1.36, 1.38 0, 0, 0];
% zpos = 1/a0*[-1.27, -1.27, -2.1, -2.1, 1.76];

% Simple model
charges = -1*[2, 2, -2, -2, 1]; % charge unit is -e
xpos = 1/a0*[-1.45, 1.45, 0, 0, 0]; % length unit is bohr radius
ypos = 1/a0*[0, 0, -1.45, 1.45, 0]; 
zpos = 1/a0*[-1.92, -1.92, -1.92, -1.92, 1.8];

V = 0 .* X;
for i = 1:length(charges)
    V = V - PotentialPointCharge(charges(i),xpos(i),ypos(i),zpos(i),X,Y,Z);
end

%% Calculate Coulomb energy for each solution
CoulombEnergy = 0.*c1_sol;

for i = 1:length(c1_sol)
    c1 = c1_sol(i);
    c2 = c2_sol(i);
    cs = cs_sol(i);
    Znuc = Z_sol(i);
    % Calculate orbitals dz2, dx2-y2 while 4s is scaled accordingly
    dz2  = radialpart(R,Znuc) .* dz2_ang(Z,R);
    dx2y2 = radialpart(R,Znuc) .* dx2y2_ang(X,Y,R);
    s = 1/sqrt(4*pi) * radialpart_4s(R,Znuc*Znuc4s/Znuc3d);
    % Calculate orbital
    orb = c1*dx2y2 + c2*dz2 + cs * s;
    % calculate Coulomb energy
    Coulomb = conj(orb) .* V .* orb;
    Coulomb(ix,iy,iz) = 1/6 * ( Coulomb(ix+1,iy,iz) + Coulomb(ix-1,iy,iz) + ...
                            Coulomb(ix,iy+1,iz) + Coulomb(ix,iy-1,iz) + ...
                            Coulomb(ix,iy,iz+1) + Coulomb(ix,iy,iz-1) );
    CoulombEnergy(i) = 1/(length(xvec)*length(yvec)*length(zvec)) * sum(Coulomb,'all');
end

%% Save results
%save(strcat('PotentialEnergy', extension,'.mat'))

%% Plot results

figure;
hold on
box on
axis square
plot([-1,1],[-1/sqrt(3),1/sqrt(3)],'red')
%plot([-1,1],[-sqrt(3),sqrt(3)],'red')
plot([-1,1],[0,0],'Color',[0,0,0]+0.7)
plot([0,0],[-1,1],'Color',[0,0,0]+0.7)
scatter(c1_sol,c2_sol,[],CoulombEnergy,'filled')
colorbar
xlim([-1.1,1.1])
ylim([-1.1,1.1])
xlabel('c1 scaling dx2-y2')
ylabel('c2 scaling dz2')
title('Coulomb Energy (Hartree)')

%% Plot Coulomb individually for the different cs values

figure;
hold on
for i = length(csvec):-1:1
    cs = csvec(i);
    filter = (cs_sol == cs);
    test = atan2(c2_sol(filter),c1_sol(filter));
    CEmeV = CoulombEnergy * 27.2114 * 1e3; %From Hartree to meV
    plot(test * 180/pi, CEmeV(filter),'.','DisplayName',strcat('cs= ',num2str(cs)))
end
legend()
box on
xlim([-180,180])
xlabel('angle \chi between (c1,c2)')
ylabel('Coulomb energy (meV)')
%title('Dependence on cs')

%% Each cs has its plot

for i = length(csvec):-1:1
    figure;
    cs = csvec(i);
    filter = (cs_sol == cs);
    test = atan2(c2_sol(filter),c1_sol(filter));
    plot(test/pi, CoulombEnergy(filter),'.')
    xlim([-1,1])
    xlabel('angle \theta between (c1,c2) (units of \pi)')
    ylabel('Coulomb energy (Hartree)')
    title(strcat('Coulomb energy for cs = ',num2str(cs)))
end


%% Plot wave-function corresponding to minima for different cs

for i = length(csvec):-1:1
    cs = csvec(i)
    % Filter the slice that corresponds to a given cs
    filter = (cs_sol == cs);
    CEi = CoulombEnergy(filter);
    c1i = c1_sol(filter);
    c2i = c2_sol(filter);
    Zi = Z_sol(filter);
    Pi = P_sol(filter);
    % Identify minimum and corresponding values
    [m,index] = min(CEi);
    c1_opt = c1i(index)
    c2_opt = c2i(index)
    Znuc = Zi(index)
    Popt = Pi(index)
    % Calculate orbitals dz2, dx2-y2 while 4s is the scaled accordingly
    dz2  = radialpart(R,Znuc) .* dz2_ang(Z,R);
    dx2y2 = radialpart(R,Znuc) .* dx2y2_ang(X,Y,R);
    s = 1/sqrt(4*pi) * radialpart_4s(R,Znuc*Znuc4s/Znuc3d);
    % Calculate orbital
    orb = c1_opt*dx2y2 + c2_opt*dz2 + cs * s;
    rho = orb .* conj(orb);
    figure;
    isosurface(X,Y,Z,rho,max(rho,[],'all')*0.1)
    daspect([1 1 1])
%     xlim([-xspace xspace])
%     ylim([-yspace yspace])
%     zlim([-zspace zspace])
    xlabel('x')
    ylabel('y')
    zlabel('z')
    title(strcat('Optimal orbital with cs = ',num2str(cs)))
end

%% Plot for Figure
cs = 0;
filter = (cs_sol == cs);
CEi = CoulombEnergy(filter);
c1i = c1_sol(filter);
c2i = c2_sol(filter);
Zi = Z_sol(filter);
Pi = P_sol(filter);
% Identify minimum and corresponding values
[m,index] = min(CEi);
c1_opt = c1i(index)
c2_opt = c2i(index)
Znuc = Zi(index)
Popt = Pi(index)
% Calculate orbitals dz2, dx2-y2 while 4s is scaled accordingly
dz2  = radialpart(R,Znuc) .* dz2_ang(Z,R);
dx2y2 = radialpart(R,Znuc) .* dx2y2_ang(X,Y,R);
s = 1/sqrt(4*pi) * radialpart_4s(R,Znuc*Znuc4s/Znuc3d);
% Calculate orbital
orb = c1_opt*dx2y2 + c2_opt*dz2 + cs * s;
rho = orb .* conj(orb);


sphereradii = 1/a0*[1.05, 1.05, 1.05, 1.05, a0];
coloratoms = [ ...
    [190 224 185]/255; [190 224 185]/255; ...
    [151 144 178]/255; [151 144 178]/255; ...
    [0.9 0.9 0.9] ];
xpos2 = 1/a0 * [-2*1.45, -2*1.45, 2*1.45, 2*1.45 ...
                -1.45, 1.45, -1.45, 1.45];
ypos2 = 1/a0 * [-1.45, 1.45, -1.45, 1.45 ...
                -2*1.45, -2*1.45, 2*1.45, 2*1.45];
zpos2 = 0.*xpos2 - 1.6/a0;

s_r = 1/a0 * 1.05 * ones(1,length(xpos2));
coloratoms2 = [coloratoms(3:4,:); coloratoms(3:4,:); coloratoms(1:2,:); coloratoms(1:2,:)];

figure;
a = isosurface(X,Y,Z,rho,max(rho,[],'all')*0.2);
p = patch(a);
isonormals(X,Y,Z,rho,p)
view(3);
set(p,'FaceColor',[237 28 36]/255);  
set(p,'EdgeColor','none');
set(p,'SpecularExponent',3);
camlight;
lighting gouraud;
daspect([1 1 1])
plotSetting = get(gca, 'NextPlot');
hold on
for i = 1:length(xpos)
    % create sphere at right position
    [xs, ys, zs, sphere] = createspheres(xpos(i), ypos(i), zpos(i), 0.95*sphereradii(i));
    % add sphere to plot
    surf(xs, ys, zs, 'edgecolor','none','facecolor', coloratoms(i,:),'SpecularStrength',0.0)
end
for i = 1:length(xpos2)
    % create sphere at right position
    [xs, ys, zs, sphere] = createspheres(xpos2(i), ypos2(i), zpos2(i),0.95* s_r(i));
    % add sphere to plot
    surf(xs, ys, zs, 'edgecolor','none','facecolor', coloratoms2(i,:),'SpecularStrength',0.0)
end
view(-37.5, 30)
%view(0, 90)
%view(90, 0)
set(gca,'xtick',[])
set(gca,'ytick',[])
set(gca,'ztick',[])
extra = 4;
xlim([-xspace-extra xspace+extra])
ylim([-yspace-extra yspace+extra])
zlim([-zspace-1 zspace+1.5])
xlabel('x')
ylabel('y')
zlabel('z')




%% Functions

function res = PotentialPointCharge(q,x0,y0,z0,X,Y,Z)
    r = sqrt( (X-x0).^2 + (Y-y0).^2 + (Z-z0).^2 );
    res = q ./ r;
end

% Functions to calculate wavefunctions

function res = radialpart(R,Znuc)

rho = Znuc * R;
res = 4/(81*sqrt(6)) .* Znuc^(3/2) .* rho.^2 .* exp(-rho/3);

end

function res = radialpart_4s(R,Znuc)

rho = Znuc * R;
res = 1/16 .* Znuc.^(3/2) .* 1/6 .* (24 - 26*rho/2 + 12*(rho/2).^2 - (rho/2).^3) .* exp(-rho/4);

end

function res = dz2_ang(Z,R)

res = 1/2 * sqrt( 5/(4*pi) ) * (3*Z.^2 - R.^2)./R.^2;

end

function res = dx2y2_ang(X,Y,R)

res = sqrt(3)/2 * sqrt( 5/(4*pi) ) * (X.^2 - Y.^2)./R.^2;

end

% Function for orbital extent

function Z = ZfromP(P)
% From P value calculate <r-3> and then corresponding Z value

mu0 = 1e-7;
muB = 9.27e-24;
muN = 5.05e-27;
mu47 = -0.78848;
mu49 = -1.104;
h = 6.626e-34;
a0 = 5.29e-11;

rm3 = h.*P.*1e6./(2.0023.*mu47./(5/2).*mu0.*muN.*muB).*a0.^3;

Z = (81 .* rm3).^(1/3);

end

% Function for plot

function [X,Y,Z,spheresXYZ] = createspheres(spherex, spherey, spherez, r)
[x, y, z] = sphere(100);
X = (x*r+spherex);
Y = (y*r+spherey);
Z = (z*r+spherez);
spheresXYZ = [X,Y,Z];
end