clear all
close all

%% Input parameters

gO = 1.653;%1.651;%1.655;%1.653;
gMg = 1.917;%1.898;%1.936;%
gz = 1.989;%1.965;%2.013;%

AMg = 18; %MHz
AO = 68; %MHz
Az = 19; %MHz

g0 = 2.0023;

Amean = 1/3*(AMg + AO + Az);

dAx0 = AO - Amean;
dAy0 = AMg - Amean;
dAz0 = Az - Amean;

dgx0 = gO - g0;
dgy0 = gMg - g0;
dgz0 = gz - g0;

%% Define parameter space
Pvec = 0:-1:-200;
alphavec = 0:0.05:1;

csvec = 0:0.2:0.8;

%% Get results for all values of cs and append solutions

% Set parameters to define which solution to keep
margin = 4;%MHz

funcheck = @(ax, ay, az)(...
    check(ax, dAx0, margin) ...
    && check(ay, dAy0, margin) ...
    && check(az, dAz0, margin));

% Initialize arrays to store solutions
cs_sol = [];
c1_sol = [];
c2_sol = [];
alpha_sol = [];
P_sol = [];

for i = 1:length(csvec)
    cs = csvec(i);
    % Get hyperfine splitting for this cs and corresponding mesh
    [Ax, Ay, Az, c1mesh, c2mesh, Pmesh, alphamesh] = ...
        CalculateHyperfine(alphavec,Pvec,cs,dgx0,dgy0,dgz0);
    % Calculate mask to filter solutions
    mask = arrayfun(funcheck, Ax, Ay, Az);
    % Append solutions
    c1_sol = [c1_sol; c1mesh(mask)];
    c2_sol = [c2_sol; c2mesh(mask)];
    alpha_sol = [alpha_sol; alphamesh(mask)];
    P_sol = [P_sol; Pmesh(mask)];
    cs_sol = [cs_sol; cs*ones(length(c1mesh(mask)),1)];
end

%% Save parameters
clear fig
%save('MabbsC2v.mat')

%% Plot solutions

figure;
hold on
box on
axis square
plot([-1,1],[-1/sqrt(3),1/sqrt(3)],'red')
plot([-1,1],[-sqrt(3),sqrt(3)],'red')
plot([-1,1],[0,0],'Color',[0,0,0]+0.7)
plot([0,0],[-1,1],'red')
scatter(c1_sol,c2_sol,[],alpha_sol,'filled')
colorbar
xlim([-1.1,1.1])
ylim([-1.1,1.1])
xlabel('c1 scaling dx2-y2')
ylabel('c2 scaling dz2')
title('Solutions with alpha values')

figure;
hold on
box on
axis square
plot([-1,1],[-1/sqrt(3),1/sqrt(3)],'red')
plot([-1,1],[-sqrt(3),sqrt(3)],'red')
plot([-1,1],[0,0],'Color',[0,0,0]+0.7)
plot([0,0],[-1,1],'red')
scatter(c1_sol,c2_sol,[],P_sol,'filled')
colorbar
xlim([-1.1,1.1])
ylim([-1.1,1.1])
xlabel('c1 scaling dx2-y2')
ylabel('c2 scaling dz2')
title('Solutions with P values')

%% Filter alpha and P values
filteralpha = (alpha_sol > 0.7);
c1_sol_filtered = c1_sol(filteralpha);
c2_sol_filtered = c2_sol(filteralpha);
alpha_sol_filtered = alpha_sol(filteralpha);
P_sol_filtered = P_sol(filteralpha);
cs_sol_filtered = cs_sol(filteralpha);

%% Plot filtered data

figure;
hold on
box on
axis square
plot([-1,1],[-1/sqrt(3),1/sqrt(3)],'red')
plot([-1,1],[-sqrt(3),sqrt(3)],'red')
plot([-1,1],[0,0],'Color',[0,0,0]+0.7)
plot([0,0],[-1,1],'red')
scatter(c1_sol_filtered,c2_sol_filtered,[],alpha_sol_filtered,'filled')
colorbar
xlim([-1.1,1.1])
ylim([-1.1,1.1])
xlabel('c1 scaling dx2-y2')
ylabel('c2 scaling dz2')
title('Solutions with alpha values')

figure;
hold on
box on
axis square
plot([-1,1],[-1/sqrt(3),1/sqrt(3)],'red')
plot([-1,1],[-sqrt(3),sqrt(3)],'red')
plot([-1,1],[0,0],'Color',[0,0,0]+0.7)
plot([0,0],[-1,1],'red')
scatter(c1_sol_filtered,c2_sol_filtered,[],P_sol_filtered,'filled')
colorbar
xlim([-1.1,1.1])
ylim([-1.1,1.1])
xlabel('c1 scaling dx2-y2')
ylabel('c2 scaling dz2')
title('Solutions with P values')

%% Define functions

function res = paramA(dgz,alpha,c1)
    res = -dgz/( 8*alpha^2*c1^2 );
end

function res = paramB(dgx,alpha,c1,c2)
    res = -dgx/( 2*alpha^2* (c1+sqrt(3)*c2)^2 );
end

function res = paramC(dgy,alpha,c1,c2)
    res = -dgy/( 2*alpha^2* (c1-sqrt(3)*c2)^2 );
end

function res = dAz(P,alpha,c1,c2,A,B,C)
    res = P * alpha^2 * ( ...
            - 16/3 * c1^2 * A ...
            + 2/3 * ( (c1+sqrt(3)*c2)^2*B + (c1-sqrt(3)*c2)^2*C ) ...
            - 4/7 *( c1^2 - c2^2 ) ...
            - sqrt(3)/7 * (sqrt(3)*c1 + c2)*(c1 - sqrt(3)*c2) * C ...
            - sqrt(3)/7 * (sqrt(3)*c1 - c2)*(c1 + sqrt(3)*c2) * B ...
            );
end

function res = dAx(P,alpha,c1,c2,A,B,C)
    res = P * alpha^2 * ( ...
            - 4/3 * (c1+sqrt(3)*c2)^2 * B ...
            + 2/3 * ( (c1-sqrt(3)*c2)^2*C + 4*c1^2*A ) ...
            + 2/7 * (c1^2 - c2^2 - 2*sqrt(3)*c1*c2) ...
            - 4*sqrt(3)/7 *c1*c2*A ...
            + sqrt(3)/7 * (sqrt(3)*c1 + c2)*(c1 - sqrt(3)*c2) * C ...
            );
end

function res = dAy(P,alpha,c1,c2,A,B,C)
    res = P * alpha^2 * ( ...
            - 4/3 * (c1-sqrt(3)*c2)^2 * C ...
            + 2/3 * ( (c1+sqrt(3)*c2)^2*B + 4*c1^2*A ) ...
            + 2/7 * (c1^2 - c2^2 + 2*sqrt(3)*c1*c2) ...
            + 4*sqrt(3)/7 *c1*c2*A ...
            + sqrt(3)/7 * (sqrt(3)*c1 - c2)*(c1 + sqrt(3)*c2) * B ...
            );
end

function res = check(value, target, error)
    if abs(value-target)<=error
        res = 1;
    else
        res = 0;
    end
end

function [Ax, Ay, Az, c1mesh, c2mesh, Pmesh, alphamesh] = ...
    CalculateHyperfine(alphavec,Pvec,cs,dgx0,dgy0,dgz0)
% Calculate the hyperfine splitting for one value of cs
% alphavec and Pvec are the vectors that define the values spanned by alpha
% and P
% dgx0, dy0 and dz0 are the g values that are used to calculate A, B and C

% define c1 and c2 based on angle between them
dtheta = 1;
theta = 0:dtheta:360-dtheta;
theta = theta*pi/180;
c1vec = sqrt(1-cs^2)*cos(theta);
c2vec = sqrt(1-cs^2)*sin(theta);

% make mesh of all parameters
Pmesh = repmat(Pvec,length(c1vec),1,length(alphavec));
c1mesh = repmat(c1vec',1,length(Pvec),length(alphavec));
c2mesh = repmat(c2vec',1,length(Pvec),length(alphavec));
alphamesh = repmat(alphavec',1,length(c1vec),length(Pvec));
alphamesh = permute(alphamesh,[2 3 1]);

% Solve equations - first determine A, B and C from g

funA = @(alpha,c1)paramA(dgz0,alpha,c1);
funB = @(alpha,c1,c2)paramB(dgx0,alpha,c1,c2);
funC = @(alpha,c1,c2)paramC(dgy0,alpha,c1,c2);

Amesh = arrayfun(funA, alphamesh, c1mesh);
Bmesh = arrayfun(funB, alphamesh, c1mesh, c2mesh);
Cmesh = arrayfun(funC, alphamesh, c1mesh, c2mesh);

% Calculate A for all possible solutions

funAx = @(P,alpha,c1,c2,A,B,C)dAx(P,alpha,c1,c2,A,B,C);
funAy = @(P,alpha,c1,c2,A,B,C)dAy(P,alpha,c1,c2,A,B,C);
funAz = @(P,alpha,c1,c2,A,B,C)dAz(P,alpha,c1,c2,A,B,C);

Ax = arrayfun(funAx, Pmesh, alphamesh, c1mesh, c2mesh, Amesh, Bmesh, Cmesh);
Ay = arrayfun(funAy, Pmesh, alphamesh, c1mesh, c2mesh, Amesh, Bmesh, Cmesh);
Az = arrayfun(funAz, Pmesh, alphamesh, c1mesh, c2mesh, Amesh, Bmesh, Cmesh);
end
