close all
clear all

% Load variables from fits
load('variablesHorizontal.mat')
load('variablesVertical.mat')

% Define field vectors
B_h = [Bx_h, Bz_h];
B_v = [Bx_v, Bz_v];

% Calculate center frequency for reference atoms
f0_hyp_h = hypf0_h + 3.5* hypdeltaf_h; 
f0_hyp_v = hypf0_v + 3.5* hypdeltaf_v;

% For both atoms at the same time
f0_both_h = [f0_h; f0_hyp_h];
f0_both_v = [f0_v; f0_hyp_v];
B_both_h = [B_h; B_h];
B_both_v = [B_v; B_v];

%% Define fit model
% coefs = (gpar, gperp)
fitgvalues = @(coefs, B)(arrayfun(...
    @(Bpar, Bperp) gfactor2D(Bpar, Bperp, coefs(1), coefs(2)), ...
    B(:,1), B(:,2)...
    ));

% define starting point and limits
fit_start = [2,2];
fit_up = [3,3];
fit_low = [0,0];

%% Fit reference horizontal
[fitg_rh, ~, res_rh, ~, ~, ~, jac_rh] = ...
    lsqcurvefit(fitgvalues, fit_start, B_h, f0_h, fit_low, fit_up);
% 95% confidence intervals for fit coefficients
conf_rh = nlparci(fitg_rh,res_rh,'jacobian',jac_rh);

%% Fit hyperfine horizontal
[fitg_hyph, ~, res_hyph, ~, ~, ~, jac_hyph] = ...
    lsqcurvefit(fitgvalues, fit_start, B_h, f0_hyp_h, fit_low, fit_up);
% 95% confidence intervals for fit coefficients
conf_hyph = nlparci(fitg_hyph,res_hyph,'jacobian',jac_hyph);

%% Fit reference+hyperfine horizontal
[fitg_bothh, ~, res_bothh, ~, ~, ~, jac_bothh] = ...
    lsqcurvefit(fitgvalues, fit_start, B_both_h, f0_both_h, fit_low, fit_up);
% 95% confidence intervals for fit coefficients
conf_bothh = nlparci(fitg_bothh,res_bothh,'jacobian',jac_bothh);

%% Fit reference vertical
[fitg_rv, ~, res_rv, ~, ~, ~, jac_rv] = ...
    lsqcurvefit(fitgvalues, fit_start, B_v, f0_v, fit_low, fit_up);
% 95% confidence intervals for fit coefficients
conf_rv = nlparci(fitg_rv,res_rv,'jacobian',jac_rv);

%% Fit hyperfine vertical
[fitg_hypv, ~, res_hypv, ~, ~, ~, jac_hypv] = ...
    lsqcurvefit(fitgvalues, fit_start, B_v, f0_hyp_v, fit_low, fit_up);
% 95% confidence intehypvals for fit coefficients
conf_hypv = nlparci(fitg_hypv,res_hypv,'jacobian',jac_hypv);

%% Fit reference+hyperfine vertical
[fitg_bothv, ~, res_bothv, ~, ~, ~, jac_bothv] = ...
    lsqcurvefit(fitgvalues, fit_start, B_both_v, f0_both_v, fit_low, fit_up);
% 95% confidence intehypvals for fit coefficients
conf_bothv = nlparci(fitg_bothv,res_bothv,'jacobian',jac_bothv);

%% Estimate additionnal field

% define fit model
ftfield = fittype( 'gfactor3D(Bpar, Bperp, phi, Bx0, By0, Bz0)', ...
    'Independent', {'Bpar','Bperp'});
% Set limits and starting coefficients (order coefs: Bx0, By0, Bz0, phi)
phiv = 16;
phih = 90-phiv;

optionsv = fitoptions(ftfield);
optionsv.Lower = [-1, -1, -1, phiv];
optionsv.Upper = [1, 1, 1, phiv];
optionsv.StartPoint = [0.001, 0.001, 0.001, phiv];

optionsh = fitoptions(ftfield);
optionsh.Lower = [-1, -1, -1, phih];
optionsh.Upper = [1, 1, 1, phih];
optionsh.StartPoint = [0.001, 0.001, 0.001, phih];

% vertical reference
[fitB_v_ref,gofB_v_ref] = fit([Bx_v, Bz_v], f0_v, ftfield, optionsv)
% vertical hyperfine
[fitB_v_hyp,gofB_v_hyp] = fit([Bx_v, Bz_v], f0_hyp_v, ftfield, optionsv)
% vertical reference + hyperfine
[fitB_v_both,gofB_v_both] = fit(B_both_v, f0_both_v, ftfield, optionsv)

% horizontal reference
[fitB_h_ref,gofB_h_ref] = fit([Bx_h, Bz_h], f0_h, ftfield, optionsh)
% horizontal hyperfine
[fitB_h_hyp,gofB_h_hyp] = fit([Bx_h, Bz_h], f0_hyp_h, ftfield, optionsh)
% horizontal reference + hyperfine
[fitB_h_both,gofB_h_both] = fit(B_both_h, f0_both_h, ftfield, optionsh)

%% Functions

function res = gfactor2D(Bpar, Bperp, gpar, gperp)
%Calculates the energy of a ESR transition for a spin 1/2 with anisotropic
%g factor. Only consider a 2D vector field (Bpar, Bperp) and assume that
%the components Bpar and Bperp are perpendicular to each other.
%   Bpar: external magnetic field parallel to the surface
%   Bperp: external magnetic field perpendicular to the surface
%   gpar: effective g factor parallel to the surface
%   gperp: effective g factor perpendicular to the surface

mub = 9.27 * 10^(-24); % J/T Bohr magneton
h = 6.626 * 10^(-34); % J*s Planck's constant
fac = mub/h * 10^(-9); % to get results in GHz
res = fac * sqrt( (gpar*Bpar).^2 + (gperp*Bperp).^2 );
end