clear all
close all

% Load variables from fits
load('variablesHorizontal.mat')
load('variablesVertical.mat')

% Hyperfine values in MHz
hypdeltaf_v = hypdeltaf_v.*1e3;
hypdeltaf_h = hypdeltaf_h.*1e3;
hypdeltafneg_h = hypdeltafneg_h*10^3;
hypdeltafneg_v = hypdeltafneg_v*10^3;
hypdeltafpos_h = hypdeltafpos_h*10^3;
hypdeltafpos_v = hypdeltafpos_v*10^3;

%% Define fit functions

% g factor from J. Kim et al, PRB 104, 174408 (2021)
gv = 1.6696;
gh = 1.9026;
gz = 1.989;

% coefs = (theta0, Apar, Aperp)
fun_vert = @(coefs, theta)...
            HyperfineTheta(theta,coefs(1),coefs(2),coefs(3),gv,gz);
fun_hor = @(coefs, theta)...
            HyperfineTheta(theta,coefs(1),coefs(2),coefs(3),gh,gz);

% Define ranges
fit_start = [0, 30, 30];
fit_up = [180, 100, 100];
fit_low = [-180, 0, 0];

%% Fit vertical data
[fit_v, ~, res_v, ~, ~, ~, jac_v] = ...
    lsqcurvefit(fun_vert, fit_start, angle_v, hypdeltaf_v, fit_low, fit_up);
% 95% confidence intervals for fit coefficients
conf_v = nlparci(fit_v,res_v,'jacobian',jac_v);


%% Fit horizontal data
[fit_h, ~, res_h, ~, ~, ~, jac_h] = ...
    lsqcurvefit(fun_hor, fit_start, angle_h, hypdeltaf_h, fit_low, fit_up);
% 95% confidence intervals for fit coefficients
conf_h = nlparci(fit_h,res_h,'jacobian',jac_h);

%% Plot fit results in a polar plot
angle_plot = -180:1:179;    

figure;
vd = polarplot((angle_v -90)*pi/180, hypdeltaf_v, 'o');
vd.MarkerFaceColor = '#0072BD';
vd.MarkerSize = 6;
hold on
vf = polarplot((angle_plot-90)*pi/180,fun_vert(fit_v, angle_plot));
vf.Color = '#0072BD';
vf.LineWidth = 1;

hd = polarplot((angle_h -90)*pi/180, hypdeltaf_h, 'o');
hd.MarkerFaceColor = '#EDB120';
hf = polarplot((angle_plot-90)*pi/180,fun_hor(fit_h, angle_plot));
hf.Color = '#EDB120';
hf.LineWidth = 1;

%% Functions

function A = HyperfineTheta(theta,theta0,Apar,Aperp,gpar,gperp)
%Calculate the hyperfine splitting in the presence of an external field
%that makes an angle theta with the perpendicular direction

theta = theta * pi./180;
theta0 = theta0 * pi./180;

% Calculate g vector
g = sqrt( (gperp.*cos(theta-theta0)).^2 + (gpar.*sin(theta-theta0)).^2 );

% Calculate hyperfine splitting
A = 1./g .* sqrt( (Aperp.*gperp.*cos(theta-theta0)).^2 ...
                + (Apar.*gpar.*sin(theta-theta0)).^2 );
end