clear all
close all
%% Load variables
% Load variables from fits
load('variablesHorizontal.mat')
load('variablesVertical.mat')

% Define field vectors
B_h = [Bx_h, Bz_h];
B_v = [Bx_v, Bz_v];

% Make hyperfine values row vectors in MHz
hypdeltaf_v = hypdeltaf_v'.*1e3;
hypdeltaf_h = hypdeltaf_h'.*1e3;
hypdeltafneg_h = hypdeltafneg_h'*10^3;
hypdeltafneg_v = hypdeltafneg_v'*10^3;
hypdeltafpos_h = hypdeltafpos_h'*10^3;
hypdeltafpos_v = hypdeltafpos_v'*10^3;


%% Define parameters
phi = 14;

Aprime_v = 23.6;
% Tip field
Btip_v = 0.110;
gamma_v = 111;
delta_v = 89;
kappa_v = delta_v - phi;

Aprime_h = 65.4;
% Tip field
Btip_h = 0.011;
gamma_h = 12;
delta_h = -phi;
kappa_h = delta_h - (90 - phi);


% phi = 12;
% 
% Aprime_v = 23.6;
% % Tip field
% Btip_v = 0.102;
% gamma_v = 101;
% delta_v = 91;
% kappa_v = delta_v - phi;
% 
% Aprime_h = 65.4;
% % Tip field
% Btip_h = 0.010;
% gamma_h = 24;
% delta_h = -phi;
% kappa_h = delta_h - (90 - phi);
% 
% phi = 16;
% 
% Aprime_v = 23.6;
% % Tip field
% Btip_v = 0.080;
% gamma_v = 106;
% delta_v = 90;
% kappa_v = delta_v - phi;
% 
% Aprime_h = 65.4;
% % Tip field
% Btip_h = 0.012;
% gamma_h = 4;
% delta_h = -phi;
% kappa_h = delta_h - (90 - phi);

%% Define fit functions

fit_start = [60, 20, 10];
fit_up = [100, 100, 180];
fit_low = [0, 0, -180];

fun_vert0 = @(Apar,Aperp,theta0,Bpar,Bperp) ...
        HyperfineBtip(Apar,Aperp,Aprime_v,Bpar,Bperp,theta0,phi,Btip_v,gamma_v,kappa_v);
fun_vert = @(coefs, B)(...
    fun_vert0(coefs(1),coefs(2),coefs(3),B(:,1),B(:,2) )...
    );

fun_hor0 = @(Apar,Aperp,theta0,Bpar,Bperp) ...
        HyperfineBtip(Apar,Aperp,Aprime_h,Bpar,Bperp,theta0,90 - phi,Btip_h,gamma_h,kappa_h);
fun_hor = @(coefs, B)(...
    fun_hor0(coefs(1),coefs(2),coefs(3),B(:,1),B(:,2) )...
    );

%% Fit vertical data

% Fit vertical data
[fit_v, ~, res_v, ~, ~, ~, jac_v] = ...
    lsqcurvefit(fun_vert, fit_start, B_v, hypdeltaf_v, fit_low, fit_up);
% 95% confidence intervals for fit coefficients
conf_v = nlparci(fit_v,res_v,'jacobian',jac_v);

%% Fit horizontal data

% Fit horizontal data
[fit_h, ~, res_h, ~, ~, ~, jac_h] = ...
    lsqcurvefit(fun_hor, fit_start, B_h, hypdeltaf_h, fit_low, fit_up);
% 95% confidence intervals for fit coefficients
conf_h = nlparci(fit_h,res_h,'jacobian',jac_h);

%% Define x axis for fits and interpolate B fields values
angle_fit = -180:1:180;
Bz_v_fit = interp1(angle_v,Bz_v,angle_fit);
Bx_v_fit = interp1(angle_v,Bx_v,angle_fit);
Bz_h_fit = interp1(angle_h,Bz_h,angle_fit);
Bx_h_fit = interp1(angle_h,Bx_h,angle_fit);

% Define field vectors
B_h_fit = [Bx_h_fit', Bz_h_fit'];
B_v_fit = [Bx_v_fit', Bz_v_fit'];

res_v = fun_vert(fit_v, B_v_fit);
res_h = fun_hor(fit_h, B_h_fit);

%% Plot vertical data
figure;
set(gca,'FontSize',18)
xticks([-180 -90 0 90 180])
yticks([10 30 50 70])
hold on;
box on
p = errorbar(angle_v, hypdeltaf_v, hypdeltafneg_v, hypdeltafpos_v, 'o');
p.Color = '#494EA1';
p.MarkerFaceColor = '#494EA1';
p.MarkerSize = 3;
s = plot(angle_fit, res_v );
s.Color = '#231F20';
s.LineWidth = 1;
xlabel('Angle (Bx, Bz)')
ylabel('Hyperfine splitting (MHz)')

%% Plot horizontal data
figure;
set(gca,'FontSize',18)
xticks([-180 -90 0 90 180])
yticks([14 18 22 26])
hold on;
box on
p = errorbar(angle_h, hypdeltaf_h, hypdeltafneg_h, hypdeltafpos_h, 'o');
p.Color = '#494EA1';
p.MarkerFaceColor = '#494EA1';
p.MarkerSize = 3;
s = plot(angle_fit, res_h );
s.Color = '#231F20';
s.LineWidth = 1;
xlabel('Angle (Bx, Bz)')
ylabel('Hyperfine splitting (MHz)')

%% Plot both data sets together
figure;
set(gca,'FontSize',18)
xticks([-180 -90 0 90 180])
yticks([10 30 50 70])
hold on;
box on
p = errorbar(angle_v, hypdeltaf_v, hypdeltafneg_v, hypdeltafpos_v, 'o');
p.Color = '#494EA1';
p.MarkerFaceColor = '#494EA1';
p.MarkerSize = 3;
s = plot(angle_fit, res_v );
s.Color = '#231F20';
s.LineWidth = 1;
q = errorbar(angle_h, hypdeltaf_h, hypdeltafneg_h, hypdeltafpos_h, 'o');
q.Color = '#494EA1';
q.MarkerFaceColor = '#494EA1';
q.MarkerSize = 3;
r = plot(angle_fit, res_h );
r.Color = '#231F20';
r.LineWidth = 1;
xlabel('Angle (Bx, Bz)')
ylabel('Hyperfine splitting (MHz)')

%% Try as polar plot
angle_plot = atan2(Bx_h_fit,Bz_h_fit) * 180 /pi;

figure;
vd = polarplot((angle_v -90)*pi/180, hypdeltaf_v, 'o');
vd.MarkerFaceColor = '#0072BD';
vd.MarkerSize = 6;
hold on
vf = polarplot((angle_plot-90)*pi/180,res_v);
vf.Color = '#0072BD';
vf.LineWidth = 1;

hd = polarplot((angle_h -90)*pi/180, hypdeltaf_h, 'o');
hd.MarkerFaceColor = '#EDB120';
hf = polarplot((angle_plot-90)*pi/180,res_h);
hf.Color = '#EDB120';
hf.LineWidth = 1;

%% Functions

function A = HyperfineBtip(Apar,Aperp,Aprime,Bpar,Bperp,theta0,phi,Btip,gamma,kappa)
% Calculate hyperfine splitting in an external field when a tip field is
% also consider.
%   lattice basis: (x,y,z)
%   phi: rotation of the plane defined by (Bpar,Bperp) w.r.t x axis
%   name the axis that defines the intersection of the two planes u
%   (u,v,z) basis is the measurement basis
%   theta0: rotation of (Bpar,Bperp) w.r.t (u,z)
%   (Apar, Aperp) effective A parameter along (u,z)
%   Aprime effective A parameter along v
%   Btip amplitude of the tip field
%   gamma angle of the tip field w.r.t to z axis
%   kappa angle of the in-plane component of the tip field w.r.t u axis
%NB: Bpar and Bperp are vector columns

Bpar = Bpar'; % make it a row vector
Bperp = Bperp'; % make it a row vector

theta0 = theta0 * pi./180; %radians
%theta0 = 0; %just to be sure it is always 0
phi = phi * pi./180; % radians
gamma = gamma * pi./180; % radians
kappa = kappa * pi./180; % radians

% g factor from J. Kim et al, PRB 104, 174408 (2021)
gy = 1.917; %gMg
gx = 1.653; %gO
gz = 1.989;

% effective g factor in (u,v,z) basis
gu = sqrt( (cos(phi)*gx)^2 + (sin(phi)*gy)^2 );
gv = sqrt( (sin(phi)*gx)^2 + (cos(phi)*gy)^2 );

% Tip field in (u,v,z) basis
Btip_uvz = Btip* [sin(gamma)*cos(kappa);...
                sin(gamma)*sin(kappa);...
                cos(gamma)];
            
% Total field in the (u,v,z) basis
Bext_uvz = [cos(theta0).*Bpar - sin(theta0).*Bperp; 0.*Bpar; cos(theta0).*Bperp + sin(theta0).*Bpar] ;
%rotate only Bpar
%Bext_uvz = [cos(theta0).*Bpar; 0.*Bpar; Bperp + sin(theta0).*Bpar] ; %
Swtch = sum(Btip_uvz .* Bext_uvz); % Calculate product and sum components
Btot = Bext_uvz + sign(Swtch).*Btip_uvz;
%do not switch tip field
%Btot = Bext_uvz + Btip_uvz;

% Calculate cosine directions
Btot2 = Btot .* Btot;
Btot2_sum = sqrt(sum(Btot2)); % sum of each column and sqrt to get norm
DirCos = Btot ./ Btot2_sum;

% Calculate effective g factor
gvec = DirCos .* [gu; gv; gz]; % row with column [lgu; ngv; mgz]
g = sqrt(sum(gvec.*gvec)); % sum of each column and sqrt to get norm

% Calculate hyperfine splitting
Avec = DirCos .* [gu; gv; gz] .* [Apar; Aprime; Aperp]; % row with column [lgxAx; ngyAy; mgzAz]
A = 1./g .* sqrt(sum(Avec.*Avec)); % sum of each column and sqrt to get norm

end