clear all
close all

%% Parameters to load data
% List of spectra and corresponding fields
Hyp_h = {'ESR_A701';'ESR_A699';'ESR_A697';'ESR_A693';'ESR_A691';'ESR_A679';'ESR_A683';'ESR_A685';'ESR_A687';'ESR_A659';'ESR_A643';'ESR_A641';'ESR_A629';'ESR_A625';'ESR_A714';'ESR_A711';'ESR_A709';'ESR_A705';'ESR_A703'};
Ref_h = {'ESR_A700';'ESR_A698';'ESR_A694';'ESR_A692';'ESR_A690';'ESR_A678';'ESR_A682';'ESR_A684';'ESR_A686';'ESR_A653';'ESR_A642';'ESR_A640';'ESR_A627';'ESR_A622';'ESR_A713';'ESR_A710';'ESR_A707';'ESR_A704';'ESR_A702'};
Bx_h = {-0.227;-0.440;-0.624;-0.776;-0.883;-0.781;-0.617;-0.435;-0.225;0.222;0.627;0.772;0.873;0.905;0.776;0.625;0.440;0.222;0.000};
Bx_h = cell2mat(Bx_h);
Bz_h = {-0.848;-0.762;-0.624;-0.448;-0.237;0.451;0.617;0.753;0.838;0.827;0.627;0.446;0.234;0.000;-0.448;-0.625;-0.762;-0.828;-0.857};
Bz_h = cell2mat(Bz_h);

sensref = {0.001;0.001;0.001;0.001;0.001;0.001;0.001;0.001;0.001;0.001;0.001;0.005;0.001;0.001;0.001;0.001;0.001;0.001;0.001};
sensref = cell2mat(sensref); %V
sens = 0.001; %V

% Calculate total field
Btot_h = sqrt(Bz_h.^2+Bx_h.^2);

% Calculate corresponding field angle
angle_h = atan2(Bx_h,Bz_h) * 180 /pi;

% Parameters
gain = 10^9;


%% Initialize arrays to store fit results for reference atom

% To store results
f0_h = zeros(size(Bx_h));
f0neg_h = zeros(size(Bx_h));
f0pos_h = zeros(size(Bx_h));
I1_h = zeros(size(Bx_h));
I1neg_h = zeros(size(Bx_h));
I1pos_h = zeros(size(Bx_h));
Ioff_h = zeros(size(Bx_h));
Ioffneg_h = zeros(size(Bx_h));
Ioffpos_h = zeros(size(Bx_h));
gamma_h = zeros(size(Bx_h));
gammaneg_h = zeros(size(Bx_h));
gammapos_h = zeros(size(Bx_h));
alpha_h = zeros(size(Bx_h));
alphaneg_h = zeros(size(Bx_h));
alphapos_h = zeros(size(Bx_h));


%% Go through spectra - fit reference and store fit values

for i = 1:size(Ref_h)
    filename = strcat(Ref_h{i},'.tf');
    
    % Open file and load data
    fid = fopen(filename);
    data = textscan(fid, '%f %f %f %f %f %f %f %f %f %f %f', 'headerlines', 17);
    fclose(fid);
    freq = data{1}*1E-9; %GHz
    lockin = data{7};
    lockin = lockin(~isnan(freq));
    freq = freq(~isnan(freq));
    
    % Convert lockin to change in current
    deltaI = lockin * (sensref(i)/10/gain) * 2*sqrt(2)*pi/4 * 10^12; %pA
    
    % Fit with a Fano function
    % intialize parameters
    [I1i, f0i] = max(deltaI);
    gammai = 0.03;
    alphai = 0.4;
    Ioffi = deltaI(1);
    % define fit model
    ft = fittype( 'FitFano(f,f0,I1,gamma,alpha,Ioff)', 'Independent', 'f');
    % Set limits and starting coefficients (order coefs: I1, Ioff, alpha, f0,gamma)
    options = fitoptions(ft);
    options.Lower = [0, -10 , -Inf, min(freq), 0];
    options.Upper = [100, 10, 10, max(freq), 0.2];
    options.StartPoint = [I1i, Ioffi, alphai, f0i, gammai];
    % Perform fit
    [myfit,gof] = fit(freq, deltaI, ft, options);
    % Plot fit results
    fig = figure;
    plot(myfit, freq, deltaI)
    title(Ref_h{i}, 'Interpreter', 'None')
    annotation('textbox',[0.2 0.2 0 0 ], 'String', strcat('R^2:',num2str(gof.rsquare,'%4.2f')))
    subtitle(strcat('(Bx, Bz)=(',num2str(Bx_h(i)),',',num2str(Bz_h(i)),') / (Btot,theta)=(',num2str(Btot_h(i)),',',num2str(angle_h(i)),')'))
    xlabel('Frequency (GHz)')
    ylabel('Delta I (pA)')
    exportgraphics(fig,fullfile('Figures',strrep(filename,'tf','png')),'Resolution',300)
    
    % Store values for field dependence
    f0_h(i) = myfit.f0;
    I1_h(i) = myfit.I1;
    Ioff_h(i) = myfit.Ioff;
    gamma_h(i) = myfit.gamma;
    alpha_h(i) = myfit.alpha;
    
    % Store values for error bars (order coefs: I1, Ioff, alpha, f0,gamma)
    conf = confint(myfit);
    I1neg_h(i) = conf(1)-myfit.I1;
    I1pos_h(i) = conf(2)-myfit.I1;
    Ioffneg_h(i) = conf(3)-myfit.Ioff;
    Ioffpos_h(i) = conf(4)-myfit.Ioff;
    alphaneg_h(i) = conf(5)-myfit.alpha;
    alphapos_h(i) = conf(6)-myfit.alpha;
    f0neg_h(i) = conf(7)-myfit.f0;
    f0pos_h(i) = conf(8)-myfit.f0;
    gammaneg_h(i) = conf(9)-myfit.gamma;
    gammapos_h(i) = conf(10)-myfit.gamma;
end


%% Plot results field dependence for reference atom
figure;
plot(angle_h, Btot_h, 'o')
xlabel('Angle Bxz')
ylabel('Total field (T)')

figure;
errorbar(angle_h, f0_h, f0neg_h, f0pos_h, 'o')
xlabel('Angle Bxz')
ylabel('Center frequency (GHz)')

figure;
errorbar(angle_h, I1_h, I1neg_h, I1pos_h, 'o')
xlabel('Angle Bxz')
ylabel('Peak intensity (pA)')

figure;
errorbar(angle_h, Ioff_h, Ioffneg_h, Ioffpos_h, 'o')
xlabel('Angle Bxz')
ylabel('Offset (pA)')

figure;
errorbar(angle_h, gamma_h, gammaneg_h, gammapos_h, 'o')
xlabel('Angle Bxz')
ylabel('Full width half maximum (GHz)')

figure;
errorbar(angle_h, alpha_h, alphaneg_h, alphapos_h, 'o')
xlabel('Angle Bxz')
ylabel('Peak asymmetry/q factor')

%% Initialize arrays to store fit results for hyperfine atom

% To store results
hypf0_h = zeros(size(Bx_h));
hypf0neg_h = zeros(size(Bx_h));
hypf0pos_h = zeros(size(Bx_h));
hypI1_h = zeros(size(Bx_h));
hypI1neg_h = zeros(size(Bx_h));
hypI1pos_h = zeros(size(Bx_h));
hypIoff_h = zeros(size(Bx_h));
hypIoffneg_h = zeros(size(Bx_h));
hypIoffpos_h = zeros(size(Bx_h));
hypdeltaf_h = zeros(size(Bx_h));
hypdeltafneg_h = zeros(size(Bx_h));
hypdeltafpos_h = zeros(size(Bx_h));

hypgof_h = zeros(size(Bx_h));

%% Now fit hyperfine data

gh = 2; % to guess hyperfine splitting
gz = 2; % to guess hyperfine splitting

Ah = 0.03; %GHz initial guess for hyperfine in plane
Az = 0.02; %GHz initial guess for hyperfine out of plane

for i = 1:size(Hyp_h)
    filename = strcat(Hyp_h{i},'.tf');
    
    % Open file and load data
    fid = fopen(filename);
    data = textscan(fid, '%f %f %f %f %f %f %f %f %f %f %f', 'headerlines', 17);
    fclose(fid);
    freq = data{1}*1E-9; %GHz
    lockin = data{7};
    
    % Convert lockin to change in current
    deltaI = lockin * (sens/10/gain) * 2*sqrt(2)*pi/4 * 10^12; %pA
    
    % Intialize parameters
    % To guess hyperfine coupling
    thetai = (angle(i) - 10) * pi/180;
    gi = sqrt( (sin(thetai)*gh).^2 + (cos(thetai)*gz).^2 );
    % Initialize parameters
    deltafi = 1/gi * sqrt( (sin(thetai)*gh*Ah).^2 + (cos(thetai)*gz*Az).^2 );
    hypdeltaf_h(i) = deltafi;
    n = 8;
    f0i = f0_h(i) - (n-1)/2 * deltafi; 
    I1i = I1_h(i)/n;
    gammai = gamma_h(i);
    alphai = alpha_h(i);
    Ioffi = Ioff_h(i);
    
    % define fit model
    ft = fittype( 'SumFitFanos(f,f0,I1,gamma,alpha,Ioff,n,deltaf)', 'Independent', 'f');
    % Set limits and starting coefficients (order coefs: I1, Ioff, alpha, deltaf, f0, gamma, n)
    options = fitoptions(ft);
    options.Lower = [0, min(deltaI) , alphai, 0, min(freq), gammai, n];
    options.Upper = [100, max(deltaI) , alphai, 1, max(freq), gammai, n];
    options.StartPoint = [I1i, Ioffi, alphai, deltafi, f0i, gammai, n];
    % Perform fit
    [myfit,gof] = fit(freq, deltaI, ft, options);
    
    hypgof_h(i) = gof.rsquare;
    % Plot fit results
    fig = figure;
    plot(myfit, freq, deltaI)
    title(Hyp_h{i}, 'Interpreter', 'None')
    annotation('textbox',[0.2 0.2 0 0 ], 'String', strcat('R^2:',num2str(gof.rsquare,'%4.2f')))
    subtitle(strcat('(Bx, Bz)=(',num2str(Bx_h(i)),',',num2str(Bz_h(i)),') / (Btot,theta)=(',num2str(Btot_h(i)),',',num2str(angle_h(i)),')'))
    xlabel('Frequency (GHz)')
    ylabel('Delta I (pA)')
    exportgraphics(fig,fullfile('Figures',strrep(filename,'tf','png')),'Resolution',300)
    
    % Store values for field dependence
    hypf0_h(i) = myfit.f0;
    hypI1_h(i) = myfit.I1;
    hypIoff_h(i) = myfit.Ioff;
    hypdeltaf_h(i) = myfit.deltaf;
    
    % Store values for error bars
    conf = confint(myfit);
    hypI1neg_h(i) = conf(1)-myfit.I1;
    hypI1pos_h(i) = conf(2)-myfit.I1;
    hypIoffneg_h(i) = conf(3)-myfit.Ioff;
    hypIoffpos_h(i) = conf(4)-myfit.Ioff;
    hypdeltafneg_h(i) = conf(7)-myfit.deltaf;
    hypdeltafpos_h(i) = conf(8)-myfit.deltaf;
    hypf0neg_h(i) = conf(9)-myfit.f0;
    hypf0pos_h(i) = conf(10)-myfit.f0;
end


%% Plot results

figure;
plot(angle_h, hypgof_h, 'o')
xlabel('Angle Bxz')
ylabel('Goodness of fit')

figure;
errorbar(angle_h, hypf0_h, hypf0neg_h, hypf0pos_h, 'o')
xlabel('Angle Bxz')
ylabel('Center frequency (GHz)')

figure;
errorbar(angle_h, hypI1_h, hypI1neg_h, hypI1pos_h, 'o')
xlabel('Angle Bxz')
ylabel('Peak intensity (pA)')

figure;
errorbar(angle_h, hypIoff_h, hypIoffneg_h, hypIoffpos_h, 'o')
xlabel('Angle Bxz')
ylabel('Offset (pA)')

figure;
errorbar(angle_h, hypdeltaf_h, hypdeltafneg_h, hypdeltafpos_h, 'o')
xlabel('Angle Bxz')
ylabel('Hyperfine splitting (GHz)')

%% Save parameters
clear fig
save('variablesHorizontal.mat')