%==========================================================================
% [main.m] Main file for B747 RECOVER PID/IMF-INDI robust design case study
%
%     Optional: 
%       [1] Valid IQCLab installation
%
%   Author: T.S.C. Pollack
%   Copyright (2024) T.S.C. Pollack
%==========================================================================

clear all; clc; init;                   

%% --- USER SETTINGS ---

% External CLAW initialization file, if available
f_claw_init = '';

% --- AIRFRAME MODEL & DESIRED DYNAMICS ---

% Specify airframe model
fname_mdl = 'b747_h30kft_M0.85_n1_xcg0.25_def_f1'; 

% Set desired C* dynamics
w0 = 1.3; D = 0.8; w3 = 1.5; w1 = 1;

% --- PLOTTING ---
plt_figs = 0;

% --- SAVE DATA ---
save_data  = 0;
annotation = '';

% --- UNCERTAINTY & DISTURBANCE ---

% Specify uncertainties
options.unc.Wunc_act   = makeweight(db2mag(-80),70,db2mag(14),0,1);
options.unc.Wunc_q     = makeweight(db2mag(-80),25,db2mag(18),0,3);
options.unc.Wunc_q     = fcn_transform_uweight(options.unc.Wunc_q,1,...
                                                'minreal');
options.unc.Wunc_nz    = minreal(makeweight(db2mag(-80),15,db2mag(50),0,4));
options.unc.Wunc_nz    = fcn_transform_uweight(options.unc.Wunc_nz,2,...
                                                'minreal');
options.unc.Wunc_alpha = tf(0);
options.unc.dMa        = 0.5;
options.unc.dMq        = 0.3;
options.unc.dMde       = 0.2;
options.unc.dZa        = 0.1;
options.unc.dZq        = 0;
options.unc.dZde       = 0;

% Specify disturbances
options.dist.w_M       = 3;    % [deg/s^2];
options.dist.w_Z       = 0.25; % [g]
options.dist.a_g       = 2;    % [deg];
options.dist.W_g       = tf(1);

% Define uncertainty block for MU-analysis
options.dksyn.blk = [1 1; 1 1; 1 1; 1 1; 1 1; 1 1; 1 1; 1 1; 1 1; 3 2];
options.dkana.blk = [1 0; 1 0; 1 0; 1 0; -1 0; -1 0; -1 0; 3 2];

% --- CONTROL LAW ---
options.claw.mode       = 'horthbfc';
options.claw.sync       = 'cc';
options.claw.aoa_mode   = 'est_nz';
options.dksyn.mode      = 'hybrid';
    
% --- SYNTHESIS ---

% D-K iteration settings; note - an important step to check whether the 
% D-K settings are appropriate is to compare the MUtab and MUDtab outputs 
% in the INFO struct
options.dksyn.n_dkiter    = 10; 
options.dksyn.RandomStart = 1;
options.dksyn.RandomPNorm = 1;
options.dksyn.eps         = 0.01; 
options.dksyn.NDfit       = 4;   
options.dksyn.W           = logspace(-2,2,71);
        
% SYSTUNE options
options.systune = systuneOptions('SoftTol',0.001,'Display','sub',...
                        'SoftScale',1/5000,'MaxIter',20,'MinDecay',1e-8);

% Performance weights
options.dksyn.weights.Wperf_te_mcv = 2*tf(1,[1 1e-2]);
options.dksyn.weights.Wperf_NP_K   = 15;

% Control activity weights
options.dksyn.weights.Wperf_u = tf(0); 
  
% --- ANALYSIS ---

% IQC analysis
options.iqc.run        = 0;
options.iqc.perf_idx   = [1,2;1,3]; 
options.iqc.npsi       = 4;
options.iqc.bftype     = 1; 
options.iqc.metric     = 'L2';

%% --- GENERATION ---

% Synthesis plant
gen_syn_plant;

% Initialize CLAWs
init_dp_claw;

% Configure CLAWs
gen_dp_claw;

% Closed-loop synthesis formulation
gen_syn_claw;

%% --- REQUIREMENTS ---
    
% Nominal performance: Hoo tuning goals
Req_NP   = TuningGoal.WeightedGain({'w_r'}, {'z_eps_mcv'},1,1);                                   
Req_NP_U = TuningGoal.WeightedGain({'w_g'}, {'z_u'},1,1);

% Robust performance: structured Hoo tuning goal
Req_RP = TuningGoal.WeightedGain({'p_act','p_m_q','p_m_alpha',...
                                  'p_m_nz','w_r','w_g','w_xi'}, ...
                                   {'q_act','q_m_q','q_m_alpha',...
                                   'q_m_nz','z_eps_mcv'},1,1);

% Uncertain model stability margin requirements
Req_RS_SM = TuningGoal.WeightedGain({'p_n_act','p_n_es','p_n_q',...
       'p_n_alpha','p_n_nz','p_act','p_m_q','p_m_alpha','p_m_nz'}, ...
                                    {'q_n_act','q_n_es','q_n_q',...
       'q_n_alpha','q_n_nz','q_act','q_m_q','q_m_alpha','q_m_nz'},1,1);

% Set model and frequency band scopes 
Req_RP.Models    = 1;
Req_RS_SM.Models = 2; 
Req_NP.Models    = 3;
Req_NP_U.Models  = 3;

Req_RP.Focus    = [options.dksyn.W(1),options.dksyn.W(end)];
Req_RS.Focus    = [options.dksyn.W(1),options.dksyn.W(end)];
Req_NP.Focus    = [options.dksyn.W(1),options.dksyn.W(end)];
Req_NP_U.Focus  = [options.dksyn.W(1),options.dksyn.W(end)];
Req_RS_SM.Focus = [options.dksyn.W(1),options.dksyn.W(end)];

% Set gamma 'tunable' as performance metric
GAMMA    = TuningGoal.Gain('g_in','g_out',1e3);

% Collect requirements
SoftReqs = GAMMA;
HardReqs = [Req_RP,Req_RS_SM,Req_NP,Req_NP_U];

% Apply soft requirements to performance models only
SoftReqs.Models = 1; 

%% --- SYNTHESIS ---

tic

% Remove analysis channels and select [RP,RS,NP] synthesis models
S_CL0_MA = CL_MA(1:end-5,1:end,:,:,[1,3,7]);

% Run hybrid approximation synthesis algorithm
[S_CL_MA,fSoft,gHard,INFO] = systune_hybrid(S_CL0_MA,K_CLAW,SoftReqs,...
                                                        HardReqs,options); 

% Get CLAW design parameters
K_CLAW   = setBlockValue(K_CLAW,S_CL_MA);

toc

%% --- CONVERSION ---

% Transfer block values to individual control blocks
K_ff     = setBlockValue(K_ff,K_CLAW);
K_vc     = setBlockValue(K_vc,K_CLAW);
K_qdot   = setBlockValue(K_qdot,K_CLAW);
K_syn    = setBlockValue(K_syn,K_CLAW);

% Transfer block values - closed-loop model arrays
CL_MA    = setBlockValue(CL_MA,K_CLAW);
UCL_MA   = setBlockValue(UCL_MA,K_CLAW);
S_UCL_MA = UCL_MA(1:end-5,1:end);

% Store performance models
UCL_P    = UCL_MA(end-6:end,end-2:end,:,:,1);
S_UCL_P  = S_UCL_MA(end-1:end,end-2:end,:,:,1);

% Store gamma and MU values
WCG_SYN   = INFO.GAMMAtab{end};
WCG_N_SYN = INFO.MUDtab{end};

%% --- PREPARE ANALYSIS MODEL ---

if any(strfind(options.claw.mode,'hort'))
    IC_BL = linearize('BL_IMF_INDI_MOD');
elseif any(strfind(options.claw.mode,'hodc'))
    IC_BL = linearize('BL_IMF_INDI_QMCV');
elseif strfind(options.claw.mode,'pi') || strfind(options.claw.mode,'pid') 
    IC_BL = linearize('BL_PID_PF');
end

% Rename I/O
IC_BL = fcn_sys_iocleaner(IC_BL);

% Connect airframe and complex uncertainties
BL  = lft(IC_BL,B747_USYS);
UBL = lft(Delta_c,BL);

% Connect analysis points
AP_act     = AnalysisPoint('AP_act');
AP_es      = AnalysisPoint('AP_es');
AP_q_m     = AnalysisPoint('AP_q_m');
AP_alpha_m = AnalysisPoint('AP_alpha_m');
AP_nz_m    = AnalysisPoint('AP_nz_m');
APs        = blkdiag(AP_act,AP_es,AP_q_m,AP_alpha_m,AP_nz_m);

A_UCL    = lft(APs,UBL);
A_UCL_R  = A_UCL(:,1);
A_UCL_AG = A_UCL(:,2);
A_UCL_D  = A_UCL(:,3);
A_UCL_P  = A_UCL(1:2,:);

%% --- MU ANALYSIS ---

% Run wcgain from synthesis performance model
[WCG_SP,WCU_SP] = wcgain(S_UCL_P); 
[WCG_AP,WCU_AP] = wcgain(A_UCL_P); 

% Get gamma for individual channels
[WCG_R_AP,WCU_R_AP]   = wcgain(A_UCL_P(1,1)); 
[WCG_AG_AP,WCU_AG_AP] = wcgain(A_UCL_P(1,2)); 
[WCG_D_AP,WCU_D_AP]   = wcgain(A_UCL_P(1,3)); 

% Run Nichols analysis
N_UCL_MA = S_UCL_MA(1:end-2,1:end-3);
[WCG_N_TAB,WCU_N_TAB] = fcn_wc_nichols(N_UCL_MA,{1e-8,1e8});
[WCG_N_MF_TAB,WCU_N_MF_TAB] = fcn_wc_nichols(N_UCL_MA,{1e-1,2e1});
[WCG_N_HF_TAB,WCU_N_HF_TAB] = fcn_wc_nichols(N_UCL_MA,{2e1,1e3});

% Obtain mu bounds for performance model  
P_UCL_RP = lftdata(S_UCL_P);
P_UCL_RS = P_UCL_RP(1:end-2,1:end-3);

[mubnds_rs,mu_rs_info] = mussv(P_UCL_RS,options.dkana.blk(1:end-1,:));    
max_mu_rs = max(mubnds_rs(:,1).ResponseData(:));

[mubnds_rp,mu_rp_info] = mussv(P_UCL_RP,options.dkana.blk);    
max_mu_rp = max(mubnds_rp(:,1).ResponseData(:));

%% --- IQC ANALYSIS ---

if options.iqc.run    
    run_iqc;    
end

%% --- PLOT RESULTS ---

% Set number of samples
options.plt.n_sample = 25;

% Set output indices
out_idx_z_eps_mcv = 1;
out_idx_q         = 3;
out_idx_nz        = 5;
out_idx_cstar     = 6;
out_idx_de        = 7;

% Set input indices
in_idx_wr         = 1;
in_idx_wg         = 2;
in_idx_wxi        = 3;

if plt_figs    
    plt_results;
end

%% --- EXPORT DATA ---

% Clear YALMIP variables
clear prob_perf

% Identify batch setting
if options.dksyn.RandomStart > 0
    rsmode = 'batch';
else
    rsmode = 'single';
end

% Set directory name
res_dir   = strcat('_RESULTS/',fname_mdl,'\',options.claw.mode,'\', ...
                    options.claw.sync,'\',options.claw.aoa_mode,'\',...
                    options.dksyn.mode,'\',rsmode,'\',annotation,'\');
export_fname = 'umbrella';

% Save data
if save_data
    if ~exist(res_dir, 'dir')
        mkdir(res_dir);
    end
   save(strcat(res_dir,export_fname)); 
end
