%==========================================================================
% [plt_uncertainty_weights.m] Plot uncertainty weights in freq. domain
%
%   Note: before using this script, an umbrella.mat file must be loaded
%
%   Author: T.S.C. Pollack
%   Copyright (2024) T.S.C. Pollack
%==========================================================================

% Specify colors
clr_order = [1,2,3,6];

% Specify line width
LineWidthSetting = 3;

% Specify font size
font_size = 20;

% Save plot yes/no
save_plt = 1;

% Create legend yes/no
add_lgd  = 1;

% Add annotation yes/no
annotate = 1;

% Set analysis point
ap = 'AP_es';

% Set frequency spectrum of interest
omega = logspace(-3,4,100);

%% --- GET DATA ---

% Actuator
[WACT_MAG,~] = bode(options.unc.Wunc_act,omega); 

% Pitch rate
[WACT_QM_MAG,~]     = bode(options.unc.Wunc_q,omega); 
[WACT_QM_BA_MAG,~]  = bode(B747_NSYS(1,1)*(1+options.unc.Wunc_q),omega);

% Vertical acceleration
[WACT_NZM_MAG,~]    = bode(options.unc.Wunc_nz,omega); 
[WACT_NZM_BA_MAG,~] = bode(B747_NSYS(3,1)*(1+options.unc.Wunc_nz),omega);

%% --- PLOT ---

close all;

% Set color theme
clr_theme = [ [0, 0.4470, 0.7410]; ...
            [0.8500, 0.3250, 0.0980]; ...
            [0.4660, 0.6740, 0.1880]; ...
            [0.4940, 0.1840, 0.5560]; ...  
            [0.6350, 0.0780, 0.1840]; ...
            [0.75, 0.75, 0.75]; ...
            [0.75, 0.75, 0]; ...
            [0, 0.75, 0.75]; ...
            [0.9290, 0.6940, 0.1250]];
        
FIG_WU = figure; FIG_WU.Position = [300 59 560 410];

% Plot combined weights
sp1 = subplot(1,1,1);
semilogx(sp1,omega,mag2db(WACT_MAG(:)),'color',clr_theme(clr_order(1),:),...
                            'LineStyle','-','LineWidth',LineWidthSetting); 
grid on; hold on;  
semilogx(sp1,omega,mag2db(WACT_QM_MAG(:)),'color',clr_theme(clr_order(2),:),...
                            'LineStyle','-','LineWidth',LineWidthSetting); 
semilogx(sp1,omega,mag2db(WACT_NZM_MAG(:)),'color',clr_theme(clr_order(3),:),...
                            'LineStyle','-','LineWidth',LineWidthSetting);      
semilogx(sp1,omega,zeros(size(omega)),'color',clr_theme(clr_order(4),:),...
                            'LineStyle','--','LineWidth',LineWidthSetting);  

xlim(omega([1,end]));
set(sp1,'TickLabelInterpreter','latex','FontSize',14)
xlabel(sp1,'Frequency (rad/s)','Interpreter','latex','FontSize',font_size);
ylabel(sp1,'Magnitude (dB)','Interpreter','latex','FontSize',font_size);

% Add legend
if add_lgd
    legend('$W_{act}$','$W_q$','$W_{n_z}$',...
        'Orientation','horizontal','NumColumns',1, ...
        'Interpreter','latex','Location','Best','FontSize',font_size-6);
end

% Spotlight Qm weight
FIG_WUQM = figure; FIG_WUQM.Position = [300 59 560 410];

sp2 = subplot(1,1,1);
semilogx(sp2,omega,mag2db(180/pi*WACT_QM_BA_MAG(:)),'color',...
   clr_theme(clr_order(2),:),'LineStyle','-','LineWidth',LineWidthSetting); 
grid on; hold on;  

xlim(omega([1,end]));
set(sp2,'TickLabelInterpreter','latex','FontSize',14)
xlabel(sp2,'Frequency (rad/s)','Interpreter','latex','FontSize',font_size);
ylabel(sp2,'Magnitude (dB)','Interpreter','latex','FontSize',font_size);

% Spotlight NZm weight
FIG_WUNZM = figure; FIG_WUNZM.Position = [300 59 560 410];

sp3 = subplot(1,1,1);
semilogx(sp3,omega,mag2db(WACT_NZM_BA_MAG(:)),'color',...
   clr_theme(clr_order(3),:),'LineStyle','-','LineWidth',LineWidthSetting); 
grid on; hold on;  

xlim(omega([1,end]));
set(sp3,'TickLabelInterpreter','latex','FontSize',14)
xlabel(sp3,'Frequency (rad/s)','Interpreter','latex','FontSize',font_size);
ylabel(sp3,'Magnitude (dB)','Interpreter','latex','FontSize',font_size);


%% Save plot

FIG_WU.Renderer    = 'painters';
FIG_WUQM.Renderer  = 'painters';
FIG_WUNZM.Renderer = 'painters';

if save_plt
    saveas(FIG_WU,strcat('_FIGURES/','FIG_WU','.fig'));
    saveas(FIG_WU,strcat('_FIGURES/','FIG_WU','.png'));
    saveas(FIG_WU,strcat('_FIGURES/','FIG_WU','.emf'));
    saveas(FIG_WU,strcat('_FIGURES/','FIG_WU'),'epsc');
    
    saveas(FIG_WUQM,strcat('_FIGURES/','FIG_WUQM','.fig'));
    saveas(FIG_WUQM,strcat('_FIGURES/','FIG_WUQM','.png'));
    saveas(FIG_WUQM,strcat('_FIGURES/','FIG_WUQM','.emf'));
    saveas(FIG_WUQM,strcat('_FIGURES/','FIG_WUQM'),'epsc');
    
    saveas(FIG_WUNZM,strcat('_FIGURES/','FIG_WUNZM','.fig'));
    saveas(FIG_WUNZM,strcat('_FIGURES/','FIG_WUNZM','.png'));
    saveas(FIG_WUNZM,strcat('_FIGURES/','FIG_WUNZM','.emf'));
    saveas(FIG_WUNZM,strcat('_FIGURES/','FIG_WUNZM'),'epsc');
end