%==========================================================================
% [plt_sim_ltv.m] Simulate and plot LTV response
%
%   Note: before using this script, an umbrella.mat file must be loaded
%
%   Author: T.S.C. Pollack
%   Copyright (2024) T.S.C. Pollack
%==========================================================================

%% --- USER SETTINGS ---

close all; clear all; clc;

% Set file name
ffname = 'FIG_SIM_LTV';

% Set data sources
res_dir = '..\..\..\..\_FIGDATA\';

f1 = strcat(res_dir,'b747_h30kft_M0.85_n1_xcg0.25\PI\batch\umbrella_plt.mat');
f2 = strcat(res_dir,'b747_h30kft_M0.85_n1_xcg0.25\PID\batch\umbrella_plt.mat');
f3 = strcat(res_dir,'b747_h30kft_M0.85_n1_xcg0.25\HB-INDI-QMCV\batch\umbrella_plt.mat');
f4 = strcat(res_dir,'b747_h30kft_M0.85_n1_xcg0.25\HB-INDI-CSTARMCV\batch\umbrella_plt.mat');

fnames{1} = f1; fnames{2} = f2; fnames{3} = f3; fnames{4} = f4; 

% Set simulation sources
s1 = '..\..\..\_SIMULATION\SIM_LTV_PID_PF.slx';
s2 = '..\..\..\_SIMULATION\SIM_LTV_PID_PF.slx';
s3 = '..\..\..\_SIMULATION\SIM_LTV_IMF_INDI_QMCV.slx';
s4 = '..\..\..\_SIMULATION\SIM_LTV_IMF_INDI_MOD.slx';

fnames{1} = f1; fnames{2} = f2; fnames{3} = f3; fnames{4} = f4; 
snames{1} = s1; snames{2} = s2; snames{3} = s3; snames{4} = s4; 

% Set file destination
fsdes = '_RESULTS/2_PID_IMF_INDI/SIM/';

% Specify theme
clr_order = [1,3,2,4];
st_order = {'-','-','--','-'};

% Specify line width
LineWidthSetting = 3;

% Specify font size
font_size = 15;

% Save plot yes/no
save_plt = 1;

% Add legend yes/no
add_lgd = 1;

% Set frequency spectrum of interest
omega = logspace(-2,2,100);

% Specify simulation settings
dt = 1e-2; T = 0:dt:15;

% Set LTV settings
A_dMa = 1;
F_dMa = 1;
P_dMa = 0;
A_dMq = 1;
F_dMq = 1;
P_dMq = pi/2;
A_dZa = 1;
A_dMde = 0;
F_dMde = 1;
P_dMde = -pi/2;

%% --- GET SIMULATION DATA ---

for p_i = 1:length(fnames)
    
    % Import controller data
    load(fnames{p_i});

    % Simulate
    sim(snames{p_i});
    
    % Extract data
    y_q_tab{p_i}  = simdata.Data(:,1);
    y_nz_tab{p_i} = simdata.Data(:,3);
    y_cstar_tab{p_i} = y_nz_tab{p_i} + kq*y_q_tab{p_i};
    y_dde_tab{p_i} = simdata.Data(:,4);
    
end

%% --- MISC ---

% Set color theme
clr_theme = [ [0, 0.4470, 0.7410]; ...
            [0.8500, 0.3250, 0.0980]; ...
            [0.4660, 0.6740, 0.1880]; ...
            [0.4940, 0.1840, 0.5560]; ...  
            [0.6350, 0.0780, 0.1840]; ...
            [0.75, 0.75, 0.75]; ...
            [0.75, 0.75, 0]; ...
            [0, 0.75, 0.75]; ...
            [0.9290, 0.6940, 0.1250]];

%% --- PLOT ---

% Create figure [left bottom width height]
FIG = figure; FIG.Position = [300 59 560*2 620];

% Simulate and plot response
for p_i = 1:length(fnames)

    if p_i == 1
        sp1 = subplot(2,2,1);
        sp1.Position = sp1.Position + [-0.02 -0.03 0.04 0.07];  
        
        [y_cstar_ref,~] = step(H_des,T);
        plot(sp1,T,y_cstar_ref,'r--','LineWidth',LineWidthSetting);
        grid on; hold on;  
    end
        
    plot(sp1,T,y_cstar_tab{p_i},'color',clr_theme(clr_order(p_i),:),...
        'LineStyle',st_order{p_i},'LineWidth',LineWidthSetting);
    
    if p_i == length(fnames)
        
        ylim(sp1,[-0.1,1.4]);
        ylabel(sp1,'$C^{*}$ (-)','Interpreter','latex','FontSize',...
                font_size);
        set(sp1,'TickLabelInterpreter','latex','FontSize',font_size)
        set(sp1,'XTickLabel',[])

        % Add legend
        if add_lgd
            legend(sp1,'Reference','PI','PID','HB-INDI ($q$ MCV)',...
                'HB-INDI ($C^*$ MCV)', 'Orientation','vertical',...
                'NumColumns',2,'Interpreter','latex','Location',...
                'southeast','FontSize',font_size);
        end
    end        

    if p_i == 1
        sp2 = subplot(2,2,2);
        sp2.Position = sp2.Position + [0 -0.03 0.04 0.07];
    end    
    
    plot(sp2,T,y_q_tab{p_i}*180/pi,'color',clr_theme(clr_order(p_i),:),...
        'LineStyle',st_order{p_i},'LineWidth',LineWidthSetting);
    grid on; hold on;
    
    if p_i == length(fnames)
        ylim(sp2,[-0.2,3.5]);
        ylabel(sp2,'Pitch rate (deg/s)','Interpreter','latex',...
                    'FontSize',font_size);
        set(sp2,'TickLabelInterpreter','latex','FontSize',font_size)
        set(sp2,'XTickLabel',[])
    end        

    if p_i == 1
        sp3 = subplot(2,2,3);
        sp3.Position = sp3.Position + [-0.02 -0.01 0.04 0.07];
    end    
    
    plot(sp3,T,y_nz_tab{p_i},'color',clr_theme(clr_order(p_i),:),...
        'LineStyle',st_order{p_i},'LineWidth',LineWidthSetting);
    grid on; hold on;
    
    if p_i == length(fnames)
        ylim(sp3,[-0.1,0.9]);
        ylabel(sp3,'Normal acceleration (g)','Interpreter','latex',...
            'FontSize',font_size);
        set(sp3,'TickLabelInterpreter','latex','FontSize',font_size)
        xlabel(sp3,'Time (s)','Interpreter','latex','FontSize',font_size);
    end    

    if p_i == 1
        sp4 = subplot(2,2,4);
        sp4.Position = sp4.Position+ [0 -0.01 0.04 0.07];
    end    
    
    plot(sp4,T,y_dde_tab{p_i},'color',clr_theme(clr_order(p_i),:),...
        'LineStyle',st_order{p_i},'LineWidth',LineWidthSetting);
    grid on; hold on;
        
    if p_i == length(fnames)
        % Plot rate limit
        plot(sp4,T,-37*ones(size(T)),'k--','LineWidth',LineWidthSetting);
        
        ylim(sp4,[-40,15]);
        xlabel(sp4,'Time (s)','Interpreter','latex','FontSize',font_size);
        ylabel(sp4,'Deflection rate (deg/s)','Interpreter','latex',...
            'FontSize',font_size);
        set(sp4,'TickLabelInterpreter','latex','FontSize',font_size)
    end  

end

%% Save plot

FIG.Renderer = 'painters';

if save_plt
    saveas(FIG,strcat(fsdes,ffname,'.fig'));
    saveas(FIG,strcat(fsdes,ffname,'.png'));
    saveas(FIG,strcat(fsdes,ffname,'.emf'));
    saveas(FIG,strcat(fsdes,ffname),'epsc');
end