%==========================================================================
% [plt_rp_ud.m] Plot input complementary sensitivity function in comparison
% to uncertainty bound
%
%   Author: T.S.C. Pollack
%   Copyright (2024) T.S.C. Pollack
%==========================================================================

%% --- USER SETTINGS ---

close all; clear all; clc;

% Set file name
ffname = 'FIG_RP_WGA_WC_CONSTR';

% Set data sources
res_dir = '..\..\..\..\_FIGDATA\';

f1 = strcat(res_dir,'b747_h30kft_M0.85_n1_xcg0.25\PI\batch\umbrella_plt.mat');
f2 = strcat(res_dir,'b747_h30kft_M0.85_n1_xcg0.25\PID\batch\umbrella_plt.mat');
f3 = strcat(res_dir,'b747_h30kft_M0.85_n1_xcg0.25\HB-INDI-QMCV\batch\umbrella_plt.mat');
f4 = strcat(res_dir,'b747_h30kft_M0.85_n1_xcg0.25\HB-INDI-CSTARMCV\batch\umbrella_plt.mat');

fnames{1} = f1; fnames{2} = f2; fnames{3} = f3; fnames{4} = f4; 

% Set file destination
fsdes = '_RESULTS/RP/DIST/';

% Specify theme
clr_order = [1,3,2,4];
st_order = {'-','-','--','-'};

% Specify line width
LineWidthSetting = 3;

% Specify font size
font_size = 15;

% Save plot yes/no
save_plt = 0;

% Set plot mode nominal/worst-case
plt_mode = 'nom';

% Create legend yes/no
add_lgd  = 0;

% Set frequency spectrum of interest
omega = logspace(-1,1.5,100);

%% --- GET SIMULATION DATA ---

for p_i = 1:length(fnames)

    % Import data
    load(fnames{p_i});

    % Select model(synthesis/analysis)
    UCL_PLT = A_UCL_AG;

    % Sample uncertainties
    UCL_SAMPLED = usample(UCL_PLT,options.plt.n_sample);

    % Sample worst-case uncertainty
    UCL_WC = usubs(UCL_PLT,WCU_AG_AP);

    % Get nominal system
    NCL = ss(UCL_PLT);

    % Get responses
    SSYS_d{p_i}  = UCL_SAMPLED(out_idx_de,1);
    WSYS_d{p_i}  = UCL_WC(out_idx_de,1);
    NSYS_d{p_i}  = NCL(out_idx_de,1);
    
end

%% --- MISC ---

% Set color theme
clr_theme = [ [0, 0.4470, 0.7410]; ...
            [0.8500, 0.3250, 0.0980]; ...
            [0.4660, 0.6740, 0.1880]; ...
            [0.4940, 0.1840, 0.5560]; ...  
            [0.6350, 0.0780, 0.1840]; ...
            [0.75, 0.75, 0.75]; ...
            [0.75, 0.75, 0]; ...
            [0, 0.75, 0.75]; ...
            [0.9290, 0.6940, 0.1250]];

%% --- PLOT ---

% Preliminaries        
FIG = figure; FIG.Position = [300 59 560 320]; 

if strcmp(plt_mode,'nom')
    
    % --- Plot nominal magnitude ---
    for p_i = 1:length(fnames)

        if p_i == 1
            sp1 = subplot(1,1,1);      
        end

        % Get response data
        [NL_MAG,~] = bode(NSYS_d{p_i},omega); 

        % Plot nominal response
        semilogx(sp1,omega,mag2db(NL_MAG(:)),'color',clr_theme(clr_order(p_i),:),'LineStyle',st_order{p_i},'LineWidth',LineWidthSetting);  
        grid on; hold on;     

    end

elseif strcmp(plt_mode,'wc')

    % --- Plot worst-case magnitude ---
    for p_i = 1:length(fnames)

        if p_i == 1
            sp1 = subplot(1,1,1);  
        end

        % Get response data
        [WC_MAG,~] = bode(WSYS_d{p_i},omega);  

        % Plot worst-case response
        semilogx(sp1,omega,mag2db(WC_MAG(:)),'color',clr_theme(clr_order(p_i),:),'LineStyle',st_order{p_i},'LineWidth',LineWidthSetting);
        grid on; hold on;  

    end

end

xlim([omega(1),omega(end)]); ylim([-20,10]);
xlabel(sp1,'Frequency (rad/s)','Interpreter','latex','FontSize',font_size);
ylabel(sp1,'Magnitude (dB)','Interpreter','latex','FontSize',font_size);
set(sp1,'TickLabelInterpreter','latex','FontSize',font_size)

% Add legend
if add_lgd
    legend(sp1,'PI','PID','HB-INDI ($q$ MCV)','HB-INDI ($C^*$ MCV)',...
        'Orientation','vertical','NumColumns',2, ...
        'Interpreter','latex','Location','NorthWest','FontSize',font_size);
end

%% Save plot

FIG.Renderer = 'painters';

if save_plt
    saveas(FIG,strcat(fsdes,ffname,'.fig'));
    saveas(FIG,strcat(fsdes,ffname,'.png'));
    saveas(FIG,strcat(fsdes,ffname,'.emf'));
    saveas(FIG,strcat(fsdes,ffname),'epsc');
end