%==========================================================================
% [sync.m] Hinf-based synchronization principle/design
%
%   Note: before using this script, an umbrella.mat file must be loaded
%
%   Author: T.S.C. Pollack
%   Copyright (2024) T.S.C. Pollack
%==========================================================================

clc;

plt_figs = 0;

% Define Laplace operator
s = tf('s');

% Specify sensitivity bounds
M_i = 2;

% Specify frequency band
wtab = [1e-1,1e4];

% Specify PM reference frequency
wref = 1.3;

% Specify dynamic contributions
H_diff = tf(2,[1 2]);

act_w_n  = 20; 
act_zeta = sqrt(1/2);
act_del  = 0; 

G_act    = tf(act_w_n^2,[1 2*act_zeta*act_w_n act_w_n^2]);

Ts      = 1/80;
G_del   = pade(exp(-Ts*s),1);
G_ZOH   = (1 - pade(exp(-Ts*s),1))/(s + 1e-4)*1/Ts;
G_alias = tf(150,[1 150]); 

% Specify synchronization elements
Kc = 0.6; 
Hs = G_act*H_diff;

% Specify nominal unsynchronized dynamics
G_NOM = G_alias*G_del*G_ZOH;

% Specify uncertain contribution
DELTA_act = options.unc.Wunc_act*Delta_c.Uncertainty.Delta_act;
DELTA_q_m = options.unc.Wunc_q*Delta_c.Uncertainty.Delta_q_m;

G_DELTA = 1*(1 + DELTA_act)*(1 + DELTA_q_m);

% Construct equivalent filter element
C_s     = 1/(1-Kc*Hs);
C_s_bar = C_s/dcgain(C_s);

% Construct equivalent controller
K_eq = C_s*Kc*Hs;

% Construct synchronization sensitivity functions
L_s = Kc*Hs*(G_NOM*G_DELTA - 1);
S_s = 1/(1 + L_s);
S_i = (1-Kc*Hs)*S_s;
L_i = 1/S_i-1;

K_S_i = K_eq*S_i;

% Get norm values
gamma_s = wcgain(ss(S_s),{wtab(1),wtab(2)});
[gamma_s_wc,u_s_wc] = wcgain(S_s,{wtab(1),wtab(2)});

gamma_i = wcgain(ss(S_i),{wtab(1),wtab(2)});
[gamma_i_wc,u_i_wc] = wcgain(S_i,{wtab(1),wtab(2)});

% Display results
gamma_s
gamma_s_wc
gamma_i
gamma_i_wc

if plt_figs
    close all; 
    % Plot
    figure; bode(ss(S_s)); grid on; hold on; bode(usubs(S_s,u_s_wc),'m');
    figure; bode(ss(L_s)); grid on; hold on;
    figure; bode(ss(S_i)); grid on; hold on; bode(usubs(S_i,u_i_wc),'m'); 
    bode(usubs(S_s,u_s_wc),'m--');
    bode(tf(M_i),'r--'); bode(tf(db2mag(-3)),'b--');
    figure; bode(ss(L_i)); grid on; hold on;
    figure; bode(ss(K_S_i)); grid on; hold on;
    figure; bode(K_eq); grid on; hold on;
    xlim(wtab)
    figure; bode(C_s_bar); grid on; 
else    
    close all; 
    figure; bode(ss(S_s)); grid on; hold on; 
    bode(tf(db2mag(-3)),'b--'); bode(usubs(S_s,u_s_wc),'m'); 
    figure; bode(ss(S_i)); grid on; hold on; 
    bode(tf(db2mag(-3)),'b--'); bode(tf(M_i),'r--');
    bode(usubs(S_i,u_i_wc),'m'); 
end