function [Delta,pe] = fcn_iqc_conv(BLKSTRUCT,NORMUNC,options)
%   FCN_IQC_CONV Convert uncertainty to IQC multiplier descriptions
%
%       Author: T.S.C. Pollack
%       Copyright (2024) T.S.C. Pollack

% Extract settings
npsi       = options.npsi;
bftype     = options.bftype;
ploc       = options.ploc;
type       = options.metric;
perf_idx   = options.perf_idx;
param_type = options.param_type;

% Get repeated block structure
BLKSTRUCT_rep = repelem({BLKSTRUCT.Name},[BLKSTRUCT.Occurrences]);

% Find Delta indices
act_idx   = find(strcmp(BLKSTRUCT_rep,'Delta_act'));
q_idx     = find(strcmp(BLKSTRUCT_rep,'Delta_q_m'));
alpha_idx = find(strcmp(BLKSTRUCT_rep,'Delta_alpha_m'));
nz_idx    = find(strcmp(BLKSTRUCT_rep,'Delta_nz_m'));

dZa_idx     = find(strcmp(BLKSTRUCT_rep,'dZa'));

dMa_idx     = find(strcmp(BLKSTRUCT_rep,'dMa'));
dMq_idx     = find(strcmp(BLKSTRUCT_rep,'dMq'));
dMde_idx    = find(strcmp(BLKSTRUCT_rep,'dMde'));

% Actuator uncertainty
if ~isempty(act_idx)
    Delta_act_ti_iqc = iqcdelta('Delta_act','InputChannel',act_idx,...
                        'OutputChannel',act_idx,'StaticDynamic','D',...
                        'NormBounds',1);          
    Delta_act_ti_iqc = iqcassign(Delta_act_ti_iqc,'ultid','Length', ...
                    npsi,'BasisFunctionType',bftype,'PoleLocation',ploc);  
else
    Delta_act_ti_iqc = {};
end

% Qb measurement uncertainty
if ~isempty(q_idx)
    Delta_q_m_ti_iqc = iqcdelta('Delta_q_m','InputChannel',q_idx,...
                        'OutputChannel',q_idx,'StaticDynamic','D',...
                        'NormBounds',1);          
    Delta_q_m_ti_iqc = iqcassign(Delta_q_m_ti_iqc,'ultid','Length',npsi,...
                    'BasisFunctionType',bftype,'PoleLocation',ploc);  
else
    Delta_q_m_ti_iqc = {};
end

% Alpha measurement uncertainty
if ~isempty(alpha_idx)
    Delta_alpha_m_ti_iqc = iqcdelta('Delta_alpha_m','InputChannel',...
                            alpha_idx,'OutputChannel',alpha_idx, ...
                            'StaticDynamic','D','NormBounds',1);          
    Delta_alpha_m_ti_iqc = iqcassign(Delta_alpha_m_ti_iqc,'ultid',...
                            'Length',npsi,'BasisFunctionType',bftype,...
                            'PoleLocation',ploc);    
else
    Delta_alpha_m_ti_iqc = {};
end

% Nz measurement uncertainty
if ~isempty(nz_idx)
    Delta_nz_m_ti_iqc = iqcdelta('Delta_nz_m','InputChannel',nz_idx,...
                            'OutputChannel',nz_idx,'StaticDynamic','D',...
                            'NormBounds',1);          
    Delta_nz_m_ti_iqc = iqcassign(Delta_nz_m_ti_iqc,'ultid','Length',...
                            npsi,'BasisFunctionType',bftype,...
                            'PoleLocation',ploc);   
else
    Delta_nz_m_ti_iqc = {};
end

% System parametric uncertainty (dZa)
if ~isempty(dZa_idx)
    
    clear Delta_p_iqc
    
    idx = dZa_idx;
    
    if strcmp(type,'L2')    
        Delta_p_iqc = iqcdelta('Delta_Za','InputChannel',idx,...
                        'OutputChannel',idx,'TimeInvTimeVar', ...
                        'TI','Bounds',NORMUNC{idx(1)}.Range);    
        Delta_p_iqc = iqcassign(Delta_p_iqc,'ultis','Length',npsi, ...
                        'RelaxationType','DG','BasisFunctionType', ...
                        bftype,'PoleLocation',ploc); 
    end
    
    Delta_Za_iqc = Delta_p_iqc;
    
else
    Delta_Za_iqc = {};
end

% System parametric uncertainty (dMa)
if ~isempty(dMa_idx)
    
    clear Delta_p_iqc
    
    idx = dMa_idx;
    
    if strcmp(type,'L2')      
                    
        if strcmp(param_type,'ltv_slow')
            
            Delta_p_iqc = iqcdelta('Delta_Ma','InputChannel',idx,...
                            'OutputChannel',idx,'TimeInvTimeVar', ...
                            'TV','Bounds',NORMUNC{idx(1)}.Range,...
                            'RateBounds',NORMUNC{idx(1)}.Ratebounds);   
                        
            Delta_p_iqc = iqcassign(Delta_p_iqc,'ultv_rb','Length',npsi,...
                            'RelaxationType','DG','BasisFunctionType',...
                            bftype,'PoleLocation',ploc);   
                        
        elseif strcmp(param_type,'ltv_fast')      
            
            La   = polydec(pvec('box',NORMUNC{idx(1)}.Range))';
            H{1} = 1;
            
            Delta_p_iqc = iqcdelta('Delta_Ma','InputChannel',idx,...
                            'OutputChannel',idx,'TimeInvTimeVar', ...
                            'TV','Structure','FB','UncertaintyMap',...
                            H,'Polytope',La);
                        
            Delta_p_iqc = iqcassign(Delta_p_iqc,'ultv',...
                                        'RelaxationType','DG');  
            
        elseif strcmp(param_type,'lti')
            
            Delta_p_iqc = iqcdelta('Delta_Ma','InputChannel',idx,...
                                'OutputChannel',idx,'TimeInvTimeVar', ...
                                'TI','Bounds',NORMUNC{idx(1)}.Range);    
                            
            Delta_p_iqc = iqcassign(Delta_p_iqc,'ultis','Length',npsi, ...
                            'RelaxationType','DG','BasisFunctionType', ...
                            bftype,'PoleLocation',ploc); 
        end
        
    end
    
    Delta_Ma_iqc = Delta_p_iqc;
    
else
    
    Delta_Ma_iqc = {};
    
end

% System parametric uncertainty (dMq)
if ~isempty(dMq_idx)
    
    clear Delta_p_iqc
    
    idx = dMq_idx;
    
    if strcmp(type,'L2') 
        if strcmp(param_type,'ltv_slow')
            
            Delta_p_iqc = iqcdelta('Delta_Mq','InputChannel',idx,...
                            'OutputChannel',idx,'TimeInvTimeVar', ...
                            'TV','Bounds',NORMUNC{idx(1)}.Range,...
                            'RateBounds',NORMUNC{idx(1)}.Ratebounds);  
                        
            Delta_p_iqc = iqcassign(Delta_p_iqc,'ultv_rb','Length',npsi,...
                            'RelaxationType','DG','BasisFunctionType',...
                            bftype,'PoleLocation',ploc);   
                        
        elseif strcmp(param_type,'ltv_fast') 
            
            La   = polydec(pvec('box',NORMUNC{idx(1)}.Range))';
            H{1} = 1;
            
            Delta_p_iqc = iqcdelta('Delta_Mq','InputChannel',idx,...
                            'OutputChannel',idx,'TimeInvTimeVar', ...
                            'TV','Structure','FB','UncertaintyMap',H,...
                            'Polytope',La);
                        
            Delta_p_iqc = iqcassign(Delta_p_iqc,'ultv',...
                                    'RelaxationType','DG');  
            
        elseif strcmp(param_type,'lti')
            
            Delta_p_iqc = iqcdelta('Delta_Mq','InputChannel',idx,...
                            'OutputChannel',idx,'TimeInvTimeVar', ...
                            'TI','Bounds',NORMUNC{idx(1)}.Range);  
                        
            Delta_p_iqc = iqcassign(Delta_p_iqc,'ultis','Length',npsi, ...
                            'RelaxationType','DG','BasisFunctionType', ...
                            bftype,'PoleLocation',ploc); 
        end   
    end
    
    Delta_Mq_iqc = Delta_p_iqc;
    
else
    
    Delta_Mq_iqc = {};
    
end

% System parametric uncertainty (dMde)
if ~isempty(dMde_idx)
    
    clear Delta_p_iqc
    
    idx = dMde_idx;
    
    if strcmp(type,'L2')
        if strcmp(param_type,'ltv_slow')
            
            Delta_p_iqc = iqcdelta('Delta_Mde','InputChannel',idx,...
                            'OutputChannel',idx,'TimeInvTimeVar', ...
                            'TV','Bounds',NORMUNC{idx(1)}.Range, ...
                            'RateBounds',NORMUNC{idx(1)}.Ratebounds);   
                        
            Delta_p_iqc = iqcassign(Delta_p_iqc,'ultv_rb','Length',npsi,...
                            'RelaxationType','DG','BasisFunctionType',...
                            bftype,'PoleLocation',ploc);   
                        
        elseif strcmp(param_type,'ltv_fast')       
            
            La   = polydec(pvec('box',NORMUNC{idx(1)}.Range))';
            H{1} = 1;
            
            Delta_p_iqc = iqcdelta('Delta_Mde','InputChannel',idx,...
                            'OutputChannel',idx,'TimeInvTimeVar', ...
                            'TV','Structure','FB','UncertaintyMap',H,...
                            'Polytope',La);
            Delta_p_iqc = iqcassign(Delta_p_iqc,'ultv',...
                                    'RelaxationType','DG');    
        elseif strcmp(param_type,'lti')
            Delta_p_iqc = iqcdelta('Delta_Mde','InputChannel',idx,...
                                'OutputChannel',idx,'TimeInvTimeVar', ...
                                'TI','Bounds',NORMUNC{idx(1)}.Range);   
            Delta_p_iqc = iqcassign(Delta_p_iqc,'ultis','Length',npsi, ...
                            'RelaxationType','DG','BasisFunctionType', ...
                            bftype,'PoleLocation',ploc); 
        end
    end
    
    Delta_Mde_iqc = Delta_p_iqc;
else
    
    Delta_Mde_iqc = {};
    
end
        
% Generate total Delta block
Delta = {Delta_act_ti_iqc, Delta_q_m_ti_iqc, Delta_alpha_m_ti_iqc, ...
        Delta_nz_m_ti_iqc, Delta_Za_iqc, Delta_Ma_iqc, Delta_Mq_iqc, ...
        Delta_Mde_iqc };
Delta = Delta(~cellfun('isempty',Delta));   

% Define performance block
if strcmp(type,'L2')
    
    pe = iqcdelta('pe','ChannelClass','P','InputChannel',...
        length(BLKSTRUCT_rep)+(perf_idx(2,1):perf_idx(2,2)), ...
        'OutputChannel',...
        length(BLKSTRUCT_rep)+(perf_idx(1,1):perf_idx(1,2)),...
        'PerfMetric','L2'); 
end
            
end