%==========================================================================
% [eq_reg.m] Get insight into equivalent regulation terms
%
%   Note: before using this script, an umbrella.mat file must be loaded
%
%   Author: T.S.C. Pollack
%   Copyright (2024) T.S.C. Pollack
%==========================================================================

close all; clc;

% Set file name
ffname = 'FIG_KEQ';

% Set file destination
fsdes = '_RESULTS/CLAW/';

% Specify theme
clr_order = [4,4,5];
st_order = {'-','-.','-'};

% Specify line width
LineWidthSetting = 3;

% Specify font size
font_size = 12;

% Save plot yes/no
save_plt = 0;

% Add legend yes/no
add_lgd = 1;

% Set frequency spectrum of interest
omega = logspace(-2,2,100);

%% --- MISC ---

% Set color theme
clr_theme = [ [0, 0.4470, 0.7410]; ...
            [0.8500, 0.3250, 0.0980]; ...
            [0.4660, 0.6740, 0.1880]; ...
            [0.4940, 0.1840, 0.5560]; ...  
            [0.6350, 0.0780, 0.1840]; ...
            [0.75, 0.75, 0.75]; ...
            [0.75, 0.75, 0]; ...
            [0, 0.75, 0.75]; ...
            [0.9290, 0.6940, 0.1250]];

%% --- GET DATA ---

% Get compensation gain
Kc = getBlockValue(K_CLAW,'Kc');

% Get compensator filter bandwidth
tau_c = getBlockValue(K_CLAW,'tau_f'); tau_c = tau_c.D;

% Get compensator filter
Hc = tau_c/(s+tau_c);

% Get Cc and equivalent regulator
Cc = 1/(1-Kc*Hc);
K_vc_eq = Cc*(K_vc + s*Kc*Hc);

% Get equivalent desired PI-gains
Ki_d = bode(K_vc_eq*s,1e-4);
Kp_d = bode(K_vc_eq,1e4);
K_vc_d = Kp_d + Ki_d/s;

% Get equivalent settings
Ki_vc = bode(K_vc*s,1e-4);
Kp_vc = bode(K_vc,1e4);
Kc_eq = 1 - ss(Ki_vc)/Ki_d
tau_eq = 1/Kc*(Kp_d-ss(Kp_vc))

%% --- PLOT ---

% Preliminaries        
FIG = figure; FIG.Position = [300 59 560 400]; sp1 = subplot(1,1,1);    

[K_VC_MAG,~] = bode(K_vc,omega); 
[K_VCE_MAG,~] = bode(K_vc_eq,omega);
[K_VCD_MAG,~] = bode(K_vc_d,omega);

semilogx(sp1,omega,mag2db(K_VC_MAG(:)),'color',clr_theme(clr_order(1),:),...
    'LineStyle',st_order{1},'LineWidth',LineWidthSetting);   
grid on; hold on; 
semilogx(sp1,omega,mag2db(K_VCE_MAG(:)),'color',clr_theme(clr_order(2),:),...
    'LineStyle',st_order{2},'LineWidth',LineWidthSetting); 
semilogx(sp1,omega,mag2db(K_VCD_MAG(:)),'color',clr_theme(clr_order(3),:),...
    'LineStyle',st_order{3},'LineWidth',LineWidthSetting);  

xlim([omega(1),omega(end)]); ylim([0,60]);
xlabel(sp1,'Frequency (rad/s)','Interpreter','latex','FontSize',font_size);
ylabel(sp1,'Magnitude (dB)','Interpreter','latex','FontSize',font_size);
set(sp1,'TickLabelInterpreter','latex','FontSize',font_size)

% Add legend
if add_lgd
    legend(sp1,'Virtual control','Equivalent virtual control',...
        'PI approximation','Orientation','vertical','NumColumns',1, ...
        'Interpreter','latex','Location','northeast','FontSize',font_size);
end

%% Save plot

FIG.Renderer = 'painters';

if save_plt
    saveas(FIG,strcat(fsdes,ffname,'.fig'));
    saveas(FIG,strcat(fsdes,ffname,'.png'));
    saveas(FIG,strcat(fsdes,ffname,'.emf'));
    saveas(FIG,strcat(fsdes,ffname),'epsc');
end