function [CL,fSoft,gHard,INFO] = systune_hybrid(CL0,K_CLAW,SoftReqs,HardReqs,options)
%   SYSTUNE_HYBRID Multi-objective robust H∞-synthesis routine for 
%   structured controllers against mixed uncertainty (based on SYSTUNE)
%
%       IMPORTANT: this version of systune_hybrid.m only accepts synthesis
%       models associated with a single flight condition.
%
%       Author: T.S.C. Pollack
%       Copyright (2024) T.S.C. Pollack

    % Initialize parameters 
    dim       = size(CL0);   
    n_ops     = [1,1]; % IMPORTANT: [DIM1,DIM2] are fixed
    n_type    = dim(end);    

    % Store IO names for later reference
    IONames.inputs  = CL0.InputName;
    IONames.outputs = CL0.OutputName;

    % Store SYSTUNE options
    tune_options = options.systune;
    
    % =*=*= MAIN LOOP =*=*=
    
    % Loop over number of random starts
    for p = 1:options.dksyn.RandomStart+1
        
        % 0. *** Perturb initial design parameters ***
        % Only for additional random starts
        if p > 1
            
            % Get design parameters
            Kfields = fields(get(CL0).Blocks);
            
            % Initialize loop
            ndp = length(Kfields);
            
            % Loop over design parameters
            for q = 1:ndp
                
               % Get design parameter
               dpp = getfield(get(CL).Blocks,Kfields{q});
               
                % Check if dpp is a tunable parameter
               if contains(class(dpp),'tunable')
                   
                   % Get if dpp is a free tunable parameter
                   if contains(class(dpp),'Gain')
                       if ~dpp.Gain.Free
                           continue
                       end
                   end
                   
                   % Check if an allowable range is specified
                   if contains(class(dpp),'Gain') 
                       ptb_flag = (dpp.Gain.Maximum ~= inf);
                   elseif contains(class(dpp),'TF') 
                       ptb_flag = any(dpp.Denominator.Maximum ~= inf);
                   end

                   % If an allowable range exists, randomly sample over
                   % this range
                   if ptb_flag
                       
                       % Gain tunable
                       if contains(class(dpp),'Gain') 
                           dpp = dpp.Gain.Maximum*rand(1);
                           
                       % TF tunable
                       elseif contains(class(dpp),'TF')
                           dpp_K = dpp.Numerator.Value(1)*rand(1);
                           dpp_d = dpp.Denominator.Maximum(end)*rand(1);
                           dpp   = dpp_K*tf([1 dpp_d],[1 dpp_d]);
                       end
                     
                   % If the range is unspecified, perturb current value 
                   % randomly up to a maximum factor as set by RandomPNorm
                   else

                       % Make state-space model for convenience
                       dpp = ss(dpp);
                       
                       % Perturb A, B, C, D matrices element-wise
                       ndpx  = size(dpp.A,1);
                       ndpu  = size(dpp.B,2);
                       ndpy  = size(dpp.C,1);

                       % A-matrix
                       dpp.A = dpp.A.*(ones(ndpx) + options.dksyn.RandomPNorm*...
                                                2*(rand(ndpx,ndpx)-0.5));

                       % B-matrix                     
                       dpp.B = dpp.B.*(ones(ndpx,ndpu) + options.dksyn.RandomPNorm*...
                                                2*(rand(ndpx,ndpu)-0.5));

                       % C-matrix
                       dpp.C = dpp.C.*(ones(ndpy,ndpx) + options.dksyn.RandomPNorm*...
                                                2*(rand(ndpy,ndpx)-0.5));

                       % D-matrix
                       dpp.D = dpp.D.*(ones(ndpy,ndpu) + options.dksyn.RandomPNorm*...
                                                2*(rand(ndpy,ndpu)-0.5));
                                            
                   end  
                   
                   % Set new design parameter dpp
                   CL0 = setBlockValue(CL0,Kfields{q},dpp);
                   
               end
            end
        end
        
        % === Initialize D-K iteration ===
        
        % Set initial closed-loop system
        CL = CL0;     
        
        % Initialize worst-case gamma tunable
        KSQGAMMA = tunableGain('SQGAMMA',1);
        KSQGAMMA.Gain.Minimum = 0;
        
        % Initialize uncertainty table
        wcup_tab  = zeros(4,1);
        
        % Set D-K termination conditions
        k = 0; count = 1; converged = 0;
        itmax = options.dksyn.n_dkiter;

        % =*=*= D-K LOOP =*=*=
        while ~converged && (k < itmax)
            
            % Increment counter
            k = k + 1;
            
            % Set gamma
            gamma_k = ss(KSQGAMMA).D^2;
            
            % Set block parameters
            CLk = CL;
            
            % Clear variables to prevent conflicts
            clear WCLk WCLGk
                
            % Loop over each model type (RP,RS,NP)
            for m = 1:n_type

                % Loop over each flight condition (DIM1)
                for i = 1:n_ops(1)

                    % Loop over each flight condition (DIM2)
                    for j = 1:n_ops(2)                   
                        
                        % *** A. Scale performance channels (U/P) ***
                        % Note: It is assumed that P corresponds to the 
                        % LAST rows and columns
                        blk        = options.dksyn.blk;
                        
                        % Robust performance constraint
                        if m == 1                            
                            LL = blkdiag(eye(sum(blk(1:end-1,1))), ...
                                  1/sqrt(gamma_k)*eye(sum(blk(end,2))));
                            LR = blkdiag(eye(sum(blk(1:end-1,2))), ...
                                  1/sqrt(gamma_k)*eye(sum(blk(end,1)))); 
                              
                        % Robust disk margin constraint
                        elseif m < n_type                       
                            LL = blkdiag(eye(sum(blk(1:end-1,1))), ...
                                  0*eye(sum(blk(end,2))));
                            LR = blkdiag(eye(sum(blk(1:end-1,2))), ...
                                  0*eye(sum(blk(end,1))));  
                              
                        % Nominal performance constraint
                        else        
                            LL = blkdiag(0*eye(sum(blk(1:end-1,1))), ...
                                  eye(sum(blk(end,2))));
                            LR = blkdiag(0*eye(sum(blk(1:end-1,2))), ...
                                  eye(sum(blk(end,1))));                              
                        end
                        
                        % Pre- and post-multiply
                        CLk_scaled = LL*CLk*LR;                        
                        
                        % *** B. Eliminate null channels for efficiency ***
                        
                        % Structure block indices
                        blk_idx_u = flip(cumsum(options.dksyn.blk,1),2);
                        blk_idx_l = blk_idx_u -(flip(options.dksyn.blk,2)-1);
                        
                        % Initialize
                        blk_mr     = [];
                        y_idx_mr   = [];
                        u_idx_mr   = [];
                        
                        % Get minimum U/P block realization
                        for l = 1:size(blk_idx_l,1)       
                            
                            y_idx = blk_idx_l(l,1):blk_idx_u(l,1);
                            u_idx = blk_idx_l(l,2):blk_idx_u(l,2);
                            
                            if getPeakGain(CLk_scaled(y_idx,u_idx,i,j,m)) ~= 0
                                blk_mr   = [blk_mr;blk(l,:)];
                                y_idx_mr = [y_idx_mr;y_idx'];
                                u_idx_mr = [u_idx_mr;u_idx'];
                            end
                            
                        end

                        % Set reduced-order CL system
                        CLk_mij_u = CLk_scaled(y_idx_mr,u_idx_mr,i,j,m);  
                        
                        % === (LOGGING) ===
                        
                        % Check if state space model is proper
                        if det(CLk_mij_u.E) == 0
                            disp('WARNING: E-matrix is singular');
                        end
                        
                        % Store number of blocks for later reference
                        n_blk_mr   = size(blk_mr,1);
                        n_u_blk_mr = n_blk_mr-1;    
                        
                        % *** C. Perform D-step ***
                        
                        % 1) Only perform if the number of U-blocks is nonzero
                        if n_u_blk_mr > 0
                        
                            % === Obtain D-scales ===
                            % Establish D-scalings for worst-case parametric
                            % uncertainty scenerio PER MODEL type; since
                            % SYSTUNE does only return a single (global)
                            % worst-case parameter set, we loop over all known
                            % scenarios to arrive at model-dependent scalings
                            
                            % Initialize
                            mmax_mu = 0;
                            
                            for w = 1:size(wcup_tab,2)

                                % Create param uncertainty structure
                                wcup.dMa  = wcup_tab(1,w); 
                                wcup.dMq  = wcup_tab(2,w);
                                wcup.dMde = wcup_tab(3,w);
                                wcup.dZa  = wcup_tab(4,w);

                                % Analyze for current worst-case parametric 
                                % uncertainty scenario
                                CLk_mij = ss(usubs(CLk_mij_u,wcup));

                                % Get D-scalings for uncertainty channels
                                [muw,logdw] = ssv(CLk_mij.A,CLk_mij.B,...
                                        CLk_mij.C,CLk_mij.D,options.dksyn.W,...
                                        blk_mr,'psv');

                                % Store worst-case tightest upper bound         
                                if max(muw) > mmax_mu    
                                    mmax_mu = max(muw); 
                                    mu = muw;
                                    logd = logdw;     
                                    CLk_mij_wc = CLk_mij;
                                end

                            end

                            % === D-scale fitting ===

                            % Remove spurious ssv estimates, if any
                            [clogd,cw_tab] = fcn_clean_logd(logd,options.dksyn.W);

                            % Initialize multipliers
                            LL = []; LR = [];

                            % Fill multipliers
                            for l = 1:n_blk_mr
                                    
                                % Rewrite to frequency response
                                logdf_i = frd(exp(clogd(l,:)),cw_tab);

                                % Multiplier fit
                                D_ii   = fitmagfrd(logdf_i,...
                                                options.dksyn.NDfit,[],mu);

                                % Assemble left and right multiplier matrices
                                LL = blkdiag(LL,D_ii*eye(blk_mr(l,2)));
                                LR = blkdiag(LR,inv(D_ii)*eye(blk_mr(l,1)));

                            end

                            % Scale relevant plant channels
                            WCLk(y_idx_mr,u_idx_mr,i,j,m) = LL*CLk(y_idx_mr,u_idx_mr,i,j,m)*LR;

                            % === (LOGGING) ===

                            % Log SSV values
                            INFO.MUtab{p,k,m,i,j} = max(mu);
                            INFO.MUDtab{p,k,m,i,j} = getPeakGain(LL*CLk_mij_wc*LR,...
                                                0.01,options.dksyn.W([1,end])); 
                              
                        % 2) If no U-blocks, set U-channels to zero for 
                        % efficiency of optimization                   
                        else
                            
                            % Omit complex uncertainty channels
                            ny   = size(CLk,1);
                            nu   = size(CLk,2);
                            nd   = 4; % IMPORTANT: hardcoded value
                            
                            LL   = eye(ny); LL(1:nd,1:nd) = 0;
                            LR   = eye(nu); LR(1:nd,1:nd) = 0;
                            WCLk(:,:,i,j,m) = LL*CLk(:,:,i,j,m)*LR;
                            
                        end
                            
                    end
                    
                end
                
                % *** D. Perform K-step ***
                
                % === Scale channels according to requirement type ===
                % Note: It is again assumed that P is specified as the 
                % LAST rows and columns
                
                % Robust performance objective
                if m == 1                            
                    LL = blkdiag(eye(sum(blk(1:end-1,1))), ...
                          1/KSQGAMMA*eye(sum(blk(end,2))));
                    LR = blkdiag(eye(sum(blk(1:end-1,2))), ...
                          1/KSQGAMMA*eye(sum(blk(end,1))));        

                    WCLGk(:,:,:,:,m) = blkdiag(LL,1)*...
                        blkdiag(WCLk(:,:,:,:,m),KSQGAMMA^2)*blkdiag(LR,1);  
                    
                % Robust disk margin constraint
                elseif m < n_type                       
                    LL = blkdiag(eye(sum(blk(1:end-1,1))), ...
                          0*eye(sum(blk(end,2))));
                    LR = blkdiag(eye(sum(blk(1:end-1,2))), ...
                          0*eye(sum(blk(end,1))));   

                    WCLGk(:,:,:,:,m) = blkdiag(LL,0)*...
                        blkdiag(WCLk(:,:,:,:,m),0)*blkdiag(LR,0);  
                   
                % Nominal performance constraint
                else     
                    
                    LL = blkdiag(0*eye(sum(blk(1:end-1,1))), ...
                          eye(sum(blk(end,2))));
                    LR = blkdiag(0*eye(sum(blk(1:end-1,2))), ...
                          eye(sum(blk(end,1))));   

                    WCLGk(:,:,:,:,m) = blkdiag(LL,0)*...
                        blkdiag(WCLk(:,:,:,:,m),0)*blkdiag(LR,0);              
                end

            end
            
            % Restore I/O names
            WCLGk.InputName = [IONames.inputs;'g_in'];
            WCLGk.OutputName = [IONames.outputs;'g_out'];
            
            % === Set worst-case parametric uncertainty scenarios ===
                        
            % Initialize loop
            ww = 0;
            HardReqs_k = HardReqs;
            
            % Fill WCU scenario tab + add to Requirement models
            for w = 1:size(wcup_tab,2)
                
                % Skip nominal model
                if isempty(find(wcup_tab(:,w),2))
                    continue
                end
                
                % Set parametric uncertainty
                wcup.dMa  = wcup_tab(1,w); 
                wcup.dMq  = wcup_tab(2,w);
                wcup.dMde = wcup_tab(3,w);
                wcup.dZa  = wcup_tab(4,w);
                
                % Increment counter
                ww = ww + 1;

                % Add scenario to model set
                for m = 1:n_type
                    
                    % Add scenario to scaled model set
                    WCLGk(:,:,:,:,m,ww+1) = usubs(WCLGk(:,:,:,:,m,1),wcup);
                    
                    % Add model index to requirement set 
                    % Important: this only works if [DIM1,DIM2] = [1,1]
                    HardReqs_k(m).Models = [HardReqs_k(m).Models; ...
                                        n_type*(ww)+HardReqs(m).Models];
                end      
                
            end
            
            % === SYSTUNE OPTIMIZATION ===
            
            % Store variables to verify convergence later
            gamma_k_1    = gamma_k;
            wcup_dim_k_1 = size(wcup_tab,2);

            % Optimize scaled plant
            [WCLGk,fSoft,gHard,ST_INFO,ST_LOG] = ...
                            systune(WCLGk,SoftReqs,HardReqs_k,tune_options);
                        
            % E. *** POST-PROCESSING *** 
            
            % Store design parameters returned by SYSTUNE
            CL     = setBlockValue(CL0,WCLGk);
            K_CLAW = setBlockValue(K_CLAW,CL);
        
            % Store worst-case gamma
            KSQGAMMA           = setBlockValue(KSQGAMMA,WCLGk);
            gamma_k            = ss(KSQGAMMA).D^2;
            INFO.GAMMAtab{p,k} = gamma_k;
            
            % Add new worst-case parametric scenario to previous scenario
            % set, if needed
            if ~isempty(ST_INFO.wcPert)
                
                % Expand know scenario table
                wcup_tab = [wcup_tab, [ST_INFO.wcPert.dMa; ...
                    ST_INFO.wcPert.dMq; ST_INFO.wcPert.dMde; ST_INFO.wcPert.dZa]];
                
                % Keep unique scenarios
                wcup_tab = unique(wcup_tab.','rows').';
            end
            
            % === (LOGGING) ===
            
            % Store controller and uncertainty table
            INFO.Ktab{p,k}   = K_CLAW;
            INFO.WCUtab{p,k} = wcup_tab;
            
            % Store synthesis info
            INFO.ST_INFO{p,k} = ST_INFO;      
            INFO.ST_LOG{p,k}  = ST_LOG; 
            
            % ===  Determine convergence status ===
            gamma_flag = (abs(gamma_k_1/gamma_k) - 1) < options.dksyn.eps;
            wcup_flag  = (size(wcup_tab,2) == wcup_dim_k_1);
            
            % Check if convergence conditions hold over multiple iterations
            if gamma_flag && wcup_flag                
                count = count + 1;
                converged = (count == 3);                
            else
                count  = 1;
                converged = 0;
            end

        end
        
    end
    
end