%==========================================================================
% [gen_syn_plant.m] Set simulation settings and generate (uncertain) 
% plant dynamics 
%
%   Author: T.S.C. Pollack
%   Copyright (2024) T.S.C. Pollack
%==========================================================================

%% --- PRELIMINARIES ---

% Load airframe model and trim info
load(strcat(mdl_pdir,'\',fname_mdl,'.mat'));
load(strcat(mdl_pdir,'\',fname_mdl,'.tri'),'-mat');

% Get trim airspeed
V0 = x0(4);

% C* settings
knz = 1; kq = 12.4; % [-]

% Simulation rate
delta_T = 0.01;

% Misc parameters
g      = 9.81; % m/s^2

%% --- SET PERFORMANCE & UNCERTAINTY WEIGHTS ---

% Uncertainty weights
W_act   = options.unc.Wunc_act;   W_act_flag   = (hinfnorm(W_act) > 0);
W_q     = options.unc.Wunc_q;     W_q_flag     = (hinfnorm(W_q) > 0);
W_alpha = options.unc.Wunc_alpha; W_alpha_flag = (hinfnorm(W_alpha) > 0);
W_nz    = options.unc.Wunc_nz;    W_nz_flag    = (hinfnorm(W_nz) > 0);

% Set performance weights
[wte_mcv_num,wte_mcv_den] = tfdata(options.dksyn.weights.Wperf_te_mcv);
W_te = options.dksyn.weights.Wperf_te_mcv;
W_u  = options.dksyn.weights.Wperf_u;

%% --- CREATE BASELINE MODEL ---

% Nominal actuator model
act_w_n  = 20; 
act_zeta = sqrt(1/2);
act_del  = 0; 

H_act = tf(act_w_n^2,[1 2*act_zeta*act_w_n act_w_n^2]);

% Create short-period model
A_lin_sp = b747_lin.A([2,5],[2,5]);
B_lin_sp = b747_lin.B([2,5],[17:21,26:29]);
C_lin_sp = b747_lin.C([2,5,69],[2,5]);
D_lin_sp = b747_lin.D([2,5,69],[17:21,26:29]);

% Rename model
B747_SYS = ss(A_lin_sp,B_lin_sp,C_lin_sp,D_lin_sp);

% Retrieve state-space matrices
[B747_SYS_A,B747_SYS_B,B747_SYS_C,B747_SYS_D] = ssdata(B747_SYS);

% Control allocation: gang elevators + engine thrust
B747_SYS_B = [sum(B747_SYS_B(:,1:4,:,:),2),B747_SYS_B(:,5,:,:),...
                                            sum(B747_SYS_B(:,6:9,:,:),2)];
B747_SYS_D = [sum(B747_SYS_D(:,1:4,:,:),2),B747_SYS_D(:,5,:,:),...
                                            sum(B747_SYS_D(:,6:9,:,:),2)];

% Short-period simplification: set B-matrix terms for alpha to zero
B747_SYS_B(2,:) = 0*B747_SYS_B(2,:);

% Add input columns for exogenous uncertainties/disturbances
B747_SYS_B = [B747_SYS_B,zeros([2,2])];
B747_SYS_D = [B747_SYS_D,zeros([3,2])];
    
% Define alpha disturbance inputs
Ma = B747_SYS_A(1,2); Za = B747_SYS_A(2,2);
Mg = -Ma*options.dist.a_g*pi/180; 
Zg = -Za*options.dist.a_g*pi/180; 
Wg = options.dist.W_g;

% Define additive disturbance terms
dM = options.dist.w_M*pi/180;  
dZ = options.dist.w_Z*1/(-V0/g*Za);

% Get ICR location 
x_s = fcn_return_icr(B747_SYS,V0);

% Set performance station
x_p = x_s;

%% --- CREATE UNCERTAIN BARE AIRFRAME MODEL ---

% Define parametric uncertainties
if options.unc.dMa ~= 0
    duMa = ureal('dMa',0,'Range',[-1,1]*options.unc.dMa);
else
    duMa = 0; dpMa = 0;
end

if options.unc.dMq ~= 0
    duMq = ureal('dMq',0,'Range',[-1,1]*options.unc.dMq);
else
    duMq = 0; dpMq = 0;
end

if options.unc.dMde ~= 0
    duMde = ureal('dMde',0,'Range',[-1,1]*options.unc.dMde);
else
    duMde = 0; dpMde = 0;
end

if options.unc.dZa ~= 0
    duZa = ureal('dZa',0,'Range',[-1,1]*options.unc.dZa);
else
    duZa = 0; dpZa = 0;
end

if options.unc.dZq ~= 0
    duZq = ureal('dZq',0,'Range',[-1,1]*options.unc.dZq);
else
    duZq = 0; dpZq = 0;
end

if options.unc.dZde ~= 0
    duZde = ureal('dZde',0,'Range',[-1,1]*options.unc.dZde);
else
    duZde = 0; dpZde = 0;
end

% Set multiplicative/additive uncertainty matrices
duA = [ duMq,   duMa; 
        duZq,   duZa]; 
duB = [ duMde, zeros(1,2), Mg, dM; 
        duZde, zeros(1,2), Zg, dZ]; 
dC  = zeros(3,2);
dD  = zeros(3,5);

% Add to nominal plant description
n = size(B747_SYS_A,2);
m = size(B747_SYS_B,2);
p = size(B747_SYS_C,1);

B747_USYS_A = umat(zeros(size(B747_SYS_A)));
B747_USYS_B = umat(zeros(size(B747_SYS_B)));
B747_USYS_C = umat(zeros(size(B747_SYS_C)));
B747_USYS_D = umat(zeros(size(B747_SYS_D)));

% Create uncertain state-space matrices
for i = 1:n
    for j = 1:n
        B747_USYS_A(i,j,:,:) = B747_SYS_A(i,j,:,:)*(1+duA(i,j));
        if j == 1
            for k = 1:p
                B747_USYS_C(k,i,:,:) = B747_SYS_C(k,i,:,:) + dC(k,i);     
            end
        end
    end
    for j = 1:m
        B747_USYS_B(i,j,:,:) = B747_SYS_B(i,j,:,:)*(1+duB(i,j));  
        if i == 1
            for k = 1:p
                B747_USYS_D(k,j,:,:) = B747_SYS_D(k,j,:,:) + dD(k,j);         
            end
        end     
        % Note: exogenous disturbance is additive
        if j >= m-1            
            B747_USYS_B(i,j,:,:) = B747_SYS_B(i,j,:,:) + duB(i,j);  
        end
    end
end

% Redefine load factor output measurement
UZa = B747_USYS_A(2,2,:,:);
B747_USYS_C(end,:,:,:) = B747_USYS_C(2,:,:,:)*-UZa*V0/g;
B747_USYS_D(end,:,:,:) = B747_USYS_D(end,:,:,:)*0; 

% Add difference between performance and ICR station nz signal
B747_USYS_C(end+1,:,:,:) = (x_p-x_s)/g*B747_USYS_A(1,:,:,:); 
B747_USYS_D(end+1,:,:,:) = (x_p-x_s)/g*B747_USYS_B(1,:,:,:); 

% Assemble uncertain system
B747_USYS = simplify(ss(B747_USYS_A,B747_USYS_B,...
                            B747_USYS_C,B747_USYS_D),'full');

% Create correponding nominal system description
B747_NSYS = ss(B747_USYS);

% Augment with input disturbance weight 
B747_USYS = B747_USYS*blkdiag(eye(3),Wg,1);