%==========================================================================
% [gen_syn_claw.m] Set/generate control law and generalized plant
% interconnection block elements
%
%   Author: T.S.C. Pollack
%   Copyright (2024) T.S.C. Pollack
%==========================================================================

%% --- PRELIMINARIES  ---

% Define Laplace operator
s = tf('s'); 

% Set reference filter 
H_des = w0^2/w3*(s+w3)/(s^2+2*D*w0*s+w0^2);
[num_tf_ddq,den_tf_ddq] = tfdata(H_des);

% Set digital artefacts
Ts      = 1/80;
H_del   = pade(exp(-Ts*s),1);
H_ZOH   = (1 - pade(exp(-Ts*s),1))/s*1/Ts; 
H_alias = tf(150,[1 150]); 

%% --- GENERATE CLAW BLOCK ELEMENTS ---

% Define pre-filter
Gq  = B747_NSYS(1,1);
Gnz = B747_NSYS(3,1);
D_pf = -zero(zpk(minreal(Gnz + kq*Gq)));
N_pf = w3;
K_pf = tf([1/N_pf 1], [1/D_pf 1]);

% On-board benchmark model settings - based on NOMINAL model
K_obm_M_q  = ss(B747_SYS_A(1,1,:,:));
K_obm_M_a  = ss(B747_SYS_A(1,2,:,:));
K_obm_M_de = ss(B747_SYS_B(1,1,:,:));

K_obm = [K_obm_M_q,K_obm_M_a,K_obm_M_de];
K_obm_d = blkdiag(K_obm_M_q,K_obm_M_a,K_obm_M_de);

% Control allocation gains
if any(strfind(options.claw.mode,'hodc'))
    K_ca  = 1/K_obm_M_de;
else
    K_ca  = 1/K_obm_M_de/kq;
end

% C* dependencies
T_th2_inv = -ss(B747_SYS_A(2,2,:,:));
Vtas      = ss(V0);

if strcmp(options.claw.aoa_mode,'est_q')                
    K_mcvd = blkdiag(T_th2_inv, Vtas/g*T_th2_inv, T_th2_inv);  
elseif strcmp(options.claw.aoa_mode,'est_nz')    
    K_mcvd = blkdiag(T_th2_inv, Vtas/g*T_th2_inv, g/Vtas*1/T_th2_inv);  
else    
    K_mcvd = blkdiag(T_th2_inv, Vtas/g*T_th2_inv);                            
end

% Assemble inversion K-block with fixed DPs
K_INV = blkdiag(K_mcvd,K_obm_d,K_ca);

% Assemble K-block with tunable TPs
K_CLAW = append(K_ff,K_vc,K_qdot,K_syn);

%% --- GENERATE GENERALIZED PLANT DESCRIPTIONS ---

if any(strfind(options.claw.mode,'hort'))
    mdl_name = 'ICOL_IMF_INDI_MOD';
elseif any(strfind(options.claw.mode,'hodc'))
    mdl_name = 'ICOL_IMF_INDI_QMCV';
elseif strfind(options.claw.mode,'pi') || strfind(options.claw.mode,'pid') 
    mdl_name = 'ICOL_PID_PF';
end

% Obtain robust performance model
NSTRUCT              = fcn_gen_nstruct('p');
IC_OL                = linearize(mdl_name);
[UCL_P,CL_P,Delta_c] = fcn_IC_LFT(IC_OL, B747_USYS, K_INV, K_CLAW);

% Obtain nominal performance model and tighten weights
[~,CL_NP,~] = fcn_IC_LFT(IC_OL, B747_NSYS, K_INV, K_CLAW);

% Tighten weights on pilot stick input channel
CL_NP(end-5-2:end-5-1,end-2) = options.dksyn.weights.Wperf_NP_K* ...
                                            CL_NP(end-5-2:end-5-1,end-2);

% Create 'uncertain' formulation as well                                        
UCL_NP = CL_NP(5:end,5:end);

% Disk margin model - actuator
NSTRUCT              = fcn_gen_nstruct('act');
IC_OL                = linearize(mdl_name);
[UCL_NACT,CL_NACT,~] = fcn_IC_LFT(IC_OL, B747_USYS, K_INV, K_CLAW);

% Disk margin model - input
NSTRUCT          = fcn_gen_nstruct('u');
IC_OL            = linearize(mdl_name);
[UCL_NU,CL_NU,~] = fcn_IC_LFT(IC_OL, B747_USYS, K_INV, K_CLAW);

% Disk margin model - pitch rate
NSTRUCT          = fcn_gen_nstruct('q_m');
IC_OL            = linearize(mdl_name);
[UCL_NQ,CL_NQ,~] = fcn_IC_LFT(IC_OL, B747_USYS, K_INV, K_CLAW);

% Disk margin model - angle of attack
NSTRUCT                  = fcn_gen_nstruct('alpha_m');
IC_OL                    = linearize(mdl_name);
[UCL_NALPHA,CL_NALPHA,~] = fcn_IC_LFT(IC_OL, B747_USYS, K_INV, K_CLAW);

% Disk margin model - normal acceleration
NSTRUCT            = fcn_gen_nstruct('nz_m');
IC_OL              = linearize(mdl_name);
[UCL_NNZ,CL_NNZ,~] = fcn_IC_LFT(IC_OL, B747_USYS, K_INV, K_CLAW);

% Reset NSTRUCT
NSTRUCT              = fcn_gen_nstruct('p');

% Create model arrays (MA)
CL_MA(:,:,:,:,1) = CL_P;
CL_MA(:,:,:,:,2) = CL_NACT;
CL_MA(:,:,:,:,3) = CL_NU;
CL_MA(:,:,:,:,4) = CL_NQ;
CL_MA(:,:,:,:,5) = CL_NALPHA;
CL_MA(:,:,:,:,6) = CL_NNZ;
CL_MA(:,:,:,:,7) = CL_NP;

UCL_MA(:,:,:,:,1) = UCL_P;
UCL_MA(:,:,:,:,2) = UCL_NACT;
UCL_MA(:,:,:,:,3) = UCL_NU;
UCL_MA(:,:,:,:,4) = UCL_NQ;
UCL_MA(:,:,:,:,5) = UCL_NALPHA;
UCL_MA(:,:,:,:,6) = UCL_NNZ;
UCL_MA(:,:,:,:,7) = UCL_NP;
    
if strcmp(options.dksyn.mode,'inner')
    
    % Obtain synthesis model
    if any(strfind(options.claw.mode,'hort'))
        mdl_name = 'ICBL_IMF_INDI_MOD';
    elseif any(strfind(options.claw.mode,'hodc'))
        mdl_name = 'ICBL_IMF_INDI_QMCV';
    elseif strfind(options.claw.mode,'pi') || strfind(options.claw.mode,'pid') 
        mdl_name = 'ICBL_PID_PF';
    end
    
    IC_BL            = linearize(mdl_name);    
    [UBL,BL,Delta_c] = fcn_IC_LFT(IC_BL, B747_USYS, K_INV, K_CLAW);
    
    % Connect analysis points
    AP_act     = AnalysisPoint('AP_act');
    AP_es      = AnalysisPoint('AP_es');
    AP_q_m     = AnalysisPoint('AP_q_m');
    AP_alpha_m = AnalysisPoint('AP_alpha_m');
    AP_nz_m    = AnalysisPoint('AP_nz_m');
    APs        = blkdiag(AP_act,AP_es,AP_q_m,AP_alpha_m,AP_nz_m);

    UCL   = lft(APs,UBL);
    CL   = lft(APs,BL);    
    
end
