function [W_out] = fcn_transform_uweight(W_in,order,varargin)
%   FCN_TRANSFORM_UWEIGHT Returns transformed uncertainty weight; this new
%   weight adds additional high-frequency gain to overcome roll-off induced
%   by multiplication with nominal system
%
%       Author: T.S.C. Pollack
%       Copyright (2024) T.S.C. Pollack

    % Get mode
    if numel(varargin) == 1
        mode = varargin{1};
        w_co = [];
    elseif numel(varargin) == 2
        mode = varargin{1};
        w_co = varargin{2};
    elseif isempty(varargin)
        mode = '';
        w_co = [];
    end

    % Get zpk data
    [Z,P,K] = zpkdata(W_in);

    if strcmp(mode,'minreal')
        if order == 1

            % Get real pole to cancel
            P_elim_idx = find(~imag(P{1}));
            P_red_idx  = find(imag(P{1}));
            P_red = P{1}(P_red_idx);
            K_red = K/abs((P{1}(P_elim_idx)));
            
            % Set HF cut-off frequency
            if isempty(w_co)
               w_co = abs((P{1}(P_elim_idx)))*10;
            end

            % Return output weight
            W_out = zpk(Z,{P_red},K_red)*tf(w_co,[1 w_co]);

        elseif order == 2

            % Get last imaginary pole pair to cancel
            P_elim = P{1}(end-1:end);
            P_red  = P{1}(1:end-2);
            K_red = K/prod(P_elim);
            
            % Set HF cut-off frequency
            if isempty(w_co)
               w_co = sqrt(prod(P_elim))*10;
            end

            % Return output weight
            W_out = zpk(Z,{P_red},K_red)*tf(w_co^2,[1 2*sqrt(1/2)*w_co w_co^2]);

        end
    else
        
        % Get number of poles
        n_p = length(P{1});
        
        % Get HF level-off frequency
        w_lo = (abs(prod(P{1})))^(1/n_p);  
            
        % Set HF cut-off frequency
        if isempty(w_co)
           w_co = w_lo*10;
        end      
        
        if order == 1
            
            % Return output weight
            W_out = minreal(W_in*tf(w_co,[1 w_co])/tf(w_lo,[1 w_lo]));

        elseif order == 2
            
            % Return output weight
            W_out = minreal(W_in*tf(w_co^2,[1 2*sqrt(1/2)*w_co w_co^2])...
                                        /tf(w_lo^2, [1 2*w_lo^2 w_lo]));

        end        
        
    end

end

