# ################################################
#   
#   geomBasicOperations.py -basic geos based geometry operations
#
#    --------
#
#    Copyright (c) 2012 - N.J. Hardebol - N.J.Hardebol@tudelft.nl  - (a)
#     (a) Delft University of Technology, Department of Geotechnology,  Stevinweg 1, 2628 CN Delft, the Netherlands
#
#   ---------
# 
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see http://www.gnu.org/licenses/.
#
# #############################################

"""The WindowCursor Operations module provide classe(s) and (member) functions for window cursor operations:
    place window within digitizing surface, walk in given direction (rows and columns) with given step size, perform and collect (spatial/stistical calculations
"""

import os , sys

import logging

import DF_Vector

import numpy
import math

##from PyQt4 import QtCore

try:
    import qgis.core as QgsCore
    import shapely
    from shapely import geos
    from shapely import wkt
    loaded_QgsCore = True
except:
    loaded_QgsCore = False
try:
    from digifract import dgfcore as  DgfCore
    loaded_DgfCore = True
except:
    loaded_DgfCore = False
#try: 
#    import OCC
#    loaded_OCC = True
#except: 
#    loaded_OCC = False


logger = logging.getLogger("DigiFractLogger.geomBasicOperations")

import __builtin__
    
def gen_Line_fromBPoint( basePntGeom,  baseLine_azim, baseLine_length  ):
    """support function for generating a line geometry from basePoint, azimuth and length parameters"""
    
    #basePntGeom = basePoint
    #-- NJH 03-08-2013: not sure if this function should check if basePoint is correct type -- it will produce error if not
    #--               it might be too memory expensive to be used in stochastic simulations 
    if not type( basePntGeom ) is QgsCore.QgsPoint:
        basePntGeom = QgsCore.QgsPoint( basePntGeom[0],  basePntGeom[1] )
    #    basePntGeom = basePoint
    #elif type( basePoint ) is DgfCore.DgfVertex:
    #    basePntGeom = basePoint
    #elif type( basePoint) is types.StringType:
    #    basePntGeom = _DgfGeometry.fromWkt( basePoint )
    #else: 
    #    print "no correct basePoint: ",  basePoint
    #    return False

    AzimRad = math.radians(  baseLine_azim )
    ##PI = math.pi
    dx =  baseLine_length * math.sin(AzimRad) ; dy = baseLine_length * math.cos(AzimRad)

    endPntGeom = QgsCore.QgsPoint( basePntGeom[0] +dx, basePntGeom[1] +dy)
    logger.debug("in gen_Line_fromBPoint() with basePntGeom: %s and endPntGeom: %s" % (basePntGeom, endPntGeom  ))
    baseLineGeom =  QgsCore.QgsGeometry.fromPolyline( [ basePntGeom , endPntGeom ]  )

    return baseLineGeom

def gen_Line_fromCPoint( cPoint,  baseLine_azim, lineLength  ):

    #-- NJH 03-08-2013: not sure if this function should check if basePoint is correct type -- it will produce error if not
    #--               it might be too memory expensive to be used in stochastic simulations 
    #if not isinstance( cPoint , QgsCore.QgsPoint) and not isinstance( cPoint , DgfCore.DgfVertex ):
    #    return False

    AzimRad = math.radians(  baseLine_azim )
    dx =  0.5*lineLength * math.sin(AzimRad) ; dy = 0.5*lineLength * math.cos(AzimRad)

    basePntGeom = QgsCore.QgsPoint( cPoint[0] -dx, cPoint[1] -dy)
    endPntGeom  = QgsCore.QgsPoint( cPoint[0] +dx, cPoint[1] +dy)
    lineGeom       =  QgsCore.QgsGeometry.fromPolyline(  [ basePntGeom , endPntGeom ]  )

    return lineGeom

##def gen_choppedProjLine(UnitBoundGeom ,  TraceLineGeom):
def crop_PLine_byPGon( pLineGeom,  PGonCropper ):
    if PGonCropper.intersects(pLineGeom):
        cropped_pLine  = pLineGeom.intersection( PGonCropper )
        return cropped_pLine
    else:  return None

def crop_PolylineSet_byPolygon( PolylineSet,  PolygonCropper ):
    """This is the new cropper of a collection of DgfGeometries of type polylines of which a subset is made and corpeed within the PolygonCropper"""
    #-- NJH 05-12-2013: different from the crop_PolylineSet_byPolygon_old implementation, this function excecutes the QgsGeometry.intersects() and QgsGeometry.intersection instead of of converting the shapely and executing shapely functions. 
    ##logger.debug( "in crop_PolylineSet_byPolygon(..) with len(PolylineSet): %s" %  len(PolylineSet) )
    ##logger.debug( "in crop_PolylineSet_byPolygon(..) with PolygonCropper: %s" % PolygonCropper )
    croppedPolylineGeoms = list()
    ##_croppedPolylineGeoms = list()
    
    logger.debug("in the new crop_PolylineSet_byPolygon with PolygonCropper.type(): %s and QgsCore.QGis.Polygon: %s" % ( PolygonCropper.type(), QgsCore.QGis.Polygon ))
    if not PolygonCropper.type() == QgsCore.QGis.Polygon:
        return croppedPolylines
    
    if PolygonCropper.area > 0:
        for pLine in PolylineSet:
            if isinstance( pLine , QgsCore.QgsGeometry):
                if pLine.intersects( PolygonCropper ):
                    croppedPolylineGeoms.append ( pLine.intersection(PolygonCropper) )
                    ##_croppedPolylines.append(pLineGeom.intersection(PolygonCropper) )
            else:
                logger.debug("in crop_PolylineSet_byPolygon with failure as type(pLine): %s is NONE of the supported types" % type(pLine) )
    
    return croppedPolylineGeoms

def crop_PolylineSet_byPolygon_old( PolylineSet,  PolygonCropper ):
    
    ##logger.debug( "in crop_PolylineSet_byPolygon(..) with len(PolylineSet): %s" %  len(PolylineSet) )
    ##logger.debug( "in crop_PolylineSet_byPolygon(..) with PolygonCropper: %s" % PolygonCropper )
    croppedPolylines = list()
    _croppedPolylines = list()
    
    if isinstance( PolygonCropper ,  QgsCore.QgsGeometry ):
        PolygonCropper = wkt.loads( str( PolygonCropper.exportToWkt()  ) )
    elif isinstance( PolygonCropper ,  str): 
        PolygonCropper   = shapely.wkt.loads( str( PolygonCropper ) )
    else:
        if not type(PolygonCropper) == shapely.geometry.Polygon():
            return croppedPolylines
    
    if PolygonCropper.area > 0:
        for pLine in PolylineSet:
            if isinstance( pLine , QgsCore.QgsGeometry):
                pLineGeom = wkt.loads( str( pLine.exportToWkt() ) )
                if pLineGeom.intersects( PolygonCropper ):
                    croppedPolylines.append ( pLineGeom.intersection(PolygonCropper) )
                    _croppedPolylines.append(pLineGeom.intersection(PolygonCropper) )
            elif isinstance( pLine ,  str ): 
                pLineGeom   = shapely.wkt.loads( str( pLine ) )
                if pLineGeom.intersects( PolygonCropper ):
                    croppedPolylines.append ( pLineGeom.intersection(PolygonCropper) )
                    _croppedPolylines.append( pLineGeom.intersection(PolygonCropper) )
            elif isinstance(pLine ,  shapely.geometry.Polygon ):
                if pLine.intersects( PolygonCropper ):
                    croppedPolylines.append (  pLine.intersection(PolygonCropper) )
            elif hasattr(pLine, "wktGeometry" ) and hasattr(pLine, "setGeometry" ):
                #elif type(pLine) == DigiFractLib.baseclasses.DF_Feature:
                #-- in case the PolylineSet is a FeatureSet with Polylines at feature.geometry()
                pLineGeom = shapely.wkt.loads( pLine.wktGeometry  )
                if pLineGeom.is_valid:
                    if pLineGeom.intersects( PolygonCropper ):
                        croppedPolyLine = pLineGeom.intersection(PolygonCropper)
                        pLine.setGeometry(QgsCore.QgsGeometry.fromWkt(croppedPolyLine.wkt ))
                        croppedPolylines.append(pLine)
                        _croppedPolylines.append(croppedPolyLine)
            else:
                logger.debug("in crop_PolylineSet_byPolygon with failure as type(pLine): %s is NONE of the supported types" % type(pLine) )
    
    return croppedPolylines

#-- NJH 05-08-2013: should be same as DgfGeometry.bgnPoint property function
##def get_lineLeftEndPoint(self.fracture)
#-- NJH 05-08-2013: should be same as DgfGeometry.endPoint property function
##def get_lineRightEndPoint(self.fracture)

    
def gen_endPointExtrusion(_geom,  _params):
    _vertices = _geom.toPolyline()
    
    side2extend = _params[0]
    style = _params[1]  #-- NJH 24-07_2015: what is this style parameter?
    extlength = _params[2]
    if len(_params) == 3: 
        _extVector = _params[3]
    else: _extVector = None
    #_intersectTolerance = extensionParams[4] ; _overlapTolerance = extensionParams[5]
    
    if side2extend == 0:
        if not _extVector:
            _extVector = DgfCore.DgfVertex( _vertices[0].x() - _vertices[1].x() , _vertices[0].y() - _vertices[1].y() )
        _l = numpy.power(numpy.power(_extVector.x(),  2) + numpy.power( _extVector.y(), 2) , 0.5)
        _extVector = DgfCore.DgfVertex( _l * _extVector.x() ,  _l * _extVector.y() )
        _newPnt    = DgfCore.DgfVertex( _vertices[0]. x() + extlength *  _extVector.x() ,  _vertices[0]. y() + extlength *  _extVector.y()  )
        _geom.insertVertex(0 ,  _newPnt)
        return _geom
    elif side2extend == 1:
        lastIdx = len(_vertices)-1
        if not _extVector: 
            _extVector = DgfCore.DgfVertex( _vertices[lastIdx].x() - _vertices[lastIdx-1].x() , _vertices[lastIdx].y() - _vertices[lastIdx-1].y() )
        _l = numpy.power(numpy.power(_extVector.x(),  2) + numpy.power( _extVector.y(), 2) , 0.5)
        _extVector = DgfCore.DgfVertex( _l * _extVector.x() ,  _l * _extVector.y() )
        _newPnt    = DgfCore.DgfVertex( _vertices[lastIdx]. x() + extlength *  _extVector.x() ,  _vertices[lastIdx]. y() + extlength *  _extVector.y()  )
        _geom.insertVertex(lastIdx+1 ,  _newPnt)
        return _geom
    
def extend_curve_to_point ( _geom,  point ):
    pass
    
    
#from shapely import ops
def translate( _geom,  _transVector ):
    ##logger.debug("in translate with _geom: %s" % _geom)
    ##logger.debug("in translate with _transVector: %s %s %s" % ( _transVector,  _transVector.x(),  _transVector.y() ) )
    #shapelyGeom = shapely.wkt.loads( str(_geom.exportToWkt()) )
    #_transgeom = shapely.ops.transform( lambda x, y, z=None: ( x + _transVector.x() , y + _transVector.y() ) , shapelyGeom )
    #_transgeom = QgsCore.QgsGeometry().fromWkt(_transgeom.wkt)
    _transgeom =_geom
    _transgeom.translate(_transVector.X(), _transVector.Y() )
    return _transgeom
    
def rotate( _geom,  _rotateAngle,  _rotateAxis='center',  use_radians=True ):
    import shapely.affinity as shapelyAff
    shapelyGeom = shapely.wkt.loads( str( _geom.exportToWkt()) )
    _rotatedgeom = shapelyAff.rotate( shapelyGeom,  _rotateAngle, _rotateAxis, use_radians )
    _rotatedgeom = QgsCore.QgsGeometry().fromWkt( _rotatedgeom.wkt )
    return _rotatedgeom

#-- NJH 16-08-2013: compare with IntersectCurve2CurveList() in rhinoLib.IntersectCurve2Curve
def intersection_curve_with_curve( curveA, curveB, input_params=None ):
    '''Calculate the intersection'''
    if not curveA or not curveB: None,  None
    #intersection_tolerance = input_params[0]
    #overlap_tolerance       = input_params[1]
    #addPoint2curve          = input_params[2]
    #splitCurve4Point         = input_params[3]
    
    pntList    = list()
    
    ##if curveA.touches(curveB):
    ##    _curveA = shapely.wkt.loads( str(curveA.exportToWkt()) )
    ##    _curveB = shapely.wkt.loads( str(curveB.exportToWkt() ) ) 
    ##    _return = _curveA.intersection(_curveB)
    ##    _geom = QgsCore.QgsGeometry()
    ##    _return = _geom.fromWkt(_return.wkt)
    ##else:
    
    _return = curveA.intersection(curveB)
    
    if  _return.type() == 0:   #-- QgsCore.QgsPoint  has numerator 0
        if _return.isMultipart():
            _pntList = _return.asGeometryCollection()
        else:
            _pntList = [_return]
        pntList.extend( _pntList )

    return pntList

def split_curve_and_curve( curveA, curveB, input_params=None ):
    _segmList = list()
    
    _return = curveA.difference(curveB)    
    if  _return.type() == 1:
        if _return.isMultipart():
            __segmList = _return.asGeometryCollection()
        else:
            __segmList = [_return ]
        _segmList.extend( __segmList )
    
    _return = curveB.difference(curveA)    
    if  _return.type() == 1:
        if _return.isMultipart():
            __segmList = _return.asGeometryCollection()
        else:
            __segmList = [_return ]
        _segmList.extend( __segmList )
        
    return _segmList

def split_curve_by_curve( curveA, curveB, input_params=None ):
    
    """""if curveA.touches(curveB):
        logger.debug("in split_curve_by_curve(..) curves touch with curveA.exportToWkt(): %s and curveB.exportToWkt(): %s" %( curveA.exportToWkt() ,  curveB.exportToWkt()))
        g = QgsCore.QgsGeometry()
        _curveA = shapely.wkt.loads( str(curveA.exportToWkt()) )
        _curveB = shapely.wkt.loads( str(curveB.exportToWkt()) ) 
        xpnt = g.fromWkt( _curveA.intersection(_curveB).wkt )
        curveApnts = curveA.asPolyline()
        curveBpnts = curveB.asPolyline()
        if not already_in_GeomList( xpnt,  [g.fromPoint(curveApnts[0]),  g.fromPoint(curveApnts[len(curveApnts)-1]), g.fromPoint(curveApnts[0]),  g.fromPoint(curveApnts[len(curveApnts)-1]) ] ):
            _return = _curveA.difference( _curveB )
            logger.debug( "in split_curve_by_curve with shapely difference operation -> _return.wkt: %s" % ( _return.wkt) )
            _geom = QgsCore.QgsGeometry()
            _return = _geom.fromWkt(_return.wkt)
            logger.debug( "in split_curve_by_curve with shapely-backconverted _return: %s %s" % ( _return,  type(_return) ) )
        else:
            return [curveA]
    else:
    """
    #_return = curveA.difference(curveB)
    _return = curveA.splitGeometry(curveB.asPolyline(),  False)
    #_return = curveA.makeDifference(curveB)
    
    #-- NJH 04-03-2016: use these 4 lines when using  curveA.difference(curveB)
    #if _return.isMultipart():
    #    return _return.asGeometryCollection()
    #else:
    #    return [curveA]
    
    #-- NJH 04-03-2016: use these 4 lines when using  curveA.splitGeometry(curveB.asPolyline(),  False)
    if _return[0] == 1:
        return [curveA]
    else:
        return  [ _return[1][0],  curveA ]
    #-- strangely enough the return[1] only holds one part of the splitted curve. The other part sits in a modified curveA
    
    #if curveA.isMultipart():
    #    return curveA.asGeometryCollection()
    #else:
    #    return [curveA]

    
def find_equals_in_GeomList( igeom,  geomList,  decimals=6):
    #-- NJH 25-02-14: consider using the python itertools.combinations or permutation functions
    geom = geomList[igeom]
    del geomList[igeom]

    for other_geom in geomList:
        #if geom.equals( other_geom ):     
        #if curveList[icurve].almost_equals( other_curve,  decimals ):
        if shapely.wkt.loads(geom.exportToWkt()).almost_equals( shapely.wkt.loads(other_geom.exportToWkt()),  decimals ):
            return True
    
    return False

def already_in_GeomList( geom,  other_geomList,  decimals=6):
    #-- NJH 16-09-2013: should function only check for geometric almost_equals (to a certain precision level) or also check for overlap?
    
    for other_geom in other_geomList:
        ##if geom.equals(other_geom):
        #if  geom.type() == 1 and other_geom.type() == 1:
        #    if almost_equals_custom(geom,  other_geom):
        #        return True
        #    else: return False
        #elif shapely.wkt.loads(str( geom.exportToWkt()) ).almost_equals( shapely.wkt.loads(str( other_geom.exportToWkt() )),  decimals ):
        if shapely.wkt.loads(str( geom.exportToWkt()) ).almost_equals( shapely.wkt.loads(str( other_geom.exportToWkt() )),  decimals ):
            ##logger.debug("geom: %s already in GeomList: %s" % (geom.exportToWkt(),  other_geom.exportToWkt() ) )
            return True
    return False

def normOverlap_withOtherGeom( _geom,  _othergeom,  _buffer_dist ):
    _buff_geom = _geom.buffer(_buffer_dist , 11 )
    intersect_geom = _buff_geom.intersection( _othergeom )
    
    _weightedOverlap = intersect_geom.length() /  _geom.length() 
    return _weightedOverlap
    
def normProximity_withOtherGeom( _geom,  _othergeom,  _buffer_dist ):
    ##print("in normProximity_withOtherGeom() with _geom,  _othergeom,  _buffer_dist: %s %s %s" % ( _geom,  _othergeom,  _buffer_dist ) )
    _buff_geom = _geom.buffer(_buffer_dist , 11 )
    logger.debug("in normProximity_withOtherGeom() with _geom.exportToWkt() & _buff_geom: %s %s" % ( _geom.exportToWkt(),   _buff_geom ) )

    intersect_geom = _buff_geom.intersection( _othergeom )
    _weightedProx = intersect_geom.length() /  ( 0.5*( _geom.length() + _othergeom.length() ) )
    logger.debug("lengths of _geom: %s | _othergeom: %s | intersect_geom: %s | _weightedProx: %s" % ( float( _geom.length() ),  float( _othergeom.length() ) ,  intersect_geom.length() ,  _weightedProx  )  )
    return _weightedProx
    
def almost_equals_custom(geom1,  geom2, _weightedProximityThreshold = 0.90,  __buffer_dist = 0.02 ):
    _weightedProximity = normProximity_withOtherGeom(geom1,  geom2,  __buffer_dist )
    if _weightedProximity > _weightedProximityThreshold:
        logger.debug( "in geom1.almost_equals_custom(geom2) almost_equals_custom with _weightedProximity: %s" % _weightedProximity )
        _weightedProximity = normProximity_withOtherGeom(geom2,  geom1,  __buffer_dist )
        if _weightedProximity > _weightedProximityThreshold:
            logger.debug( "in geom2.almost_equals_custom(geom1) with _weightedProximity: %s" % _weightedProximity )
            return True
    return False
    
def intersects_curve2curveCollection( curve,  otherCurveList):
    intersects_curveIdxList = list()
    
    for i_other_curve in range(len(otherCurveList)):
        other_curve = otherCurveList[i_other_curve]
        if not isinstance( other_curve ,  QgsCore.QgsGeometry ):
            other_curve = other_curve.geometry()
        
        _return = curve.intersects(other_curve)
        if _return:
            intersects_curveIdxList.append(i_other_curve)
    
    return intersects_curveIdxList

def intersections_curve2curveCollection( icurve,  curveList,  otherCurveList):
    logger.debug("in intersections_curve2curveCollection with len(curveList): %s" % len(curveList))
    input_params = None
    __builtin__.recurrencyLvl +=1
    pntList    = list()
    i_other_curve = 0
    _increment = 1
    
    while i_other_curve < len( otherCurveList ):
        #logger.debug( "at beginning of segmentize_curve2curveCollection() with icurve/len(curveList): %s / %s | i_other_curve: %s " % ( icurve,len(curveList),  i_other_curve ) )
        
        #-- the incoming curve that  gets cut by any other_curve that may touch or intersect. The icurve is fixed while going through while loop, however
        #--                   however the curve might get updated as it is cropped when intersecting another_curve in the curveList. If not, then you get in endless loop!
        if not isinstance( curveList[icurve] ,  QgsCore.QgsGeometry ):
            curve = curveList[icurve].geometry()
        else: curve =curveList[icurve]
        
        if not isinstance( otherCurveList[i_other_curve] , QgsCore.QgsGeometry  ):
            other_curve = otherCurveList[i_other_curve] .geometry()
        else: other_curve = otherCurveList[i_other_curve]
        
        __pntList   = intersection_curve_with_curve(curve,  other_curve)
        
        ##if len(__pntList) > 0:
            ####logger.debug( "in IntersectCurve2CurveList(..) with i_other_curve:%s | len(__pntList): %s" % (i_other_curve,  len(__pntList) ) )
        for __pnt in __pntList:
            if not already_in_GeomList( __pnt, pntList  ):
                pntList.append( __pnt )
        
        i_other_curve+= _increment
        
    return pntList

#-- NJH 04-01-2013: the above intersections_curve2curveCollection only returns a pntList, while 
#--                                        segmentize_curve2curveCollection returns curveList,  pntList, attribList

#-- NJH 16-08-2013: compare with IntersectCurve2CurveList() in rhinoLib.healing2DNetwork_algorithms
def segmentize_curve2curveCollection( curveList, otherCurveList, i_curve, i_othercurve ):
    
    #logger.debug("in segmentize_curve2curveCollection with len(curveList): %s" % len(curveList) )
    input_params = None
    __builtin__.recurrencyLvl +=1
    
    _thresholdLength=1.0e-4
    
    pntList    = list()
    attribList = list()
    oldCurveListLength = len(curveList)
    
    #notreturned_from_recurrent = True
    #while notreturned_from_recurrent and i_othercurve < len( otherCurveList ):
    _i_curve_increment = 1
    _i_othercurve_increment = 1
    while i_othercurve < len( otherCurveList ):
        #-- the incoming curve that  gets cut by any other_curve that may touch or intersect. The i_curve is fixed while going through while loop, however
        #--                   however the curve might get updated as it is cropped when intersecting another_curve in the curveList. If not, then you get in endless loop!
        if not  isinstance( curveList[i_curve] ,  QgsCore.QgsGeometry ):
            curve = curveList[i_curve].geometry()
        else: curve =curveList[i_curve]
        
        if not isinstance( otherCurveList[i_othercurve] ,  QgsCore.QgsGeometry ):
            other_curve = otherCurveList[i_othercurve] .geometry()
        else: other_curve = otherCurveList[i_othercurve]
        
        _i_othercurve_increment = 1
        
        #logger.debug( "at beginning of segmentize_curve2curveCollection() with i_curve/len(curveList): %s / %s | i_othercurve: %s : %s" % ( i_curve,len(curveList),  i_othercurve, other_curve.exportToWkt()  ) )
        
        __segmList = list(); __pntList = list()
        #-- how to deal with the fact that the curveList contains the identical curve as part of its list: 
        #--                     check for equal in list and then ignore
        #-- NJH 14-09-2013: overlap between curve and othercurve should not accur between two different fractures in a Network-- the network is eroneous!!
        #-- NJH 14-09-2013: otherwise overlap can only be found in a Network if curve equals othercurve
        #-- NJH 16-09-2013: However, the curveList is updated with segmentized curves, instead the otherCurveList may contain all the non-segementized of the network (which is not updated) 
        #--                               and therefore overlap can occur between a curve and a othercurve. In that case curve should not be segmentized with a non-segmentized overlapping curve. That causes precision-related intersections that are meaningless.
        ##logger.debug("in segmentize_curve2curveCollection() with curve[%s]: %s | %s & other_curve[%s]: %s | %s" % ( i_curve,  curve, curve.exportToWkt(),   i_othercurve,  other_curve,  other_curve.exportToWkt() ) )
        _overlapWithOtherCurve= normOverlap_withOtherGeom(curve,  other_curve,  0.01 )
        
        ##logger.debug( "in segmentize_curve2curveCollection(..) curve.almost_equals( other_curve ) gives false while almost_equals_custom give True")
        ##logger.debug( "in segmentize_curve2curveCollection(..) with curve: %s and other_curve: %s " % ( curve.exportToWkt() , other_curve.exportToWkt()  ))
        ##if almost_equals_custom(curve,  other_curve) or curve.intersection(other_curve).type() == 1 :
        ##if almost_equals_custom(curve,  other_curve,  0.10 ,  0.01) or curve.intersection(other_curve).type() == 1 :
        
        #-- segmentizing of curve with othercurve should not be done in case of 'overlap' or almost_equality', only in case of intersection.
        #--                                  
        if _overlapWithOtherCurve > 0.99 or wkt.loads( str(curve.exportToWkt() ) ).almost_equals( wkt.loads( str(other_curve.exportToWkt() ) )) :
            #logger.debug( "<<%s<< step_-4.1 with both curve.almost_equals( other_curve ) giving True" % ( __builtin__.recurrencyLvl ) )
            __segmList = [curve]; __pntList = list()
        else:
            __pntList   = intersection_curve_with_curve(curve,  other_curve)
            __segmList = split_curve_by_curve(curve,  other_curve ) 
            #logger.debug( "<<%s<< step_-4.2 the curve - othercurve not equal -> after splitting len(__segmList): %s " % ( __builtin__.recurrencyLvl ,  len(__segmList) ) )

        if len(__pntList) > 0:
            ##logger.debug( "in IntersectCurve2CurveList(..) with i_othercurve:%s | len(__pntList): %s" % (i_othercurve,  len(__pntList) ) )
            for __pnt in __pntList:
                if not already_in_GeomList( __pnt, pntList  ):
                    pntList.append( __pnt )
        
        #-- NJH 16-09-2013: intersection information of the idx-es of curve and othercurve must be recorded in order to build network-graphs -> to be worked out further.
        if len(__segmList) > 1:
            attribList.append( {i_curve: i_othercurve} )
        
            ##logger.debug( "<<%s<< step_-3 curve[%s] intersecting with othercurve[%s] ->  len(__segmList) > 1 : %s " % ( __builtin__.recurrencyLvl, i_curve,  i_othercurve, len(__segmList) ) )
            if len(__segmList) > 2: logger.debug( "<<%s<< step_-2 with split function gives more than two segments back!!" % ( __builtin__.recurrencyLvl ) )
            ##logger.debug( "in IntersectCurve2CurveList(..) with i_curve: %s | i_othercurve:%s | len(__segmList): %s" % (i_curve, i_othercurve,  len(__segmList) ) )
            
            for isegm in range(len( __segmList )):
                #logger.debug("<<%s<< step_-1 with isegm / nsegm: %s / %s " % ( __builtin__.recurrencyLvl,  isegm,  len(__segmList) ) )
                _bl_do_add2curveList = True
                if already_in_GeomList( __segmList[ isegm ],  curveList,  _thresholdLength ):
                    _bl_do_add2curveList = False
                elif __segmList[ isegm ].length() < _thresholdLength:
                    _bl_do_add2curveList = False
            
                if _bl_do_add2curveList:
                    _i_curve_increment = 0
                    #_i_othercurve_increment = 0 #-- if segments are added to (or replace curve in) curveList, then the _i_othercurve_increment should be kept zero i_othercurve should stay same, to check agian for the esgmeents if they intersect with the other_curve
                                          #--   hey, no, if the former curve did not intersect with the othercurve for i_othercurve upto the current i_othercurve, so will not tdop their derived segments. So simply contiue with next i_othercurve
                    ##for isegm in range(len( __segmList )):
                    ## #logger.debug("<<%s<< step_-1 with isegm / nsegm: %s / %s " % ( __builtin__.recurrencyLvl,  isegm,  len(__segmList) ) )
                    ##
                    ## #if not already_in_GeomList( __segmList[ isegm ],  curveList,  1.0e-4 ) and  __segmList[ isegm ].length() > 1e-4:
                    ##    if __segmList[ isegm ].length() < 1.0:
                    ##        logger.debug("<<%s<< step_-0.2 segm[%s] has length %s b/c intersection curve[%s] with othercurve[%s] " % ( __builtin__.recurrencyLvl, isegm,  __segmList[ isegm ].length() ,  i_curve,  i_othercurve  ) )
                    ##        logger.debug("<<%s<< step_-0.2 segm[%s]: %s curve: %s | othercurve: %s" % ( __builtin__.recurrencyLvl, i_curve,  __segmList[ isegm ].exportToWkt(),  curve.exportToWkt() , other_curve.exportToWkt() ) )
                    ##    #logger.debug("<<%s<< step_0.1 with isegm / nsegm: %s / %s with %s (big enough and) not yet in curveList " % ( __builtin__.recurrencyLvl,  isegm,  len(__segmList),  __segmList[ isegm ].exportToWkt() ) )
                    ##    #logger.debug("<<%s<< step_0.1 with isegm / nsegm: %s / %s with %s curve: %s | other_curve: %s " % ( __builtin__.recurrencyLvl,  isegm,  len(__segmList),  __segmList[ isegm ].exportToWkt(),  curve.exportToWkt() , other_curve.exportToWkt() ) )
    
                    #-- NJH 04-03-2016: why again adding __pnt in __pntList to the pntList?? 
                    #--                Already did that ~20lines up. Redundant/eroneous repetition?!!
                    ##  if len(__pntList) > 0:
                    ##      for __pnt in __pntList:
                    ##          if not already_in_GeomList( __pnt,  pntList,  1.0e-6 ):
                    ##              pntList.append( __pnt )
                        #if len (___attribList) > 0:
                        #    for ___attrib in ___attribList:
                        #        attribList.append(___attrib)
                        
                    if isegm == 0:
                            del curveList[i_curve]
                            curveList.insert((i_curve+isegm),  __segmList[ isegm ] ) 
                            #logger.debug("<<%s<< step_+4.1 isegm/nsegms: %s/%s - replaced first curve in curveList by new segm -> curveListLength: %s and oldCurveListLength: %s" % (  __builtin__.recurrencyLvl,  isegm,  len(__segmList),  len( curveList ) ,  oldCurveListLength  ) )
                    else:
                            curveList.insert(( i_curve+isegm ),  __segmList[ isegm ] ) 
                            #logger.debug("<<%s<< step_+4.1 isegm/nsegms: %s/%s - after inserting new segm to curveListLength: %s and oldCurveListLength: %s" % (  __builtin__.recurrencyLvl,  isegm,  len(__segmList),  len( curveList ) ,  oldCurveListLength  ) )
                    #logger.debug("<<%s<< step_+5 with curveListLength: %s and oldCurveListLength: %s" % (  __builtin__.recurrencyLvl,  len( curveList ),  oldCurveListLength  ) )
                        
                        #-- NJH 11-09 -- !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! a new attempt
                        #curveList,  ___pntList,  ___attribList = segmentize_curve2curveCollection( icurve+isegm  , curveList ,  list(otherCurveList) )
                        #logger.debug("<<%s<< step_+6 with curveListLength: %s and oldCurveListLength: %s" % (  __builtin__.recurrencyLvl,  len( curveList ),  oldCurveListLength  ) )
                        #if len( curveList ) > oldCurveListLength: notreturned_from_recurrent = False
                ##else:
                    ##logger.debug("<<%s<< step_0.2 with isegm/nsegms: %s/%s - segm (too small or) already exists in curveList length: %s and oldCurveListLength: %s" % (  __builtin__.recurrencyLvl,  isegm, len(__segmList), len( curveList ),  oldCurveListLength ) )

            #else: pass
            #return _pntList, curveList
        
        ##else:
        ##    attribList.append( {i_curve: None} )
        ##    logger.debug( "<<%s<< step_-3 curve[%s] no intersects with othercurve[%s] -> no segm (%s) 2add2 curveList: %s" % ( __builtin__.recurrencyLvl, icurve,  i_othercurve,  len(__segmList) ,  len( curveList ) ) )
        
        i_othercurve+= _i_othercurve_increment
    
    #logger.debug("<<%s<< step_end returning old/new len(curveList): %s / %s" % (  __builtin__.recurrencyLvl, oldCurveListLength, len( curveList ) ) )
    __builtin__.recurrencyLvl -=1
    i_curve+= _i_curve_increment
    return curveList,  pntList, attribList,  i_curve, i_othercurve
    
def get_intersectionTopologyList( curveList,  otherCurveList ):
    
    logger.debug("in get_intersectionTopologyList() with len(curveList): %s" % len(curveList) )
    input_params = None
    __builtin__.recurrencyLvl +=1
    
    terminationDict = dict()
    icurve = 0
    oldCurveListLength = len(curveList)
    
    while icurve < len( curveList ):
        logger.debug( "at beginning of get_intersectionTopologyList() with icurve/len(curveList): %s / %s " % ( icurve,len(curveList) ) )
        
        #-- the incoming curve that  gets cut by any other_curve that may touch or intersect. The icurve is fixed while going through while loop, however
        #--                   however the curve might get updated as it is cropped when intersecting another_curve in the curveList. If not, then you get in endless loop!
        if not  isinstance( curveList[icurve] ,  QgsCore.QgsGeometry ):
            curve = curveList[icurve].geometry()
        else: curve =curveList[icurve]
        
        terminationDict[icurve] = dict()
        terminationDict[icurve]['crosspoints'] = list()
        terminationDict = get_endPointType( terminationDict, curveList, icurve, otherCurveList )

        #if len(__pntList) > 0:
        #    ##logger.debug( "in IntersectCurve2CurveList(..) with i_other_curve:%s | len(__pntList): %s" % (i_other_curve,  len(__pntList) ) )
        #    for __pntKey,  __pntValue in __pntList.items():
        #        if not already_in_GeomList( __pntDict.key(), pntList  ):
        #            pntDict[__pntKey] = __pntValue
        
        icurve+= 1
    
    #logger.debug("<<%s<< step_end returning old/new len(curveList): %s / %s" % (  __builtin__.recurrencyLvl, oldCurveListLength, len( curveList ) ) )
    __builtin__.recurrencyLvl -=1
    return terminationDict

def get_intersectionTopologyList( terminationsTable,  curveList,  otherCurveList,  _outcropDomain,  _precision=0.01 ):
    
    logger.debug("in get_intersectionTopologyHDFTable() with len(curveList): %s" % len(curveList) )
    input_params = None
    __builtin__.recurrencyLvl +=1
    
    icurve = 0
    oldCurveListLength = len(curveList)
    
    while icurve < len( curveList ):
        logger.debug( "at beginning of get_intersectionTopologyList() with icurve/len(curveList): %s / %s " % ( icurve,len(curveList) ) )
        
        #-- the incoming curve that  gets cut by any other_curve that may touch or intersect. The icurve is fixed while going through while loop, however
        #--                   however the curve might get updated as it is cropped when intersecting another_curve in the curveList. If not, then you get in endless loop!
        if not  isinstance( curveList[icurve] ,  QgsCore.QgsGeometry ):
            curve = curveList[icurve].geometry()
        else: curve =curveList[icurve]
        
        terminationsTable = get_endPointType( terminationsTable, curveList, icurve, 
                            otherCurveList,  _outcropDomain,  _precision )

        icurve+= 1
    
    #logger.debug("<<%s<< step_end returning old/new len(curveList): %s / %s" % (  __builtin__.recurrencyLvl, oldCurveListLength, len( curveList ) ) )
    __builtin__.recurrencyLvl -=1
    
    return terminationsList
 

def gen_convexHull_onGeomCollection(_geoms):
    _pointset = shapely.geometry.multipoint.MultiPoint()
    
    ##print "in gen_convexHull_onGeomCollection with _geoms: %s %s" % ( type(_geoms),  _geoms )
    ##logger.debug("in gen_convexHull_onGeomCollection with _geoms:%s %s" % ( type(_geoms),  _geoms ) )
    
    if isinstance( _geoms , list ):
        ##logger.debug("in gen_convexHull_onGeomCollection with type(_geoms[0] )%s %s" % ( type(_geoms[0]),  _geoms[0] ) )
        if isinstance( _geoms[0],  shapely.geometry.point.Point):
            _geoms = shapely.geometry.multipoint.MultiPoint(_geoms)
        elif isinstance( _geoms[0],  DgfCore.DgfVertex ):
            _geoms = shapely.geometry.multipoint.MultiPoint(_geoms)
        else:
            _geoms = shapely.geometry.multipoint.MultiPoint(_geoms)
    
    logger.debug("in gen_convexHull_onGeomCollection with casted _geoms:%s %s" % ( type(_geoms),  _geoms ) )
    if isinstance( _geoms , shapely.geometry.multipoint.MultiPoint ):
        _pointset = _geoms
    elif isinstance( _geoms , shapely.geometry.multilinestring.MultiLineString ):
        for _line in _geoms:
            _pointset.extend( _line.coords() )
    else:
        return None
    
    _convexhullGeom = _pointset.convex_hull
    
    return _convexhullGeom

def get_interceptAngle(vec1,  vec2,  form='deg'):
    if not vec1 or not vec2: return None
    _angle = vec1.dot( vec2 )
    _angle = numpy.arccos(_angle)
    if form  =='deg':
        _angle = numpy.degrees(_angle)
    
    #-- to determine if clockwise angle (positive) of vec1->vec2 or anticlockwise (negative) 
    #--         this only applies to mapview vectors, with corss prodcuti of shape [ 0, 0 , 1] clockswise and [0,0,-1] anticlockwise
    #logger.debug("vec1.cross(vec2): %s | length vec1, vec2: %s %s" % ( vec1.cross(vec2),  vec1.length(),  vec2.length() ) )
    crossprod = vec1.cross(vec2)
    if crossprod[2]  < 0.0:
        _angle = -1.0 * _angle
    
    return _angle
   
    
def getSegmentList( geometry ):
    segmentList = list()
    iVertex = 0
    if geometry.type() ==  1:
        if geometry.isMultipart():
            print "isMultipart()"
        else:
            vertexList_per_part = geometry.asPolyline()
            logger.debug("in getSegmentList() with len(vertexList_per_part)%s" %  len(vertexList_per_part) )
            while iVertex < len(vertexList_per_part) -1:
                #newSegm = QgsCore.QgsGeometry()
                newSegm = QgsCore.QgsGeometry.fromPolyline( [ QgsCore.QgsPoint ( vertexList_per_part[iVertex][0] ,  vertexList_per_part[iVertex][1] )
                                                   , QgsCore.QgsPoint ( vertexList_per_part[iVertex+1][0] ,  vertexList_per_part[iVertex+1][1] ) ] )
                segmentList.append(newSegm)
                iVertex+=1
    elif geometry.type() ==  2:
        for vertexList_per_part in geometry.asPolygon():
            while iVertex < len(vertexList_per_part) -1:
                #newSegm = QgsCore.QgsGeometry()
                newSegm = QgsCore.QgsGeometry.fromPolyline( [ QgsCore.QgsPoint ( vertexList_per_part[iVertex][0] ,  vertexList_per_part[iVertex][1] )
                                                   , QgsCore.QgsPoint ( vertexList_per_part[iVertex+1][0] ,  vertexList_per_part[iVertex+1][1] ) ] )
                segmentList.append(newSegm)
                iVertex+=1
    
    logger.debug("in getSegmentList() with len(segmentList): %s"  %  len(segmentList))
    #logger.debug("in getSegmentList() with segmentList: %s"  %  [geom.exportToWkt() for geom in segmentList] )
    return segmentList

#-- NJH 14-02-2014: taken  out of fractGenerators stochFractGenerator_nonSelfIntersecting class
#-- NJH 15-08-2013: the self.modelDomain_withExclusions seems to get easily geometrically invalid during 
#--                                the geom_outer.difference(geom_inner) operation in gen_excludedArea(..)
def gen_excludedArea(self, _modelDomain_withExclusions,  fractSet,  _azim,  _length ):
    logger.debug("in stochFractSet_withExcludedArea.gen_excludedArea() with initial _modelDomain: %s " % _modelDomain_withExclusions.exportToWkt() )            
    
    from shapely import wkt
    
    #for _fract in self.get_fractSet():
    for _fract in fractSet:
        logger.debug("in stochFractSet_withExcludedArea.gen_excludedArea() with existing_fract.geometry(): %s " % _fract.geometry().exportToWkt() )            
        _fractZoneExtrPnts = []
        for _centre in _fract.geometry().asPolyline():
            _futureFracShape = gen_Line_fromCPoint( _centre,  _azim,  _length )
            _futureFracShape = crop_PLine_byPGon(  _futureFracShape , self.modelDomain  )
            logger.debug("in stochFractSet_withExcludedArea.gen_excludedArea() with _futureFracShape.geometry(): %s " % _futureFracShape.exportToWkt() )
            _fractZoneExtrPnts.extend ( _futureFracShape.asPolyline() )
        logger.debug("in stochFractSet_withExcludedArea.gen_excludedArea() with _fractZoneExtrPnts: %s" % _fractZoneExtrPnts )
        _area = gen_convexHull_onGeomCollection ( _fractZoneExtrPnts )
        #self.excludedArea.append(_area)
        logger.debug("in stochFractSet_withExcludedArea.gen_excludedArea() with _an excluded_area: %s" % _area.wkt )
        _geom = _modelDomain_withExclusions.difference( QgsCore.QgsGeometry.fromWkt( _area.wkt ) )
        #_geom_alt = wkt.loads(str(_modelDomain_withExclusions.exportToWkt())).difference(_area)
        #logger.debug("in stochFractSet_withExcludedArea.gen_excludedArea() with the difference() returned _geom: %s %s" % (_geom.isGeosValid(),  _geom.exportToWkt()) )
        #logger.debug("in stochFractSet_withExcludedArea.gen_excludedArea() with the alt_ difference() returned _geom_alt:%s %s" % ( _geom_alt.is_valid,  _geom_alt.wkt ) )
        if _geom: _modelDomain_withExclusions  = _geom
    
    ##self._modelDomain_withExclusions = _modelDomain_withExclusions
    logger.debug("in gen_excludedArea() with modelDomain_withExclusions: %s" % _modelDomain_withExclusionsexportToWkt())
    return _modelDomain_withExclusions

# ############################################################################

def SegmAt( _geom,  _u):
    if _u > _geom.length():
        return None
    iPrev=0
    nVert = 0
    ##print "in SegmAt with _geom.type(): %s" % _geom.type()
    if _geom.type() == 1:
        ##print "in SegmAt with _geom.asPolyline(): %s" % _geom.asPolyline()
        nVert = len( _geom.asPolyline() )
    _pntList = []
    _pntList.append( _geom.vertexAt( iPrev ) )
    for iCurr in range( 1,  nVert ):
        _pntList.append( _geom.vertexAt(  iCurr ) )
        _lineLength = QgsCore.QgsGeometry.fromPolyline( _pntList ). length()
        #logger.debug("in SegmAt with _lineLength: %s and _u %s" % (_lineLength,  _u) )
        if  _lineLength >= _u:
            _segm =  [ _geom.vertexAt( iPrev ) ,  _geom.vertexAt( iCurr ) ]
            break;
    return _segm
    
def PointAt( _geom,  _u):
    if _u > _geom.length():
        return False
    
    nVert = 0
    if _geom.type() == 1:
        nVert = len( _geom.asPolyline() )
        
    iPrev=0; iCurr=0
    _pntList = []
    _pntList.append( _geom.vertexAt( iPrev ) )
    for iCurr in range( 1,  nVert ):
        _pntList.append( _geom.vertexAt( iCurr ) )
        _lineLength = QgsCore.QgsGeometry.fromPolyline( _pntList ). length()
        if  _lineLength >= _u:
            _segm =  [_geom.vertexAt( iPrev ) ,  _geom.vertexAt( iCurr ) ]
            _segmLength = QgsCore.QgsGeometry.fromPolyline(_segm).length()
            logger.debug( "_u: %s | _lineLength: %s | _segmLength: %s " % ( _u ,  _lineLength  , _segmLength ) )
            segmCroppingFactor = ( _u - 1.0* (  _lineLength - _segmLength ) ) / _segmLength
            break;
        iPrev = iCurr
    
    #logger.debug("in PointAt() with segmCroppingFactor: %s" % ( segmCroppingFactor ) )
    #logger.debug("in PointAt() with _geom.vertexAt( iPrev ): %s" % ( _geom.vertexAt( iPrev ) ) )
    _dirVPnt = segmentDirVPnt( _segm )
    #logger.debug("in PointAt() with _dirVPnt( iPrev ): %s" % ( _dirVPnt ) )
    #_resizedDirVPnt = DgfCore.DgfVertex( _dirVPnt.x()*_segmLength*segmCroppingFactor , _dirVPnt.y()*_segmLength*segmCroppingFactor  )
    _resizedDirVPnt = DgfCore.DgfVertex( _dirVPnt[0]*_segmLength*segmCroppingFactor , _dirVPnt[1]*_segmLength*segmCroppingFactor  )
    #logger.debug("in PointAt() with _resizedDirVPnt: %s" % ( _resizedDirVPnt ) )
    
    #_point =  vertexAt( _geom, iPrev ) + _dirVPnt.multiply( segmCroppingFactor )
    #_point =  DgfCore.DgfVertex( _geom.vertexAt( iPrev ).x() + _resizedDirVPnt.x() , 
    #                                _geom.vertexAt( iPrev ).y() + _resizedDirVPnt.y() ) 
    _point =  DgfCore.DgfVertex( _geom.vertexAt( iPrev )[0] + _resizedDirVPnt[0] , 
                                    _geom.vertexAt( iPrev )[1]+ _resizedDirVPnt[1] ) 
    return _point

def intersectsAt(_geom,  _othergeom):
    _vertexIdxs = list() #_- two vertex indexes between which the intersection of _geom and _othergeom occurs
    if not _geom.intersects(_othergeom):
        return None,  None,  None,  None
    
    nVert = 0
    if _othergeom.type() == 1:
        nVert = len( _othergeom.asPolyline() )
        
    iCurr=0
    _incremVertexList = []
    _incremVertexList.append( _othergeom.vertexAt( iCurr ) )
    for iCurr in range( 1,  nVert ):
        _vertexPair = []
        _pnt = None
        _u = None
        _vertexIdxs = None
        _theSegm = None
        _vertexPair.append( _othergeom.vertexAt( iCurr-1 ) )  
        _vertexPair.append( _othergeom.vertexAt( iCurr ) )
        _segmGeom = QgsCore.QgsGeometry.fromPolyline( _vertexPair )
        if  _segmGeom.intersects(_geom):
            _incremVertexList.append( _othergeom.vertexAt( iCurr ) )
            _theSegm = [DgfCore.DgfVertex(_vertexPair[0].x(), _vertexPair[0].y(), 0.0 ), DgfCore.DgfVertex(_vertexPair[1].x(), _vertexPair[1].y(), 0.0 )]
            _incremLineString = QgsCore.QgsGeometry.fromPolyline( _incremVertexList )
            _pnt = _segmGeom.intersection(_geom )
            _vertexIdxs = [iCurr-1, iCurr]
            _u = _incremLineString.length() + _segmGeom.length()
            break
    
    return _pnt,  _u,  _vertexIdxs,  _theSegm
    
def NormalAt( _geom,  _u ):
    _vectorPnt = segmentDirVPnt(  SegmAt( _geom,  _u) )  
    return QgsCore.QgsPoint( _vectorPnt[1], -1*_vectorPnt[0] )
    
def DirVectAt(_geom,  _u):
    ##logger.debug("_dirVect_other: %s %s " % ( _dirVect_other,  type(_dirVect_other) ) )
    _segm = SegmAt(_geom, _u)
    if not _segm: return None
    return segmentDirVPnt( _segm )
    
def segmentDirVPnt( _segm ):
    _vector = numpy.array( [ _segm[1][0] - _segm[0][0]  ,  _segm[1][1] - _segm[0][1] ] )
    _length = numpy.sqrt( _vector[0]**2 + _vector[1] **2)
    #logger.debug("in segmentDirVPnt() before normalizing with _length: %s" % _length)
    _vector = _vector / _length
    #logger.debug("in segmentDirVPnt() after normalizing with _length: %s" % numpy.sqrt( _vector[0]**2 + _vector[1] **2) )
    return DgfCore.DgfVertex( _vector[0],  _vector[1] )
