"""DF_Vector"""
# ################################################
#   
#   DigiFractLib.baseclasses-- an interface to the DigiFract baseclasses and datamodel 
#   and other DigiFractLib modules for digital field acquisition of fractures.
#
#    --------
#   
#  Copyright (c) 2014 (a) - N.J. Hardebol, 
#     (a) Delft University of Technology, Department of Geotechnology,  Stevinweg 1, 2628 CN Delft, the Netherlands
#
#   ---------
# 
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# #############################################

import numpy
import logging
logger = logging.getLogger("DigiFractLogger.DF_Vector")

from digifract import dgfcore as  DgfCore

def strikeVec2surfNVec( _strikeVec ,  _phi=90.0 ):
    """create a vector for a polyline in 2-d space (on a  map)"""
    #-- for getting the _x & _y normal to the strikevector , do a swap: (a,b) -> (-b,a)
    _x = -1*_strikeVec.y() ; _y =  _strikeVec.x()
    xylength = (_x**2.0 + _y**2.0)**0.5
    _z = xylength / numpy.tan(numpy.radians( _phi ) )
    ##logger.debug("in pline2vec2d() with dipang: %s xylength: %s and _z: %s" % ( _phi,  xylength, _z ) )
    _vector = DF_Vect3D( _x  ,  _y ,  _z )
    ##logger.debug("in pline2vec2d() with resulting non-normalized vector: %s" % ( _vector ) )
    
    return _vector.normalized()
    
def pline2vec2d( _pline ,  _plunge=0.0 ):
    """create a vector for a polyline in 2-d space (on a  map)"""
    _x = None ; _y = None
    #if isinstance(_pline,  QgsCore.QgsGeometry ):
    # NJH 05-02-2016: let's simply assume that _pline is of the right type. 
    #    There is no implementation, in case _pline is not.
    #     Also it simplifies the module dependencies. Commenting out the QgsCore.QgsGeometry, removes dependency on QGIS.
    if (_pline.type() == 1) and not _pline.isMultipart():
        _pline = _pline.asPolyline()
        _x =_pline[-1].x() - _pline[0].x()  ;  _y = _pline[-1].y() - _pline[0].y()
    elif (_pline.type() == 1) and _pline.isMultipart():
        #-- NJH 05-11-2014: better implement the vector extraction from a multipolyline-- now just take the first linestring in list!!!!
        __pline = _pline.asMultiPolyline()[0]
        logger.debug("in pline2vec2d with a Multipart _pline: %s of which only considering the first part: %s " % ( _pline.exportToWkt(),  __pline ) )
        _x = __pline[-1].x() - __pline[0].x()  ; _y =  __pline[-1].y() - __pline[0].y()
    else:
        logger.debug("in pline2vec2d with a _pline: %s %s %s" % ( _pline.type() ,  _pline.isMultipart(),  _pline.exportToWkt(), ) )
        _pline = _pline.asPolyline()
        _x = _pline[-1].x() - _pline[0].x()  ;  _y = _pline[-1].y() - _pline[0].y() 
    
    if _x is None: return None
     
    xylength = (_x**2.0 + _y**2.0)**0.5
    _z = xylength * numpy.tan(numpy.radians( _plunge ) )
    ##logger.debug("in pline2vec2d() with _plunge: %s xylength: %s and _z: %s" % ( _plunge,  xylength, _z ) )
    _vector = DF_Vect3D( _x  ,  _y ,  _z )
    ##logger.debug("in pline2vec2d() with resulting non-normalized vector: %s" % ( _vector ) )
    
    return _vector.normalized()

#def azim2vec2d(azim):
#    """convert Azimuth orientation into 2d-vector in mapview"""
#    return DF_Vect3D( numpy.cos(numpy.radians(azim)) ,  numpy.sin(numpy.radians(azim)),  0.0 )
    
def orient2vector( _theta,  _phi=0.0,  type='line' ):
    """_theta is the azimuth (dipdir) orientation (0-360d) 
        and _phi the dipang of a surface or plunge of a line (0-90d)
    """
    if type=='poleplane': 
        x_ =  numpy.sin(numpy.radians( _theta ) ) * numpy.sin(numpy.radians( _phi ) )
        y_ =  numpy.cos(numpy.radians( _theta ) ) * numpy.sin(numpy.radians( _phi) )
        z_ =  numpy.cos(numpy.radians( _phi ) )
    else: 
        x_ =  numpy.sin(numpy.radians( _theta ) ) * numpy.cos(numpy.radians( _phi ) )
        y_ =  numpy.cos(numpy.radians( _theta ) ) * numpy.cos(numpy.radians( _phi) )
        z_ =  numpy.sin(numpy.radians( _phi ) )
    _vector = DF_Vect3D( x_ , y_ ,  z_ ).normalized()
    return _vector

def vector2angles( _vector ):
    """returns theta as the azimuth (dipdir) orientation (0-360d) and 
        phi being the dipang of a surface or plunge of a line (0-90d)
    """
        
    if isinstance(_vector, list ):
        _vector = DF_Vect3D( _vector[0], _vector[1], _vector[2] )
    
    _vector = _vector.normalized()
    _x = _vector.x()
    _y = _vector.y()
    _z = _vector.z()
    
    logger.debug("in vector2orient() with _x: %s ; _y: %s ; _z: %s" % ( _x,  _y,  _z) )
    if _z is not None:
        if _z < 0:
            _x = -1* _x ;  _y = -1* _y ; _z = -1* _z
    logger.debug("in vector2orient() with _x: %s ; _y: %s ; _z: %s" % ( _x,  _y,  _z) )
    
    phi = numpy.degrees( numpy.arccos( _z )  )
    _flat_vector = DF_Vect3D( _vector.x(),  _vector.y(),  0.0 ).normalized()
    theta = numpy.degrees( numpy.arcsin( _flat_vector.x() ) )

#    if type == 'poleplane':
#        phi = numpy.degrees( numpy.arccos( _z )  )
#        _flat_vector = DF_Vect3D( _vector.x(),  _vector.y(),  0.0 ).normalized()
#        theta = numpy.degrees( numpy.arcsin( _flat_vector.x() ) )
#    elif type == 'strikeline' or 'strikeASstrike' or 'strike2pole':
#        phi = 90.0      #-- in case a strikeline vector is given, then even its plunge holds no information on the dipangle of the plane. So we assume the plane is verticle?
#        _flat_vector = DF_Vect3D( _vector.y(), -1*_vector.x(),  0.0 ).normalized() #-- from strikeline, get the dipdir under 90dgr angle by [a,b] -> [b,-a] swap
#        theta = numpy.degrees( numpy.arcsin( _flat_vector.x() ) )
#        logger.debug( "in vector2orient() for strikeline with vector (%s) ->  theta: %s" % ( _flat_vector,  theta  ) )
#    else:                                   #-- t
#        phi = numpy.degrees( numpy.arcsin( _z )  )
#        _flat_vector = DF_Vect3D( _vector.x(),  _vector.y(),  0.0 ).normalized()
#        theta = numpy.degrees( numpy.arcsin( _flat_vector.x() ) )
#    logger.debug( "in vector2orient() with non-converted phi: %s | theta: %s" % (phi,  theta  ) )
    
    if _y < 0: theta = 180.0-theta

    if theta < 0: theta = 360.0 + theta
    if theta >= 360.0: theta = theta - 360.0
    logger.debug( "in vector2orient() with phi: %s | theta: %s" % (phi,  theta  ) )
    
    return theta, phi
    
def vector2orient(_vector,  type='strikeline'):
    """
      returns an orientation object that is meant for planes. This is conform dipdir, dipang. 
    """
    import orientLib
    
    theta, phi = vector2angles(_vector )
    
    if type == 'poleplane' or type == 'strikeASstrike':
        _orient = orientLib.orientation.Orientation( theta, phi )
    elif type == 'strikeline' or 'strike2pole':
        _dipdir = theta+90.0
        if _dipdir >= 360.0: _dipdir = _dipdir-360.0
        _orient = orientLib.orientation.Orientation( _dipdir, phi )
    else:
        raise ValueError("type of orient unknown : %s" % type )

    return _orient

class DF_Pnt3D( DgfCore.DgfVertex ):
    def __init__(self, _x=0.0, _y=0.0, _z=0.0):
        DgfCore.DgfVertex.__init__( self,  _x, _y, _z )
        
    def x(self): return self[0]
    def y(self): return self[1]
    def z(self): return self[2]
    
    def normalized(self):
        if self.length() == 0.0:
            return self
        _x = self[0] / self.length()
        _y = self[1] / self.length()
        _z = self[2] / self.length()
        return DF_Vect3D(_x,  _y, _z)

    def __sub__(self, _other ):
        return DF_Vect3D( self[0] - _other[0],  self[1]- _other[1], self[2] - _other[2] )
    
    def __add__(self, _other ):
        return DF_Vect3D(_other[0]+self[0],  _other[1]+self[1], _other[2]+self[2])
    
    def __mul__(self, _other ):
        return DF_Vect3D(_other*self[0],  _other*self[1], _other*self[2])
        
    def __rmul__(self, _other ):
        return DF_Vect3D(_other*self[0],  _other*self[1], _other*self[2])

#class DF_Pnt3D( vec3.vec3 ):
#    def __init__(self ,  _x  ,  _y , _z ):
#        vec3.vec3.__init__(self,  _x  ,  _y , _z)
#    def X(self): return self.x
#    def Y(self): return self.y
#    def Z(self): return self.z
    
#class DF_Pnt3D( OCC.gp.gp_Pnt ):
#    def __init__(self ,  _x  ,  _y , _z ):
#        OCC.gp.gp_Pnt.__init__(self,  _x  ,  _y , _z)


class DF_Vect3D( DgfCore.DgfVertex ):
    def __init__(self, _x=0.0, _y=0.0, _z=0.0):
        if isinstance(_x , DgfCore.DgfVertex ):
            DgfCore.DgfVertex.__init__( self,  _x[0], _x[1], _x[2] )
        else:
            DgfCore.DgfVertex.__init__( self,  _x, _y, _z )
        
        #-- theta = longitutde or azimuth
        #-- phi =sin of  latitude or dipang

    def length(self):
        return numpy.power( numpy.power(self[0], 2) +  numpy.power(self[1], 2) +  numpy.power(self[2], 2) ,  0.5 )

    def azim(self):
        _azim = vector2orient(self).dipdir
        return _azim
    
    def plunge(self):
        #-- dipdir only makes sense for a plane; so assume that the vector represents normal_vector of surface
        _normself = self.normalized()
        ##_phi = numpy.power( numpy.power( _normself[0],  2) +numpy.power( _normself[1] ,  2) ,  0.5 )
        ##_dipang = numpy.degrees( numpy.arcsin( _phi ) )
        logger.debug("in DF_Vector.plunge() debug with _normself.z(): %s and numpy.arcsin( _normself.z() ): %s" %  (  _normself.z() , numpy.arcsin( _normself.z() )) )
        _plunge = numpy.degrees( numpy.arcsin( _normself.z() ) ) 
        logger.debug("in DF_Vector.plunge() debug with _plunge: %s" %  ( _plunge) )
        return _plunge
    
    def orient(self):
        return vector2orient(self)
    
    def dot(self,  _other):
        return self[0]*_other[0] + self[1]*_other[1] + self[2]*_other[2] 
    
    def cross(self,  _other):
        _a = self[1]*_other[2] - self[2]*_other[1] 
        _b = self[2]*_other[0] - self[0]*_other[2] 
        _c = self[0]*_other[1] - self[1]*_other[0]
        return DF_Vect3D(_a,  _b, _c)
    
    def set_fromQgsPoint(self,  _qgsPnt):
        self[0] = _qgsPnt.x()
        self[1] = _qgsPnt.y()
        self[2] = 0.0

    def normalized(self):
        if self.length() == 0.0:
            return self
        _x = self[0] / self.length()
        _y = self[1] / self.length()
        _z = self[2] / self.length()
        return DF_Vect3D(_x,  _y, _z)
        
    def __add__(self, _other ):
        return DF_Vect3D( _other[0]+self[0],  _other[1]+self[1], _other[2]+self[2] )
    
    def __sub__(self, _other ):
        return DF_Vect3D( self[0] - _other[0],  self[1] - _other[1], self[2] - _other[2] )
    
    def __mul__(self, _other ):
        return DF_Vect3D(_other*self[0],  _other*self[1], _other*self[2])
        
    def __rmul__(self, _other ):
        return DF_Vect3D(_other*self[0],  _other*self[1], _other*self[2])
    
    def x(self):
        return self[0]
    
    def y(self):
        return self[1]
    
    def z(self):
        return self[2]
        
    def ortho(self):
        """Returns an orthogonal vector.

        Returns a vector that is orthogonal to self (where
        self*self.ortho()==0).

        >>> a=vec3(1.0, -1.5, 0.8)
        >>> print round(a*a.ortho(),8)
        0.0
        """
        pass
    
    def __repr__(self):
        return "DF_Vec3D( %1.4f , %1.4f , %1.4f)" % ( self.x() ,  self.y() ,  self.z() )

    def __str__(self):
        fmt="%1.4f"
        return "(%1.4f ,%1.4f , %1.4f )" % ( self.x() ,  self.y() , self.z()  )

#try:
#    import OCC
#    OCC_loaded = True
#    class DF_Vect3D( OCC.gp.gp_Vec):
#        def __init__(self ,  _x  ,  _y , _z ):
#            OCC.gp.gp_Vec.__init__(self,  _x  ,  _y , _z)
#except:
#    OCC_loaded = False
#    import vec3
#    class DF_Vect3D(vec3.vec3):
#        def __init__(self ,  _x  ,  _y , _z ):
#            vec3.vec3.__init__(self,  _x  ,  _y , _z)
#        def X(self): return self.x
#        def Y(self): return self.y
#        def Z(self): return self.z
#        def Normalized(self): return self.normalize()
