"""DF_GeometryCollection"""
# ################################################
#   
#   DigiFractLib-- an interface to the DigiFract baseclasses and datamodel 
#   and other DigiFractLib modules for digital field acquisition of fractures.
#
#    --------
#   
#  Copyright (c) 2014 (a) - N.J. Hardebol, 
#     (a) Delft University of Technology, Department of Geotechnology,  Stevinweg 1, 2628 CN Delft, the Netherlands
#
#   ---------
# 
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# #############################################

import qgis.core as QgsCore
try:
    from digifract import dgfcore as  DgfCore
    loaded_DgfCore = True
except:
    loaded_DgfCore = False
    
##import DF_Feature

import logging
logger = logging.getLogger("DigiFractLogger.DF_GeometryCollection")

def resize_nd_shift_geom( _geom,  _params ):
    """support function for resizing (scaling) and shifting a geometry"""

    #-- NJH 16-01-2015: for now only _geom of type point are handled. 
    qgsPoint = _geom
    
    #--  scaling factors for rescale the x-coordinates to 0,1.0
    if 'x_scalefactor' in _params:
        XScaleFactor = _params['x_scalefactor'] 
    else: XScaleFactor = _params['newXSize'] / _params['oldXSize']
    #newXSize = 2.0
    
    if 'y_scalefactor' in _params:
        YScaleFactor = _params['y_scalefactor']
    else: YScaleFactor = _params['newYSize'] / _params['oldYSize']
    #newYSize = oldYSize * XScaleFactor
    
    #-- shift and multiply factor to have center in the 'middle' of domain stretching form -1.0<->0.0<->+1.0
    if 'oldX_shift' in _params:
        oldX_shift   =  _params['oldX_shift']    ##-2000.0
    else: oldX_shift = 0.0
    if 'oldY_shift' in _params:
        oldY_shift   =  _params['oldY_shift']    ##1500.0
    else: oldY_shift = 0.0
    
    if 'newX_shift' in _params:
        newX_shift = _params['newX_shift']     ##0.0        #--  with extent of 2.0, have zero in middle.
    else: newX_shift = 0.0
    if 'newY_shift' in _params:
        newY_shift = _params['newY_shift']     ##0.0
    else: newY_shift = 0.0
    #ynew_shift = float( oldSizeY ) / float( oldSizeX )
    
    _resizedX = ( qgsPoint[0] + oldX_shift ) * XScaleFactor + newX_shift
    ##logger.debug("in add_QgsGeom with qgsPoint[0]: %s and _resizedX: %s" % (qgsPoint[0],  _resizedX))
    _resizedY = ( qgsPoint[1] + oldY_shift ) * YScaleFactor  + newY_shift
    ##logger.debug("in add_QgsGeom with qgsPoint[1]: %s and _resizedY: %s" % (qgsPoint[1],  _resizedY))

    return DgfCore.DgfVertex( _resizedX ,  _resizedY ) 

    
def Property(func):
    """This defines a decorator tag and should not be called directly"""
    return property(**func())

class DF_GeometryCollection(list):
    """Fracture python class holding a collection of DgfGeometries and with pointers to abVertexSet and xyzVertexSet"""
    def __init__ ( self, name,  pFeatureSet ):
        list.__init__(self)  
        self.name =  name   
        
class DF_GeometryCollection( DgfCore.DgfGeometrySet ):
    """Fracture python class holding a collection of DgfGeometries and with pointers to abVertexSet and xyzVertexSet"""
    def __init__ ( self, name,  pFeatureSet ,  _vertexSetMode = 'AB' ):
        DgfCore.DgfGeometrySet.__init__( self, name )

        self.pFeatureSet = pFeatureSet
        #self.set_dimmode('2D')
        self.set_vertexSetMode(_vertexSetMode)
    
    def set_vertexSetMode(self,  _mode ):
        """member function for setting the vertexSet mode to AB (relative 2D coordinates) or UV for meter coordinates."""
        self.vertexSetMode = _mode
        logger.debug("in set_vertexSetMode with _mode: %s %s " % ( _mode,  self.vertexSetMode ) )
        if self.vertexSetMode == 'AB':
            self.associate(self.pFeatureSet.pOutcropSpace.abVertexSet )
            self.pVertexSet = self.pFeatureSet.pOutcropSpace.abVertexSet
        if self.vertexSetMode == 'UV':
            self.associate(self.pFeatureSet.pOutcropSpace.uvVertexSet )
            self.pVertexSet = self.pFeatureSet.pOutcropSpace.uvVertexSet 
        if self.vertexSetMode == 'XYZ':
            self.associate( self.pFeatureSet.pOutcropSpace.xyzVertexSet )
            self.pVertexSet = self.pFeatureSet.pOutcropSpace.xyzVertexSet
        logger.debug("in set_vertexSetMode with self.pVertexSet: %s" % ( self.pVertexSet.name() ) )

#    def set_dimmode(self,  _mode ):
#        self.dimmode = _mode
#        if self.dimmode == '2D':
#            self.associate(self.pFeatureSet.pOutcropSpace.abVertexSet )
#            self.pVertexSet = self.pFeatureSet.pOutcropSpace.abVertexSet 
#        if self.dimmode == '3D':
#            self.associate( self.pFeatureSet.pOutcropSpace.xyzVertexSet )
#            self.pVertexSet = self.pFeatureSet.pOutcropSpace.xyzVertexSet 
        
    def add_QgsGeom(self,  _geom,  _rescalingParams = None ):
        """member function that converts a 2D QGis geometry into a DigiFract geometry (i.e. adds its vertices to the 2DPointSet"""
        #_dgfGeom = DgfCore.DgfGeometrySet()  #-- NJH 23-04-2015: DgfGeometrySet should be renamed to DgfGeometry
        
        if  not _geom: 
            ##logger.debug("in add_QgsGeom() with _geom: %s" % _geom )
            return None
        if _geom.type() == 0: #-- QgsGeoemtryType: 0 = Point, 1 = LineString, 2 =
            qgsPoint =_geom.asPoint()
            if _rescalingParams: 
                _dgfVertex = resize_nd_shift_geom( qgsPoint , _rescalingParams )
            else:  _dgfVertex = DgfCore.DgfVertex( qgsPoint[0] ,  qgsPoint[1] )
            vertexIdx = self.pVertexSet.add( _dgfVertex )
            logger.debug("in add_QgsGeom() - about to add the vertexIdx of a Point to _DgfGeometrySet that are returned from the _2DVertexSet: %s" % ( vertexIdx ) )
            #_idx = _dgfGeom.add( vertexIdx )
            geomIdx = self.add( vertexIdx )
        elif _geom.type() == 1: #-- QgsGeoemtryType: 0 = Point, 1 = LineString, 2 =
            vertexSetIndexList = list()
            for qgsPoint in _geom.asPolyline():
                if _rescalingParams: 
                    _dgfVertex = resize_nd_shift_geom( qgsPoint , _rescalingParams )
                else:  _dgfVertex = DgfCore.DgfVertex( qgsPoint[0] ,  qgsPoint[1] ) 
                _vertexIdx = self.pVertexSet.add( _dgfVertex )
                vertexSetIndexList.append( _vertexIdx ) 
            ##logger.debug("in add_QgsGeom() - about to add the vertexSetIndexList of a Polyline to _DgfGeometrySet that are returned from the _2DVertexSet: %s" % ( vertexSetIndexList ) )
            #_idx = _dgfGeom.add(vertexSetIndexList)
            geomIdx = self.add(vertexSetIndexList)
        elif _geom.type() == 2: #-- QgsGeoemtryType: 0 = Point, 1 = LineString, 2 =
            ##aDgfPolygon = DgfCore.DgfGeometrySet()
            vertexSetIndexList = list()
            for ring in _geom.asPolygon():
                logger.debug("in add_QgsGeom() with _geom.type(): %s | ring: %s" % ( _geom.type(),  ring ) )
                for iPoint in range( len( ring )-2 ):
                    qgsPoint = ring[iPoint]
                    if _rescalingParams: 
                        _dgfVertex = resize_nd_shift_geom( qgsPoint , _rescalingParams )
                    else:  _dgfVertex = DgfCore.DgfVertex( qgsPoint[0] ,  qgsPoint[1] ) 
                    _vertexIdx = self.pVertexSet.add( _dgfVertex )
                    #logger.debug("in add_QgsGeom() after pVertexSet.add of DgfVertex and  _vertexIdx: %s" % ( _vertexIdx ) )
                    vertexSetIndexList.append( _vertexIdx )
                vertexSetIndexList.append( vertexSetIndexList[0] )
                #-- For Polygon, the last vertexIDx should be the first vertexIdx
                logger.debug("in add_QgsGeom() after pVertexSet.add..ing polygon ring with vertexSetIndexList: %s" % ( vertexSetIndexList ) )
                ##aDgfPolygon.add( vertexSetIndexList )
                #_idx = _dgfGeom.add( vertexSetIndexList )
                #geomIdx = _dgfGeom.add( vertexSetIndexList )
                geomIdx = self.add( vertexSetIndexList )
                ##logger.debug("in add_QgsGeom() after adding vertexSetIndexList" )
        
        #self.append( _dgfGeom )
        #geomIdx = self.__len__() - 1
        ##logger.debug("in add_QgsGeom() - at the end")
        return geomIdx
