""" DF_Geometry """
# ################################################
#   
#   DigiFractLib-- an interface to the DigiFract baseclasses and datamodel 
#   and other DigiFractLib modules for digital field acquisition of fractures.
#
#    --------
#   
#  Copyright (c) 2014 (a) - N.J. Hardebol, 
#     (a) Delft University of Technology, Department of Geotechnology,  Stevinweg 1, 2628 CN Delft, the Netherlands
#
#   ---------
# 
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# #############################################
"""The DataModel module provides classes to define the core data classes used by the fracQgis plugin"""

import os
import numpy

from PyQt4 import QtCore
import qgis.core as QgsCore
from digifract import dgfcore as DgfCore

import logging
logger = logging.getLogger("DigiFractLogger.Dgf_Geometry")

try: 
    from OCC import GeomAbs
    from OCC import gp
    from OCC import TopoDS
    loaded_OCC = True
except: 
    loaded_OCC = False

from twoDGeomBaseLib.core  import geomBasicOperations

try:
    from threeDGeomLib import OCC_GeomMdlFunctions
    loaded_threeDGeom = True
except:
    logger.debug("DigiFract will run without OCC_GeomMdlFunctions")
    loaded_threeDGeom = False

def Qgs2DgfGeometry( _QgsGeom ,  _DgfGeom , _2DPointSet ):
    """support function that converts a 2D QGis geometry into a DigiFract geometry (i.e. adds its vertices to the 2DPointSet"""
    #-- NJH 16-08-2012: Can we do without having _2DPointSet as input parameter?
    #--               Can the _2DPointSet be called as accessible _DgfGeom.data-member (provided function is member function), 
    #--                      in fact a reference since the poinSet is not physically stored in _DgfGeom
    #--  NJH 05-05-2014: Ask Martijn if it is a good idea to have the vSet2D  and vSet3D SIP wrapped so that their pointer can be returned?  
    ##GeosGeom   = wkb.loads( _QgsGeom.asWkb() )
    logger.debug("in Qgs2DgfGeometry() - what is the _QgsGeom.type(): %s" % _QgsGeom.type() )
    #if _QgsGeom.type() == "LineString":
    #    _QgsGeom.asPolyline()
    PntIndexList = []
    for qgsPoint in _QgsGeom.asPolyline():
        PntIndexList.append( _2DPointSet.add( DgfCore.DgfVertex( qgsPoint[0] ,  qgsPoint[1] ) )  )
    logger.debug("in Qgs2DgfGeometry() - about to add the PntIndexList to _DgfGeom that are returned from the _2DPointSet: %s" % ( PntIndexList ) )
    _DgfGeom.add(PntIndexList)
    return True
    
def Dgf2QgsGeometry( _verticesIds ,  _2DVertexSet ):
    """support function that converts a 2D DgfGeom geometry into a QGis geometry"""
    logger.debug("in Dgf2QgsGeometry() with _verticesIds: %s" % ( _verticesIds ) )
    #_success,  _vertexList = _2DVertexSet.getSome(  _verticesIds  )
    _vertexList = DgfCore.DgfVertexSet( 'newVertexList' , DgfCore.DgfVertex.TWO_DEE )
    _success = _2DVertexSet.getSome(  _vertexList,  _verticesIds  )
    logger.debug("in Dgf2QgsGeometry() with _vertexList: %s" % ( _vertexList ) )
    
    #-- generate QgsGeometry as point
    if len(_verticesIds) == 1:
        _QgsGeom = QgsCore.QgsGeometry().fromPoint( QgsCore.QgsPoint( _vertexList[0][0] ,  _vertexList[0][1]) )
    #-- generate QgsGeometry as linestring
    elif not _verticesIds[0] == _verticesIds[-1]:
        _QgsGeom = QgsCore.QgsGeometry().fromPolyline( [QgsCore.QgsPoint( _vertex[0] ,  _vertex[1]) for _vertex in _vertexList ] )
    #-- generate QgsGeometry as polygon
    else:
        _QgsGeom = QgsCore.QgsGeometry().fromPolygon(  [QgsCore.QgsPoint( _vertex[0] ,  _vertex[1]) for _vertex in _vertexList ]  )
    logger.debug("in Dgf2QgsGeometry() with _QgsGeom.exportToWkt(): %s" % ( _QgsGeom.exportToWkt() ) )
    
    return _QgsGeom

#NJH 23-11-2015: quickly re-implemented. the Qgs-> OCC is actually useful to still have in addition to the Dgf->OCC
def Qgs2OCCGeometry( _DgfGeom ): 
    """support function that converts a Qgs geometry  into a 3D OpenCascade geometry"""
    PntIndexList = [] 
    #-- NJ 03-08 this function is not yet properly implemented 
    #--                how to get coords of OCC_Geom? 
    
    #-- NJH 14-02-2013: below assumes that coord is 2D, how about 3D OCC_Geom? 
    if _DgfGeom.type() == 0: 
        _pnt = gp.gp_Pnt( _DgfGeom.asPoint().x() , _DgfGeom.asPoint().y(), 0.0 )  
        return OCC_GeomMdlFunctions.make_OCCVertex( _pnt ) 
    
    elif _DgfGeom.type() == 1: 
        print "_DgfGeom.type() equals 1, thus asPolyline()" 
        if not _DgfGeom.isMultipart(): 
            print "not isMultipart()" 
            uvVerticesList = _DgfGeom.asPolyline() 
        else: uvVerticesList = _DgfGeom.asPolyline()[0]
    elif _DgfGeom.type() == 2: 
        print "_DgfGeom.type() equals 2, thus asPolygon()" 
        if _DgfGeom.isMultipart(): 
            print "isMultipart()" 
        else: uvVerticesList = _DgfGeom.asPolygon()[0] #-- in case of MultiPolygon this will not work 
    print "uvVerticesList: ",  uvVerticesList 
    
    pntList = [] 
    for vertex in uvVerticesList: 
        #-- in case of future DgfVertex implementation instead of pg.pg_Pnt: 
        #abPoint = DgfCore.DgfVertex() 
        #abVertexSet.get(abPoint,  index) 
        pntList.append ( gp.gp_Pnt( vertex[0] , vertex[1], 0.0 ) ) 
    wire = OCC_GeomMdlFunctions.make_OCCWire( pntList ) 
    return wire
                
def OCC2DgfGeometry( OCC_Geom ,  _DgfGeom , _3DPointSet ):
    """support function that converts a 3D OpenCascade geometry into a DigiFract geometry (i.e. adds its vertices to the 3DPointSet"""
    PntIndexList = []
    #-- NJ 03-08 this function is not yet properly implemented
    #--                how to get coords of OCC_Geom?
    
    #-- NJH 14-02-2013: below assumes that coord is 2D, how about 3D OCC_Geom?
    for coord in OCC_Geom.coords:
        PntIndexList.append( _3DPointSet.add( DgfCore.DgfVertex( coord[0] ,  coord[1]  ) )  )
    _DgfGeom.add(PntIndexList)
    return True

if loaded_threeDGeom:

    def Dgf2OCCGeometry( _DgfGeom,  _vertexSet ,  _translateVector=None ):
        """support function that converts a DigiFract geometry  into a 3D OpenCascade geometry"""
        ##logger.debug( "in Dgf2OCCGeometry with _DgfGeom: %s"  % _DgfGeom)
        ##logger.debug( "in Dgf2OCCGeometry with _vertexSet: %s %s %s" % (type(_vertexSet),  _vertexSet,  dir(_vertexSet) ) )
        #-- NJH type() == 0 -> a Point
        #-- NJH 29-09-2014: our current DgfGeometry is not QgsGeometry based and thus has no type() and isMultipart() member functions
        
        _vertexBuffer = DgfCore.DgfVertex( )
        
        ##logger.debug("in Dgf2OCCGeometry with _DgfGeom: %s | type(_DgfGeom): %s | len(_DgfGeom): %s | _DgfGeom(0): " % ( _DgfGeom,  type(_DgfGeom),  len(_DgfGeom) ) )
        #if _DgfGeom.type() == 0:
        if len(_DgfGeom) == 0: 
            return None
        elif len(_DgfGeom) == 1:
            _OCCGeom = OCC_GeomMdlFunctions.make_OCCVertex( gp.gp_Pnt( _DgfGeom[0] )  )
            return _OCCGeom
        else:
            xyzVertexList = list()
            for iVertex in _DgfGeom:
                ##_success = _vertexSet.get( _vertexBuffer,  iVertex) 
                if _translateVector:
                    _vertexBuffer = _vertexSet.get( iVertex ) + _translateVector
                    logger.debug("in Dgf2OCCGeometry() with _vertexBuffer = _vertexSet.get( iVertex ) + _translateVector: %s %s %s " % ( _vertexBuffer , _vertexSet.get( iVertex ) ,  _translateVector ) )
                else: _vertexBuffer = _vertexSet.get( iVertex )
                logger.debug("in Dgf2OCCGeometry with vertex[%s]: %s" % ( iVertex,  _vertexBuffer ) )
                xyzVertexList.append ( gp.gp_Pnt( _vertexBuffer[0] , _vertexBuffer[1], _vertexBuffer[2] ) )
            ##logger.debug( "in Dgf2OCCGeometry with xyzVertexList: %s" % xyzVertexList )
            #-- see if the DgfGeom's start and end vertex are the same. if True then a Polygon that will be returned as a 'face', if false then a polyline, returned as a wire
            if not _DgfGeom[0] == _DgfGeom[-1]:
                _OCCGeom = OCC_GeomMdlFunctions.make_OCCWire( xyzVertexList )
                logger.debug( "in Dgf2OCCGeometry with _OCCGeom (wire): %s" % _OCCGeom )
                return _OCCGeom
            else:
                #-- NJH still returns a OCCWire, not a face
                _OCCGeom = OCC_GeomMdelFunctions.make_OCCWire(xyzVertexList)     
                return _OCCGeom


def Property(func):
    """This defines a decorator tag and should not be called directly"""
    return property(**func())

# #####################
# Martijn-- NJH 16-10-2013: Consider DgfVertex,  DgfVector and DgfSegm classes with calculus member functions implemented: 
#                                              normal_vector etc...  in- out prducts, summation of DgfVertex + DgfVector and DgfVertex + DgfSegm  etc... 
#   derived from / inspired by (openCascade) OCC.gp implementation
# ##################

class DF_Vert(DgfCore.DgfVertex):
    def __init__ (self,_x=None, _y=None, _z=None ):
        if not _x:
            DgfCore.DgfVertex.__init__(self )
        else:
            DgfCore.DgfVertex.__init__(self,  _x, _y, _z)
        
class DF_Vect(DgfCore.DgfVertex):
    def __init__ (self,_x=None, _y=None, _z=None ):
        if not _x:
            DgfCore.DgfVertex.__init__(self )
        else:
            DgfCore.DgfVertex.__init__(self,  _x, _y, _z)


#-- for later
class DF_Point():  #-- / wrapped from DgfVertex c++ implementation class. See if c++ and python wrapped calss should have these different naming convenctions Dgf andd DF_ 
    def __init__ (self, parent, _vertex ):
        #xyzVertexSet, abVertexSet,
        pass
class DF_Segm():
    def __init__ (self, parent, xyzVertexSet, abVertexSet ):
        pass
class DF_Linestring():
    def __init__ (self, parent, xyzVertexSet, abVertexSet ):
        pass
class DF_Polygon():
    def __init__ (self, parent, xyzVertexSet, abVertexSet ):
        pass
class DF_SplineCurve():
    def __init__ (self, parent, xyzVertexSet, abVertexSet ):
        pass
class DF_TriangleSurface():
    def __init__ (self, parent, xyzVertexSet, abVertexSet ):
        pass
class DF_SplineSurface():
    def __init__ (self, parent, xyzVertexSet, abVertexSet ):
        pass
        
class DF_Geometry( DgfCore.DgfGeometrySet ):
    """DigiFract Geometry base class holding all geometry aspects including 2DQgsGeometry and 3D_OCC_Shape/Geom aspects and orientation aspects"""
    #-- NJH 17-11-2012 -- hmm should orientation be an geometric aspect or a feature attribute??
    def __init__ (self, parentGeometryCollection, _geometry,  xyzVertexSet=None ):
        self.parentGeometryCollection = parentGeometryCollection    #-- is the GeometryCollection  that sits in the holder (i.e. the parent
                                                                                                    #that has abVertexSet and xyzVertexSet

       #-- set Qgs_2DGeom / OCC_Geom / _QgsFeatureAttributeMap
        if isinstance( _geometry ,  QgsCore.QgsGeometry ):
            DgfCore.DgfGeometrySet.__init__( self, "Geom", self.getVertexSet, self.parentGeometryCollection.xyzVertexSet )
            Qgs2DgfGeometry( _geometry , self,  self.getVertexSet )
        
        #-- NJH 10-12-2013: the below five lines still have to be properly implemented, including the function OCC2DgfGeometry
        elif loaded_OCC:
            if isinstance( _geometry , TopoDS.TopoDS_Shape ):
                DgfCore.DgfGeometrySet.__init__( self, "Geom", self.getVertexSet, self.parentGeometryCollection.xyzVertexSet )
                if self.parentGeometryCollection.Dim == 'three':
                    pass
                else:
                    OCC2DgfGeometry( _geometry , self,  self.getVertexSet  )
            else:
                #-- NJH 07-05-205: Still need to look into this parsing of TopDS sapes anyways. 
                pass
        else:
            pass
            ##_DgfGeometry.__init__( self, vertexSet,  xyzVertexSet )

        #-- NJH 09-098-2013-- in case DF_Geometry should support spline geometry geneartion and return.
        #--                               in that case store some default spline rendering params....
        #self.SplineApproxParams = { 'Tol3d': 0.2 , 'Order': GeomAbs.GeomAbs_C0 ,
        #                         'MaxSegments': 1 , 'MaxDegree': 8 }

# #####################
# Martijn-- NJH 09-10-2013: determine what 2d and 3d geometry representations we return that are made of vertices in (a,b)-space and (x,y,z)-space
#                                               +  in what dataformats: asQgs, asWkt, asOCC, asgOcad()
#                                        then implement member functions like 
#                                           def asQgsGeom()    in true 2-D (flattened) space in (u,v) coordinate system, a translation of the (a,b) parameter space coordinates. 
#                                           def asWktGeom()    in true 2-D (flattened) space in (u,v) coordinate system
#                                           def asOCCGeom()    in 3-D space in (x,y,z) coordinate system
#                                           def asgOcadGeom() in 3-D space in (x,y,z) coordinate system
#
# ##################
##    def asQgsGeom()
##    def asWktGeom()
##    def asOCCGeom()
##    def asgOcadGeom() 
##    etc..

    @Property
    def getVertexSet( ):
        """property function to OCC TopoDS representation of 3D geometry"""
        def fget( self ):
            if self.parentGeometryCollection.vertexSetMode == 'UV':
                return self.parentGeometryCollection.uvVertexSet 
            else:
                return self.parentGeometryCollection.abVertexSet
        return locals()
    
    @Property
    def asTopoDSxyz( ):
        """property function to OCC TopoDS representation of 3D geometry"""
        def fget( self ):
            ##if self.type() == linestring:  #-- DgfGeometry has no type function yet
            indexList = self.get(0)
            pntList = []
            for index in indexList:
                xyzPoint = DgfCore.DgfVertex()
                self.parentGeometryCollection.xyzVertexSet.get(xyzPoint,  index)
                pntList.append ( gp.gp_Pnt( xyzPoint[0] , xyzPoint[1], xyzPoint[2] ) )
        
            logger.debug("in asTopoDSxyz() property function with pntList: %s" % pntList)
            wire = OCC_GeomMdlFunctions.make_OCCWire( pntList )
            logger.debug( "in asTopoDSxyz() property function with wire: %s" %  wire )
            return wire
        return locals()
    
    @Property
    def asTopoDSab( ):
        """property function to OCC TopoDS representation of 3D geometry"""
        def fget( self ):
            ##if self.type() == linestring:  #-- DgfGeometry has no type function yet
            indexList = self.get(0)
            pntList = []
            for index in indexList:
                _vertex = self.getVertexSet.get( index)
                pntList.append ( gp.gp_Pnt( _vertex[0] , _vertex[1], 0.0 ) )
        
            logger.debug("in asTopoDSab() property function with pntList: %s" % pntList)
            wire = OCC_GeomMdlFunctions.make_OCCWire( pntList )
            logger.debug( "in asTopoDSab() property function with wire: %s" %  wire )
            return wire
        return locals()
        
    @Property
    def asWKT_ab():
    #def as2Dwkt():
        """property function to 2D wkt (string) description of self"""
        def fget( self ):
            if self and self.wkbSize() != 0 :
                #--NJH 22-08-2012: there should be a ToWkt() function that always simply returns the 2D vertices from the 2D abVertexSet
                #--                                        Not a function that returns a 2D or 3D version depending on the setDimension(...)                   
                _wktGeometry = ''
                pntIdxList = self.get(0)
                pntslist_string = ''
                for pntIdx in pntIdxList[:-1]:
                    ##abPoint = DgfCore.DgfVertex()
                    _vertex = self.getVertexSet.get( pntIdx )
                    pntslist_string = pntslist_string + ' %s %s ,' % ( _vertex[0] , _vertex[1] )
                    
                _wktGeometry = "LINESTRING (%s)" % pntslist_string
                return _wktGeometry
            else: return None
        return locals()
    
    @Property
    def asWKT_xyz():
        """property function to 3d xyz wkt (string) description of self"""
        def fget( self ):
            #-- see for three-dee wkt goemetry descriptions: http://workshops.boundlessgeo.com/postgis-intro/3d.html
            if self and self.wkbSize() != 0 :
                _wktGeometryZ = ''
                pntIdxList = self.get(0)
                pntslist_string = ''
                for pntIdx in pntIdxList[:-1]:
                    xyzPoint = DgfCore.DgfVertex()
                    self.parentGeometryCollection.xyzVertexSet.get(xyzPoint,  pntIdx)
                    pntslist_string = pntslist_string + ' %s %s %s ,' % ( xyzPoint[0] , xyzPoint[1], xyzPoint[2] )
                xyzPoint = DgfCore.DgfVertex()
                self.parentGeometryCollection.xyzVertexSet.get(xyzPoint,  pntIdxList[-1])
                pntslist_string = pntslist_string + ' %s %s %s ,' % ( xyzPoint[0] , xyzPoint[1], xyzPoint[2] )
                #if type == 'linestring': 
                #    _wktGeometryZ = "LINESTRING Z (%s)" % pntslist_string
                #if type == 'polygon': 
                #    _wktGeometryZ = "POLYGON Z (%s)" % pntslist_string
                _wktGeometryZ = "LINESTRING Z (%s)" % pntslist_string
                return _wktGeometryZ
            else: return None
        return locals()

    @Property
    def as3DTopoDS():
        """property function returning the 3D (x,y,z) geometry representation of self"""
        def fget( self ):
            pass
        return locals()
        
    @Property
    def centerPointWkt():
        """property function to centroid of self.geometry"""
        def fget( self ):
            pass
            ##centroid = self.geometry().centroid()
            ##return centroid.exportToWkt()
        return locals()

# #####################
# Martijn-- NJH 09-10-2013: implement in c++: SegmAt(distance _u double) and PointAt(distance _u double) as DgfGeometry c++ member functions
# ##################
    #-- NJH 01-08-2013: give the segment at distance _u along DF_Geometry
    def SegmAt( self,  _u):
        _segm = geomBasicOperations.SegmAt(self, _u)
        return _segm
    
    #-- NJH 01-08-2013: give the point at distance _u along DF_Geometry
    #-- NJH 09-10-2013: different from VertexAt() instead of VertexAt(Index int), which gives the vertex at given index, 
    #--                              this PointAt(distance float) gives a Point precisely at distance along geometry from its start.
    def PointAt( self,  _u):
        _point = geomBasicOperations.PointAt(self, _u)
        return _point
    
# #####################
# Martijn-- NJH 09-10-2013: Consider a better worked-out c++ implementation of (possibly) a class Segm and dir() and normal() vectors
# ##################
    #--NJH 01-08-2013: VPnt is vector stored as QgsPoint() (currently for pragmatic reasons)
    #-- NJH 04-10-2013: It might be better to have a seperate DgfVector class that can be cast into a  QgsPoint as well as as an OCC gp_vec3
    #-- NJH 09-10-2013: This below implementation may invite to be restructured: have a class segment with member functions dirvector() and normalvector()
    def segmentDirVPnt( self,  _segm ):
        return geomBasicOperations.segmentDirVPnt( _segm )

    def segmentNormalVPnt( self,  _pline,  idxPnt1, idxPnt2 ):
        ##_geom = QgsCore.QgsGeometry.fromPolyline( [ _pnt1 ,  _pnt2 ] )
        _vectorPnt = self.segmentDirVPnt(  [ _pline.vertexAt( idxPnt1 ) , _pline.vertexAt( idxPnt2 ) ] )
        return QgsCore.QgsPoint( _vectorPnt.y(), -1*_vectorPnt.x() )

    def NormalAtIdxPoint_ofLine(self, _pline,  idxPnt ):
        atportion = _u / self.length()
        idxBefore = int() ; idxAfter = int()
        _pline.adjacentVertices (IdxPnt, idxBefore, idxAfter)
        QgsCore.QgsGeometry.fromPolyline( [ ] )
        #-- NJH 09-10-2013: and the rest of the implementation??
        
    def NormalAt( self, _u):
        _vectorPnt = self.segmentDirVPnt(  self.SegmAt( _u ) )
        return QgsCore.QgsPoint( _vectorPnt.y(), -1*_vectorPnt.x() )

# #####################
# NJH 09-10-2013: The below bgnPoint and endPoint functions stay as python implementations and their implementation will be reconsidered.
# ##################
    @Property
    def bgnPoint():
        """property function to begin point of self.geometry"""
        #-- NJH 09-10-2013: currently implemented succh that the bgnPoint is the Point with lowest y-value.
        #--       This might make sense for vertical outcrops, where you define bgnPoint as lowest vertical point. 
        #--       Otherwise bgnPoint should simply be the vertex in goemetry with lowest index.
        #--       Therefore, when geologic object geoemtries are stored, take care at that point that order of vertex is in bottom-up order (if preferred).
        def fget( self ):
            bgnPoint = self.vertexAt( 0 )
            endPoint = self.vertexAt( len( self.asPolyline() )-1 )
            if bgnPoint.y < endPoint.y:  
                return bgnPoint  #-- begin Point with lowest y is bgnPoint
            else: return endPoint   #-- begin Point with lowest y is endPoint
        return locals()

    #-- NJH 25-07-2013: this function should be unnecessary as the DgfVertex will get member functions .asQgsGeom(), asWktGeom(), asOCCGeom(), .asgOcadGeom() etc..
    @Property
    def bgnPointWkt():
        """property function to begin point of self.geometry"""
        def fget( self ):
            return bgnPoint.wellKnownText()   #-- begin Point with lowest y is endPoint
        return locals()

    @Property
    def endPoint():
        """property function to end point of self.geometry"""
        def fget( self ):
            bgnPoint = self.vertexAt( 0 )
            endPoint = self.vertexAt( len( self.asPolyline() )-1 )
            if endPoint.y > bgnPoint.y:  # -- find the Point with lowest y-value
                return endPoint    #-- end Point with highest y is endPoint
            else: return bgnPoint   #-- end Point with highest y is bgnPoint
        return locals()

    #-- NJH 25-07-2013: this function should be unnecessary as the DgfVertex will get member functions .asQgsGeom(), asWktGeom(), asOCCGeom(), .asgOcadGeom() etc.
    @Property
    def endPointWkt():
        """property function to end point of self.geometry"""
        def fget( self ):
            return endPoint.wellKnownText()   #-- end Point with highest y is bgnPoint
        return locals()
        
    @Property
    def centerPoint():
        """property function to centroid of self"""
        def fget( self ):
            from shapely import wkt
            _wktGeom = wkt.loads( self.wktGeometry )
            centroid = _wktGeom.centroid
            return centroid()
        return locals()
    
    #-- NJH 25-07-2013: this function should be unnecessary as the DgfVertex will get member functions .asQgsGeom(), asWktGeom(), asOCCGeom(), .asgOcadGeom() etc..
    @Property
    def centerPointWkt():
        """property function to centroid of self"""
        def fget( self ):
            from shapely import wkt
            _wktGeom = wkt.loads( self.wktGeometry )
            centroid = _wktGeom.centroid
            return centroid.to_wkt()
        return locals()


#-- might contain some usefull code to be still integrated in the above; otherwise redundant
class DigiFract_blabla(object):
    """DigiFract Geometry base class holding all geometry aspects including 2DQgsGeometry and 3D_OCC_Shape/Geom aspects and orientation aspects"""
    def __init__ (self, parent=None,  _QgsGeometry=None , _TopoDS_shape = None,   _QgsFeatureAttributeMap=None):
        print "in init DigiFract_Feature"
        if parent:                                         #-- the Question -> should a feature 'know' of itself to what parent it belongs, whether this parent be a dataLayer which contains multiple of the same Features  or the outcropStation relative to whcih the Feature coordiantes are defined.
            self.parent_outcropStation = parent           #-- for instance, it is better to store the AttributeNameList per dataLayer instead of per Feature... many features share the same set of attributes - i.e. the de facto DataLayer with one attribute table. The names of the attributes are stored at higher level. Yet for a feature you may want to get the 'fractype' attribute. How would the feature know under what index this attribute is stored unless it asks its parent for the index?
            print "set parent: ",  parent,  self.parent_outcropStation
        else:
            print "no parent set"
            self.parent_outcropStation = None
        
        if _QgsGeometry:
            self.QgsGeometry = _QgsGeometry
            ##self.PlanarGraph = GEOS.PlanarGraph()
        else:
            self.QgsGeometry = None
        
        if _TopoDS_shape:
            self.TopoDS_shape = _TopoDS_shape
        else:  self.TopoDS_shape = None
        
        if _QgsFeatureAttributeMap:
            self.AttributeList = _QgsFeatureAttributeMap
        else:
            self.AttributeList = None
        
        if self.parent_outcropStation and self.AttributeList:
            
            self.orient = orientation.Orientation(self.dipdir , self.dipang )
        else: 
            self.orient = orientation.Orientation( None , None )

    def set_qgsGeometry(self ,  _qgsGeometry ):
        self.QgsGeometry = _qgsGeometry

    def set_occShape(self ,  _shape ):
        self.TopoDS_shape = _shape
        
    def get_qgsGeometry(self):
        pass
    
    def get_geosGeometry(self):
        pass
    
    def get_topoDSShape(self):
        if self.TopoDS_shape:
           return self.TopoDS_shape
        elif self.parent_outcropStation and self.QgsGeometry:
            shape = geommodeler.translate_2Dto3DGeometry( self.parent_outcropStation ,  self.QgsGeometry )
            return shape
        else: return false
    
    def get_topoDSShape_asWire(self):
        if self.TopoDS_shape.ShapeType() == 5:
            return TopoDS.TopoDS_Wire( self.TopoDS_shape )
        else: return None
            
    def get_topoDSShape_asFace(self):
        if self.TopoDS_shape.ShapeType() == 4:
            return TopoDS.TopoDS_face( self.TopoDS_shape ) 
        else: return None
    
    def get_topoDSShape_asShell(self):
        print "self.TopoDS_shape: ",  self.TopoDS_shape,  dir(self.TopoDS_shape)
        print "self.TopoDS_shape.IsNull(): ",  self.TopoDS_shape.IsNull()
        print "self.TopoDS_shape.ShapeType(): ",  self.TopoDS_shape.ShapeType()
        if self.TopoDS_shape.ShapeType() == 3:
            return TopoDS.TopoDS_shell( self.TopoDS_shape) 
        else: return None
    
    def get_SplineCurve(self):
                #-- check out Module BRepAdaptor :: Class BRepAdaptor_Surface
        if self.TopoDS_shape:
           shape = self.TopoDS_shape
        elif self.parent_outcropStation and self.QgsGeometry:
            shape = geommodeler.translate_2Dto3DGeometry( self.parent_outcropStation ,  self.QgsGeometry )
        else: return false
    
        print "shape: ",  shape
        if shape.ShapeType() == 5:
            Tol3d = self.SplineApproxParams['Tol3d'] ; Order = self.SplineApproxParams['Order']
            MaxSegments = self.SplineApproxParams['MaxSegments'] ; MaxDegree = self.SplineApproxParams['MaxDegree']
            curve = geommodeler.convert_OCCWire2SplineCurve( shape , Tol3d , Order , MaxSegments , MaxDegree )
            return curve
        else: return False
        
    def Adapt2Geom_Surface(self):
        #-- check out Module BRepAdaptor :: Class BRepAdaptor_Surface
        print "self.TopoDS_shape: ",  self.TopoDS_shape,  repr(self.TopoDS_shape),  dir(self.TopoDS_shape)
        if self.TopoDS_shape.ShapeType() == 4:
            adaptor = BRepAdaptor.BRepAdaptor_Surface( TopoDS.TopoDS_face( self.TopoDS_shape )  )
            surface = adaptor.Surface().Surface().GetObject()
            print ##"adaptor.Surface(): ",  surface,  dir(surface),  adaptor.Surface().Surface()
            return surface
