# ################################################
#   
#   fractModelSets_class
#
#    --------
#
#    Copyright (c) 2012 - 
#       N.J. Hardebol - N.J.Hardebol@tudelft.nl  - (a)
#
#     (a) Delft University of Technology, Department of Geotechnology,  Stevinweg 1, 2628 CN Delft, the Netherlands
#
# #############################################

import os
import logging
import numpy


import qgis.core as QgsCore
from PyQt4 import QtCore
from digifract import dgfcore as  DgfCore

##import DigiFractLib
import twoDGeomBaseLib.core

from stochSimLib import fractGenerators

from twoDGeomBaseLib.core import DF_GeometryCollection

logger = logging.getLogger("DigiFractLogger.fractModelSets")

class Sfm_modelSpaceCollection( object ):
    """    LocationSpace python base class holding basePoints, baseLines and orientation description of the digitizing surface in 2.5Dim
    """
    def __init__(self , _dataSource , _domainBound ):
        pass


class Sfm_modelSpace( object ):
    """    OutcropSpace python base class holding basePoints, baseLines and orientation description of the digitizing surface in 2.5Dim
    """
    def __init__(self , _dataSource , _domainBound ):
        self.set_dataSource( _dataSource)
        
        self.set_domainBound( _domainBound )
        
        #self.xyzVertexSet   = DgfCore.DgfVertexSet("xyzVertexSet", DgfCore.DgfVertex.THREE_DEE) #-- implemented as (private) data-member in c++ class
        self.abVertexSet     = DgfCore.DgfVertexSet("abVertexSet", DgfCore.DgfVertex.TWO_DEE) #-- implemented as (private) data-member in c++ class
        self.uvVertexSet     = DgfCore.DgfVertexSet("uvVertexSet", DgfCore.DgfVertex.TWO_DEE) #-- implemented as (private) data-member in c++ class
    
    def set_dataSource(self,  _dataSource):
        self.dataSource = _dataSource
    
    def set_domainBound(self,  _bound ):
        """member function for adding domain boundary"""
        logger.debug("in Sfm_modelSpace.set_domainBound() with _bound: %s to be added" % ( _bound ) )
        
        self.domainBound = QgsCore.QgsGeometry( _bound )
        logger.debug("in Sfm_modelSpace.set_domainBound() with self.domainBound: %s" % ( self.domainBound ) )

        return True
    
    def connect2models(self,  _mdlNameList):
        pass
    

class DF_GeometryCollection( DgfCore.DgfGeometrySet ):
    """Fracture python class holding a collection of DgfGeometries and with pointers to abVertexSet and xyzVertexSet"""
    def __init__ ( self, name,  pFeatureSet ,  _vertexSetMode = 'UV' ):
        DgfCore.DgfGeometrySet.__init__( self, name )

        self.pFeatureSet = pFeatureSet
        #self.set_dimmode('2D')
        self.set_vertexSetMode(_vertexSetMode)
    
    def set_vertexSetMode(self,  _mode ):
        """member function for setting the vertexSet mode to AB (relative 2D coordinates) or UV for meter coordinates."""
        self.vertexSetMode = _mode
        logger.debug("in set_vertexSetMode with _mode: %s %s " % ( _mode,  self.vertexSetMode ) )
        if self.vertexSetMode == 'UV':
            self.associate(self.pFeatureSet.parentNetwork.dataSpace.uvVertexSet )
            self.pVertexSet = self.pFeatureSet.parentNetwork.dataSpace.uvVertexSet 
        if self.vertexSetMode == 'XYZ':
            self.associate( self.pFeatureSet.parentNetwork.dataSpace.xyzVertexSet )
            self.pVertexSet = self.pFeatureSet.parentNetwork.dataSpace.xyzVertexSet
        logger.debug("in set_vertexSetMode with self.pVertexSet: %s" % ( self.pVertexSet.name() ) )
        
    def add_QgsGeom(self,  _geom ):
        """member function that converts a 2D QGis geometry into a DigiFract geometry (i.e. adds its vertices to the 2DPointSet"""
        #_dgfGeom = DgfCore.DgfGeometrySet()  #-- NJH 23-04-2015: DgfGeometrySet should be renamed to DgfGeometry
        
        if  not _geom: 
            ##logger.debug("in add_QgsGeom() with _geom: %s" % _geom )
            return None
        
        print("in add_QgsGeom() with _geom.type() : %s : %s " % (_geom.type(),  _geom.exportToWkt() ) )
        if _geom.type() == 0: #-- QgsGeoemtryType: 0 = Point, 1 = LineString, 2 =
            qgsPoint =_geom.asPoint()
            _dgfVertex = DgfCore.DgfVertex( qgsPoint[0] ,  qgsPoint[1] )
            vertexIdx = self.pVertexSet.add( _dgfVertex )
            logger.debug("in add_QgsGeom() - about to add the vertexIdx of a Point to _DgfGeometrySet that are returned from the _2DVertexSet: %s" % ( vertexIdx ) )
            #_idx = _dgfGeom.add( vertexIdx )
            geomIdx = self.add( vertexIdx )
        elif _geom.type() == 1: #-- QgsGeoemtryType: 0 = Point, 1 = LineString, 2 =
            vertexSetIndexList = list()
            for qgsPoint in _geom.asPolyline():
                _dgfVertex = DgfCore.DgfVertex( qgsPoint[0] ,  qgsPoint[1] ) 
                _vertexIdx = self.pVertexSet.add( _dgfVertex )
                vertexSetIndexList.append( _vertexIdx ) 
            geomIdx = self.add(vertexSetIndexList)
        elif _geom.type() == 2: #-- QgsGeoemtryType: 0 = Point, 1 = LineString, 2 =
            vertexSetIndexList = list()
            for ring in _geom.asPolygon():
                logger.debug("in add_QgsGeom() with _geom.type(): %s | ring: %s" % ( _geom.type(),  ring ) )
                for iPoint in range( len( ring )-2 ):
                    qgsPoint = ring[iPoint]
                    _dgfVertex = DgfCore.DgfVertex( qgsPoint[0] ,  qgsPoint[1] ) 
                    _vertexIdx = self.pVertexSet.add( _dgfVertex )
                    vertexSetIndexList.append( _vertexIdx )
                vertexSetIndexList.append( vertexSetIndexList[0] )
                #-- For Polygon, the last vertexIDx should be the first vertexIdx
                logger.debug("in add_QgsGeom() after pVertexSet.add..ing polygon ring with vertexSetIndexList: %s" % ( vertexSetIndexList ) )
                geomIdx = self.add( vertexSetIndexList )
                ##logger.debug("in add_QgsGeom() after adding vertexSetIndexList" )
        
        return geomIdx


#class fractSets(list):
class fractSets(QgsCore.QgsVectorLayer ):
    def __init__ ( self, _dataSpace,  _fractSets=list(),  _name = 'all',  providerName = 'ogr' ):
        #list.__init__(self, _fractSets )
        self.dataSpace = _dataSpace
        self._name = _name
        self.Names2IdxDict = dict()
        self.fractSets = list()
        
        logger.debug("in fractSets() with self.dataSpace.dataSource: %s" % ( self.dataSpace.dataSource ) )
        _fractSetFilePath = os.path.normpath( os.path.join(  self.dataSpace.dataSource ,  "fract_%s.shp" % ( self._name ) ) )
        logger.debug("in fractSets() with _fractSetFilePath: %s" % ( _fractSetFilePath ) )
        
        if providerName == 'memory':
            QgsCore.QgsVectorLayer.__init__( self, _fractSetFilePath, self._name, providerName )
        elif isinstance( _fractSetFilePath , str ):
            if os.path.isfile(_fractSetFilePath):
                QgsCore.QgsVectorLayer.__init__( self,  _fractSetFilePath, self._name, providerName )
            else:
                _lyr = QgsCore.QgsVectorLayer( "LineString?field=id:integer&field=DgfGeomIdx:integer&field=LevelNm:string(10)&field=SetNm:string(15)&field=ClusterIdx:integer" , self._name , "memory")
                QgsCore.QgsVectorFileWriter.writeAsVectorFormat( _lyr, _fractSetFilePath ,  'System', QgsCore.QgsCoordinateReferenceSystem( 28992, QgsCore.QgsCoordinateReferenceSystem.EpsgCrsId ) )
                QgsCore.QgsVectorLayer.__init__( self,  _fractSetFilePath, self._name, providerName )
        else:
            logger.warning( "failed to initiate a DF_FeatureCollection  with _fractSetFilePath: %s of type: %s" % ( _fractSetFilePath ,  type(_fractSetFilePath) ) )
            raise ValueError("currently not possible to initiate fractModelSet %s of type: %s" % ( _fractSetFilePath , type(_fractSetFilePath) ) )

        logger.debug("in fractModelSets.__init__ with self.dataProvider(): %s" % self.dataProvider() )
        logger.debug("in fractModelSets.__init__ with self.dataProvider().geometryType(): %s" % self.dataProvider().geometryType() )
        logger.debug("in fractModelSets.__init__ with self.dataProvider().fields(): %s" % [field.name() for field in self.dataProvider().fields() .toList() ])

    def add_set(self, _fractSet ):
        ##logger.debug( "in fractNetwork.add_set() with _fractSet: %s %s" % ( _fractSet._name,  _fractSet.featureCount() ) )
        self.fractSets.append( _fractSet )
        self.Names2IdxDict[_fractSet._name] = len(self.fractSets) - 1
        return True
    
    def find_set(self,  _name):
        idx = self.find_Idx4Name(_name)
        return self.fractSets[idx]
    
    def replace_set(self,  _name,  _set):
        if not _name == _set._name:
            _set.set_name( _name )
        idx = self.find_Idx4Name(_name)
        self[idx] = _set
    
    def remove_set(self,  _name ):
        idx = self.find_Idx4Name(_name)
        if idx: 
            del self.fractSets[idx]
            return True
        return False
        
    def find_Idx4Name(self, _name ):
        _selectedIdx = None
        for fractSetIdx in range( len(self.fractSets) ):
            logger.debug("in fractNetwork.find_set with _name to find: %s - looping through fractSet at name: %s" % (_name, self.fractSets[fractSetIdx]._name ) )
            if self.fractSets[fractSetIdx]._name == _name:
                _selectedIdx = fractSetIdx
                logger.debug("in fractNetwork.find_set %s  equals %s" % (_name, self.fractSets[fractSetIdx]._name ) )
                break
        logger.debug("in fractNetwork.find_set  - about to return selectedSet: %s" % ( _selectedIdx ) )
        return _selectedIdx
    
    def get_fractures_ofSetName(self,  _setName ):
        selectedFractures = list(self.getFeatures(  QgsCore.QgsFeatureRequest().setFilterExpression( u'"SetNm" = \'%s\'' % ( _setName ) )) )
        return selectedFractures
        
    def get_fractures_ofFracSetIdxs(self,  _fracSetIdxs ):
        ##logger.debug( "in get_fractures_ofFracSetIdxs() with _fracSetIdxs: %s" % _fracSetIdxs )
        ##logger.debug( "in get_fractures_ofFracSetIdxs() with _fracSet namelist: %s" % [self.fractSets[_Idx].name for _Idx in _fracSetIdxs] )
        mergedFractList = list()
        for _Idx in _fracSetIdxs:
            _name = self.fractSets[_Idx].name
            _fracs = self.get_fractures_ofSetName( self.fractSets[_Idx].name )
            mergedFractList.extend( _fracs )
        return mergedFractList
        
    def nextFreeFeatureID( self ):
        """member function that gives a new (unused) feature_id"""
        #-- NJH 02-04-2014: how about  using simply the QgsVectorLayer.findFreeId() function i.e. the fId.
        #--                             No-- findFreeId is a private member function! 
        #--                             Still can we somehow find the fId that will be assigned to the feature before we committed (or even created) the new feature?
        if self.dataProvider().featureCount() == 0:
            return 0
        elif  len(list(self.getFeatures())) == 0:
            return 0
        else:
            list_of_IDs = []
            for feat in self.getFeatures():
                list_of_IDs.append( feat.id() )
        maxFractID = max( list_of_IDs )
        return maxFractID  + 1

class networkLevel(object):
    """
    Todo:
    """
    def __init__ ( self, _parentNetwork,  _name,  _fractSets,  _superpositionprobs ):
    #def __init__ ( self, _parentNetwork,  _name,  _fractSets,  _p21s,  _superpositionprobs ):
        self.parentNetwork = _parentNetwork
        self.name = _name
        self.fractSetIdxs = None
        self.set_fractSetIdxs(_fractSets)    #-- in C++ implementation this should be a pointer to the list of vertices
        self.superpositionprobs = None
        self.set_superpositionprops( _superpositionprobs )
    
    def deepcopy(self):
        import copy
        logger.debug( "in networkLevel.copy()" )
        _copy = networkLevel( self.parentNetwork,  self.name, copy.deepcopy( self.fractSetIdxs ), copy.deepcopy( self.superpositionprobs ) )
        logger.debug( "in networkLevel.copy() with self.name: %s | self.fractSetIdxs: %s" % (_copy.name,  _copy.fractSetIdxs) )
        return _copy        
    
    def set_fractSetIdxs(self,  _fractSets):
        """ """
        logger.debug( "in networkLevel.set_fractSetIdxs() with _fractSets: %s" %  _fractSets )
        if isinstance( _fractSets[0]  ,  str ):
            _fractSets = [self.parentNetwork.find_Idx4Name(_name) for _name in _fractSets]
            self.fractSetIdxs = _fractSets
        if isinstance( _fractSets[0]  ,  int ):
            self.fractSetIdxs = _fractSets
        return True
    
    def set_superpositionprops(self,  _probabilities):
        if  _probabilities is not None:
            _sum = numpy.sum(_probabilities)
            self.superpositionprobs = _probabilities/_sum
            return True
        else: 
            return False
    
    def get_cumul_probs(self):
        return numpy.cumsum(self.superpositionprobs)
    
    def remove_set(self,  _set):
        j =None
        if isinstance(_set,  str):
            _set = _parentNetwork.find_Idx4Name(_set)
        
        if not isinstance(_set,  int):
            return False
        
        for l,  _idx in enumerate( self.fractSetIdxs ):
            #logger.debug("in remove_set with self.fractSetIdxs: %s and _setIdx: %s" % (self.fractSetIdxs,  _set ) )
            if _set == _idx:
                #logger.debug("succesful - found _set in self.fractSetIdxs: %s == %s" % (self.fractSetIdxs,  _set ) )
                j = l
                break;
        
        if j is None:
            logger.debug("in remove_set - failed to remove set: %s " % _set )
            return False
        
        del self.fractSetIdxs[j]
        ##del self.p21s[j]
        _superpositionprobs = numpy.delete(self.superpositionprobs, [j])
        self.set_superpositionprops( _superpositionprobs )
        #logger.debug( "in remove_set - successfully removed set: %s" % (_set) )
        return True
    
#    def get_p21(self, fractSet):
#        """get the p21 for a fracture set with of given index within this levelSet"""
#        if isinstance(fractSet,  str):
#            _fractSetIdx = find_Idx4Name(_set)
#            for l,  _idx in enumerate( self.fractSetIdxs ):
#                if _fractSetIdx == _idx:
#                    return self.p21s[ l ]
#        if isinstance(fractSet,  int):
#            for l,  _idx in enumerate( self.fractSetIdxs ):
#                if fractSet == _idx:
#                    return self.p21s[ l ]
#        return None
    
class fractNetwork(fractSets):
    """
        Class for (stochastic modelling of fracture network that consists of multiple fracturesets.
        Derived from the fractSets baseclass
        
        Todo:
            Why should this class be initiated with _fractSets. 
            Those can also be assigned later. Giving _fractSets during initation has led to some bugs in which the set was shared between different instances of this class.
            The fractSets is now initiated with a None for _fractSets. Just to avoid bug issues of shared data members between different isntances of this the fractSets class
            
            (2) What reason for having three successive classes: fractModelSets inherits from fractNetwork, fractNetwork is derived from fractSets.
                
                Why these three classes. Maybe merge fractNetwork and fractSets? Or merge fractNetwork and fractModelSets?
                       Or, keep them seperate but better specify fractSets are just collection of fractSet(s). 
                       fractNetwork has the implementation of fractLevels, hierarchy, topology, future path implementation?
    
    """
    #-- a set of fractureSets that exhibit a certain spatial, topological order
    def __init__ ( self, _dataSpace,  _fractSets,  _name ):
        ##self.dataSpace = _dataSpace
        fractSets.__init__(self, _dataSpace, None,  _name )
        ##self.extend( _fractSets )
        self.levels = list()
        ##self.p21s = {}
        ##self.superpositions = {}
        self.Names2IdxDict = {}

    def add_Level(self,  _levelName,  _fractSetNameList=None,  _superpositionprobs=None,  _blabla=None):
        #-- NJH 04-10-2013: if _levelName already exists as key in self.levels then that one is overwritten by new list()
        
        _networkLevel = networkLevel( self,  _levelName , _fractSetNameList , _blabla, _superpositionprobs  )
        for ilevel,  level in enumerate( self.levels ):
            if level.name == _networkLevel.name:
                self.levels[ilevel] = _networkLevel
                return True
        self.levels.append(_networkLevel)
        logger.debug( "in add_Level() with self.levels: %s" % self.levels )
        return True
    
    def get_Level4fractSetName(self, _fractSetName):
        _fractSetIdx = self.Names2IdxDict[ _fractSetName ] 
        ##logger.debug( "in fractModelSets.get_Level() with _fractSetName: %s  _fractSetIdx: %s" % ( _fractSetName,  _fractSetIdx) )
        for ilevel,  level in enumerate( self.levels ):  #-- levels.iteritems DO NOT return fractSetNames but fractSetIdxes...!!!!
            ##logger.debug( "in fractModelSets.get_Level() with lvlName, level.fractSetIdxs: %s | %s " % ( level.name,  level.fractSetIdxs ) )
            if _fractSetIdx in level.fractSetIdxs:
                ##logger.debug( "found %s as being the levelName to which set %s belongs" % ( level.name,  _fractSetIdx ) )
                return level
        return None

    def get_fractures_ofSameLevel(self,  _levelName=None):
        ##logger.debug( "in get_fractures_ofSameLevel() with _levelName: %s" % _levelName )
        if not _levelName:
            return []
        mergedFractSet = DigiFractLib.fractdatamodel.fracture.FractureSet( self.dataSpace )
        for level in self.levels:
            if level.name == _levelName:
                for _fractSetIdx in level.fractSetIdxs:
                    _fractSet = self[_fractSetIdx]
                    mergedFractSet.add_collection( _fractSet )
                return mergedFractSet
        return list()
    
#    def get_allFractures(self,  _exclusionIdxs=None):
#        logger.debug( "in get_fractures_ofSameLevel() with _levelName: %s" % _levelName )
#        ##mergedFractSet = DigiFractLib.fractdatamodel.fracture.FractureSet( self.dataSpace )
#        mergedFractSet =  fractModelSet( self,  self.name, providerName = 'memory')
#        for _fractIdx,  _fractSet in enumerature( self.fractSets ):
#            _fractSet = self[_fractSetIdx]
#            mergedFractSet.add_collection( _fractSet )
#            return mergedFractSet
#        return list()
        
    def concatenate(self):
        #_collection = DigiFractLib.fractdatamodel.fracture.FractureSet(self.dataSpace)
        _collection = QgsCore.QgsVectorLayer()
        for _set in self:
            #logger.debug("in fractNetwork.concatenate() with type(_set): %s" % _set)
            #_collection.add_features(_set['sim_set'])
            _collection.addFeatures(list(_set['sim_set'].getFeatures() ) ) 
        return _collection

    def indexConversion(self,  setIdx,  fractIdx,  flattenedIdx):
        if flattenedIdx is not None:
            _Idx
            for _setIdx in range(len(fractSet)):
                _prevIdx = _Idx
                _Idx+=len(fractSet[_setIdx])
                if flattenedIdx < _Idx:
                    fractIdx = flattenedIdx - _prevIdx
                    return _setIdx,  fractIdx
        else:
            while _setIdx <= setIdx:
                flattenedIdx+=len(fractSet[_setIdx])
            return flattenedIdx+fractIdx

    #def __iter__(self):
    #    self.iFractSet = 0
    #    self.iFract = 0
    #    return  self
    #def next(self):
    def allfractIterator(self , Idxs2exclude=[]):
        if not self.iFract == len(self[self.iFractSet])-1:
            self.iFract+=1
            return self[self.iFractSet][self.iFract]
        else: 
            if not self.iFractSet == len(self)-1:
                self.iFractSet+=1
                self.iFract = 0
                return self[self.iFractSet][self.iFract]
            else:
                raise StopIteration
                
    def get_allFractureFeat( self ,  exclusionIdxs = [] ):
        fractFeatures = list()
        for _iFractSet,  _fractSet in enumerate( self ):
            if not _iFractSet in exclusionIdxs:
                fractFeatures.extend( [frac for frac in _fractSet.getFeatures() ] )
        return fractFeatures
        
    def get_allFractureGeoms( self ,  exclusionIdxs = [] ):
        """return all fracture geometries across different (level) sets
        
            Previously this was a challenge as each fractureset was stored seperately, a joint list had to be merged.
            WIth all fracturesets tof the entwork stored together, get_allFractureGeoms is easy.
        """
#        fractGeoms = list()
#        for _iFractSet,  _fractSet in enumerate( self.fractSets ):
#            if not _iFractSet in exclusionIdxs:
#                fractGeoms.extend( [QgsCore.QgsGeometry( frac.geometry() ) for frac in _fractSet.getFeatures() ] )
#        return fractGeoms
        return [ QgsCore.QgsGeometry( frac.geometry() ) for frac in self.getFeatures() ]


class fractModelSets( fractNetwork ):
    """
        Class for (stochastic modelling of fracture network that consists of multiple fracturesets.
        Derived from the fractNetwork baseclass
        
        Todo:
            Why should this class be initiated with _fractSets. Those can also be assigned later. Giving _fractSets during initation has led to some bugs in which the set was shared between different instances of this class.
    """
    def __init__ ( self, _dataSpace,  _fractSets=[],  _simulationName ='mySimulation' ):
        fractNetwork.__init__(self ,  _dataSpace,  _fractSets,  _simulationName )
        ##if len(_fractSets) > 0:
        ##    self.extend(_fractSets)
    
    #-- fractSets (inherited from fractNetwork) sets the name
    ##    self.set_simulationName( _simulationName )
    ##
    ##def set_simulationName( self,  _name ):
    ##    self.name = _name
    
    # NJH 26-07-16: rename the below function to def add_simFractSet
    def add_simFractSet_n(self, _name, **_generator_specs ):
        print "in fractModelSets.add_simFractSet_n() with _generator_specs: %s" % _generator_specs
        _fractSet = fractModelSet( self,  _name,  **_generator_specs )
        logger.debug( "the instance of type fractModelSet  is set with 'name' = %s and generator to %s" % ( _fractSet._name , _fractSet.generator ) )
        self.add_set(_fractSet)
        return True

    # NJH 26-07-16: rename the below function to def get_simFractSet
    def get_simFractSet_n(self, _name ):
        for fractSet in self:
            print "%s =?= %s " % ( fractSet.name, _name )
            if fractSet.name() == _name:
                print "in get_simFractSet() with fractSet['name'] equals _name, thus return %s" % fractSet
                return fractSet
    
    def set_simFractSet(self, _name, _fractSetPointer):
        for fractSet in self:
            print "%s =?= %s " % ( fractSet['name'], _name )
            if fractSet['name'] == _name:
                fractSet['sim_set'] = _fractSetPointer
                print "the 'simFractSet' of fractSet with 'name' = %s is set to %s" % ( fractSet['name'] , fractSet['sim_set'])
    
    def add_Level(self,  _levelName,  _fractSetNameList=None, _superpositionprobs=None,  _blabla=None,  ):
        #-- NJH 04-10-2013: if _levelName already exists as key in self.levels then that one is overwritten by new list()
        
        _networkLevel = networkLevel( self,  _levelName , _fractSetNameList, _superpositionprobs  )
        #-- find if new _networkLevel of given name already exists, then it is overwritten.
        for ilevel,  level in enumerate( self.levels ):
            if level.name == _networkLevel.name:
                self.levels[ilevel] = _networkLevel
                return True
        self.levels.append(_networkLevel)
        logger.debug( "in add_Level() with self.levels: %s" % self.levels )
        return True
    
    def set_P21Limit(self, _name,  _p21):
        for fractSet in self.fractSets:
            if fractSet._name == _name:
                fractSet.set_P21Limit( _p21 )
                logger.debug("Successfully set p21 of %s for fractSet with name %s " % ( fractSet.P21Limit,  _name ) )
                return True
        logger.debug("Failed to set p21 Cannot find fractSet with name %s " % ( _name ) )
        return False
        
    def get_P21Limit(self, _name,  ):
        for fractSet in self.fractSets:
            if fractSet._name == _name:
                return fractSet.get_P21Limit(  )
        return None
    
    def set_PlacerConstraint(self, _name,  _constraint):
        for fractSet in self:
            print "%s =?= %s " % ( fractSet['name'], _name )
            if fractSet['name'] == _name:
                fractSet['placerConstraint'] = _constraint

#import networkx
class segmFractNetwork(fractNetwork):
    """
        class for fracture network that consists of multiple fracturesets.
        The fracture linestrings are segmentized at intersection points.
        
        Todo:
            Only a rudimentary implementation. Not meant to function yet.
    """
    
    def __init__ ( self, _dataSpace,  _fractSets=[] ):
        fractNetwork.__init__(self )
        if len(_fractSets) > 0:
            self.extend( _fractSets )
        self.dataSpace = _dataSpace
        
        #self.graph = networkx.
        self.xpoints = list()
        self.segments = list()
        

#class fractModelSet( QgsCore.QgsVectorLayer ):
class fractModelSet( object ):
    """
    a fracture set datalayer for (stochastic) fracture modelling: fractures can be incrementally added and stored.
    inherited from QgsVectorLayer for on-the-fly storage as GIS format (Shapefile, or other ogr format).
    
    Attributes:
        parentNetwork:
        name:
        providerName: the type of data(base) provider: ogr (for shapefiles)
        **generator_specs
    """
    
    ##def __init__( self, _parentNetwork=None,  _name=None, providerName = 'ogr',  **_generator_specs  ):
    def __init__( self, _parentNetwork=None,  _name=None,  **_generator_specs  ):
        self.set_parentNetwork(_parentNetwork)
        self.set_name(_name)
         
        ##self.GeometryCollection = DF_GeometryCollection( self._name,  self ,  'UV' )
        ##for _ifeat,  _feat in enumerate( self.getFeatures() ):
        ##    _geom = _feat.geometry()
        ##    if _geom:
        ##        self.GeometryCollection.add_QgsGeom( _geom  )
        ##    else: 
        ##        logger.warning("in DF_FeatureCollection.__init__() - found a feature (ifeat=%s) in datasource: %s that has no geometry assigned!!" % ( _ifeat , dataSource ) )     
        
        #success = self.dataProvider().addAttributes( [ QgsCore.QgsField( "id", QtCore.QVariant.Int, "integer" ),  QgsCore.QgsField( "DgfGeomIdx", QtCore.QVariant.Int, "integer" )  ] )
        print("in fractModelSet.__init__ debug1")
        self.set_generator( **_generator_specs )
        
    def set_parentNetwork(self,  _parentNetwork):
        self.parentNetwork = _parentNetwork
    
    def set_name(self,  _name):
        self._name = _name
        self.name = _name
    
    def name(self):
        return self.name
    
    def set_generator(self,  **kargs):
        logger.debug( "in fractModelSet.set_generator() with kargs: %s" % kargs )
        print("in fractModelSet.set_generator with kargs: %s" % kargs)
        
        if 'generator_type' in kargs.keys():
            _generator_type = kargs['generator_type']
            print("in fractModelSet.set_generator with just set _generator_type: %s" % _generator_type)
        else:
            logger.debug("in fractModelSet.set_generator() without a generator_type argument specified")
            return False
        #if 'parentFractModelSet' in kargs.keys():
        #    _parentFractModelSet = kargs['parentFractModelSet']
        #else: _parentFractModelSet = self
        if 'mode_randCentroid' in kargs.keys():
            _mode_randCentroid = kargs['mode_randCentroid']
        else: _mode_randCentroid = None
        if 'modelDomain' in kargs.keys():
            _modelDomain = kargs['modelDomain']
        else: _modelDomain = None
        
        if _generator_type == 'randomFracture':
            self.generator = fractGenerators.stochFracGenerator( 
                                self, _mode_randCentroid, _modelDomain ) 
        elif _generator_type == 'placedAtHOrderFract':
            logger.debug("at start of in fractModelSet.set_generator() of type 'placedAtHOrderFract'" )
            if 'placerConstraint' in kargs.keys():
                _placerConstraints = kargs['placerConstraint']
                logger.debug("in fractModelSet.set_generator() with _placerConstraints: %s" % _placerConstraints )
                if _placerConstraints is None:
                    logger.debug("in fractModelSet.set_generator() of %s failed to setgenerator of type 'placedAtHOrderFract' b/c the placerConstraint refers to unknown fractSet: %s" % ( self.name,  kargs['placerConstraint'] ) )
                    return False
            else:
                logger.debug("in fractModelSet.set_generator() of %s failed to setgenerator of type 'placedAtHOrderFract' b/c no placerConstraint key argument is found." % self.name)
                return False
            self.generator = fractGenerators.stochFractGenerator_placedAtHOrderFract( 
                                self, _mode_randCentroid, _modelDomain,  _placerConstraints  ) 
            logger.debug("in fractModelSet.set_generator() just set the generator of type 'placedAtHOrderFract': %s" % self.generator )
        
        elif _generator_type == 'nonHigherOrderIntersecting':
            logger.debug("at start of in fractModelSet.set_generator() of type 'placedAtHOrderFract'" )
            #if 'placerConstraint' in kargs.keys():
            #    _placerConstraints = kargs['placerConstraint']
            #    logger.debug("in fractModelSet.set_generator() with _placerConstraints: %s" % _placerConstraints )
            #    if _placerConstraints is None:
            #        logger.debug("in fractModelSet.set_generator() of %s failed to setgenerator of type 'placedAtHOrderFract' b/c the placerConstraint refers to unknown fractSet: %s" % ( self.name,  kargs['placerConstraint'] ) )
            #        return False
            #else:
            #    logger.debug("in fractModelSet.set_generator() of %s failed to setgenerator of type 'placedAtHOrderFract' b/c no placerConstraint key argument is found." % self.name)
            #    return False
            self.generator = fractGenerators.stochFractGenerator_nonHigherOrderIntersecting( 
                                self, _mode_randCentroid, _modelDomain  ) 
            logger.debug("in fractModelSet.set_generator() just set the generator of type 'placedAtHOrderFract': %s" % self.generator )

        elif _generator_type == 'noSelfIntersecting':
            #logger.debug("about to set the self.generator to stochFractGenerator_nonSelfIntersecting with _parentFractModelSet: %s" % _parentFractModelSet)
            self.generator = fractGenerators.stochFractGenerator_nonSelfIntersecting( 
                                self, _mode_randCentroid, _modelDomain ) 
        elif _generator_type == 'noEqualLvlIntersecting':
            self.generator = fractGenerators.stochFractGenerator_nonSelfIntersecting( 
                                self, _mode_randCentroid, _modelDomain ) 
            self.generator.set_nonSameLevelIntersect_condition(True)
        
        logger.debug("in set_generator() with parentFractModelSet.parentNetwork.dataProvider().dataSourceUri(): %s " % self.generator.parentFractModelSet.parentNetwork.dataProvider().dataSourceUri() )
    
    def set_P21Limit(self,  _p21):
        self.P21Limit = _p21
    
    def get_P21Limit(self):
        return self.P21Limit
    
    def nextFreeFeatureID( self ):
        """member function that gives a new (unused) feature_id"""
        return self.parentNetwork.nextFreeFeatureID()
        
    def addFeature(self,  _feat):
        #_feat
        self.parentNetwork.addFeature(_feat)
        return True
        

