
import logging
import numpy
from scipy import stats
import random
import math

import StoSim_algorithms
import twoDGeomBaseLib

#-- NJH 08-03-2016 Below imported powerlaw module is from https://github.com/jeffalstott/powerlaw based on http://tuvalu.santafe.edu/~aaronc/powerlaws/
#--     !!!Consider significant difference between two powerlaw implementations !!:
#--        scipy.stats.powerlaw has a positive powerlaw component. Cannot be assigned a negative one!!
#--        instead the 'powerlaw' module from https://github.com/jeffalstott/powerlaw
#--        uses a negative power law when giving a positive alpha: alpha * x ^ -alpha
import powerlaw

try:
    import shapely
    from shapely import geometry
    shapely_loaded = True
except:
    shapely_loaded = False

try:
    import qgis.core as QgsCore
    loaded_QgsCore = True
except:
    loaded_QgsCore = False
try:
    from digifract import dgfcore as  DgfCore
    loaded_DgfCore = True
except:
    loaded_DgfCore = False

logger = logging.getLogger("DigiFractLogger.RandomFracture")
  
def gen_RandomFracture( parentFractureSet, centroid, _size, fracttype=None, _dipdir=None ):
    _strike = _dipdir
    if _strike >= 360.0: _strike = _strike - 360.0
    fractGeom = twoDGeomBaseLib.core.geomBasicOperations.gen_Line_fromCPoint( centroid,  _strike, _size  )
    
    myDigiSurface = None
    #_fract = DigiFractLib.baseclasses.DF_Feature.DF_Feature( parentFractModelSet, _fractGeom,  [ self.parentFractModelSet.nextFreeFeatureID(),  -99 ] )
    fractFeat = QgsCore.QgsFeature(  parentFractureSet.parentNetwork.nextFreeFeatureID() )
    fractFeat.initAttributes( len(parentFractureSet.parentNetwork.dataProvider().attributeIndexes() ) )
    fractFeat.setAttributes( [parentFractureSet.parentNetwork.nextFreeFeatureID(),  -99, parentFractureSet.parentNetwork.name,  parentFractureSet.name,  None] )
    fractFeat.setFeatureId( parentFractureSet.parentNetwork.nextFreeFeatureID() )
    fractFeat.setGeometry(fractGeom)
    #_fract.setAttributes([ self.parentFractModelSet.nextFreeFeatureID(),  -99 ])
    logger.debug("in gen_RandomFracture() with parentFractureSet.parentNetwork.nextFreeFeatureID(): %s and fractFeat.id(): %s" % ( parentFractureSet.parentNetwork.nextFreeFeatureID(),  fractFeat.id() ) )
    return fractFeat

class SimulationError(Exception):
    def __init__(self, _value,  _centroid):
        self.cause = _value
        self.centroid = _centroid
    def __str__(self):
        return repr(self.value)

class stochFract_baseOperator( object ):
    """random Fracture simulation base class"""
    def __init__ ( self, parentFractModelSet,  mode_randCentroid='RANDinArea', modelDomain=None ):
    #-- NJH01-10-2013: the arg mode_randCentroid at initation has become redundant. Therefore replace with next init:
    #def __init__ ( self, parentFractModelSet, modelDomain=None )
        self.set_parentFractModelSet(parentFractModelSet)
        
        ##self.set_mode_randCentroid( mode_randCentroid )
        self.set_modelDomain( modelDomain )
        
        self.centroidGenerator = None
        
        self.reset_fracture()

    def reset_fracture(self):
        self.centroid = None
        self.size = None
        self.azim = None
        self.failureState = ''
        self.fracture = None
        
    def set_parentFractModelSet(self,  _set):
        self.parentFractModelSet = _set
    
    def get_fractSet(self,  _name):
        return self.parentNetwork.get_simFractSet( _name )
        
    def set_fracture(self , _fracture=None):
        self.fracture = _fracture
        self.fractAreaMassProps = None  #-- centroid and other fracture surface properties are stored here so that they are only calculated once
        ##self.centroid = None
    
    def set_modelDomain( self, domain ):
        self.modelDomain = domain
        
    ##def set_mode_randCentroid( self, mode='RANDinArea' ):
    ##    #-- mode choice: 'RANDinArea', 'RANDinVOL', 'RANDfromSURF', 'RANDfromEXTSURF'
    ##    if mode: self.mode_randCentroid = mode
    ##    else: return False
    ##    return True
    
    def set_randCentroidGenerator(self, mode_randCentroid='RANDinArea',  _params = None ):
        #-- implemented in the derived classes
        pass 

    def set_orientGenerator(self, type_ = 'vonmises',  _params = None ):
        """set the orientation distribution function for stochastic sampling an azimuth direction for the stochastic fracture"""
        if _params is not None:
            self.orientGenParams = _params
        
        if type_ == 'vonmises':
            _mean = self.orientGenParams[0] ; _kappa= self.orientGenParams[1]
            _std_dev = self.orientGenParams[2]; _samplesize = self.orientGenParams[3]
            self.orientGenerator = lambda: stats.vonmises.rvs( [_kappa],loc=_mean,  scale=_std_dev , size=_samplesize)
            logger.debug("just set the self.orientGenerator of type vonmisses with _kappa: %s | loc: %s | scale: %s | size: %s" % ( _kappa, _mean, _std_dev , _samplesize ) )
        elif type_ == 'uniform':
            _loc = self.orientGenParams[0] ; _scale = self.orientGenParams[1]
            self.orientGenerator = lambda: stats.uniform.rvs( loc=_loc, scale=_scale, size=1 )
            logger.debug("just set the self.orientGenerator of type uniform")

        return True
    
    def get_orientGenParams(self):
        return self.orientGenParams
    
    def set_sizeGenerator(self, type_ = 'lognorm',  _params = None ):
        if type_ == 'uniform':
            self.sizeGenerator = lambda: stats.uniform.rvs( loc=_params[0],  scale=_params[1]-_params[0] ,  size=1)
        if type_ == 'lognorm':
            if _params is not None:
                _mean = _params[0]; _std_dev= _params[1]; _samplesize = _params[2]
            else:
                _mean = 0.5 ; _std_dev= 1.0 ;  _samplesize = 1 #_samplesize = 1000
            #self.sizeGenerator = lambda: scipy.stats.lognorm.rvs( [_std_dev],  scale=numpy.exp(_mean) ,  size=_samplesize)
            self.sizeGenerator = lambda: stats.lognorm.rvs( [_std_dev],  scale=_params[0] ,  size=_samplesize)
        if type_ == 'expon':
            if _params is not None:
                _mean = _params[0] ; _scale= _params[1] ; _samplesize = _params[2]
            else:
                _mean = 0.5 ; _scale= 1.0 ;  _samplesize = 1 #_samplesize = 1000
            self.sizeGenerator = lambda: stats.expon.rvs( [_mean],  scale=_scale ,  size=_samplesize)
            #if self.sizeGenerator < 0.3:
            #    self.sizeGenerator = 0.3
            #else:
            #    self.sizeGenerator = self.sizeGenerator    
        if type_ == 'powerlaw':
            if _params is not None:
                #-- _params[0] = power law exponent -- choose a positive exponent for negativve powerlaw when using the powerlaw module!!
                #--  _params[1] = xmin
                _shape = _params[0]; _mean= _params[1]; _std_dev=_params[2]; _samplesize = _params[3]
            else:
                _shape = 0.4 ; _mean= 150.0; _std_dev=80.0;  _samplesize = 1 #_samplesize = 1000
            #-- NJH 08-03-2016 !!!Consider significant difference between two powerlaw implementations !!:
            #--        scipy.stats.powerlaw has a positive powerlaw component. Cannot be assigned a negative one!!
            #--        instead the 'powerlaw' module from https://github.com/jeffalstott/powerlaw based on http://tuvalu.santafe.edu/~aaronc/powerlaws/
            #--        uses a negative power law when giving a positive alpha: alpha * x ^ -alpha
            #self.sizeGenerator = lambda: scipy.stats.powerlaw.rvs( _shape, loc=_mean ,  scale=_std_dev,   size=_samplesize)
            if _params[1]:
                _law = powerlaw.Power_Law(xmin=_params[1], xmax=_params[2], parameters=[_params[0]])
            else:
                _law = powerlaw.Power_Law(parameters=[_params[0]])
            self.sizeGenerator = lambda: _law.generate_random( _samplesize )   
        if type_ == 'fixed':
            self.sizeGenerator = lambda:  [ _params[0] ]

    def set_nonIntersecting_fractSets(self,  fractSetNames,  intersect_probab=None,  abut_probab=None):
        #-- NJH 29-11-2013: currently the probabilities are not used
        logger.debug("in set_nonIntersecting_fractSets() with fractSetNames: %s" % fractSetNames)
        self.fractSetIdxs_not2Intersect = list()
        for _name in fractSetNames:
            _idx = self.parentFractModelSet.parentNetwork.find_Idx4Name( _name)
            if _idx is not None:
                self.fractSetIdxs_not2Intersect.append( _idx )
        if intersect_probab:
            pass
        if abut_probab:
            pass
        logger.debug("in set_nonIntersecting_fractSets() with self.fractSetIdxs_not2Intersect: %s" % self.fractSetIdxs_not2Intersect)
      
        return True
        
    def set_buffer_dist(self,  _dist):
        self.buffer_dist = _dist
        return True
    
    def translate( self ):
        pass
    
    def trim(self, fracture=None):
        if not fracture:
            fracture = self.fracture
        
        fractGeom = twoDGeomBaseLib.core.geomBasicOperations.crop_PLine_byPGon(  fracture.geometry() , self.modelDomain )
        
        if fractGeom is None: 
            self.set_fracture( None )
            return False
        #fractFeat = DigiFractLib.baseclasses.DF_Feature.DF_Feature(self.parentFractModelSet, fractGeom,  [self.parentFractModelSet.nextFreeFeatureID(),  -99] )
        fractFeat = QgsCore.QgsFeature( self.parentFractModelSet.parentNetwork.nextFreeFeatureID() )
        fractFeat.initAttributes( len(self.parentFractModelSet.parentNetwork.dataProvider().attributeIndexes() ) )
        fractFeat.setAttributes( [self.parentFractModelSet.parentNetwork.nextFreeFeatureID(),  -99, self.parentFractModelSet.parentNetwork.name,  self.parentFractModelSet.name,  None] )
        fractFeat.setGeometry(fractGeom)
    
        self.set_fracture( fractFeat )
        return True
    
    def gen_randCentroid( self ):
        """"""
        self.centroid = self.centroidGenerator()
    
    def gen_randSize( self ):
        self.size = self.sizeGenerator()[0]
    
    def gen_randOrient( self ):
        """pick stochastically an orientation from distribution. Orientation is used as strike by the stochastic fracture generator"""
        self.azim = self.orientGenerator()[0]
    
class stochFracGenerator( stochFract_baseOperator ):
    """random Fracture simulation base class"""
    def __init__ ( self, parentFractModelSet, mode_randCentroid,  modelDomain=None ):
        stochFract_baseOperator.__init__( self, parentFractModelSet,  mode_randCentroid, modelDomain ) 
        if mode_randCentroid:
            self.mode_randCentroid = mode_randCentroid
        else:
            self.mode_randCentroid = 'RANDinArea'
       
        #-- NJH 01-08-2013: placerConstraint - a constraint that sets the random fractures in their place
##        if self.mode_randCentroid == 'RANDinArea' and self.modelDomain:
##            _box = self.modelDomain.boundingBox()
##            modelBox = [ [ _box.xMinimum() ,  _box.xMaximum()  ] ,  [ _box.yMinimum() ,  _box.yMaximum() ] ]
##            logger.debug("In randomFracture with modelDomain boundingbox as placerConstraint: %s" % repr( modelBox ) )
##            self.set_placerConstraint( modelBox )
##        else:
##            logger.debug("failed to create a placerConstraint upon initation randomFracture: %s | %s" % (self.mode_randCentroid , self.modelDomain))
##        logger.debug( "after set_placerConstraint with self.placerConstraint: %s" % self.placerConstraint )
        
        self.set_randCentroidGenerator( mode_randCentroid )
        self.set_sizeGenerator()
        self.set_orientGenerator('vonmises',  [0.0 ,  10.0, 1.0,  1 ])
    
    def set_randCentroidGenerator(self, mode_randCentroid='RANDinArea',  _params = None ):
        #if not self.mode_randCentroid:
            #mode_randCentroid = self.mode_randCentroid
        #-- NJH 01-08-2013: only when the right 3D volumetric operators are developed--
        #if mode_randCentroid == 'RANDinVOL':
        #    self.centroidGenerator = lambda: StoSim_algorithms.Gen_RandomPoint_inVolume( self.modelDomain , self.placerConstraint )
        #if mode_randCentroid == 'RANDinArea':
        self.centroidGenerator = lambda: StoSim_algorithms.Gen_RandomPoint_in2DPlane( self.modelDomain )
            #-- NJH 22-10-2013: the randomPoint Generator may not not only allow centroids that fall within the modelDomain, but also those cnetroid for which a segment of the line falls within the modelDomain
            #self.centroidGenerator = lambda: StoSim_algorithms.Gen_RandomPoint_in2DPlane( self.modelDomain,  self.azim,  self.size )
    
    def gen_randCentroid( self ):
        self.centroid = self.centroidGenerator( )
        
    def generate(self):
        self.gen_randSize()
        self.gen_randOrient()
        self.gen_randCentroid()
        #logger.debug( "self.size: %s | self.azim(degr): %s" % ( self.size, math.degrees( self.azim ) ) )
        _fract = gen_RandomFracture( self.parentFractModelSet,  self.centroid, self.size, None, math.degrees( self.azim ) )
        logger.debug("in stochFracGenerator.generate() with new random _fract: %s"% _fract.geometry().exportToWkt() )
        self.set_fracture( _fract )
        self.trim()
        return self.fracture

#class randomFracture_atSystSpacing( stochFract_baseOperator ):
class stochFractGenerator_atSystSpacing( stochFract_baseOperator ):
    """random Fracture simulation base class"""
    def __init__ ( self, parentFractModelSet,  modelDomain=None ):
        stochFract_baseOperator.__init__( self, parentFractModelSet, modelDomain,  placerConstraint ) 
        
        if placerConstraint:
            logger.debug("In randomFracture with a trackline as placerConstraint: %s" % repr( placerConstraint ) )
            self.set_placerConstraint( placerConstraint )
        else:
            logger.debug("failed to create a placerConstraint upon initation randomFracture: %s | %s" % (self.mode_randCentroid , self.modelDomain))
    
        #logger.debug( "after set_placerConstraint with self.placerConstraint: %s" % self.placerConstraint )
        self.set_randCentroidGenerator( mode_randCentroid )
        self.set_sizeGenerator()
        self.set_orientGenerator('vonmises',  [0.0 ,  10.0, 1.0,  1 ])    
    
    def set_randCentroidGenerator(self, mode_randCentroid='RANDinArea',  _params = None ):          
        self.spacingGenerator = lambda _mean, _std_dev,  _samplesize, _prev_u  : _spacingGenerator( 
                                                                                                _mean, _std_dev,  _samplesize, _prev_u )
        self.current_spacing = 0.0

        self.centroidGenerator = lambda _spacing : StoSim_algorithms.Gen_SystSpacedPoint_in2DPlane( self.modelDomain ,  self.placerConstraint ,  
                                                            _spacing )
        self.centroid = None
        
    def _spacingGenerator( _mean, _std_dev,  _samplesize, _prev_u ):
                _spacing = stats.norm.rvs( loc=_mean,  scale=_std_dev ,  size=_samplesize)
                return _prev_u + _spacing[0]
    
    def gen_randCentroid( self , _u=None  ):
        self.centroid = self.centroidGenerator( _u)
        if self.centroid: return True
        else: return False
        
    def generate(self):
        self.reset_fracture()
        
        self.gen_randSize()
        self.gen_randOrient()
        
        #if not _params: 
        _params = [ 30.0,  7.0,  1 ] 
        _mean = _params[0] ; _std_dev = _params[1] ; _samplesize = _params[2]
        self.current_spacing = self.spacingGenerator( _mean, _std_dev, _samplesize ,  self.current_spacing )
        withinSpacingTrack  = self.gen_randCentroid(self.current_spacing)
        logger.debug("in RandomFracture.generate() with withinSpacingTrack: %s | self.current_spacing: %s" % ( withinSpacingTrack ,  self.current_spacing) )
        if not withinSpacingTrack: return withinSpacingTrack

        #logger.debug( "self.size: %s | self.azim: %s | math.degrees(self.azim): %s" % ( self.size, self.azim,  math.degrees( self.azim ) ) )
        _fract = gen_RandomFracture( self.parentFractModelSet,  self.centroid, self.size, None, math.degrees( self.azim ) )
        
        self.set_fracture( _fract )
        self.trim()
        
        return self.fracture
        
#class stochFractSet_nonSelfIntersecting( stochFract_baseOperator ):
class stochFractGenerator_nonSelfIntersecting( stochFract_baseOperator ):
    """stochastic fracture generator for fractures should not intersect within the same set
    
    Todo: Determine in how far the _nonSelfIntersecting (how about equalLevelIntersect) and _nonHigherOrderInteresecting generators can be combined.
            Namely, the sets of fractures (whether self, equal-level or higher-order) can simply be specified. That is the only difference?
    """
    def __init__ ( self, parentFractModelSet, mode_randCentroid=None, modelDomain=None ):
        stochFract_baseOperator.__init__( self, parentFractModelSet,  mode_randCentroid, modelDomain )
        
        self.excludedArea = geometry.multipolygon.MultiPolygon()
        self.modelDomain_withExclusions = self.modelDomain
        
        self.set_sizeGenerator()
        
        self.set_orientGenerator('vonmises',  [0.0 ,0.05 , 0.05,  1 ])
        
        self.set_randCentroidGenerator( mode_randCentroid )
        
        self.reset_fracture()
    
    def set_proximity_condition(self,  _cond = False,  _weightedProxLimit = 0.4 ):
        self.proximity_condition = _cond
        self.weightedProxLimit = _weightedProxLimit
        return True
    
    def set_nonSameLevelIntersect_condition(self,  _cond = False ):
        self.nonSameLevelIntersect_condition = _cond
        #-- BUG NJH 11-08-2015: the fracturs of same level should be indicated 
        #--      so that the fractSetIdxs_not2Intersect variable is set
        ##self.set_nonIntersecting_fractSets(self.parentFractModelSet.)
        return True

    def set_randCentroidGenerator(self,  _params = None ):

        #_modelDomain = self.modelDomain.difference(self.excludedArea)
        _modelDomain = self.modelDomain_withExclusions
        
        self.centroidGenerator = lambda: StoSim_algorithms.Gen_RandomPoint_in2DPlane( _modelDomain )
        
        return True
        
    def gen_randCentroid( self ):
        self.centroid = self.centroidGenerator( )
        if self.centroid: return True
        else: return False

    def translate_2fitconditions( self,  _geom=None):
        if not _geom: _geom = self.fracture
        
        #-- instead of running the intersections_curve2curveCollection() the below 10-code-lines are executed and address both the
        #--        nonSameLevelIntersect_condition and proximity_condition
        #_pntList, _segmList = twoDGeomBaseLib.core.geomBasicOperations.intersections_curve2curveCollection( _geom , self.fractSet ) 
            
       # _pntList    = list()
       #_segmList = list()
        
        #logger.debug( "in translate_2fitconditions() with self.fractSet: %s" % self.fractSet )
        
        #for other_curve in self.fractSet:
        _levelName = self.parentFractModelSet.parentNetwork.get_Level4fractSetName( self.parentFractModelSet.name ).name
        ##logger.debug( "in translate_2fitconditions(..) with _levelName: %s " % ( _levelName ) )
        #logger.debug( "in translate_2fitconditions(..) self.parentFractModelSet.parentNetwork: %s " % ( self.parentFractModelSet.parentNetwork ) )
        #fracturesNot2Intersect = self.parentFractModelSet.parentNetwork.get_fractures_ofSameLevel( _levelName )
        fracturesNot2Intersect = self.parentFractModelSet.parentNetwork.get_fractures_ofFracSetIdxs( self.fractSetIdxs_not2Intersect )
        ##logger.debug( "in translate_2fitconditions with number of fracturesNot2Intersect: %s " %  ( len(fracturesNot2Intersect) ) )
        
        #-- NJH 12-08-2015: apply proximity condition to all fractures not only to the ones not to intersect,
        #--                             since for this nonSelfIntersection, then only fprximity condition is applied to fracs of same set.
        logger.debug("in stochFractGenerator_nonSelfIntersecting with self.proximity_condition and self.buffer_dist: %s %s" % ( self.proximity_condition, self.buffer_dist )  )
        for _other_geom in self.parentFractModelSet.parentNetwork.get_allFractureGeoms():
            ##logger.debug( "in translate_2fitconditions() applying proximity condition with _geom: %s and other_geom: %s " %  ( _geom, _other_geom ) )
            if self.proximity_condition:
                _weightedProx = twoDGeomBaseLib.core.geomBasicOperations.normProximity_withOtherGeom( _other_geom,  _geom,  self.buffer_dist)
                #logger.debug( "in IntersectCurve2CurveList(..) with _weighted proximity: %s " % ( _weightedProx ) )
                if _weightedProx > self.weightedProxLimit:
                    self.failureState = 'failed_weighted_proximity_%s' % _weightedProx
                    return False
        
        for other_curve in fracturesNot2Intersect:
            if not isinstance( other_curve ,  QgsCore.QgsGeometry ):
                other_curve = other_curve.geometry()
            
            if self.nonSameLevelIntersect_condition:
                _segmList = twoDGeomBaseLib.core.geomBasicOperations.split_curve_by_curve(  _geom, other_curve, None )
                #logger.debug( "in translate_2fitconditions with equal_level_intersection and consequent _segmList: %s %s" % ( len(_segmList),  [segm.exportToWkt() for segm in _segmList ] ) )
                if len(_segmList) > 1:
                    #-- the _geom will be translated in the direction of the smallest segment of the two in the _segmList
                    #-- NJH 02-11-2013: only translate when the interesection occurs within less than 20% of the linelength to endPoint
                    if not _segmList[0].length()/_geom.length() < 0.2 or not  _segmList[1].length()/_geom.length() < 0.2:
                        self.failureState = 'failed_samelevelintersect_toolarge_overlap'
                        return False
                    if _segmList[0].length()  < _segmList[ len(_segmList)-1 ].length():
                        transSegm = _segmList[0].asPolyline()
                    else: 
                        transSegm = _segmList[ len(_segmList)-1 ].asPolyline()
                    #logger.debug( "in translate_2fitconditions with transSegm: %s " % transSegm )
                    transVector = DgfCore.DgfVector(transSegm[1].x() - transSegm[0].x() , transSegm[1].y() - transSegm[0].y() )
                    _geom = twoDGeomBaseLib.core.geomBasicOperations.translate( _geom,  transVector )
                    
                    #-- check again if the translated _geom  intersects with no other curve now that it is translated:
                    #-- NJH 17-02-2014 !!! wrong impmentation-- the shifted curve should be checked against all other-fractures, not just the one it initially interesected with.
                    #--                                 chances are that the shifted fracture intersects ith fractures that the loop already passed.
                    #--        better not fix this , but go for another implementation entirely-- i.e. by implementing an excluded area-- 
                    #--                   the centroid is only randomly placed in an an area with the excluded-area alrady excluded beforehand.
                    _segmList = twoDGeomBaseLib.core.geomBasicOperations.split_curve_by_curve(  _geom, other_curve, None )
                    #logger.debug( "in translate_2fitconditions with ar re-check after translation -> equal_level_intersection and consequent _segmList: %s %s" % ( len(_segmList),  [segm.exportToWkt() for segm in _segmList ] ) )
                    if len(_segmList) > 1:
                        self.failureState = 'failed_shiftedfracture_stillintersects'
                        return False
            #geomBasicOperations.translate( _geom,  _transVector )
        
        return _geom
        
    def generate(self):
        self.reset_fracture()
        
        self.gen_randSize()
        self.gen_randOrient()
        
        #-- 15-08-2013: the self.modelDomain_withExclusions seems to get easily geometrically invalid during 
        #--                   the geom_outer.difference(geom_inner) operation in gen_excludedArea(..)
        #twoDGeomBaseLib.core.geomBasicOperations.gen_excludedArea( self.get_fractSet(), self.modelDomain_withExclusions, self.azim ,  self.size)
        #self.set_randCentroidGenerator()
        #logger.debug( "in stochFractSet_withExcludedArea.generate() with self.modelDomain_withExclusions: %s" % self.modelDomain_withExclusions )
        #-- NJH 15-08-2013: just a test
        #for i in range(5):
        #    _pnt = StoSim_algorithms.Gen_RandomPoint_of2DPlane_withHoles( self.modelDomain_withExclusions )
        #    logger.debug( "in stochFractSet_withExcludedArea.generate() with just a pnt on ring: %s" % _pnt )
        
        self.gen_randCentroid()
        
        logger.debug( "self.size: %s | self.azim(degr): %s" % ( self.size, math.degrees( self.azim ) ) )
        #_fract = gen_RandomFracture( self.centroid, self.size, None, math.degrees( self.azim ) )
        _fractGeom = twoDGeomBaseLib.core.geomBasicOperations.gen_Line_fromCPoint( self.centroid,  math.degrees( self.azim ), self.size  )        
        logger.debug( "in stochFractGenerator_nonSelfIntersecting.generate() with _fractGeom: %s is empty? %s" %  ( _fractGeom,  _fractGeom.isGeosEmpty()) ) 

        #if not _fractGeom:
        if  _fractGeom.isGeosEmpty():
            return False
        if not _fractGeom:
            return False
        _fractGeom = self.translate_2fitconditions( _fractGeom )
        if not _fractGeom:
            return False
        logger.debug( "in stochFractGenerator_nonSelfIntersecting.generate() with self.parentFractModelSet: %s" % self.parentFractModelSet ) 
        #fractFeat = DigiFractLib.baseclasses.DF_Feature.DF_Feature(self.parentFractModelSet, _fractGeom,  [self.parentFractModelSet.nextFreeFeatureID(),  -99] )
        
        fractFeat = QgsCore.QgsFeature( self.parentFractModelSet.parentNetwork.nextFreeFeatureID() )
        fractFeat.initAttributes( len(self.parentFractModelSet.parentNetwork.dataProvider().attributeIndexes() ) )
        fractFeat.setAttributes( [self.parentFractModelSet.parentNetwork.nextFreeFeatureID(),  -99, self.parentFractModelSet.parentNetwork.name,  self.parentFractModelSet.name,  None] )
        fractFeat.setGeometry(_fractGeom)

        logger.debug( "in stochFractGenerator_nonSelfIntersecting.generate() with _fractFeat.isValid(): %s" % fractFeat.isValid() )
        self.set_fracture( fractFeat )
        logger.debug("stochFractSet_nonSelfIntersecting.generate with _fractFeat: %s and self.fracture: %s %s" % ( fractFeat ,  self.fracture,  self.fracture.geometry().exportToWkt() ) )
        self.trim()
        if self.fracture:
            ##logger.debug("stochFractSet_nonSelfIntersecting.generate with self.fracture: %s" % ( self.fracture ) )
            ##self.fractSet.append( self.fracture )
            return True
        else:
            self.failureState = 'failed_at finish'
            logger.debug("in randomFracture_placedAtHOrderFract.generate(): failed because no fracture satisfied the conditions")
            return False

class stochFractGenerator_nonHigherOrderIntersecting( stochFract_baseOperator ):
    """random Fracture simulation base class"""
    def __init__ ( self, parentFractModelSet, mode_randCentroid=None, modelDomain=None ):
        stochFract_baseOperator.__init__( self, parentFractModelSet,  mode_randCentroid, modelDomain )
        
        self.excludedArea = shapely.geometry.multipolygon.MultiPolygon()
        self.modelDomain_withExclusions = self.modelDomain
        
        self.set_sizeGenerator()
        
        self.set_orientGenerator('vonmises',  [0.0 ,0.05 , 0.05,  1 ])
        
        self.set_randCentroidGenerator( mode_randCentroid )
    
    def set_proximity_condition(self,  _cond = False,  _weightedProxLimit = 0.4 ):
        self.proximity_condition = _cond
        self.weightedProxLimit = _weightedProxLimit
        return True
    
    def set_nonSameLevelIntersect_condition(self,  _cond = False ):
        self.nonSameLevelIntersect_condition = _cond
        
        return True
        
    def set_placerConstraint(self, constraints = None):
        logger.debug("set_placerConstraint() with the placing constraints : %s" % constraints  )
        self.higherLevel_fractSetIdxs = list()
        for fracSetNm in constraints:
            _idx = self.parentFractModelSet.parentNetwork.find_Idx4Name( fracSetNm )
            logger.debug("in set_placerConstraint() with fractSetIdx %s for fracSetNm: %s" % (fracSetNm,  _idx))
            self.higherLevel_fractSetIdxs.append(_idx) 
        logger.debug("in set_placerConstraint() with self.higherLevel_fractSetIdxs: %s" % self.higherLevel_fractSetIdxs )
        return True
    
    #-- NJH 29-11-2013: this below member function will not help much in precising non-intersection relationships. Just define each fracture explicitly...
    ##def set_nonHigherLevelIntersect_condition(self,  _cond = False ):
    ##    self.nonSameLevelIntersect_condition = _cond
    ##    return True

    def set_randCentroidGenerator(self,  _params = None ):

        #_modelDomain = self.modelDomain.difference(self.excludedArea)
        _modelDomain = self.modelDomain_withExclusions
        
        self.centroidGenerator = lambda: StoSim_algorithms.Gen_RandomPoint_in2DPlane( _modelDomain )
        
        return True
        
    def gen_randCentroid( self ):
        self.centroid = self.centroidGenerator( )
        if self.centroid: return True
        else: return False

    def generate(self):
        self.reset_fracture()
        
        self.gen_randCentroid()
        self.gen_randSize()
        self.gen_randOrient()
        
        logger.debug( "self.size: %s | self.azim(degr): %s" % ( self.size, math.degrees( self.azim ) ) )
        #_fract = gen_RandomFracture( self.centroid, self.size, None, math.degrees( self.azim ) )
        fractGeom = twoDGeomBaseLib.core.geomBasicOperations.gen_Line_fromCPoint( self.centroid,  math.degrees( self.azim ), self.size  )        
        fractGeom = twoDGeomBaseLib.core.geomBasicOperations.crop_PLine_byPGon(  fractGeom , self.modelDomain )
        
        #_higherLevel_fractures = self.parentFractModelSet.parentNetwork.get_fractures_ofFracSetIdxs(self.higherLevel_fractSetIdxs)
        fracturesNot2Intersect = self.parentFractModelSet.parentNetwork.get_fractures_ofFracSetIdxs( self.fractSetIdxs_not2Intersect )
        logger.debug( "in stochFractGenerator_nonHigherOrderIntersecting.generate() with len(fracturesNot2Intersect): %s" % ( len( fracturesNot2Intersect ) ) )

        segmList = list()
        segmList.append( fractGeom)
        logger.debug( "in stochFractGenerator_nonHigherOrderIntersecting.generate() with the wkt of fractGeom to be segmentized: %s" % ( fractGeom.exportToWkt() ) )
        i_fract = 0
        i_otherfract = 0
        while i_fract < len( segmList ):
            segmList,  pntList,  attribList,  i_fract,  i_otherfract = twoDGeomBaseLib.core.geomBasicOperations.segmentize_curve2curveCollection( segmList, fracturesNot2Intersect,  i_fract, i_otherfract  )
            #pntList = twoDGeomBaseLib.core.geomBasicOperations.intersections_curve2curveCollection( i_fract,  segmList, _higherLevel_fractures  )
            i_otherfract = 0
            ##i_fract +=1
        
        logger.debug("in stochFractGenerator_nonHigherOrderIntersecting.generate() with len(segmList): %s" % len(segmList) )
        logger.debug("in stochFractGenerator_nonHigherOrderIntersecting.generate() with [segmList]: %s" % [segm.exportToWkt() for segm in segmList] )
        logger.debug("in stochFractGenerator_nonHigherOrderIntersecting.generate() with QgsCore.QgsGeometry.fromPoint(self.centroid): %s" % QgsCore.QgsGeometry.fromPoint(self.centroid).exportToWkt() )
        
        changedGeom2Segm = False
        _cPntGeom = QgsCore.QgsGeometry.fromPoint(self.centroid)
        logger.debug("in stochFractGenerator_nonHigherOrderIntersecting.generate() with _cPntGeom: %s" % (_cPntGeom.exportToWkt() ) )
        #-- NJH 04-03-2015: it might be faster to select the segment on which the centerPoint sits during the segmentation.
        #--                       Now segments that will be thrown out, are further segmentized first. That is unnessary extra work.
        #--            It would require a further refined segmentize_curve2curveCollection() function which may not be generic enough
        for _segm in segmList:
            if _segm.intersects( _cPntGeom.buffer(0.01, 6) ):
                fractGeom = _segm
                changedGeom2Segm = True
                break;
        logger.debug("Did we change the fracture Geom by finding Segment that holds centroid? %s" % changedGeom2Segm )
        
        ##logger.debug( "in stochFractGenerator_nonHigherOrderIntersecting with fractGeom: %s and other_curve: %s " %  ( fractGeom, fracturesNot2Intersect ) )
        if self.proximity_condition:
            for i_other_curve in range(len(fracturesNot2Intersect)):
                other_fractGeom = fracturesNot2Intersect[i_other_curve].geometry()
                _weightedProx = twoDGeomBaseLib.core.geomBasicOperations.normProximity_withOtherGeom( other_fractGeom, fractGeom, self.buffer_dist )
                #logger.debug( "in IntersectCurve2CurveList(..) with _weighted proximity: %s " % ( _weightedProx ) )
                if _weightedProx > self.weightedProxLimit:
                    self.failureState = 'failed_weighted_proximity_%s' % _weightedProx
        
        for _segm in segmList:
            if _segm.intersects( _cPntGeom.buffer(0.01, 6) ):
                fractGeom = _segm
                changedGeom2Segm = True
                break;
        logger.debug("Did we change the fracture Geom by finding Segment that holds centroid? %s" % changedGeom2Segm )
        
        #-- Try again after the fracture is segmentized
        ##logger.debug( "in stochFractGenerator_nonHigherOrderIntersecting with fractGeom: %s and other_curve: %s " %  ( fractGeom, fracturesNot2Intersect ) )
        if self.proximity_condition:
            for i_other_curve in range(len(fracturesNot2Intersect)):
                other_fractGeom = fracturesNot2Intersect[i_other_curve].geometry()
                _weightedProx = twoDGeomBaseLib.core.geomBasicOperations.normProximity_withOtherGeom( other_fractGeom, fractGeom, self.buffer_dist )
                #logger.debug( "in IntersectCurve2CurveList(..) with _weighted proximity: %s " % ( _weightedProx ) )
                if _weightedProx > self.weightedProxLimit:
                    self.failureState = 'failed_weighted_proximity_%s' % _weightedProx
                    return False
        
        logger.debug( "in stochFractGenerator_nonHigherOrderIntersecting.generate() with self.parentFractModelSet: %s" % self.parentFractModelSet ) 
        logger.debug( "in stochFractGenerator_nonHigherOrderIntersecting.generate() with self.parentFractModelSet.nextFreeFeatureID(): %s" % self.parentFractModelSet.nextFreeFeatureID() ) 
        #fractFeat = DigiFractLib.baseclasses.DF_Feature.DF_Feature(self.parentFractModelSet, fractGeom,  [self.parentFractModelSet.nextFreeFeatureID(),  -99] )
        fractFeat = QgsCore.QgsFeature( self.parentFractModelSet.parentNetwork.nextFreeFeatureID() )
        fractFeat.initAttributes( len(self.parentFractModelSet.parentNetwork.dataProvider().attributeIndexes() ) )
        fractFeat.setAttributes( [self.parentFractModelSet.parentNetwork.nextFreeFeatureID(),  -99, self.parentFractModelSet.parentNetwork.name,  self.parentFractModelSet.name,  None] )
        fractFeat.setGeometry(fractGeom)
        self.set_fracture( fractFeat )
        logger.debug("stochFractGenerator_nonHigherOrderIntersecting.generate with _fractFeat: %s and self.fracture: %s" % ( fractFeat ,  self.fracture ) )
        self.trim()
        if self.fracture:
            ##logger.debug("stochFractSet_nonSelfIntersecting.generate with self.fracture: %s" % ( self.fracture ) )
            ##self.fractSet.append( self.fracture )
            return True
        else:
            self.failureState = 'failed_at finish'
            logger.debug("in stochFractGenerator_nonHigherOrderIntersecting.generate(): failed because no fracture satisfied the conditions")
            return False
            
#class randomFracture_placedAtHOrderFract( stochFract_baseOperator ):
class stochFractGenerator_placedAtHOrderFract( stochFract_baseOperator ):
    """
    Stochastic fracture generator for placing fracture at an higher order fracture.
    The fracture is placed with one end-point at at a random position along a randomly picked higher order fracture.
    The fracture end-point will have an abutment probability and if not abutting, then a randomly offset off fram the higher order fracture is applied.
    
    Attributes:
        * parentFractModelSet: the fracture set into which the new random fracture is generated
        * mode_randCentroid: (NJH - check: not sure in how far this variable is used for this _placedAtHOrderFract generator)
        * modelDomain: the domain into which the random fracture is to be placed.
        * placerConstraint: a list of higher order fractures, from which one is picked at which the new random fracture is placed with a probability-to-abut and offset-distance.
    
    Todo:
    
    """
    
    def __init__ ( self, parentFractModelSet, mode_randCentroid=None, modelDomain=None,  placerConstraint=None):
        stochFract_baseOperator.__init__( self, parentFractModelSet,  mode_randCentroid, modelDomain )
        
        self.set_offsetGenerator()
        self.set_placerConstraint( placerConstraint )
        self.set_NIntersectFract()
        
    def reset_fracture(self):
        self.centroidGenerator = None
        self.failureState = ''
        self.fracture = None
        self.offset = None
        
    def set_placerConstraint(self, constraints = None):
        """the list of (higher order) fractures at which the new random fracture is placed"""
        logger.debug("set_placerConstraint() with the placing constraints : %s" % constraints  )
        self.higherLevel_fractSetIdxs = list()
        for fracSetNm in constraints:
            _idx = self.parentFractModelSet.parentNetwork.find_Idx4Name( fracSetNm )
            logger.debug("in set_placerConstraint() with fractSetIdx %s for fracSetNm: %s" % (fracSetNm,  _idx))
            self.higherLevel_fractSetIdxs.append(_idx) 
        logger.debug("in set_placerConstraint() with self.higherLevel_fractSetIdxs: %s" % self.higherLevel_fractSetIdxs )
        return True
    
    def set_proximity_condition(self,  _cond = False,  _weightedProxLimit = 0.4 ):
        self.proximity_condition = _cond
        self.weightedProxLimit = _weightedProxLimit
        return True

    def set_nonSameLevelIntersect_condition(self,  _cond = False ):
        self.nonSameLevelIntersect_condition = _cond
        return True
    
    #-- NJH 29-11-2013: this below member function will not help much in precising non-intersection relationships. Just define each fracture explicitly...
    ##def set_nonHigherLevelIntersect_condition(self,  _cond = False ):
    ##    self.nonSameLevelIntersect_condition = _cond
    ##    return True
    
    def set_NIntersectFract(self,  _NIntersectFract=None):
        self.NIntersectFract = _NIntersectFract
        return True
    
    def pick(self,  _higherLevel_fractures ):
        ##logger.debug("in stochFractGenerator_placedAtHOrderFract about to pick from self.higherLevel_fractSetIdxs: %s" % self.higherLevel_fractSetIdxs)
               #-- NJH 25-08-16: avoid too many if-else checks in the stoch simulation, slows down
        #if not _higherLevel_fractures:
        #    _higherLevel_fractures = self.parentFractModelSet.parentNetwork.get_fractures_ofFracSetIdxs(self.higherLevel_fractSetIdxs)
        #if not _higherLevel_fractures: 
        #    return False
        _randInt = random.randint(0,len(_higherLevel_fractures)-1)
        _fract = _higherLevel_fractures[_randInt]
        logger.debug("in stochFractGenerator_placedAtHOrderFract about to pick from _higherLevel_fractures: %s with picked Idx %s" % ( len(_higherLevel_fractures),  _randInt ) )
        #_fract = random.choice( _higherLevel_fractures )
        if not _fract: 
            return False
        self.fract_asPlacerConstraint = _fract
        return _fract
    
    def set_offsetGenerator(self,  _params = None ):
        
        if _params is None:
            self.abutChance        = 1.0
            self.offset_exponent   = 10.0
            self.cutoff_distance    = 20.0
    
        else:
            self.abutChance        = _params[0]
            self.offset_exponent   = _params[1]
            self.cutoff_distance    = _params[2]
        
        #self.offsetGenerator = lambda: self.cutoff_distance * numpy.random.exponential( self.offset_exponent )
        self.offsetGenerator = lambda: stats.expon.rvs( loc=self.cutoff_distance,  scale=self.offset_exponent )

        return True
    
    def gen_Offset(self,  _u,  _offset_angle =90.0 ):
        
        _offset_dist = StoSim_algorithms.Gen_RandomDistance( self.abutChance, self.cutoff_distance, self.offsetGenerator( )   )
        _2cross = StoSim_algorithms.Gen_RandomSign()
        _placementSign = StoSim_algorithms.Gen_RandomSign()
        
        #_normal = DigiFractLib.baseclasses.DF_Geometry.DF_Geometry( self.parentFractModelSet.parentNetwork.digiSpace, self.fract_asPlacerConstraint.geometry() ).NormalAt(_u)
        _normal = twoDGeomBaseLib.core.geomBasicOperations.NormalAt( self.fract_asPlacerConstraint.geometry(),  _u )
        #_normal = twoDGeomBaseLib.core.geomBasicOperations.NormalAt( self.fract_asPlacerConstraint.geometry(),  _u)
        
        ##logger.debug("in randomFracture_placedAtHOrderFract.gen_Offset() with _placementSign: %s | self.offset_dist: %s" % ( _placementSign,  _offset_dist ) )
        logger.debug( "_placementSign: %s | _normal: (%s,%s) " % ( _placementSign, _normal.x() ,  _normal.y() ) )
        _xoffset ,  _yoffset = _normal.x() * _placementSign * _offset_dist ,  _normal.y() * _placementSign * _offset_dist
        ##logger.debug("in randomFracture_placedAtHOrderFract.gen_Offset() with _offset_dist: %s and (_xoffset,_yoffset) : (%s , %s) " % (_offset_dist, _xoffset ,  _yoffset ) )
        
        self.offset =  twoDGeomBaseLib.core.DF_Vector.DF_Vect3D( _xoffset, _yoffset )
        return True
    
    def gen_cropping(self , other_fractSet):
        pass
        #for other_highlvl_curve in other_fractSet:
        #                
        
    def generate(self):
        
        logger.debug("in randomFracture_placedAtHOrderFract with self.proximity_condition: %s" % self.proximity_condition )
 
        self.reset_fracture ()
        
        _higherLevel_fractures = self.parentFractModelSet.parentNetwork.get_fractures_ofFracSetIdxs(self.higherLevel_fractSetIdxs)
        logger.debug( "in randomFracture_placedAtHOrderFract.generate() with len(_higherLevel_fractures): %s" % ( len( _higherLevel_fractures ) ) )
        if not self.pick( _higherLevel_fractures ): 
            logger.debug("in randomFracture_placedAtHOrderFract.generate(): failed successfull pick of _higherLevel_fractures: failed to pick higher order fracture")
            self.centroid = QgsCore.QgsGeometry ()
            self.failureState = 'failed_fract_pick'
            return False
        
        #-- a fractBasePoint is generated relative to the above picked higher order fracture (set as fract_asPlacerConstraint ) from self.fractSet_asPlacerConstraint
        logger.debug("in randomFracture_placedAtHOrderFract.generate(): with self.fract_asPlacerConstraint: %s %s" % ( type(self.fract_asPlacerConstraint), self.fract_asPlacerConstraint ) )

        _u = StoSim_algorithms.RandomUOnLine( self.fract_asPlacerConstraint.geometry() )  
        self.gen_Offset( _u )
        
        ##logger.debug("in randomFracture_placedAtHOrderFract.generate() with self.offset: %s and : %s" % ( self.offset,  self.NIntersectFract ))
        offsetLength = numpy.power( numpy.power(self.offset.x(), 2) +  numpy.power( self.offset.y() , 2 ) ,  0.5)
        if offsetLength < self.cutoff_distance and self.NIntersectFract is not None:
            self.size = numpy.power( numpy.power(self.modelDomain.boundingBox().height(), 2) +  numpy.power( self.modelDomain.boundingBox().width() , 2 ) ,  0.5)
        else:
            self.gen_randSize()
            
        #_point  =  DigiFractLib.baseclasses.DF_Geometry.DF_Geometry(None, None, self.fract_asPlacerConstraint.geometry() ).PointAt(_u)
        _point  =  twoDGeomBaseLib.core.geomBasicOperations.PointAt(  self.fract_asPlacerConstraint.geometry(),  _u)

        logger.debug("in randomFracture_placedAtHOrderFract.generate() with offset from HOrderFract of _x ,  _y: %s ,  %s " % ( self.offset.x() ,  self.offset.y()) )
        _x ,  _y =  _point[0] + self.offset[0] ,  _point[1] + self.offset[1]
        ##logger.debug("in randomFracture_placedAtHOrderFract.generate() with offset from HOrderFract of _x ,  _y: %s ,  %s " % ( _x ,  _y) )
        fractBPoint =  DgfCore.DgfVertex(  _x ,  _y  )
        
        _directionSign   = StoSim_algorithms.Gen_RandomSign()
        
        self.gen_randOrient()
        ##logger.debug( "self.size: %s | self.azim: %s | math.degrees(self.azim): %s" % ( self.size, self.azim,  math.degrees( self.azim ) ) )
        
        _azim_deg = math.degrees( self.azim )
        if _directionSign == -1: _azim_deg += 180.0
        if _azim_deg >=360.0: _azim_deg += -360.0
            
        fractGeom = twoDGeomBaseLib.core.geomBasicOperations.gen_Line_fromBPoint( fractBPoint, _azim_deg , self.size  )
        ##logger.debug( "in randomFracture_placedAtHOrderFract.generate() with _x:%s , _y: %s fractBPoint: %s and  fractGeom: %s" % ( _x ,  _y ,  fractBPoint , fractGeom.exportToWkt() ) )
        fractGeom = twoDGeomBaseLib.core.geomBasicOperations.crop_PLine_byPGon(  fractGeom , self.modelDomain )
        if not fractGeom:
            self.failureState = 'failed_fact_is_none to higher-order frac id %s' % (self.fract_asPlacerConstraint.id())
            self.centroid = QgsCore.QgsPoint()
            logger.debug("in randomFracture_placedAtHOrderFract.generate(): failed as fractGeom is None")
            return False
        
        segmList = list()
        segmList.append(fractGeom)
        i_fract = 0
        i_otherfract =0
        while i_fract < len( segmList ):
            segmList,  pntList,  attribList, i_fract, i_otherfract  = twoDGeomBaseLib.core.geomBasicOperations.segmentize_curve2curveCollection( segmList, _higherLevel_fractures,  i_fract,  i_otherfract )
            #pntList = twoDGeomBaseLib.core.geomBasicOperations.intersections_curve2curveCollection( i_fract,  segmList, _higherLevel_fractures  )
            ##i_fract +=1
        logger.debug( "in randomFracture_placedAtHOrderFract.generate() with trial segmList: %s" % segmList )
        ##logger.debug( "in randomFracture_placedAtHOrderFract.generate() with trial attribList: %s" % attribList )

        nearestFeatIdx = None
        nearestFractDistance = 1.0e8
        ##nearestPntOnOtherFeat = None
        
        #logger.debug("in randomFracture_placedAtHOrderFract.generate() - segmList as wkt: %s" % [ str( segm.exportToWkt() ) for segm in segmList ])
        #logger.debug("in randomFracture_placedAtHOrderFract.generate() - fractBPoint: %s %s" % ( fractBPoint.x() ,  fractBPoint.y() )  )
        _BPointGeom = QgsCore.QgsGeometry() ; _BPointGeom.fromPoint( QgsCore.QgsPoint( fractBPoint[0] ,  fractBPoint[1])  )
        #logger.debug("in randomFracture_placedAtHOrderFract.generate() - check if first segment intersects the fractBPoint: %s" % segmList[0].intersects(_BPointGeom) )
        ##logger.debug("in randomFracture_placedAtHOrderFract.generate() with segmList: %s" % segmList)
        if self.NIntersectFract == 0:
            fractGeom = segmList[0]
            #if fractGeom.length() < 1.0:
            if fractGeom.length() < self.cutoff_distance:
                logger.debug("the geometry had a length smaller than the self.cutoff_distance")
                if len(segmList) > 1:
                    if fractGeom.length() >= self.cutoff_distance:
                        logger.debug("but hey the other segment of the segmList is the right one: %s %s" % (segmList[1].length(), segmList[1].exportToWkt()  ) )
                        fractGeom = segmList[1]
                    else:
                        self.centroid = fractGeom.centroid().asPoint()
                        self.failureState = 'failed_fact_cropping1'
                        logger.debug("in randomFracture_placedAtHOrderFract.generate(): failed as fractGeom is None")
                        return False
                else:
                    self.centroid = fractGeom.centroid().asPoint()
                    self.failureState = 'failed_fact_cropping2'
                    logger.debug("in randomFracture_placedAtHOrderFract.generate(): failed as fractGeom is None")
                    return False
            ##if fractGeom.coords[1]
        elif self.NIntersectFract is None:
            #logger.debug( "in randomFracture_placedAtHOrderFract.generate() - self.NIntersectFract: %s and fractGeom: %s" % ( self.NIntersectFract, fractGeom) )

            #self.gen_cropping( self.higherLevel_fractSet )
            fractGeomCoords = fractGeom.asPolyline()
            
            for ifract in range(len( _higherLevel_fractures )):
                other_fractGeom = _higherLevel_fractures[ifract].geometry()
                _dist = other_fractGeom.closestSegmentWithContext (fractGeomCoords[ len(fractGeomCoords)-1 ] )
                if _dist < nearestFractDistance: 
                    nearestFractDistance = _dist
                    nearestFeatIdx = ifract
            
            if self.cutoff_distance > nearestFractDistance:
                logger.debug("Fracture[%s] is found at a distance of %s from endPoint that closer than the cutoff_distance %s" % ( nearestFeatIdx, nearestFractDistance, self.cutoff_distance ) )
                if fractGeom.intersects( _higherLevel_fractures[nearestFeatIdx].geometry() ):
                    fractGeom = twoDGeomBaseLib.core.geomBasicOperations.split_curve_by_curve( fractGeom,  _higherLevel_fractures[nearestFeatIdx].geometry() )
                else:
                    _extLength = numpy.power( numpy.power(modelDomain.boundingBox().height(), 2) +  numpy.power( modelDomain.boundingBox().width() , 2 ) ,  0.5)
                    _fractGeom = twoDGeomBaseLib.core.geomBasicOperations.gen_endPointExtrusion( _fractGeom,  [1 , None,  _extLength ] )
                    _fractGeom = twoDGeomBaseLib.core.geomBasicOperations.intersection_curve_with_curve(_fractGeom, _higherLevel_fractures[nearestFeatIdx].geometry() )
                    fractGeom = _fractGeom[0]
                    logger.debug("in randomFracture_placedAtHOrderFract.generate() - check if _fractGeom holds the fractBPoint: %s" % fractGeom.contains(fractBPoint) )
            #self.abutChance
        else:
            self.centroid = fractGeom.centroid().asPoint()
            self.failureState = 'failed_not_implemented1'
            logger.debug("in randomFracture_placedAtHOrderFract.generate(): failed with the option self.NIntersectFract : %s is not implemented yet and therefore self.generate() returns False" % self.NIntersectFract )
            return False
        
        logger.debug("in randomFracture_placedAtHOrderFract.generate() after applying abutment ->  fractGeom: %s" % fractGeom.exportToWkt() )
        
        #-- NJH 25-08-2016: this codelines prevent small fractures of length smaller than buffer_dist.
        #--   However this also prevents any fracture to form between two closely spaded higher order fracture, an unbridageble zone no chance for percolation!!cc
        #if fractGeom.length() < self.buffer_dist:
        #    self.centroid = fractGeom.centroid().asPoint()
        #    self.failureState = 'failed_toosmall to higher-order frac id %s' % (self.fract_asPlacerConstraint.id())
        #    logger.debug( "in randomFracture_placedAtHOrderFract.generate(): failed as fractGeom.length() < self.buffer_dist" )
        #    return False
        
        fracturesNot2Intersect = self.parentFractModelSet.parentNetwork.get_fractures_ofFracSetIdxs( self.fractSetIdxs_not2Intersect )
        logger.debug("in randomFracture_placedAtHOrderFract.generate() with self.fractSetIdxs_not2Intersect: %s and len( fracturesNot2Intersect ): %s" % ( self.fractSetIdxs_not2Intersect,  len( fracturesNot2Intersect ) ) )
        for i_other_curve in range(len(fracturesNot2Intersect)):
            other_fractGeom = fracturesNot2Intersect[i_other_curve].geometry()
            ##if not isinstance( other_fractGeom ,  QgsCore.QgsGeometry ):
            ##    other_fractGeom = other_fractGeom.geometry()
        
            #if  self.nonSameLevelIntersect_condition:
            _return = fractGeom.intersects(other_fractGeom)
            #_return = twoDGeomBaseLib.core.geomBasicOperations.intersects_curve2curveCollection( fractGeom,  self.parentFractModelSet )
            #logger.debug("in randomFracture_placedAtHOrderFract.generate() - the proposed fracture has %s %s intersections with same order fractures" % ( len(_return),  _return ) )
            #if len(_return) > 0:
            if _return:
                #logger.debug("in randomFracture_placedAtHOrderFract.generate() - the proposed fracture %s and first other_racture that intersects: %s" % ( fractGeom.exportToWkt(), self.parentFractModelSet[ _return[0] ].geometry().exportToWkt() ) )
                self.centroid = fractGeom.centroid().asPoint()
                self.failureState = 'failed_samelvlintersect'
                logger.debug("in randomFracture_placedAtHOrderFract.generate(): failed because of nonSameLevelIntersect_condition condition")
                return False
        
        logger.debug("in stochFractGenerator_placedAtHOrderFract with self.proximity_condition and self.buffer_dist: %s %s" % ( self.proximity_condition, self.buffer_dist )  )
        if self.proximity_condition:
            #fracs2StayAwayFrom = [ self.parentFractModelSet.parentNetwork.getFeatures() ]
            #for i_other_curve in range(len(fracs2StayAwayFrom)):
            for i_other_curve in range(len(fracturesNot2Intersect)):
                other_fractGeom = fracturesNot2Intersect[i_other_curve].geometry()

                _weightedProx = twoDGeomBaseLib.core.geomBasicOperations.normProximity_withOtherGeom( fractGeom, other_fractGeom, self.buffer_dist )
                if _weightedProx > self.weightedProxLimit:
                    self.centroid = fractGeom.centroid().asPoint()
                    self.failureState = 'failed_proximitycond to higher-order frac id %s' % (self.fract_asPlacerConstraint.id())
                    logger.debug("in randomFracture_placedAtHOrderFract.generate(): failed for centroid: %s because of proximity condition between new and existing fracture: %s %s  = higher-order frac id %s'" % ( self.centroid, other_fractGeom.exportToWkt(), fractGeom.exportToWkt(), self.fract_asPlacerConstraint.id() ))
                    return None

        ###fractFeat = DigiFractLib.fractdatamodel.fracture.Fracture( self.parentFractModelSet,  fractGeom,  None )
        #fractFeat = DigiFractLib.baseclasses.DF_Feature.DF_Feature(self.parentFractModelSet, fractGeom,  [self.parentFractModelSet.nextFreeFeatureID(),  -99] )
        fractFeat = QgsCore.QgsFeature( self.parentFractModelSet.parentNetwork.nextFreeFeatureID() )
        fractFeat.initAttributes( len(self.parentFractModelSet.parentNetwork.dataProvider().attributeIndexes() ) )
        fractFeat.setAttributes( [self.parentFractModelSet.parentNetwork.nextFreeFeatureID(),  -99, self.parentFractModelSet.parentNetwork._name,  self.parentFractModelSet.name,  None] )
        fractFeat.setGeometry(fractGeom)
        #logger.debug("randomFracture_placedAtHOrderFract.generate with _fract: %s" % _fract)
        if fractFeat:
            self.set_fracture( fractFeat )
            self.centroid = fractFeat.geometry().centroid().asPoint()
            self.failureState  = ''
            ##success = self.trim()
            logger.debug("randomFracture_placedAtHOrderFract.generate with self.fracture: %s" % self.fracture)
            return self.fracture
        else:
            self.failureState = 'failed_at finish'
            self.centroid = fractGeom.centroid().asPoint()
            logger.debug("in randomFracture_placedAtHOrderFract.generate(): failed because no fracture satisfied the conditions")
            return False

class bootstrapFracture_placedAtHOrderFract( stochFract_baseOperator ):
    """random Fracture simulation class"""
    def __init__ ( self, parentFractModelSet, mode_randCentroid=None, modelDomain=None, placerConstraint=None ):
        _name = None
        randomFracture.__init__( self, parentFractModelSet, mode_randCentroid, modelDomain, placerConstraint )
        
        #-- pick a fracture from a fractureDataCollection
        
