---
jupyter:
  jupytext:
    formats: ipynb,md
    text_representation:
      extension: .md
      format_name: markdown
      format_version: '1.2'
      jupytext_version: 1.7.1
  kernelspec:
    display_name: Python [conda env:parity_diagram]
    language: python
    name: conda-env-parity_diagram-py
---

```python
import matplotlib.pyplot as plt
import matplotlib.colors as colors
from matplotlib.gridspec import GridSpec, GridSpecFromSubplotSpec
from ipywidgets import interactive, SelectionSlider
import numpy as np
import pickle
import glob
import pandas as pd
import xarray as xr
import cbo
```

<!-- #region toc-hr-collapsed=true toc-nb-collapsed=true -->
# Fig. S1
<!-- #endregion -->

## Load and prepare the data

```python
filename = 'raw_data/fig_S01.csv'
data = pd.read_csv(filename).set_index(['bias', 'pg']).to_xarray()
gdata = cbo.get_conductance_data(
    data,
    gain = 1e6,
    Rseries = 8798,
    dV = 10e-6 / np.sqrt(2),
    dim_attrs = {
    'pg' : {'long_name' :  r'Plunger gate voltage', 'units' : 'mV'},
    'bias' : {'long_name' : r'Bias voltage $V$', 'units' : r'$\mu$V'}},
    method='fit',
    plot=True
)
```

## Prepare figure

```python
# Set font sizes
plt.rc('font', size=14)          # controls default text sizes
plt.rc('axes', titlesize=14)     # fontsize of the axes title
plt.rc('axes', labelsize=14)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=12)    # fontsize of the tick labels
plt.rc('ytick', labelsize=12)    # fontsize of the tick labels
plt.rc('legend', fontsize=12)    # legend fontsize
plt.rc('figure', titlesize=14)  # fontsize of the figure title

figsize =  (2 * (3 + 3/8), (4 + 3/8))
fig, ax = plt.subplots(figsize=figsize)
divnorm = colors.TwoSlopeNorm(vmin=np.min(gdata['XR']).item(), vcenter=0, vmax=np.max(gdata['XR']).item())
gdata['XR'].plot(ax=ax, norm=divnorm, cmap='RdGy', rasterized=True, cbar_kwargs={'label' : r'Conductance [$e^2/h$]'})
plt.savefig("figures/supp_figure_coulomb_diamond.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

<!-- #region toc-hr-collapsed=true toc-nb-collapsed=true -->
# Fig. S2 - critical field measurement
<!-- #endregion -->

## Load and prepare the data

```python
filename = 'raw_data/fig_S02.csv'
data = pd.read_csv(filename).set_index(['bias', 'field']).to_xarray()
gdata = cbo.get_conductance_data(
    data,
    gain = 1e6,
    Rseries = 8798,
    dV = 10e-6 / np.sqrt(2),
    dim_attrs = {
    'field' : {'long_name' :  r'Magnetic Field $B$', 'units' : 'T'},
    'bias' : {'long_name' : r'Bias voltage $V$', 'units' : r'$\mu$V'}},
    method='fit',
    plot=True
)
```

## Prepare figure

```python
# Set font sizes
plt.rc('font', size=14)          # controls default text sizes
plt.rc('axes', titlesize=14)     # fontsize of the axes title
plt.rc('axes', labelsize=14)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=12)    # fontsize of the tick labels
plt.rc('ytick', labelsize=12)    # fontsize of the tick labels
plt.rc('legend', fontsize=12)    # legend fontsize
plt.rc('figure', titlesize=14)  # fontsize of the figure title

figsize =  (2 * (3 + 3/8), (4 + 3/8))
fig, ax = plt.subplots(figsize=figsize)
gdata['XR'].plot(ax=ax, vmin=0, cmap='Greys', rasterized=True, cbar_kwargs={'label' : r'Conductance [$e^2/h$]'})

plt.savefig("figures/supp_figure_critical_field.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

<!-- #region toc-hr-collapsed=true toc-nb-collapsed=true -->
# Fig. S3 - gate vs gate map
<!-- #endregion -->

## Load and prepare the data

```python
filename = 'raw_data/fig_S03.csv'
data = pd.read_csv(filename).set_index(['pg', 'rtg']).to_xarray()
gdata = cbo.get_conductance_data(
    data,
    gain = 1e6,
    Rseries = 8798,
    dV = 5e-6 / np.sqrt(2),
    dim_attrs = {
    'pg' : {'long_name' :  r'Plunger gate voltage', 'units' : 'mV'},
    'rtg' : {'long_name' : r'Right tunnel gate voltage', 'units' : r'mV'}},
    method='fit',
    plot=True
)
```

```python
filename = 'raw_data/fig_S03b.csv'
data = pd.read_csv(filename).set_index(['pg', 'ltg']).to_xarray()
gdatab = cbo.get_conductance_data(
    data,
    gain = 1e6,
    Rseries = 8798,
    dV = 5e-6 / np.sqrt(2),
    dim_attrs = {
    'pg' : {'long_name' :  r'Plunger gate voltage', 'units' : 'mV'},
    'ltg' : {'long_name' : r'Left tunnel gate voltage', 'units' : r'mV'}},
    method='fit',
    plot=True
)
```

## Prepare figure

```python
# Set font sizes
plt.rc('font', size=14)          # controls default text sizes
plt.rc('axes', titlesize=14)     # fontsize of the axes title
plt.rc('axes', labelsize=14)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=12)    # fontsize of the tick labels
plt.rc('ytick', labelsize=12)    # fontsize of the tick labels
plt.rc('legend', fontsize=12)    # legend fontsize
plt.rc('figure', titlesize=14)  # fontsize of the figure title

figsize =  (4 * (3 + 3/8), (4 + 3/8))
fig = plt.figure(constrained_layout=True, figsize=figsize)
gs = GridSpec(1, 2, figure=fig)
ax0 = fig.add_subplot(gs[0,0])
ax1 = fig.add_subplot(gs[0,1])

gdata['XR'].transpose().plot(
    ax=ax0,
    vmax=5,
    cmap='Greys',
    rasterized=True,
    cbar_kwargs={'label' : r'Conductance [$e^2/h$]', 'aspect' : 60}
)
#
pg_init = -1800
tg_init = -310
pg_final = 1800
tg_final = -490
ax0.plot((pg_init, pg_final), (tg_init, tg_final), marker='o', c='firebrick', ls='--')
ax0.set_ylim([-500, 0])

gdatab['XR'].transpose().plot(
    ax=ax1,
    robust=True, vmin=0, vmax=5,
    cmap='Greys',
    rasterized=True,
    cbar_kwargs={'label' : r'Conductance [$e^2/h$]', 'aspect' : 60}
)


plt.savefig("figures/supp_figure_gate_vs_gate.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

<!-- #region toc-hr-collapsed=true toc-nb-collapsed=true -->
# Fig. S4 - large conductance map vs tunnel gate in Region I
<!-- #endregion -->

## Load and prepare the data

```python
filename = 'raw_data/fig_S04.csv'
data = pd.read_csv(filename).set_index(['rtg', 'field']).to_xarray()
data = data.set_coords('pg')
data['pg'] = data['pg'].mean('field') # Removes one trivial dimension
gdata = cbo.get_conductance_data(
    data,
    gain = 1e6,
    Rseries = 8798,
    dV = 5e-6 / np.sqrt(2),
    dim_attrs = {
        'field' : {'long_name' :  r'Magnetic Field $B$', 'units' : 'T'},
        'rtg' : {'long_name' : r'Right tunnel gate voltage', 'units' : r'mV'},
        'pg' : {'long_name' :  r'Plunger gate voltage', 'units' : 'mV'}
    },
    method='fit',
    plot=True
)
```

## Peak finding

```python
# Peak finding - rotated data
peaks = cbo.get_peaks(
    array = gdata['XR'],
    coord = 'rtg',
    filter_params={'window_length' : 11, 'polyorder' : 3},
    peak_params={'distance' : 3},
    height_threshold=1,
    zscore_threshold=-2
)
```

## Prepare figure

```python
# Set font sizes
plt.rc('font', size=14)          # controls default text sizes
plt.rc('axes', titlesize=14)     # fontsize of the axes title
plt.rc('axes', labelsize=14)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=12)    # fontsize of the tick labels
plt.rc('ytick', labelsize=12)    # fontsize of the tick labels
plt.rc('legend', fontsize=12)    # legend fontsize
plt.rc('figure', titlesize=14)  # fontsize of the figure title

# Prepare grid for the figure
figsize =  (4 * (3 + 3/8), 2 * (4 + 3/8))
fig = plt.figure(constrained_layout=True, figsize=figsize)
gs = GridSpec(2, 2, figure=fig, width_ratios=[3,1])
ax0 = fig.add_subplot(gs[0,:])
ax1 = fig.add_subplot(gs[1,0])
ax2 = fig.add_subplot(gs[1,1], sharey=ax1)

im0 = gdata['XR'].isel({'field' : 0}).plot(ax=ax0, label=fr'$B={gdata.field.values[0]:2g}$ T')
gdata['XR'].isel({'field' : -11}).plot(ax=ax0, label=fr'$B={gdata.field.values[-11]:2g}$ T')
ax0.set_ylabel(ylabel=r"Conductance [$e^2/h$]")
ax0.set_title('')
ax0.set_ylim(0, 2.5)
ax0.set_xlim([-150,150])
ax0.legend()

im1 = gdata['XR'].transpose().plot(
    ax=ax1,
    cmap='Greys',
    rasterized=True,
    robust=True,
    add_colorbar=False
)

cb1 = plt.colorbar(
    mappable=im1,
    ax=ax1,
    extend='max',
    label=r"Conductance [$e^2/h$]",
    aspect=60
)


m, b = np.polyfit(gdata.rtg.values, gdata.pg.values, deg=1)
def tg_to_pg(x): return m * x + b
def pg_to_tg(x): return (x - b) / m

secax = ax1.secondary_xaxis('top', functions=(tg_to_pg, pg_to_tg))
secax.set_xlabel('Plunger gate voltage [mV]', labelpad=10)

im2 = cbo.return_spacing_histogram(peaks, coord='field', nbins=35, rng=(0, 8), dim='rtg').plot(
    ax=ax2,
    cmap='gist_heat_r',
    rasterized=True,
    add_colorbar=False
)
ax2.set_xlabel(r"Peak spacing [mV]")
ax2.set_ylabel("")
cb2 = plt.colorbar(
    mappable = im2,
    ax = ax2,
    label = r"Counts",
    aspect = 60
)

fig.text(0.01, 1.01, "(a)", fontsize=14)
fig.text(0.01, 0.50, "(b)", fontsize=14)
fig.text(0.72, 0.50, "(c)", fontsize=14)

plt.savefig("figures/supp_figure_tg_measurement_region_I.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

<!-- #region toc-hr-collapsed=true toc-nb-collapsed=true -->
# Fig. S5 - large conductance map vs tunnel gate in Region III
<!-- #endregion -->

## Load and prepare the data

```python
filename = 'raw_data/fig_S05.csv'
data = pd.read_csv(filename).set_index(['rtg', 'field']).to_xarray()
data = data.set_coords('pg')
data['pg'] = data['pg'].mean('field') # Removes one trivial dimension
gdata = cbo.get_conductance_data(
    data,
    gain = 1e6,
    Rseries = 8798,
    dV = 5e-6 / np.sqrt(2),
    dim_attrs = {
        'field' : {'long_name' :  r'Magnetic Field $B$', 'units' : 'T'},
        'rtg' : {'long_name' : r'Right tunnel gate voltage', 'units' : r'mV'},
        'pg' : {'long_name' :  r'Plunger gate voltage', 'units' : 'mV'}
    },
    method='fit',
    plot=True
)
```

## Peak finding

```python
# Peak finding - rotated data
peaks = cbo.get_peaks(
    array = gdata['XR'],
    coord = 'rtg',
    filter_params={'window_length' : 11, 'polyorder' : 3},
    peak_params={'distance' : 3},
    height_threshold=1,
    zscore_threshold=-2
)
```

## Prepare figure

```python
# Set font sizes
plt.rc('font', size=14)          # controls default text sizes
plt.rc('axes', titlesize=14)     # fontsize of the axes title
plt.rc('axes', labelsize=14)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=12)    # fontsize of the tick labels
plt.rc('ytick', labelsize=12)    # fontsize of the tick labels
plt.rc('legend', fontsize=12)    # legend fontsize
plt.rc('figure', titlesize=14)  # fontsize of the figure title

# Prepare grid for the figure
figsize =  (4 * (3 + 3/8), 2 * (4 + 3/8))
fig = plt.figure(constrained_layout=True, figsize=figsize)
gs = GridSpec(2, 2, figure=fig, width_ratios=[3,1])
ax0 = fig.add_subplot(gs[0,:])
ax1 = fig.add_subplot(gs[1,0])
ax2 = fig.add_subplot(gs[1,1], sharey=ax1)

im0 = gdata['XR'].isel({'field' : 0}).plot(ax=ax0, label=fr'$B={gdata.field.values[0]:2g}$ T')
gdata['XR'].isel({'field' : 20}).plot(ax=ax0, label=fr'$B={gdata.field.values[20]:2g}$ T')
ax0.set_ylabel(ylabel=r"Conductance [$e^2/h$]")
ax0.set_title('')
ax0.set_ylim(0, 3)
ax0.set_xlim([-550,-350])
ax0.legend()

im1 = gdata['XR'].transpose().plot(
    ax=ax1,
    cmap='Greys',
    rasterized=True,
    robust=True,
    add_colorbar=False
)

cb1 = plt.colorbar(
    mappable=im1,
    ax=ax1,
    extend='max',
    label=r"Conductance [$e^2/h$]",
    aspect=60
)


m, b = np.polyfit(gdata.rtg.values, gdata.pg.values, deg=1)
def tg_to_pg(x): return m * x + b
def pg_to_tg(x): return (x - b) / m

secax = ax1.secondary_xaxis('top', functions=(tg_to_pg, pg_to_tg))
secax.set_xlabel('Plunger gate voltage [mV]', labelpad=10)

im2 = cbo.return_spacing_histogram(peaks, coord='field', nbins=25, rng=(0, 5), dim='rtg').plot(
    ax=ax2,
    cmap='gist_heat_r',
    rasterized=True,
    add_colorbar=False
)
ax2.set_xlabel(r"Peak spacing [mV]")
ax2.set_ylabel("")
cb2 = plt.colorbar(
    mappable = im2,
    ax = ax2,
    label = r"Counts",
    aspect = 60
)

fig.text(0.01, 1.01, "(a)", fontsize=14)
fig.text(0.01, 0.50, "(b)", fontsize=14)
fig.text(0.72, 0.50, "(c)", fontsize=14)

plt.savefig("figures/supp_figure_tg_measurement_region_III.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

<!-- #region toc-hr-collapsed=true toc-nb-collapsed=true -->
# Fig S6: See `figures_main_text.md`
<!-- #endregion -->

# Fig. S7 -  second phase diagram


## Load and prepare the data

```python
files = glob.glob("raw_data/fig_S07/*.csv")
files.sort()
dfs = [pd.read_csv(file) for file in files]
data = pd.concat(dfs).set_index(['field', 'pg_fine', 'pg_coarse']).to_xarray()
data = data.set_coords('rtg')
data['rtg'] = data['rtg'].mean(['field', 'pg_fine']) # Removes trivial dimensions
data
```

```python
fields = data.field.values
plungers = data.pg_coarse.values
dim_attributes = {
    'pg_coarse' : {'long_name' :  'Plunger gate voltage', 'units' : r'mV'},
    'pg_fine' : {'long_name' :  'Plunger gate voltage', 'units' : r'mV'},
    'field' : {'long_name' : r'Magnetic Field', 'units' : 'T'}

}
darrays = []
for (i, f) in enumerate(fields):
    darray = []
    for (j, p) in enumerate(plungers):
        d = data.sel(dict(field=f, pg_coarse=p), method='nearest')
        dnew = cbo.get_conductance_data(
            d, gain=1e6, Rseries=8798, dV=5e-6 / np.sqrt(2),
            dim_attrs=dim_attributes,
            method='minimum',
            plot=False
        )
        darray.append(dnew)
    darrays.append(darray)
gdata = xr.combine_nested(darrays, concat_dim=['field', 'pg_coarse'])
```

```python
cbo.plot_and_compare_conductance_data(gdata)
```

```python
# Cut first two values of pg coarse, which are unreliable due to noise after the large MDAC jump
gdata = gdata.isel({'pg_coarse' : slice(2,91)})
```

## Peak spacing analysis

```python
# Peak finding - rotated data
peaks_rotated = cbo.get_peaks(
    array = gdata['XR'],
    coord = 'pg_fine',
    filter_params={'window_length' : 11, 'polyorder' : 3},
    peak_params={'distance' : 2},
    height_threshold=1,
    zscore_threshold=-2
)
results_rotated = cbo.get_aggregate_peak_info(peaks_rotated)
```

```python
for var in results_rotated.data_vars:
    results_rotated[var].plot()
    plt.show()
```

## Preparing the figure

```python
figsize =  (2 * (3 + 3/8), 1.5 * (4 + 3/8))
fig = plt.figure(constrained_layout=True, figsize=figsize)

# Set font sizes
plt.rc('font', size=14)          # controls default text sizes
plt.rc('axes', titlesize=14)     # fontsize of the axes title
plt.rc('axes', labelsize=14)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=12)    # fontsize of the tick labels
plt.rc('ytick', labelsize=12)    # fontsize of the tick labels
plt.rc('legend', fontsize=12)    # legend fontsize
plt.rc('figure', titlesize=14)  # fontsize of the figure title

# Prepare plot grid
gs = GridSpec(1, 1, figure=fig)
ax1 = fig.add_subplot(gs[0,0])

# ax1: Field versus plunger
im1 = xr.plot.plot(
    darray = results_rotated["median_spacing"],
    ax=ax1,
    cmap='Blues',
    robust=True,
    rasterized=True,
    add_colorbar=False)

cb1 = plt.colorbar(
    mappable=im1,
    ax=ax1,
    location='top',
    extend='max',
    shrink=0.75,
    label=r"Median Peak Spacing [mV]",
    aspect=60
)

ax1.set_xlabel(r"Plunger Gate voltage [mV]")
ax1.set_ylabel(r"Magnetic Field $B$ [T]")

# Arrows defining regions
ax1.annotate(
    "",
    xy=(-1720, 0.9),
    xytext=(-750, 0.9),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3")
)
ax1.annotate(
    "", xy=(-750, 0.9), xytext=(0, 0.9),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3")
)
ax1.annotate(
    "", xy=(0, 0.9), xytext=(1800, 0.9),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3")
)

# Region labels
ax1.text(-1360, 0.95, "I", size=16)
ax1.text(-375, 0.95, "II", size=16)
ax1.text(900, 0.95, "III", size=16)

# Show figure
plt.savefig("figures/supp_figure_phase_diagram.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

# Fig. S8 : More Coulomb oscillations in region II


## Load and prepare data

```python
filename = 'raw_data/fig_S08.csv'
data = pd.read_csv(filename, sep=",").set_index(['field', 'pg']).to_xarray()
data
```

```python
dim_attributes = {
    'pg' : {'long_name' :  r'Plunger gate voltage', 'units' : r'mV'},
    'field' : {'long_name' : r'Magnetic Field $B$', 'units' : 'T'}

}
gdata = cbo.get_conductance_data(data, gain=1e6, Rseries=8798, dV=10e-6 / np.sqrt(2), dim_attrs=dim_attributes, method='fit')
```

```python
# Crop the data to focus only on complete Coulomb valleys to extract peak spacings
gdata = gdata.isel({'pg' : np.where(gdata.pg > -688.5)[0]})
gdata = gdata.isel({'pg' : np.where(gdata.pg < -682)[0]})
```

<!-- #region toc-hr-collapsed=true toc-nb-collapsed=true -->
## Peak finding
<!-- #endregion -->

```python
peaks = cbo.get_peaks(
    gdata['XR'],
    coord='pg',
    filter_params={'window_length' : 15, 'polyorder' : 5},
    peak_params={'distance' : 2},
    height_threshold=0.1,
    zscore_threshold=-3
).drop('height_zscore', axis=1)

# Also filter by absolute height of the peak
peaks = peaks.loc[peaks['height'] >= 0.01]

# Sort peaks by field, then plunger
peaks = peaks.sort_values(by = ['field', 'pg'])
```

```python
pgs = peaks.pg.values
bs = peaks.field.values
gdata['XR'].plot(vmin=0, cmap='Greys')
plt.scatter(pgs, bs, c='red', marker='.')
```

### Outliers selection

```python
cbo.plot_conductance_data_with_peaks(gdata['XR'], peaks)
```

```python
# By manual inspection, we select outliers or boundary points that we don't need

# outliers
to_remove = [
    (0.8, 0),
    (0.76, 0)
]

# boundary points right side, first group
for b in np.arange(0.41, 0.47+1e-9, 0.01):
    to_remove.append((np.round(b, 2), -1))

# We also notice shallow side peaks that we can add by hand even though they were not picked up by the peak finder
dvpg = np.diff(gdata.pg.values)[0]
to_add = [
    (0.33, peaks.loc[peaks['field']==0.33].pg.values[7] + 5 * dvpg),
    (0.37, peaks.loc[peaks['field']==0.37].pg.values[0] + 11 * dvpg),
    (0.37, peaks.loc[peaks['field']==0.37].pg.values[1] + 13 * dvpg),
    (0.37, peaks.loc[peaks['field']==0.37].pg.values[2] + 12 * dvpg),
    (0.37, peaks.loc[peaks['field']==0.37].pg.values[5] + 13 * dvpg),
    (0.38, peaks.loc[peaks['field']==0.38].pg.values[0] + 13 * dvpg),
    (0.38, peaks.loc[peaks['field']==0.38].pg.values[1] + 13 * dvpg),
    (0.38, peaks.loc[peaks['field']==0.38].pg.values[2] + 10 * dvpg),
    (0.38, peaks.loc[peaks['field']==0.38].pg.values[3] + 11 * dvpg),
    (0.38, peaks.loc[peaks['field']==0.38].pg.values[4] + 15 * dvpg),
    (0.39, peaks.loc[peaks['field']==0.39].pg.values[2] + 10 * dvpg),
    (0.39, peaks.loc[peaks['field']==0.39].pg.values[3] + 12 * dvpg),
    (0.39, peaks.loc[peaks['field']==0.39].pg.values[4] + 11 * dvpg),
    (0.39, peaks.loc[peaks['field']==0.39].pg.values[5] + 15 * dvpg),
    (0.40, peaks.loc[peaks['field']==0.40].pg.values[0] + 12 * dvpg),
    (0.40, peaks.loc[peaks['field']==0.40].pg.values[1] + 12 * dvpg),
    (0.40, peaks.loc[peaks['field']==0.40].pg.values[2] + 12 * dvpg),
    (0.40, peaks.loc[peaks['field']==0.40].pg.values[3] + 13 * dvpg),
    (0.40, peaks.loc[peaks['field']==0.40].pg.values[4] + 24 * dvpg),
    (0.41, peaks.loc[peaks['field']==0.41].pg.values[2] + 12 * dvpg),
    (0.41, peaks.loc[peaks['field']==0.41].pg.values[3] + 13 * dvpg),
    (0.41, peaks.loc[peaks['field']==0.41].pg.values[6] + 29 * dvpg),
    (0.42, peaks.loc[peaks['field']==0.42].pg.values[8] + 28 * dvpg),
]

# Find plunger values for the points to be removed
_tmp = []
for (b, idx) in to_remove:
    p = peaks.loc[peaks['field']==b]
    new_point = (b, p.pg.values[idx])
    _tmp.append(new_point)
to_remove = _tmp

# Remove rows corresponding to the outliers
rows_to_remove = []
for (item, row) in peaks.iterrows():
    point = (row['field'], row['pg'])
    if point in to_remove:
        rows_to_remove.append(item)
peaks = peaks.drop(rows_to_remove)

# Add peaks
for (b, pg) in to_add:
    height = gdata['XR'].sel(field=b, pg=pg, method='nearest').values
    peaks = peaks.append({'field' : b, 'pg' : pg, 'height' : height}, ignore_index=True)

# Remove all peaks at B=0.53T (gate jump)
peaks = peaks[peaks.field != 0.53]
# Remove all peaks for B smaller than 0.33T
# This is because identifying peaks and valley parity for B ~ 0.2-0.3 T is difficult and thus we prefer not to commit
peaks = peaks[peaks.field > 0.32]
```

```python
pgs = peaks.pg.values
bs = peaks.field.values
gdata['XR'].plot(vmin=0, cmap='Greys')
plt.scatter(pgs, bs, c='red', marker='.')
plt.show()
```

## Peak spacings

We have 5 complete odd valleys and four even valleys starting at $B=0.33$ T and ending at $B=0.8$T.

```python
even_spacings = np.zeros((5, 48))
odd_spacings = np.zeros((5, 48))
# Field values run from 0.3 to 0.8, but skipping 0.53T
field_vals = np.round(np.arange(0.33, 0.8+1e-9, 0.01),2)

for i, (b, group) in enumerate(peaks.groupby('field')):
    group = group.sort_values('pg')
    spacings = np.diff(group.pg.values)
    if b < 0.53:
        even_spacings[:-1,i] = spacings[1::2]
        even_spacings[-1] = np.nan
        odd_spacings[:,i] = spacings[::2]
    elif b > 0.53:
        even_spacings[:-1,i+1] = spacings[1::2]
        even_spacings[-1,i+1] = np.nan
        odd_spacings[:,i+1] = spacings[::2]
# Fill 0.53 T values as nans
odd_spacings[:,20] = np.empty((5, )).fill(np.nan)
even_spacings[:,20] = np.empty((5, )).fill(np.nan)

# Load all the spacings data into a single xarray
peak_spacings = xr.Dataset(
    data_vars = {
        'even' : (['field', 'label'], even_spacings.T),
        'odd'  : (['field', 'label'], odd_spacings.T),
    },
    coords = {
        'field' : ('field', field_vals),
        'label' : ('label', [1, 2, 3, 4, 5]),
    }
)
peak_spacings['diff'] = peak_spacings['even'] - peak_spacings['odd']
```

## Peak heights

```python
lambda_left = np.zeros((4, 48))
lambda_right = np.zeros((4, 48))
for i, (b, group) in enumerate(peaks.groupby('field')):
    group = group.sort_values('pg')
    hs = group.height.values #boundary: discard first peak
    if b < 0.53:
        lambda_left[:,i] = np.array([hs[2*m+1] / (hs[2*m] + hs[2*m + 1]) for m in range(4)])
        lambda_right[:,i] = np.array([hs[2*m+1] / (hs[2*m+2] + hs[2*m + 1]) for m in range(4)])
    if b > 0.53:
        lambda_left[:,i+1] = np.array([hs[2*m+1] / (hs[2*m] + hs[2*m + 1]) for m in range(4)])
        lambda_right[:,i+1] = np.array([hs[2*m+1] / (hs[2*m+2] + hs[2*m + 1]) for m in range(4)])
lambda_left[:,20] = np.empty((4, )).fill(np.nan)
lambda_right[:,20] = np.empty((4, )).fill(np.nan)

# Load all the spacings data into a single xarray
peak_lambdas = xr.Dataset(
    data_vars = {
        'left' : (['field', 'label'], lambda_left.T),
        'right' : (['field', 'label'], lambda_right.T)
    },
    coords = {
        'field' : ('field', field_vals),
        'label' : ('label', [1, 2, 3, 4]),
    }
)
peak_lambdas['avg'] = 0.5 * (peak_lambdas['left'] + peak_lambdas['right'])

# Interpolate the peak lambda to find the magnetic field values at which it is equal to 0.5
bs = peak_lambdas.field.values
bvalues = []
binterp = np.linspace(0.4, 0.5, 10001)
for valley_idx in range(1,5):
    lambdas = peak_lambdas.sel(label=valley_idx).avg.values
    bidx = np.argmin(np.abs(np.interp(binterp, bs, lambdas) - 0.5))
    bvalues.append(binterp[bidx])
```

## Prepare the figure

```python
figsize =  (2 * (3 + 3/8), 2 * (4 + 3/8))
fig = plt.figure(constrained_layout=True, figsize=figsize)

# Set font sizes
plt.rc('font', size=14)          # controls default text sizes
plt.rc('axes', titlesize=14)     # fontsize of the axes title
plt.rc('axes', labelsize=14)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=12)    # fontsize of the tick labels
plt.rc('ytick', labelsize=12)    # fontsize of the tick labels
plt.rc('legend', fontsize=12)    # legend fontsize
plt.rc('figure', titlesize=14)  # fontsize of the figure title

# Prepare plot grid
gs = GridSpec(2, 2, figure=fig, height_ratios=[2,1], width_ratios=[1.8,1])
ax1 = fig.add_subplot(gs[0,0])
gs2 = GridSpecFromSubplotSpec(1, 2, subplot_spec=gs[1,0])
ax2 = fig.add_subplot(gs2[0,0])
ax3 = fig.add_subplot(gs2[0,1], sharey=ax2)
gs3 = GridSpecFromSubplotSpec(4, 1, subplot_spec=gs[:,1])
ax4_0 = fig.add_subplot(gs3[0,0])
ax4_1 = fig.add_subplot(gs3[1,0], sharex=ax4_0)
ax4_2 = fig.add_subplot(gs3[2,0], sharex=ax4_0)
ax4_3 = fig.add_subplot(gs3[3,0], sharex=ax4_0)

# ax1: Conductance, field versus plunger
im1 = xr.plot.plot(
    darray = gdata['XR'],
    ax=ax1,
    cmap='binary',
    robust=True,
    rasterized=True,
    add_colorbar=False)

cb1 = plt.colorbar(
    mappable=im1,
    ax=ax1,
    location='top',
    extend='both',
    shrink=0.75,
    label=r"Conductance $G$ [$e^2/h$]",
    aspect=60
)

# Arrows defining extent of bottom panels
ax1.annotate(
     "",
     xy=(-688.3, 0.33),
     xytext=(-688.3, 0.7),
     arrowprops=dict(arrowstyle="<->", connectionstyle="arc3")
)

# ax2 : peak spacing variation
sel = peak_spacings['diff'].isel({'field' : slice(0, 38), 'label' : slice(0,4)})
im2 = xr.plot.plot(
    darray = sel,
    ax=ax2,
    cmap='coolwarm',
    robust=True,
    xticks=[1,2,3,4],
    rasterized=True,
    add_colorbar=False
)
ax2.scatter(range(1, 5), bvalues, marker='x', c='k', lw=0.8)
ax2.set_ylabel(r"$B$ [T]")
ax2.set_xlabel(r"Valley index $i$")

cb2 = plt.colorbar(
    mappable=im2,
    ax=ax2,
    location='top',
    label=r"$S_{e,i}-S_{o,i}$ [mV]",
    aspect=40
)

# ax3 : relative peak height variation
im3 = xr.plot.plot(
    darray = peak_lambdas['avg'].isel({'field' : slice(0, 38)}),
    ax=ax3,
    cmap='coolwarm',
    robust=True,
    vmin=0.30,
    vmax=0.70,
    xticks=[1,2,3,4],
    rasterized=True,
    add_colorbar=False
)
ax3.scatter(range(1, 5), bvalues, marker='x', c='k', lw=0.8)
ax3.set_ylabel("")
ax3.set_xlabel(r"Valley index $i$")
plt.setp(ax3.get_yticklabels(), visible=False)
cb3 = plt.colorbar(
    mappable=im3,
    ax=ax3,
    location='top',
    label=r"Asymmetry $\Lambda_{i}$",
    aspect=40
)

# Add vertical grid lines to enhance that x axis is discrete
for x in [1.5, 2.5, 3.5]:
    ax2.axvline(x, c='k', lw=0.5)
    ax3.axvline(x, c='k', lw=0.5)

# ax4: peak spacings and height oscillations for one valley
for (i, ax) in enumerate([ax4_0, ax4_1, ax4_2, ax4_3]):
    ps = peak_spacings.isel({'label' : i, 'field' : slice(0, 38)})
    ls = peak_lambdas.isel({'label' : i, 'field' : slice(0, 38)})
    axt = ax.twinx()
    xr.plot.scatter(
        ds = ps,
        y = 'even',
        x = 'field',
        marker='x',
        c='red',
        ax = ax,
        label=r"$S_e$"
    )
    xr.plot.scatter(
        ds = ps,
        y = 'odd',
        x = 'field',
        marker='x',
        c='blue',
        ax = ax,
        label=r"$S_o$"
    )
    xr.plot.scatter(
        ds = ls,
        y = 'avg',
        x = 'field',
        ax = axt,
        marker='.',
        c = 'black'
    )
    ax.axvline(bvalues[i], c='black', lw=0.8, ls='--', dashes=(5, 5))
    # Text label for the valley
    ax.text(0.6, 0.30, rf"$i={i+1}$", size=12)
    # set up a legend
    if i==0:
        ax.scatter([], [], marker='.', c='black', label=r'$\Lambda$')
        ax.legend(
            bbox_to_anchor = (0, 1.02, 1, 1.02),
            loc='lower left',
            mode='expand',
            handletextpad=0.05,
            borderaxespad=0,
            ncol=3,
            frameon=False
        )
        pass

    ax.set_xlabel("")
    ax.set_ylabel("")
    ax.set_ylim([0.25, 1.0])
    ax.set_yticks([0.25, 0.5, 0.75])
    axt.set_ylim([0.25, 0.62])
    axt.set_yticks([0.3, 0.4, 0.5, 0.6])
    axt.set_ylabel("")
    if i != 3:
        plt.setp(ax.get_xticklabels(), visible=False)
    axt.grid()
    ax.grid(axis='x')
ax4_3.set_xlabel("$B$ [T]")

# Valley indices
ax1.text(-687.4, 0.7, r"$e_1$", size=12)
ax1.text(-686.2, 0.7, r"$e_2$", size=12)
ax1.text(-684.9, 0.7, r"$e_3$", size=12)
ax1.text(-683.7, 0.7, r"$e_4$", size=12)

ax1.text(-688, 0.75, r"$o_1$", size=12)
ax1.text(-686.75, 0.75, r"$o_2$", size=12)
ax1.text(-685.5, 0.75, r"$o_3$", size=12)
ax1.text(-684.3, 0.75, r"$o_4$", size=12)

# Panel labels
fig.text(0.01, 0.97, "(a)", fontsize=14)
fig.text(0.63, 0.97, "(b)", fontsize=14)
fig.text(0.01, 0.32, "(c)", fontsize=14)

# Show figure
plt.savefig("figures/supp_figure_coulomb_oscillations_I.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

# Fig. S9 : More Coulomb oscillations in region II


## Load and prepare data

```python
filename = 'raw_data/fig_S09.csv'
data = pd.read_csv(filename, sep=",").set_index(['field', 'pg']).to_xarray()
```

```python
dim_attributes = {
    'pg' : {'long_name' :  r'Plunger gate voltage', 'units' : r'mV'},
    'field' : {'long_name' : r'Magnetic Field $B$', 'units' : 'T'}

}
gdata = cbo.get_conductance_data(data, gain=1e6, Rseries=8798, dV=5e-6 / np.sqrt(2), dim_attrs=dim_attributes, method='fit')
```

```python
# Crop the data to focus only on complete Coulomb valleys to extract peak spacings
gdata = gdata.isel({'pg' : np.where(gdata.pg > -257.5)[0]})
```

<!-- #region toc-hr-collapsed=true toc-nb-collapsed=true -->
## Peak finding
<!-- #endregion -->

```python
peaks = cbo.get_peaks(
    gdata['XR'], coord='pg',
    filter_params={'window_length' : 15, 'polyorder' : 5},
    peak_params={'distance' : 2},
    height_threshold=0.1,
    zscore_threshold=-3
).drop('height_zscore', axis=1)

# Also filter by absolute height of the peak
peaks = peaks.loc[peaks['height'] >= 0.01]

# Sort peaks by field, then plunger
peaks = peaks.sort_values(by = ['field', 'pg'])

pgs = peaks.pg.values
bs = peaks.field.values
fig = plt.figure()
gdata['XR'].plot(vmin=0, cmap='Greys')
plt.scatter(pgs, bs, c='red', marker='.')
```

### Outliers selection

```python
cbo.plot_conductance_data_with_peaks(gdata['XR'], peaks)
```

```python
# By manual inspection, we select outliers or boundary points that we don't need

# outliers
to_remove = [
    (0.03, 3),
    (0.05, 3),
    (0.06, 0),
    (0.07, 0),
    (0.11, 5),
    (0.12, 4),
    (0.17, 4),
    (0.21, 4),
    (0.27, 1),
    (0.27, 6),
    (0.29, 3),
    (0.35, 0),
    (0.35, -1),
    (0.36, 4),
    (0.36, 6),
    (0.53, 1),
    (0.54, 1),
    (0.68, 0),
    (0.68, 6),
    (0.68, 11)
]

#boundary points right side, first group
for b in np.arange(0.40, 0.70+1e-9, 0.01):
    to_remove.append((np.round(b, 2), 0))

# We also notice shallow side peaks that we can add by hand even though they were not picked up by the peak finder
dvpg = np.diff(gdata.pg.values)[0]
to_add = [
    (0.37, peaks.loc[peaks['field']==0.37].pg.values[0] + 8 * dvpg),
    (0.37, peaks.loc[peaks['field']==0.37].pg.values[1] + 10 * dvpg),
    (0.49, peaks.loc[peaks['field']==0.49].pg.values[0] + 17 * dvpg),
    (0.50, peaks.loc[peaks['field']==0.50].pg.values[0] + 17 * dvpg),
    (0.51, peaks.loc[peaks['field']==0.51].pg.values[0] + 18 * dvpg),
]

# Find plunger values for the points to be removed
_tmp = []
for (b, idx) in to_remove:
    p = peaks.loc[peaks['field']==b]
    new_point = (b, p.pg.values[idx])
    _tmp.append(new_point)
to_remove = _tmp

# Remove rows corresponding to the outliers
rows_to_remove = []
for (item, row) in peaks.iterrows():
    point = (row['field'], row['pg'])
    if point in to_remove:
        rows_to_remove.append(item)
peaks = peaks.drop(rows_to_remove)

# Add peaks
for (b, pg) in to_add:
    height = gdata['XR'].sel(field=b, pg=pg, method='nearest').values
    peaks = peaks.append({'field' : b, 'pg' : pg, 'height' : height}, ignore_index=True)

# Remove all peaks for B higher than 0.68
# This region is noisy and not so interesting
peaks = peaks[peaks.field < 0.68]

pgs = peaks.pg.values
bs = peaks.field.values
fig = plt.figure()
gdata['XR'].plot(vmin=0, cmap='Greys')
plt.scatter(pgs, bs, c='red', marker='.')
```

# Peak spacings

We have 5 complete odd valleys and five even valleys

* The five even valleys go all the way from B=0 to B=0.68 T
* The five odd valleys appear at B=0.37 T

```python
even_spacings = np.zeros((5, 69))
odd_spacings = np.zeros((5, 69))
# Field values run from 0.3 to 0.8, but skipping 0.53T
field_vals = gdata.field.values[:69]

for i, (b, group) in enumerate(peaks.groupby('field')):
    group = group.sort_values('pg')
    spacings = np.diff(group.pg.values)
    if b < 0.37:
        even_spacings[:,i] = spacings
        odd_spacings[:,i] = np.empty((5, )).fill(np.nan)
    else:
        even_spacings[:,i] = spacings[1::2]
        odd_spacings[:,i] = spacings[::2]

# Load all the spacings data into a single xarray
peak_spacings = xr.Dataset(
    data_vars = {
        'even' : (['field', 'label'], even_spacings.T),
        'odd'  : (['field', 'label'], odd_spacings.T),
    },
    coords = {
        'field' : ('field', field_vals),
        'label' : ('label', [1, 2, 3, 4, 5]),
    }
)
peak_spacings['diff'] = peak_spacings['even'] - peak_spacings['odd']
```

## Peak heights

```python
lambda_left = np.zeros((5, 31))
lambda_right = np.zeros((5, 31))
sel = peaks.loc[peaks['field'] > 0.36 + 1e-9]
for i, (b, group) in enumerate(sel.groupby('field')):
    group = group.sort_values('pg')
    hs = group.height.values
    lambda_left[:,i] = np.array([hs[2*m+1] / (hs[2*m] + hs[2*m + 1]) for m in range(5)])
    lambda_right[:,i] = np.array([hs[2*m+1] / (hs[2*m+2] + hs[2*m + 1]) for m in range(5)])

# Load all the spacings data into a single xarray
peak_lambdas = xr.Dataset(
    data_vars = {
        'left' : (['field', 'label'], lambda_left.T),
        'right' : (['field', 'label'], lambda_right.T)
    },
    coords = {
        'field' : ('field', gdata.field.values[37:68]),
        'label' : ('label', [1, 2, 3, 4, 5]),
    }
)
peak_lambdas['avg'] = 0.5 * (peak_lambdas['left'] + peak_lambdas['right'])

# Interpolate the peak lambda to find the magnetic field values at which it is equal to 0.5
bs = peak_lambdas.field.values
bvalues = []
binterp = np.linspace(0.4, 0.5, 10001)
for valley_idx in range(1,6):
    lambdas = peak_lambdas.sel(label=valley_idx).avg.values
    bidx = np.argmin(np.abs(np.interp(binterp, bs, lambdas) - 0.5))
    bvalues.append(binterp[bidx])
```

## Prepare the figure

```python
figsize =  (2 * (3 + 3/8), 2 * (4 + 3/8))
fig = plt.figure(constrained_layout=True, figsize=figsize)

# Set font sizes
plt.rc('font', size=14)          # controls default text sizes
plt.rc('axes', titlesize=14)     # fontsize of the axes title
plt.rc('axes', labelsize=14)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=12)    # fontsize of the tick labels
plt.rc('ytick', labelsize=12)    # fontsize of the tick labels
plt.rc('legend', fontsize=12)    # legend fontsize
plt.rc('figure', titlesize=14)  # fontsize of the figure title

# Prepare plot grid
gs = GridSpec(2, 2, figure=fig, height_ratios=[2,1], width_ratios=[1.8,1])
ax1 = fig.add_subplot(gs[0,0])
gs2 = GridSpecFromSubplotSpec(1, 2, subplot_spec=gs[1,0])
ax2 = fig.add_subplot(gs2[0,0])
ax3 = fig.add_subplot(gs2[0,1], sharey=ax2)
gs3 = GridSpecFromSubplotSpec(5, 1, subplot_spec=gs[:,1])
ax4_0 = fig.add_subplot(gs3[0,0])
ax4_1 = fig.add_subplot(gs3[1,0], sharex=ax4_0)
ax4_2 = fig.add_subplot(gs3[2,0], sharex=ax4_0)
ax4_3 = fig.add_subplot(gs3[3,0], sharex=ax4_0)
ax4_4 = fig.add_subplot(gs3[4,0], sharex=ax4_0)

# ax1: Conductance, field versus plunger
im1 = xr.plot.plot(
    darray = gdata['XR'],
    ax=ax1,
    cmap='binary',
    robust=True,
    rasterized=True,
    add_colorbar=False)

cb1 = plt.colorbar(
    mappable=im1,
    ax=ax1,
    location='top',
    extend='both',
    shrink=0.75,
    label=r"Conductance $G$ [$e^2/h$]",
    aspect=60
)

#Arrows defining extent of bottom panels
ax1.annotate(
     "",
     xy=(-257.2, 0.36),
     xytext=(-257.2, 0.69),
     c='white',
     arrowprops=dict(color='black', arrowstyle="<->", connectionstyle="arc3")
)


# ax2 : peak spacing variation
sel = peak_spacings['diff'].isel({'field' : slice(37, 68), 'label' : slice(0,5)})
im2 = xr.plot.plot(
    darray = sel,
    ax=ax2,
    cmap='coolwarm',
    robust=True,
    xticks=[1,2,3,4],
    rasterized=True,
    add_colorbar=False
)
ax2.scatter(range(1, 6), bvalues, marker='x', c='k', lw=0.8)
ax2.set_ylabel(r"$B$ [T]")
ax2.set_xlabel(r"Valley index $i$")

cb2 = plt.colorbar(
    mappable=im2,
    ax=ax2,
    location='top',
    label=r"$S_{e,i}-S_{o,i}$ [mV]",
    aspect=40
)

# ax3 : relative peak height variation
im3 = xr.plot.plot(
    darray = peak_lambdas['avg'],
    ax=ax3,
    cmap='coolwarm',
    robust=True,
    vmin=0.42,
    vmax=0.58,
    xticks=[1,2,3,4],
    rasterized=True,
    add_colorbar=False
)
ax3.scatter(range(1, 6), bvalues, marker='x', c='k', lw=0.8)
ax3.set_ylabel("")
ax3.set_xlabel(r"Valley index $i$")
plt.setp(ax3.get_yticklabels(), visible=False)
cb3 = plt.colorbar(
    mappable=im3,
    ax=ax3,
    location='top',
    label=r"Asymmetry $\Lambda_{i}$",
    aspect=40
)

# Add vertical grid lines to enhance that x axis is discrete
for x in [1.5, 2.5, 3.5, 4.5]:
    ax2.axvline(x, c='k', lw=0.5)
    ax3.axvline(x, c='k', lw=0.5)

#ax4: peak spacings and height oscillations for one valley
for (i, ax) in enumerate([ax4_0, ax4_1, ax4_2, ax4_3, ax4_4]):
    ps = peak_spacings.isel({'label' : i, 'field' : slice(37, 68)})
    ls = peak_lambdas
    axt = ax.twinx()
    xr.plot.scatter(
        ds = ps,
        y = 'even',
        x = 'field',
        marker='x',
        c='red',
        ax = ax,
        label=r"$S_e$"
    )
    xr.plot.scatter(
        ds = ps,
        y = 'odd',
        x = 'field',
        marker='x',
        c='blue',
        ax = ax,
        label=r"$S_o$"
    )
    xr.plot.scatter(
        ds = ls,
        y = 'avg',
        x = 'field',
        ax = axt,
        marker='.',
        c = 'black'
    )
    ax.axvline(bvalues[i], c='black', lw=0.8, ls='--', dashes=(5, 5))
    # set up a legend
    if i==0:
        ax.scatter([], [], marker='.', c='black', label=r'$\Lambda$')
        ax.legend(
            bbox_to_anchor = (0, 1.02, 1, 1.02),
            loc='lower left',
            mode='expand',
            handletextpad=0.05,
            borderaxespad=0,
            ncol=3,
            frameon=False
        )
        pass

    ax.set_xlabel("")
    ax.set_ylabel("")
    ax.set_ylim([0.2, 1.22])
    ax.set_yticks([0.3, 0.6, 0.9, 1.2])
    axt.set_ylim([0.42, 0.58])
    axt.set_yticks([0.45, 0.5, 0.55])
    axt.set_ylabel("")
    if i != 4:
        plt.setp(ax.get_xticklabels(), visible=False)
    axt.grid()
    ax.grid(axis='x')
ax4_4.set_xlabel("$B$ [T]")

# Valley indices
ax1.text(-256.3, 0.3, r"$e_1$", size=12)
ax1.text(-255, 0.3, r"$e_2$", size=12)
ax1.text(-253.8, 0.3, r"$e_3$", size=12)
ax1.text(-252.6, 0.3, r"$e_4$", size=12)
ax1.text(-251.3, 0.3, r"$e_5$", size=12)

ax1.text(-256.6, 0.47, r"$o_1$", size=12)
ax1.text(-255.3, 0.47, r"$o_2$", size=12)
ax1.text(-254.1, 0.47, r"$o_3$", size=12)
ax1.text(-252.8, 0.47, r"$o_4$", size=12)
ax1.text(-251.5, 0.47, r"$o_5$", size=12)

# Panel labels
fig.text(0.01, 0.97, "(a)", fontsize=14)
fig.text(0.63, 0.97, "(b)", fontsize=14)
fig.text(0.01, 0.32, "(c)", fontsize=14)

# Show figure
plt.savefig("figures/supp_figure_coulomb_oscillations_II.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

# Fig. S10: Coulomb oscillations in region III


## Load and prepare data

```python
filename = 'raw_data/fig_S10.csv'
data = pd.read_csv(filename, sep=",").set_index(['field', 'pg']).to_xarray()
data
```

```python
dim_attributes = {
    'pg' : {'long_name' :  r'Plunger gate voltage', 'units' : r'mV'},
    'field' : {'long_name' : r'Magnetic Field $B$', 'units' : 'T'}

}
gdata = cbo.get_conductance_data(data, gain=1e6, Rseries=8798, dV=10e-6 / np.sqrt(2), dim_attrs=dim_attributes, method='fit')
```

## Peak finding

```python
# Peak finding - rotated data
peaks = cbo.get_peaks(
    array = gdata['XR'],
    coord = 'pg',
    filter_params={'window_length' : 11, 'polyorder' : 3},
    peak_params={'distance' : 3},
    height_threshold=1,
    zscore_threshold=-2
)
```

## Prepare the figure

```python
# Set font sizes
plt.rc('font', size=14)          # controls default text sizes
plt.rc('axes', titlesize=14)     # fontsize of the axes title
plt.rc('axes', labelsize=14)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=12)    # fontsize of the tick labels
plt.rc('ytick', labelsize=12)    # fontsize of the tick labels
plt.rc('legend', fontsize=12)    # legend fontsize
plt.rc('figure', titlesize=14)  # fontsize of the figure title

# Prepare grid for the figure
figsize =  (3 * (3 + 3/8), 1 * (4 + 3/8))
fig = plt.figure(constrained_layout=True, figsize=figsize)
gs = GridSpec(1, 2, figure=fig, width_ratios=[2,1])
ax0 = fig.add_subplot(gs[0,0])
ax1 = fig.add_subplot(gs[0,1], sharey=ax0)

im0 = gdata['XR'].plot(
    ax=ax0,
    cmap='Greys',
    rasterized=True,
    robust=True,
    add_colorbar=False
)

cb0 = plt.colorbar(
    mappable=im0,
    ax=ax0,
    extend='max',
    label=r"Conductance [$e^2/h$]",
    aspect=60
)

im1 = cbo.return_spacing_histogram(peaks, coord='field', nbins=35, rng=(0, 1.2), dim='pg').plot(
    ax=ax1,
    cmap='gist_heat_r',
    rasterized=True,
    add_colorbar=False
)
ax1.set_xlabel(r"Peak spacing [mV]")
ax1.set_ylabel("")
cb1 = plt.colorbar(
    mappable = im1,
    ax = ax1,
    label = r"Counts",
    aspect = 60
)

plt.savefig("figures/supp_figure_coulomb_oscillations_region_III.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

# Fig. S11: Coulomb oscillations in region I

```python
filename = 'raw_data/fig_S11.csv'
data = pd.read_csv(filename, sep=",").set_index(['field', 'rtg']).to_xarray()
```

```python
dim_attributes = {
    'rtg' : {'long_name' :  r'Right tunnel gate voltage', 'units' : r'mV'},
    'field' : {'long_name' : r'Magnetic Field $B$', 'units' : 'T'}

}
gdata = cbo.get_conductance_data(data, gain=1e6, Rseries=8798, dV=10e-6 / np.sqrt(2), dim_attrs=dim_attributes, method='fit')
```

```python
# Peak finding - rotated data
peaks = cbo.get_peaks(
    array = gdata['XR'],
    coord = 'rtg',
    filter_params={'window_length' : 11, 'polyorder' : 3},
    peak_params={'distance' : 3},
    height_threshold=1,
    zscore_threshold=-2
)
```

```python
# Set font sizes
plt.rc('font', size=14)          # controls default text sizes
plt.rc('axes', titlesize=14)     # fontsize of the axes title
plt.rc('axes', labelsize=14)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=12)    # fontsize of the tick labels
plt.rc('ytick', labelsize=12)    # fontsize of the tick labels
plt.rc('legend', fontsize=12)    # legend fontsize
plt.rc('figure', titlesize=14)  # fontsize of the figure title

# Prepare grid for the figure
figsize =  (3 * (3 + 3/8), 1 * (4 + 3/8))
fig = plt.figure(constrained_layout=True, figsize=figsize)
gs = GridSpec(1, 2, figure=fig, width_ratios=[2,1])
ax0 = fig.add_subplot(gs[0,0])
ax1 = fig.add_subplot(gs[0,1], sharey=ax0)

im0 = gdata['XR'].plot(
    ax=ax0,
    cmap='Greys',
    rasterized=True,
    robust=True,
    vmax=1.5,
    add_colorbar=False
)

cb0 = plt.colorbar(
    mappable=im0,
    ax=ax0,
    extend='max',
    label=r"Conductance [$e^2/h$]",
    aspect=60
)

im1 = cbo.return_spacing_histogram(peaks, coord='field', nbins=35, rng=(0, 6), dim='rtg').plot(
    ax=ax1,
    cmap='gist_heat_r',
    rasterized=True,
    add_colorbar=False
)
ax1.set_xlabel(r"Peak spacing [mV]")
ax1.set_ylabel("")
cb1 = plt.colorbar(
    mappable = im1,
    ax = ax1,
    label = r"Counts",
    aspect = 60
)

plt.savefig("figures/supp_figure_coulomb_oscillations_region_I.pdf", bbox_inches='tight', transparent=True)
plt.show()
```
