---
jupyter:
  jupytext:
    formats: ipynb,md
    text_representation:
      extension: .md
      format_name: markdown
      format_version: '1.2'
      jupytext_version: 1.7.1
  kernelspec:
    display_name: Python [conda env:parity_diagram]
    language: python
    name: conda-env-parity_diagram-py
---

```python
import matplotlib.pyplot as plt
from matplotlib.gridspec import GridSpec, GridSpecFromSubplotSpec
import matplotlib.image as mpimg
from matplotlib.patches import Polygon
from ipywidgets import interactive, SelectionSlider
from scipy.signal import find_peaks
import numpy as np
import pickle
import glob
import pandas as pd
import xarray as xr
import cbo
```

<!-- #region toc-hr-collapsed=true toc-nb-collapsed=true toc-hr-collapsed=true toc-nb-collapsed=true -->
# Fig. 2
<!-- #endregion -->

## Load and prepare the data

```python
files = glob.glob("raw_data/fig_2/*.csv")
files.sort()
dfs = [pd.read_csv(file) for file in files]
data = pd.concat(dfs).set_index(['field', 'pg_fine', 'pg_coarse']).to_xarray()
data = data.set_coords('rtg')
data['rtg'] = data['rtg'].mean(['field', 'pg_fine']) # Removes trivial dimensions
data
```

```python
fields = data.field.values
plungers = data.pg_coarse.values
dim_attributes = {
    'pg_coarse' : {'long_name' :  'Plunger gate voltage', 'units' : r'mV'},
    'pg_fine' : {'long_name' :  'Plunger gate voltage', 'units' : r'mV'},
    'field' : {'long_name' : r'Magnetic Field', 'units' : 'T'}

}
darrays = []
for (i, f) in enumerate(fields):
    darray = []
    for (j, p) in enumerate(plungers):
        d = data.sel(dict(field=f, pg_coarse=p), method='nearest')
        dnew = cbo.get_conductance_data(
            d, gain=1e6, Rseries=8798, dV=5e-6 / np.sqrt(2),
            dim_attrs=dim_attributes,
            method='minimum',
            plot=False
        )
        darray.append(dnew)
    darrays.append(darray)
gdata = xr.combine_nested(darrays, concat_dim=['field', 'pg_coarse'])
```

```python
cbo.plot_and_compare_conductance_data(gdata)
```

```python
# Cut first two values of pg coarse, which are unreliable due to noise after the large MDAC jump
gdata = gdata.isel({'pg_coarse' : slice(2,91)})
```

## Peak spacing analysis

```python
# Peak finding - rotated data
peaks_rotated = cbo.get_peaks(
    array = gdata['XR'],
    coord = 'pg_fine',
    filter_params={'window_length' : 11, 'polyorder' : 3},
    peak_params={'distance' : 2},
    height_threshold=1,
    zscore_threshold=-2
)
results_rotated = cbo.get_aggregate_peak_info(peaks_rotated)
```

```python
for var in results_rotated.data_vars:
    results_rotated[var].plot()
    plt.show()
```

```python
total_hist = cbo.return_spacing_histogram(peaks_rotated, coord='field', nbins=31, rng=(0, 1.5))
total_hist.plot(cmap='magma_r')
plt.title("All data")
plt.show()

first_boundary=-1000
second_boundary=500
regions = [peaks_rotated['pg_coarse'] <= first_boundary,
           (peaks_rotated['pg_coarse'] > first_boundary) & (peaks_rotated['pg_coarse'] <= second_boundary),
           peaks_rotated['pg_coarse'] > second_boundary]

# Peak spacing histogram vs field
if first_boundary is not None:
    regions_hists = []
    for (n, region) in enumerate(regions):
        hist = cbo.return_spacing_histogram(peaks_rotated[region], coord='field', nbins=31, rng=(0, 1.5))
        regions_hists.append(hist)
        hist.plot(cmap='magma_r')
        plt.title(f"Region {n+1}")
        plt.show()
else:
    regions_hists = None
```

## Preparing the figure

```python
figsize =  (2 * (3 + 3/8), 2 * (4 + 3/8))
fig = plt.figure(constrained_layout=True, figsize=figsize)

# Set font sizes
plt.rc('font', size=14)          # controls default text sizes
plt.rc('axes', titlesize=14)     # fontsize of the axes title
plt.rc('axes', labelsize=14)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=12)    # fontsize of the tick labels
plt.rc('ytick', labelsize=12)    # fontsize of the tick labels
plt.rc('legend', fontsize=12)    # legend fontsize
plt.rc('figure', titlesize=14)  # fontsize of the figure title

# Prepare plot grid
gs = GridSpec(4, 3, figure=fig)
ax1 = fig.add_subplot(gs[:-1,:])
ax2 = fig.add_subplot(gs[-1,0])
ax3 = fig.add_subplot(gs[-1,1], sharey=ax2)
ax4 = fig.add_subplot(gs[-1,2], sharey=ax2)

# ax1: Field versus plunger
im1 = xr.plot.plot(
    darray = results_rotated["median_spacing"],
    ax=ax1,
    cmap='Blues',
    vmin=0.25, vmax=1.25,
    robust=True,
    rasterized=True,
    add_colorbar=False)

cb1 = plt.colorbar(
    mappable=im1,
    ax=ax1,
    location='top',
    extend='max',
    shrink=0.75,
    label=r"Median Peak Spacing [mV]",
    aspect=60
)

ax1.set_xlabel(r"Plunger Gate voltage [mV]")
ax1.set_ylabel(r"Magnetic Field $B$ [T]")

# Arrows defining regions
ax1.annotate(
    "",
    xy=(-1720, 0.85),
    xytext=(-1000, 0.85),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3")
)
ax1.annotate(
    "", xy=(-1000, 0.85), xytext=(500, 0.85),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3")
)
ax1.annotate(
    "", xy=(500, 0.85), xytext=(1800, 0.85),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3")
)

# Region labels
ax1.text(-1360, 0.8, "I", size=16)
ax1.text(-250, 0.8, "II", size=16)
ax1.text(1150, 0.8, "III", size=16)


# ax2, ax3, ax4: peak spacing histograms
for (region, ax, reg_num) in zip(regions_hists, (ax2, ax3, ax4), ("I", "II", "III")):
    im = region.plot(
        ax=ax,
        cmap='gist_heat_r',
        add_colorbar=False,
        rasterized=True
    )
    ax.set_yticks([0, 0.2, 0.4, 0.6, 0.8])
    ax.set_xlabel(r"Peak spacing [mV]")
    ax.set_ylabel("")
    cb2 = plt.colorbar(
        mappable = im,
        ax = ax,
        label = r"Counts, region " + reg_num,
        location = 'top',
        aspect = 30
    )

# Field label for the three histograms
ax2.set_ylabel(r"$B$ [T]")

# Hide tick labels in ax3 and ax4
plt.setp(ax3.get_yticklabels(), visible=False)
plt.setp(ax4.get_yticklabels(), visible=False)

# Panel labels
fig.text(0.01, 0.95, "(a)", fontsize=14)
fig.text(0.01, 0.30, "(b)", fontsize=14)

# Show figure
plt.savefig("figures/figure2.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

## Preparing supplementary figure S6

```python
figsize =  (2 * (3 + 3/8), 2 * (3 + 3/8))
fig, ax = plt.subplots(constrained_layout=True, figsize=figsize)

im1 = xr.plot.plot(
    darray = results_rotated["std_spacing"],
    ax=ax,
    cmap='Blues',
    robust=True,
    rasterized=True,
    add_colorbar=False)

cb1 = plt.colorbar(
    mappable=im1,
    ax=ax,
    location='top',
    extend='max',
    shrink=0.75,
    label=r"Standard deviation peak spacing [mV]",
    aspect=60
)

ax.set_xlabel(r"Plunger Gate voltage [mV]")
ax.set_ylabel(r"Magnetic Field $B$ [T]")

# Arrows defining regions
ax.annotate(
    "",
    xy=(-1720, 0.85),
    xytext=(-1000, 0.85),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3")
)
ax.annotate(
    "", xy=(-1000, 0.85), xytext=(500, 0.85),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3")
)
ax.annotate(
    "", xy=(500, 0.85), xytext=(1800, 0.85),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3")
)

# Region labels
ax.text(-1360, 0.8, "I", size=16)
ax.text(-250, 0.8, "II", size=16)
ax.text(1150, 0.8, "III", size=16)

# Show figure
plt.savefig("figures/supp_figure_std_spacings.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

# Fig. 4


## Load and prepare data

```python
filename = 'raw_data/fig_4.csv'
data = pd.read_csv(filename, sep=",").set_index(['field', 'pg']).to_xarray()
data
```

```python
dim_attributes = {
    'pg' : {'long_name' :  r'Plunger gate voltage', 'units' : r'mV'},
    'field' : {'long_name' : r'Magnetic Field $B$', 'units' : 'T'}

}
gdata = cbo.get_conductance_data(data, gain=1e6, Rseries=8798, dV=10e-6 / np.sqrt(2), dim_attrs=dim_attributes, method='fit')
```

```python
# Crop the data to focus only on complete Coulomb valleys to extract peak spacings
gdata = gdata.isel({'pg' : np.where(gdata.pg < -665)[0]})
gdata = gdata.isel({'pg' : np.where(gdata.pg > -672.5)[0]})
```

<!-- #region toc-hr-collapsed=true toc-nb-collapsed=true -->
## Peak finding
<!-- #endregion -->

```python
# Discard points below 0.1T due to jump, we are not interested in that range anyway
gdata_peaks = gdata.isel({'field' : np.where(gdata.field.values > 0.1)[0]})
```

```python
peaks = cbo.get_peaks(
    gdata_peaks['XR'],
    coord='pg',
    filter_params={'window_length' : 15, 'polyorder' : 5},
    peak_params={'distance' : 2},
    height_threshold=0.1,
    zscore_threshold=-3
).drop('height_zscore', axis=1)

# Also filter by absolute height of the peak
peaks = peaks.loc[peaks['height'] >= 0.01]

# Sort peaks by field, then plunger
peaks = peaks.sort_values(by = ['field', 'pg'])

pgs = peaks.pg.values
bs = peaks.field.values
gdata['XR'].plot(vmin=0, cmap='Greys')
plt.scatter(pgs, bs, c='red', marker='.')
```

### Outliers selection

```python
cbo.plot_conductance_data_with_peaks(gdata['XR'], peaks)
```

```python
# By manual inspection, we select outliers or boundary points that we don't need

# outliers
to_remove = [
    (0.23, -1),
    (0.23, -2),
    (0.35, 0),
    (0.39, 5),
    (0.4, 0),
    (0.4, 4),
    (0.68, -1),
    (0.47, -1),
    (0.48, -1),
    (0.49, -1),
    (0.50, -1),
    (0.51, -1),
    (0.52, -1),
    (0.69, -1)
]

# boundary points top_left corner
for b in np.arange(0.66, 0.8+1e-9, 0.01):
    to_remove.append((np.round(b, 2), 0))
# boundary points right side, first group
for b in np.arange(0.16, 0.21+1e-9, 0.01):
    to_remove.append((np.round(b, 2), -1))
for b in np.arange(0.24, 0.34+1e-9, 0.01):
    to_remove.append((np.round(b, 2), -1))
# boundary points right side, second group
for b in np.arange(0.47, 0.52+1e-9, 0.01):
    to_remove.append((np.round(b, 2), -1))

# We also notice a few side peaks that we can add by hand even though they were not picked up by the peak finder
sel = peaks.loc[peaks['field']==0.47]
dvpg = np.diff(gdata.pg.values)[0]
to_add = [
    (0.36, -670.125),
    (0.47, sel.pg.values[0] + 9 * dvpg),
    (0.47, sel.pg.values[1] + 8 * dvpg),
    (0.47, sel.pg.values[2] + 9 * dvpg),
    (0.47, sel.pg.values[5] + 10 * dvpg),
]

# Find plunger values for the points to be removed
_tmp = []
for (b, idx) in to_remove:
    p = peaks.loc[peaks['field']==b]
    new_point = (b, p.pg.values[idx])
    _tmp.append(new_point)
to_remove = _tmp

# Remove rows corresponding to the outliers
rows_to_remove = []
for (item, row) in peaks.iterrows():
    point = (row['field'], row['pg'])
    if point in to_remove:
        rows_to_remove.append(item)
peaks = peaks.drop(rows_to_remove)

# Add peaks
for (b, pg) in to_add:
    height = gdata_peaks['XR'].sel(field=b, pg=pg, method='nearest').values
    peaks = peaks.append({'field' : b, 'pg' : pg, 'height' : height}, ignore_index=True)
```

```python
pgs = peaks.pg.values
bs = peaks.field.values
gdata['XR'].plot(vmin=0, cmap='Greys')
plt.scatter(pgs, bs, c='red', marker='.')
plt.show()
```

## Peak spacings

We have 4 complete even valleys and for odd valleys, pluse one incomplete odd valley.

* Four even valleys denoted E1, E2, E3, E4, E5 begin at $B=0$, disapperar when $B=0.37$ and appear again at $B=0.48$.
* Four odd valleys denotes by O1, O2, O3, O4, O5 appear at $B=0.35$.
* A fifth even valley begins when $B=0.48$

```python
even_spacings = np.zeros((5, gdata_peaks.field.size))
odd_spacings = np.zeros((5, gdata_peaks.field.size))

for i, (b, group) in enumerate(peaks.groupby('field')):
    group = group.sort_values('pg')
    spacings = np.diff(group.pg.values)
    # 2e even valleys at low fields
    if b < 0.35:
        even_spacings[:-1, i] = spacings
    # parity inversion
    elif 0.35 <= b < 0.37:
        even_spacings[:-1, i] = spacings[::2][:-1]
        odd_spacings[:,i] = spacings[1::2]
    # 2e odd valleys at intermediate fields
    elif 0.37 <= b < 0.47:
        odd_spacings[:,i] = spacings
    # even-odd regime
    else:
        even_spacings[:,i] = spacings[::2]
        odd_spacings[:,i] = spacings[1::2]

# Load all the spacings data into a single xarray
peak_spacings = xr.Dataset(
    data_vars = {
        'even' : (['field', 'label'], even_spacings.T),
        'odd'  : (['field', 'label'], odd_spacings.T),
    },
    coords = {
        'field' : ('field', gdata_peaks.field),
        'label' : ('label', [1, 2, 3, 4, 5]),
    }
)
peak_spacings['diff'] = peak_spacings['even'] - peak_spacings['odd']
```

## Peak heights

```python
lambda_left = np.zeros((5, 34)) # 33 field values between 0.47 T and 0.8 T
lambda_right = np.zeros((5, 34)) # 33 field values between 0.47 T and 0.8 T
sel = peaks.loc[peaks['field'] > 0.46]
for i, (b, group) in enumerate(sel.groupby('field')):
    group = group.sort_values('pg')
    hs = group.height.values # boundary: discard first peak
    lambda_left[:,i] = np.array([hs[2*m+1] / (hs[2*m] + hs[2*m + 1]) for m in range(5)])
    lambda_right[:,i] = np.array([hs[2*m+1] / (hs[2*m+2] + hs[2*m + 1]) for m in range(5)])

# Load all the peak height data into a single xarray
peak_lambdas = xr.Dataset(
    data_vars = {
        'left' : (['field', 'label'], lambda_left.T),
        'right' : (['field', 'label'], lambda_right.T)
    },
    coords = {
        'field' : ('field', np.arange(0.47, 0.8+1e-9, 0.01)),
        'label' : ('label', [1, 2, 3, 4, 5]),
    }
)
peak_lambdas['avg'] = 0.5 * (peak_lambdas['left'] + peak_lambdas['right'])

# Interpolate the peak lambda to find the magnetic field values at which it is equal to 0.5
bs = peak_lambdas.field.values
bvalues = []
binterp = np.linspace(0.47, 0.6, 10001)
for valley_idx in range(1,6):
    lambdas = peak_lambdas.sel(label=valley_idx).avg.values
    bidx = np.argmin(np.abs(np.interp(binterp, bs, lambdas) - 0.5))
    bvalues.append(binterp[bidx])
```

## Prepare the figure

```python
figsize =  (2 * (3 + 3/8), 2 * (4 + 3/8))
fig = plt.figure(constrained_layout=True, figsize=figsize)

# Set font sizes
plt.rc('font', size=14)          # controls default text sizes
plt.rc('axes', titlesize=14)     # fontsize of the axes title
plt.rc('axes', labelsize=14)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=12)    # fontsize of the tick labels
plt.rc('ytick', labelsize=12)    # fontsize of the tick labels
plt.rc('legend', fontsize=12)    # legend fontsize
plt.rc('figure', titlesize=14)  # fontsize of the figure title

# Prepare plot grid
gs = GridSpec(2, 2, figure=fig, height_ratios=[2,1], width_ratios=[1.8,1])
ax1 = fig.add_subplot(gs[0,0])
gs2 = GridSpecFromSubplotSpec(1, 2, subplot_spec=gs[1,0])
ax2 = fig.add_subplot(gs2[0,0])
ax3 = fig.add_subplot(gs2[0,1], sharey=ax2)
gs3 = GridSpecFromSubplotSpec(5, 1, subplot_spec=gs[:,1])
ax4_0 = fig.add_subplot(gs3[0,0])
ax4_1 = fig.add_subplot(gs3[1,0], sharex=ax4_0)
ax4_2 = fig.add_subplot(gs3[2,0], sharex=ax4_0)
ax4_3 = fig.add_subplot(gs3[3,0], sharex=ax4_0)
ax4_4 = fig.add_subplot(gs3[4,0], sharex=ax4_0)

# ax1: Conductance, field versus plunger
im1 = xr.plot.plot(
    darray = gdata['XR'],
    ax=ax1,
    cmap='binary',
    vmin=0,
    robust=True,
    rasterized=True,
    add_colorbar=False)

cb1 = plt.colorbar(
    mappable=im1,
    ax=ax1,
    location='top',
    extend='both',
    shrink=0.75,
    label=r"Conductance $G$ [$e^2/h$]",
    aspect=60
)

# Arrows defining extent of bottom panels
ax1.annotate(
    "",
    xy=(-672.3, 0.45),
    xytext=(-672.3, 0.7),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3")
)

# ax2 : peak spacing variation
sel = peak_spacings['diff'].isel({'field' : np.where(peak_spacings.field.values < 0.71)[0]})
sel = sel.isel({'field' : np.where(sel.field.values > 0.46)[0]})
im2 = xr.plot.plot(
    darray = sel,
    ax=ax2,
    cmap='coolwarm',
    robust=True,
    xticks=[1,2,3,4,5],
    rasterized=True,
    add_colorbar=False
)
ax2.scatter(range(1, 6), bvalues, marker='x', c='k', lw=0.8)
ax2.set_ylabel(r"$B$ [T]")
ax2.set_xlabel(r"Valley index $i$")

cb2 = plt.colorbar(
    mappable=im2,
    ax=ax2,
    location='top',
    label=r"$S_{e,i}-S_{o,i}$ [mV]",
    aspect=40
)

# ax3 : relative peak height variation
im3 = xr.plot.plot(
    darray = peak_lambdas['avg'].isel({'field' : np.where(peak_lambdas.field.values < 0.71)[0]}),
    ax=ax3,
    cmap='coolwarm',
    robust=True,
    vmin=0.38,
    vmax=0.62,
    xticks=[1,2,3,4,5],
    rasterized=True,
    add_colorbar=False
)
ax3.scatter(range(1, 6), bvalues, marker='x', c='k', lw=0.8)
ax3.set_ylabel("")
ax3.set_xlabel(r"Valley index $i$")
plt.setp(ax3.get_yticklabels(), visible=False)
cb3 = plt.colorbar(
    mappable=im3,
    ax=ax3,
    location='top',
    label=r"Asymmetry $\Lambda_{i}$",
    aspect=40
)

# Add vertical grid lines to enhance that x axis is discrete
for x in [1.5, 2.5, 3.5, 4.5]:
    ax2.axvline(x, c='k', lw=0.5)
    ax3.axvline(x, c='k', lw=0.5)


# ax4: peak spacings and height oscillations for each valley
for (i, ax) in enumerate([ax4_0, ax4_1, ax4_2, ax4_3, ax4_4]):
    ps = peak_spacings.isel({'label' : i, 'field' : slice(36, 60)})
    ls = peak_lambdas.isel({'label' : i, 'field' : slice(0, 24)})
    axt = ax.twinx()
    xr.plot.scatter(
        ds = ps,
        y = 'even',
        x = 'field',
        marker='x',
        c='red',
        ax = ax,
        label=r"$S_e$"
    )
    xr.plot.scatter(
        ds = ps,
        y = 'odd',
        x = 'field',
        marker='x',
        c='blue',
        ax = ax,
        label=r"$S_o$"
    )
    xr.plot.scatter(
        ds = ls,
        y = 'avg',
        x = 'field',
        ax = axt,
        marker='.',
        c = 'black'
    )
    ax.axvline(bvalues[i], c='black', lw=0.8, ls='--', dashes=(5, 5))
    # Text label for the valley
    ax.text(0.62, 0.88, rf"$i={i+1}$", size=12)
    # set up a legend
    if i==0:
        ax.scatter([], [], marker='.', c='black', label=r'$\Lambda$')
        ax.legend(
            bbox_to_anchor = (0, 1.02, 1, 1.02),
            loc='lower left',
            mode='expand',
            handletextpad=0.05,
            borderaxespad=0,
            ncol=3,
            frameon=False
        )
        pass

    ax.set_xlabel("")
    ax.set_ylabel("")
    ax.set_ylim([0.19, 1.01])
    ax.set_yticks([0.3, 0.6, 0.9])
    axt.set_ylim([0.35, 0.65])
    axt.set_yticks([0.4, 0.5, 0.6])
    axt.set_ylabel("")
    if i != 4:
        plt.setp(ax.get_xticklabels(), visible=False)
    axt.grid()
    ax.grid(axis='x')
ax4_4.set_xlabel("$B$ [T]")

# Valley indices
# Region labels
ax1.text(-671.7, 0.65, r"$e_1$", size=12)
ax1.text(-670.4, 0.65, r"$e_2$", size=12)
ax1.text(-669.2, 0.65, r"$e_3$", size=12)
ax1.text(-667.88, 0.65, r"$e_4$", size=12)
ax1.text(-666.6, 0.65, r"$e_5$", size=12)

ax1.text(-671.1, 0.6, r"$o_1$", size=12)
ax1.text(-669.85, 0.6, r"$o_2$", size=12)
ax1.text(-668.6, 0.6, r"$o_3$", size=12)
ax1.text(-667.4, 0.6, r"$o_4$", size=12)
ax1.text(-666.1, 0.6, r"$o_5$", size=12)

# Panel labels
fig.text(0.01, 0.97, "(a)", fontsize=14)
fig.text(0.63, 0.97, "(b)", fontsize=14)
fig.text(0.01, 0.32, "(c)", fontsize=14)

# Show figure
plt.savefig("figures/figure4.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

# Fig. 1

```python
# Set font sizes
plt.rc('font', size=14)          # controls default text sizes
plt.rc('axes', titlesize=14)     # fontsize of the axes title
plt.rc('axes', labelsize=14)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=12)    # fontsize of the tick labels
plt.rc('ytick', labelsize=12)    # fontsize of the tick labels
plt.rc('legend', fontsize=12)    # legend fontsize
plt.rc('figure', titlesize=14)  # fontsize of the figure title
data_folder =  'sim_data/'

figsize = (2 * (3 + 3/8), 7)
fig = plt.figure(constrained_layout = True, figsize=figsize)

gs = GridSpec(3, 1, figure=fig, height_ratios = [1.5, 1, 1])

# ax1 : image of the device
img=mpimg.imread('figures/device.png')[10:380,20:961,:]
ax1_small = fig.add_axes([0.1,0.6,0.8925,0.35])
ax1_small.imshow(img)
ax1_small.axes.get_xaxis().set_visible(False)
ax1_small.axes.get_yaxis().set_visible(False)

# ax2, ax3 : image of the simulated device
gs2 = GridSpecFromSubplotSpec(1, 2, subplot_spec=gs[1, 0], width_ratios = [2.6,1])
ax2 = plt.subplot(gs2[0,0])

color_dict = dict(
    substrate = 'gray',
    wire = 'burlywood',
    superconductor = 'darkseagreen',
    oxide = 'lightgray',
    plunger = 'darkslateblue',
    left_contact = 'khaki',
    right_contact = 'khaki',
    left_cutter = 'darkslateblue',
    right_cutter = 'darkslateblue',
)

with open('sim_data/longitudinal_cut.pkl', 'rb') as f:
    patch_dict = pickle.load(f)

for name, vert in patch_dict.items():
    ax2.add_patch(Polygon(vert, fc=color_dict[name], ec='k'))
ax2.set_xlim([-0.8, 0.8])
ax2.set_ylim([-0.05, 0.25])
ax2.set_xlabel(r'X [$\mu$m]')
ax2.set_ylabel(r'Z [$\mu$m]')
ax2.set_yticks([0, 0.1, 0.2])
ax2.set_xticks([-0.5, 0, 0.5])

with open(data_folder+'middle_cut.pkl', 'rb') as f:
    patch_dict = pickle.load(f)

ax3 = plt.subplot(gs2[0,1])
for name, vert in patch_dict.items():
    ax3.add_patch(Polygon(vert, fc=color_dict[name], ec='k'))
ax3.set_ylim([-0.05, 0.25])
ax3.set_xlim([-0.18, 0.18])
ax3.set_xticklabels(['', '-0.1', '0', '0.1'])
ax3.set_xlabel(r'Y [$\mu$m]')

# ax4, ax5: linecuts from Fig. 4 dataset showing 2e and 1e oscillations


gs3 = GridSpecFromSubplotSpec(1, 2, subplot_spec=gs[2,0])
ax4 = plt.subplot(gs3[0, 0])
gtrace = gdata['XR'].isel(dict(field = 0, pg = slice(0, -50)))
gtrace.plot(ax=ax4, c='k')
ax4.set_title(rf"$B=0$ T")
ax4.set_yticks([0, 0.05, 0.1])
ax4.set_ylim([0., 0.15])
ax4.margins(x=0)
ax4.set_ylabel(r'$G$ [$e^2/h$]')

# Arrow defining the 2e spacing
peaks_idxs = find_peaks(gtrace.values, height=0.05)[0]
peaks_pos = gtrace.pg.values[peaks_idxs]
ax4.annotate(
    "",
    xy=(peaks_pos[0]-0.1, 0.135),
    xytext=(peaks_pos[1]+0.1, 0.135),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3")
)
ax4.text(0.5 * (peaks_pos[0]+peaks_pos[1])-0.25, 0.110, r"$2e$", size=14)

ax5 = plt.subplot(gs3[0, 1])
gtrace = gdata['XR'].isel(dict(field = 60, pg = slice(0, -50)))
gtrace.plot(ax=ax5, c='k')
ax5.set_title('')
ax5.set_ylim([0.1, 0.25])
ax5.margins(x=0)
ax5.set_yticks([0.1, 0.15, 0.2])
ax5.set_title(rf"$B=0.6$ T")
ax5.set_ylabel('')

# Arrow defining the 2e spacing
peaks_idxs = find_peaks(gtrace.values, height=0.15)[0]
peaks_pos = gtrace.pg.values[peaks_idxs]
ax5.annotate(
    "",
    xy=(peaks_pos[2]-0.1, 0.212),
    xytext=(peaks_pos[3]+0.1, 0.212),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3")
)
ax5.text(peaks_pos[2], 0.225, r"$1e$", size=14)

# Panel labels
fig.text(0.02, 0.92, "(a)", fontsize=14)
fig.text(0.02, 0.57, "(b)", fontsize=14)
fig.text(0.02, 0.27, "(c)", fontsize=14)
fig.text(0.53, 0.27, "(d)", fontsize=14)

plt.savefig('figures/figure1.pdf', dpi=200, transparent=True, bbox_inches='tight')
plt.show()
```
