---
jupyter:
  jupytext:
    formats: ipynb,md
    text_representation:
      extension: .md
      format_name: markdown
      format_version: '1.2'
      jupytext_version: 1.6.0
  kernelspec:
    display_name: Python [conda env:py38n] *
    language: python
    name: conda-env-py38n-py
---

```python
# %matplotlib notebook
import matplotlib.pyplot as plt
import matplotlib.gridspec as gridspec
import numpy as np
import matplotlib.patches as patches
from matplotlib.patches import Polygon
from matplotlib.path import Path
from matplotlib.gridspec import GridSpec, GridSpecFromSubplotSpec
from matplotlib.colors import Normalize
import pickle
import pandas as pd
```

```python
# Set font sizes
plt.rc('font', size=14)          # controls default text sizes
plt.rc('axes', titlesize=14)     # fontsize of the axes title
plt.rc('axes', labelsize=14)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=12)    # fontsize of the tick labels
plt.rc('ytick', labelsize=12)    # fontsize of the tick labels
plt.rc('legend', fontsize=12)    # legend fontsize
plt.rc('figure', titlesize=14)  # fontsize of the figure title
data_folder =  'sim_data/'
```

```python
def get_extent(x, y):
    return [x[0]-(x[1]-x[0])/2, x[-1]+(x[-1]-x[-2])/2, y[0]-(y[1]-y[0])/2, y[-1]+(y[-1]-y[-2])/2]
```

# Fig. 3, main text

```python
figsize =  (4 * (3 + 3/8), (4 + 3/8))
fig = plt.figure(constrained_layout=True, figsize=figsize)

gs = GridSpec(1, 3, figure=fig)
ax1 = fig.add_subplot(gs[0,0])
ax2 = fig.add_subplot(gs[0,1])
gs2 = GridSpecFromSubplotSpec(2, 3, subplot_spec=gs[0, 2])

# Special plungers
plungers = [0.6025, 0.715, 1.0875]
d = np.load('sim_data/topo_pd.npz', allow_pickle=True)
paths = d['paths']

# Electrostatics
dfe = pd.read_hdf('sim_data/v_cutter04.hdf')
dfe1 = pd.read_hdf('sim_data/v_denseB.hdf')

# ax1: energy spectrum of the island
d = np.load('sim_data/gap_ndis.npz')

im = ax1.imshow(1e6*d['data'].T[::-1, :], aspect='auto',
           extent=get_extent(d['V'], d['Bv']),
           cmap='Blues', vmin=0., vmax=40., interpolation='none')
ax1.set_ylabel(r'$B$ [T]')
ax1.set_xlabel(r'$V_\mathrm{pg}$ [V]')
ax1.set_xlim([0.4, 1.2])
for j, p in enumerate(plungers):
    ax1.plot([p, p], [1.01, 0.94], color=f'C{j}', lw=3)
ax1.text(1.15, 0.8, r'$L=1\,\mu$m', horizontalalignment='right', bbox=dict(facecolor='gray', alpha=0.2, edgecolor='none'))

for path in paths:
    patch = patches.PathPatch(Path(path), facecolor='none', lw=1, edgecolor='gray', linestyle='--')
    ax1.add_patch(patch)

# Arrows defining regions
ax1.annotate(
    "",
    xy=(0.4, 0.02),
    xytext=(0.6, 0.02),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3", color='w')
)
ax1.annotate(
    "", xy=(0.6, 0.02), xytext=(0.9, 0.02),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3", color='w')
)
ax1.annotate(
    "", xy=(0.9, 0.02), xytext=(1.2, 0.02),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3", color='w')
)

# Region labels
ax1.text(0.5, 0.025, "I", size=16, horizontalalignment='center', verticalalignment='bottom', color='w')
ax1.text(0.75, 0.025, "II", size=16, horizontalalignment='center', verticalalignment='bottom', color='w')
ax1.text(1.05, 0.025, "III", size=16, horizontalalignment='center', verticalalignment='bottom', color='w')


cb = plt.colorbar(im, ax=ax1, label=r'$E_\mathrm{min}$ [$\mu$eV]', location='top', extend='max')

# ax2: gap from the bandstructure
d = np.load('sim_data/topo_pd.npz')
im = ax2.imshow(1e6*d['data'][::-1, ::-1].T, aspect='auto',
           extent=get_extent(d['V'], d['Bv']),
           cmap='RdBu', vmin=-45., vmax=45., interpolation='none')
ax2.set_yticklabels([])
ax2.set_xlabel(r'$V_\mathrm{pg}$ [V]')
ax2.set_xlim([0.4, 1.2])
for j, p in enumerate(plungers):
    ax2.plot([p, p], [1.0, 0.94], color=f'C{j}', lw=3)
ax2.text(1.15, 0.8, r'$L=\infty$', horizontalalignment='right', bbox=dict(facecolor='gray', alpha=0.2, edgecolor='none'))

bbox = ax2.get_position().get_points()
cb = plt.colorbar(im, ax=ax2, location='top', label = r'$\mathcal{Q}\, E_\mathrm{gap}$ [$\mu$eV]', extend='max')

for path in paths:
    patch = patches.PathPatch(Path(path), facecolor='none', lw=1, edgecolor='gray', linestyle='--')
    ax2.add_patch(patch)

# Arrows defining regions
ax2.annotate(
    "",
    xy=(0.4, 0.02),
    xytext=(0.6, 0.02),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3", color='w')
)
ax2.annotate(
    "", xy=(0.6, 0.02), xytext=(0.9, 0.02),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3", color='w')
)
ax2.annotate(
    "", xy=(0.9, 0.02), xytext=(1.2, 0.02),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3")
)

# Region labels
ax2.text(0.5, 0.025, "I", size=16, horizontalalignment='center', verticalalignment='bottom', color='w')
ax2.text(0.75, 0.025, "II", size=16, horizontalalignment='center', verticalalignment='bottom', color='w')
ax2.text(1.05, 0.025, "III", size=16, horizontalalignment='center', verticalalignment='bottom')


# Cross sections:
with open('sim_data/middle_cut.pkl', 'rb') as f:
    patch_dict = pickle.load(f)

color_dict = dict(
    substrate = 'gray',
    wire = 'burlywood',
    superconductor = 'darkseagreen',
    oxide = 'lightgray',
    plunger = 'darkslateblue',
    left_contact = 'khaki',
    right_contact = 'khaki',
    left_cutter = 'darkslateblue',
    right_cutter = 'darkslateblue',
)

axl = []
for i, V_plunger in enumerate(plungers):
    d = np.load(data_folder+f'potyz_V{V_plunger}.npz')
    with plt.rc_context({'axes.edgecolor' : f'C{i}', 'axes.linewidth' : 2.}):
        ax = plt.subplot(gs2[0, i])
        axl.append(ax)
        pol = Polygon(patch_dict['wire'])
        Y, Z = np.meshgrid(d['y'], d['z'])
        pot = d['pot']
        pot[::-1, :][np.logical_not(pol.contains_points(np.array([Y.flatten()/1e3, Z.flatten()/1e3]).T).reshape(Y.shape))] = np.nan
        im = ax.imshow(1e3*pot[:, :], extent=get_extent(d['y']/1e3, d['z']/1e3),
                   cmap='PuOr', vmin=-100, vmax=100., zorder=3, interpolation='none')
        for name, vert in patch_dict.items():
            ax.add_patch(Polygon(vert, fc=color_dict[name], ec='k'))
        ax.add_patch(Polygon(patch_dict['wire'], fc='none', ec='k', zorder=4))
        ax.set_xlim([-0.08, 0.08])
        ax.set_ylim([-0.01, 0.15])
        if i > 0:
            ax.set_yticklabels([])
        else:
            ax.set_ylabel(r'Z [$\mu$m]')
        ax.set_xticks([-0.05, 0., 0.05])
        ax.set_xticklabels(['-0.05', '0', '0.05'])
        ax.set_xlabel(r'Y [$\mu$m]')

cb = plt.colorbar(im, ax=axl, location='top', label=r'$\Phi$ [meV]')



# Oscillations
from matplotlib.collections import LineCollection

for n, V_plunger in enumerate(plungers):
    gr = dfe1.groupby(['V_plunger']).get_group(V_plunger)
    Bv = np.array(gr.B)
    E1 = np.array(gr.min_energy)
    E2 = np.array(gr.second_energy)
    Bv = np.concatenate((Bv, [1.]))
    E1 = np.concatenate((E1, [0.]))
    E2 = np.concatenate((E2, [0.]))
    sign = 1
    for i in range(1, len(E1)-2):
        if np.abs(-E1[i-1]+2*E1[i]-E1[i+1]) > np.abs(-E1[i-1]+2*E1[i]+E1[i+1]):
            E1[i+1:] *= -1

    with plt.rc_context({'axes.edgecolor' : f'C{n}', 'axes.linewidth' : 2.}):
        ax = plt.subplot(gs2[1, n])
        ax.plot(Bv, 1e6*E1, 'b')
        ax.plot(Bv,-1e6*E1, 'r')
        ax.plot(Bv, 1e6*E2, 'gray')
        ax.plot(Bv,-1e6*E2, 'gray')
        ax.set_xlim([0., 1.])
        ax.set_ylim([-30., 30.])
        ax.grid()
        ax.set_xlabel(r'$B$ [T]')
        ax.set_yticks([-30, -20, -10, 0, 10, 20, 30])
        if n==0:
            ax.set_ylabel(r'$E$ [$\mu$eV]')
            ax.set_yticklabels(['', '-20', '', '0', '', '20', ''])
        else:
            ax.set_yticklabels([])
        ax.set_xticks([0., 0.25, 0.5, 0.75, 1.0])
        ax.set_xticklabels(['0', '', '0.5', '', '1'])


# Panel labels
fig.text(0.01, 0.95, "(a)", fontsize=14)
fig.text(0.34, 0.95, "(b)", fontsize=14)
fig.text(0.67, 0.95, "(c)", fontsize=14)
fig.text(0.67, 0.48, "(d)", fontsize=14)

plt.savefig("figures/figure3.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

# Fig. S10, energy gap in disordered system

```python
fig = plt.figure(figsize=(4 * (3 + 3/8), (4 + 3/8)))
plt.subplots_adjust(left=0.08, right=0.9, bottom=0.15, top=0.9, wspace=0.1, hspace=0.3)

gs = gridspec.GridSpec(1, 2)

ax = plt.subplot(gs[0, 0])
d = np.load('sim_data/gap_mfp500.npz')
im = plt.imshow(1e6*d['data'].T[::-1, :], aspect='auto',
           extent=get_extent(d['V'], d['Bv']),
           cmap='Blues', vmin=0., vmax=40., interpolation='none')
plt.ylabel(r'$B$ [T]')
plt.xlabel(r'$V_\mathrm{pg}$ [V]')
ax.set_title(r'$l_e = 500$ nm')
plt.xlim([0.4, 1.2])

ax = plt.subplot(gs[0, 1])
d = np.load('sim_data/gap_mfp200.npz')
im = plt.imshow(1e6*d['data'].T[::-1, :], aspect='auto',
           extent=get_extent(d['V'], d['Bv']),
           cmap='Blues', vmin=0., vmax=40., interpolation='none')
plt.xlabel(r'$V_\mathrm{pg}$ [V]')
ax.set_title(r'$l_e = 200$ nm')
ax.set_yticklabels([])
plt.xlim([0.4, 1.2])

bbox = ax.get_position().get_points()
cbax = fig.add_axes([bbox[1, 0]+0.01, bbox[0, 1], 0.01, bbox[1, 1]-bbox[0, 1]])
cb = plt.colorbar(im, extend='max', cax=cbax)
cb.set_label(r'$E_\mathrm{min}$ [$\mu$eV]', labelpad=14)
cb.set_ticks([0, 10, 20, 30, 40])

plt.savefig("figures/supp_figure_mfp.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

# Fig. S11, topological coherence length

```python
fig = plt.figure(figsize=(6, 5))
plt.subplots_adjust(top=0.85, bottom=0.13, right=0.95)
ax = plt.gca()
d = np.load('sim_data/coherence_length.npz')
im = plt.imshow(d['dataxi']/1e3, aspect='auto',
           extent=get_extent(d['V'], d['Bv']),
           cmap='pink', vmin=0., vmax=1., interpolation='none')
data0 = 1e6*d['datapd'][::-1, ::-1].T
plt.contour(d['datapd'][::-1, :].T,
            extent=get_extent(d['V'], d['Bv']),
            levels=[0.],
            colors=['gray'],
            linestyles=['-'],
            linewidths=[0.25],
            antialiased=True)
# Normalize the colors b/w 0 and 1, we'll then pass an MxNx4 array to imshow
colors = Normalize(0, 40)(data0)
cmap = plt.get_cmap('gray_r')
colors = cmap(colors)
colors[:, :, -1] = colors[-1, :, 0]
colors[:, :, :3] = [1., 1., 1.]
plt.imshow(colors, aspect='auto',
           extent=get_extent(d['V'], d['Bv']), zorder=2, interpolation='none')
plt.ylabel(r'$B$ [T]')
plt.xlabel(r'$V_\mathrm{pg}$ [V]')
plt.xlim([0.4, 1.2])

bbox = ax.get_position().get_points()
cbax = fig.add_axes([bbox[0, 0], bbox[1, 1]+0.015, bbox[1, 0]-bbox[0, 0], 0.025])
cb = plt.colorbar(im, extend='max', cax=cbax, orientation='horizontal')
cbax.xaxis.set_ticks_position('top')
cbax.xaxis.set_label_position('top')
cb.set_label(r'$\xi_\mathrm{topo}$ [$\mu$m]')

plt.savefig("figures/supp_figure_xi.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

# Fig. S12, orbital effects

```python
fig = plt.figure(figsize=(6, 5))
plt.subplots_adjust(top=0.85, bottom=0.13, right=0.95)
ax = plt.gca()
gr = dfe.groupby(['mfp', 'orbital', 'V_cutter']).get_group((np.inf, False, 0.4))
data = gr.pivot(index='V_plunger', columns='B', values='min_energy')
data.values[:, -1] = 0.
im = plt.imshow(1e6*data.values.T[::-1, :], aspect='auto',
           extent=get_extent(data.index, data.columns),
           cmap='Blues', vmin=0., vmax=40., interpolation='none')
plt.ylabel(r'$B$ [T]', labelpad=8)
ax.set_xlabel(r'$V_\mathrm{pg}$ [V]')
ax.set_xticks([0.4, 0.6, 0.8, 1.0, 1.2])
# ax.set_yticks([0., 0.4, 0.8])
ax.set_yticks([0.2, 0.6, 1.0], minor=True)
plt.xlim([0.4, 1.2])
ax.set_ylim(ymin=0.)

bbox = ax.get_position().get_points()
cbax = fig.add_axes([bbox[0, 0], bbox[1, 1]+0.015, bbox[1, 0]-bbox[0, 0], 0.025])
cb = plt.colorbar(im, extend='max', cax=cbax, orientation='horizontal')
cbax.xaxis.set_ticks_position('top')
cbax.xaxis.set_label_position('top')
cb.set_ticks([0, 10, 20, 30, 40])
cb.set_label(r'$E_\mathrm{min}$ [$\mu$eV]', )


plt.savefig("figures/supp_figure_no_orbital.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

# Fig. S12, depletion layer

```python
fig = plt.figure(figsize=(6, 5))
plt.subplots_adjust(top=0.85, bottom=0.13, right=0.95)
ax = plt.gca()
gr = pd.read_hdf('sim_data/v_depletion.hdf')
data = gr.pivot(index='V_plunger', columns='B', values='min_energy')
data.values[:, -1] = 0.
im = plt.imshow(1e6*data.values.T[::-1, :], aspect='auto',
           extent=get_extent(data.index, data.columns),
           cmap='Blues', vmin=0., vmax=40., interpolation='none')
plt.ylabel(r'$B$ [T]', labelpad=8)
ax.set_xlabel(r'$V_\mathrm{pg}$ [V]')
ax.set_xticks([0.4, 0.6, 0.8, 1.0, 1.2])
# ax.set_yticks([0., 0.4, 0.8])
ax.set_yticks([0.2, 0.6, 1.0], minor=True)
plt.xlim([0.4, 1.2])
ax.set_ylim(ymin=0.)

bbox = ax.get_position().get_points()
cbax = fig.add_axes([bbox[0, 0], bbox[1, 1]+0.015, bbox[1, 0]-bbox[0, 0], 0.025])
cb = plt.colorbar(im, extend='max', cax=cbax, orientation='horizontal')
cbax.xaxis.set_ticks_position('top')
cbax.xaxis.set_label_position('top')
cb.set_ticks([0, 10, 20, 30, 40])
cb.set_label(r'$E_\mathrm{min}$ [$\mu$eV]', )

# Arrows defining regions
ax.annotate(
    "",
    xy=(0.4, 0.02),
    xytext=(0.78, 0.02),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3", color='w')
)
ax.annotate(
    "", xy=(0.78, 0.02), xytext=(1.2, 0.02),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3", color='w')
)

# Region labels
ax.text(0.6, 0.05, "I", size=16, horizontalalignment='center', verticalalignment='bottom', color='w')
ax.text(1.0, 0.05, "III", size=16, horizontalalignment='center', verticalalignment='bottom', color='k', bbox=dict(facecolor='w', ec='none', alpha=0.5))

plt.savefig("figures/supp_figure_depletion.pdf", bbox_inches='tight', transparent=True)
plt.show()
```

# Fig S13, gap at B=0

```python
d = np.load('sim_data/gap_ndis.npz')
dmfp5 = np.load('sim_data/gap_mfp500.npz')
dmfp2 = np.load('sim_data/gap_mfp200.npz')
d1 = np.load('sim_data/topo_pd.npz')
d2 = np.load('sim_data/topo_pd_depletion.npz')
gr = pd.read_hdf('sim_data/v_depletion.hdf')
data = gr.pivot(index='V_plunger', columns='B', values='min_energy')

gap_acc = 1e6*d['data'][:, 0]
gap_acc[gap_acc>250] = 250
gap_mfp5 = 1e6*dmfp5['data'][:, 0]
gap_mfp5[gap_mfp5>250] = 250
gap_mfp2 = 1e6*dmfp2['data'][:, 0]
gap_mfp2[gap_mfp2>250] = 250
gap_depl = 1e6*data.values[:, 0]
gap_depl[gap_depl>250] = 250

fig = plt.figure(figsize=(12, 5))
gs = gridspec.GridSpec(1, 2)

ax1 = plt.subplot(gs[0, 0])
plt.plot(d['V'], gap_acc, 'C0', label='$L=1 \mu$m \n $l_e=\infty$')
plt.plot(d['V'], gap_mfp5, 'C1', label='$L=1 \mu$m \n $l_e=500$nm')
plt.plot(d['V'], gap_mfp2, 'C2', label='$L=1 \mu$m \n $l_e=200$nm')
plt.plot(d1['V'], 1e6*d1['data'][::-1, 0], 'k--', label='$L=\infty, l_e=\infty$')
ax1.set_title('$W=50$ meV')

# Arrows defining regions
ax1.annotate(
    "",
    xy=(0.4, -5),
    xytext=(0.6, -5),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3", color='k')
)
ax1.annotate(
    "", xy=(0.6, -5), xytext=(0.9, -5),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3", color='k')
)
ax1.annotate(
    "", xy=(0.9, -5), xytext=(1.2, -5),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3")
)

# Region labels
ax1.text(0.5, 2, "I", size=16, horizontalalignment='center', verticalalignment='bottom', color='k')
ax1.text(0.75, 2, "II", size=16, horizontalalignment='center', verticalalignment='bottom', color='k')
ax1.text(1.05, 2, "III", size=16, horizontalalignment='center', verticalalignment='bottom')

plt.legend(loc='center left')

ax1.set_ylabel(r'$E_\mathrm{gap}$ [$\mu$eV]')
ax1.set_xlabel(r'$V_\mathrm{pg}$ [V]')

ax2 = plt.subplot(gs[0, 1], sharex=ax1, sharey=ax1)
plt.plot(d['V'], gap_depl, label='$L=1 \mu$m$, l_e=\infty$')
plt.plot(d2['V'], d2['data'][::-1, 0], 'k--', label='$L=\infty, l_e=\infty$')
ax2.set_title('$W=-25$ meV')

plt.legend()

ax2.set_xlim([0.4, 1.2])
ax2.set_xlabel(r'$V_\mathrm{pg}$ [V]')
# ax2.set_ylabel(r'$E_\mathrm{gap}$ [$\mu$eV]')

# Arrows defining regions
ax2.annotate(
    "",
    xy=(0.4, -5),
    xytext=(0.78, -5),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3", color='k')
)
ax2.annotate(
    "", xy=(0.78, -5), xytext=(1.2, -5),
    arrowprops=dict(arrowstyle="<->", connectionstyle="arc3")
)

# Region labels
ax2.text(0.6, 2, "I", size=16, horizontalalignment='center', verticalalignment='bottom', color='k')
ax2.text(1.0, 2, "III", size=16, horizontalalignment='center', verticalalignment='bottom')

fig.text(0.07, 0.9, "(a)", fontsize=14)
fig.text(0.5, 0.9, "(b)", fontsize=14)
plt.savefig("figures/supp_figure_gap.pdf", bbox_inches='tight', transparent=True)
```

```python

```

```python

```
