% cite: Doelz et al, Journal of Scientific Computing (2022) 90:94 https://doi.org/10.1007/s10915-021-01757-9
%% some parameters
%parpool(8); % create a parallel pool
SRef=2; % number of spherical refinements
RRef=2; % number of spatial refinements
g=0.7; % Henyey Greenstein anisotropy parameter
tol=1e-8;  % tolerance for iterative schemes
maxIt=500; % maximum allowed iterations of the schemes

innerIt=4; % number of steps l for P_1^l
N_cor=4; % N should be even -> d_N(N) = number of eigenvectors considered in subspace correction.
N_pre=5; % should be odd
d_N=@(N)(N+1).*(N+2)/2;     % d_N
do_project_to_sphere=1; % 0 or 1 for obtaining a triangulation of the sphere.

%% construct spatial triangulation T_h^R
[p,e,t]=meshstruc(8,8);  % create structured mesh for unit square with 8x8 points
p(1,:)=7*p(1,:); p(2,:)=7*p(2,:);   % scale unit square to [0,7]x[0,7]

geomDSA.level=0;
geomE0.level=0;
geomDSA.p=p;geomDSA.e=e;geomDSA.t=t;
for ii=1:RRef
    [p,e,t,P]=refinemesh_eig([],p,e,t); % uniformly refine the spatial mesh
    geomDSA.parent=geomDSA;
    geomDSA.P=kron(speye(d_N(N_cor)),P);
    geomDSA.p=p;geomDSA.t=t;geomDSA.e=e;
    geomDSA.level=ii;
    
    geomE0.parent=geomE0;
    geomE0.P=P;
    geomE0.p=p;geomE0.t=t;geomE0.e=e;
    geomE0.level=ii;
end

%% construct spherical triangulation T_h^S
[p_sphere,t_sphere]=create_octahedron(); % load octahedron
radius=1;                   % define radius of sphere
for iter=1:SRef
    [p_sphere,t_sphere]=refine_sphere_mesh(p_sphere,t_sphere,radius,do_project_to_sphere);
end

%% define dimensions
np=size(p,2);               % n_R^+
nt=size(t,2);               % n_R^-
nse=size(t_sphere,2);       % n_S^+
nso=3*nse;                  % n_S^-
fprintf('np=%5d nt=%5d nt_sphere=%4d #dofs=%1.3e g=%1.2f\n',np,nt,nse,np*nse+nt*nso,g);

%% load example: Lattice problem
[Qe,Qo,mus,mua]=setup_checkerboard_parameter_functions(nse,p,t);
scale=1;Qe=Qe/scale;
c=max(mus./(mua+mus)); % convergence rate parameter \|\sigma_s/\sigma_t\|_\infty (if no subspace correction is applied)

%% assemble spatial FEM matrices
tic
[H1,L2]=assemble_interior(p,t,1,1);           % H1, L2 inner products for pcw linears
vol_spatial_triangles = get_vol_elem(p,t);    % area of elements of T_h^R
L2_const=diag(sparse(vol_spatial_triangles)); % L2 inner product for pcw constants
[~,Mae]=assemble_interior(p,t,[],mua);        % M_a^+
[~,Mse]=assemble_interior(p,t,[],mus);        % M_s^+
Mao=diag(sparse(mua.*vol_spatial_triangles)); % M_a^-
Mso=diag(sparse(mus.*vol_spatial_triangles)); % M_s^-
Mtoi=diag(sparse(1./(vol_spatial_triangles.*(mua+mus)))); % (M_a^- + M_s^-)^{-1}
[dx,dy]=assemble_dxdy(p,t);                   % partial derivative operators

%% assemble spherical FEM matrices
A1 = assemble_constant_linear_discontinuous_sphere_curved(p_sphere,t_sphere,@(x)x(1,:)); % A_1
A2 = assemble_constant_linear_discontinuous_sphere_curved(p_sphere,t_sphere,@(x)x(2,:)); % A_2
Ide=assemble_constant_discontinuous_sphere_curved(p_sphere,t_sphere,1); % I^+ Gramian for even fct on sphere
Ido=assemble_linear_discontinuous_sphere_curved(p_sphere,t_sphere,1);   % I^- Gramian for odd  fct on sphere
[Se,So]=assemble_hg_curved(p_sphere,t_sphere,g); % spherical convolutions: angular part of scattering operator

% symmetrize matrices for eigenvalue solvers
Ide = 0.5*(Ide + Ide');
Ido = 0.5*(Ido + Ido');
Se=0.5*(Se+Se');
So=0.5*(So+So');

%% tweaking of Se and So to shift the spectrum (ensure that the analytical spectral bounds are respected)
fprintf('compute eigenvalues of S^+, S^- ... ');
fprintf('even ...  ');
eige=eigs(@(x)Se*x,size(Se,1),Ide,1,'LA','IsFunctionSymmetric',1,'IsSymmetricDefinite',1);
fprintf('odd ...\n');
eigo=eigs(@(x)So*x,size(So,1),Ido,1,'LA','IsFunctionSymmetric',1,'IsSymmetricDefinite',1);
if eige>1
    Se=(1/eige)*Se; % max eigenvalue one
    fprintf('Change: rho(S^+) from %1.4f to 1   ',eige)
end
if eigo>g
    So=(g/eigo)*So;
    fprintf(' rho(S^-) from %1.4g to g=%1.2f',eigo,g)
end
fprintf('\n')

%% setup tensor product operators
fprintf('Assemble systems in '),
alpha=1/(1-c*g);
Idoi=Ido\speye(nso); % inverse mass matrix: block-diagonal with 3x3 blocks

% A' (M^-)^{-1} A = AMiA = sum_{k=1}^{n_S^+} block_k
% angular parts
d11=diag(A1'*Idoi*A1); % A1'*Idoi*A1 is diagonal
d12=diag(A1'*Idoi*A2); % A1'*Idoi*A2 is diagonal
d22=diag(A2'*Idoi*A2); % A2'*Idoi*A2 is diagonal
% second order spatial derivative operators
dxx=dx'*(Mtoi*dx); % int_R 1/(mua+mus) dx phi_i dx phi_j
dxy=dx'*(Mtoi*dy); % int_R 1/(mua+mus) dy phi_i dx phi_j
dyy=dy'*(Mtoi*dy); % int_R 1/(mua+mus) dy phi_i dy phi_j

AMiA=cell(nse,1); % allocate A'((M^-)^{-1} A)
MEven=cell(nse,1); % allocate M^+
E_zero=cell(nse,1); % = E0
dme=diag(Ide); % Ide is diagonal
R=assemble_boundary_sn_cell(p,e,t,p_sphere,t_sphere); % boundary matrix R
% assemble block-diagonal operators for E_zero
for k=1:nse
    AMiA{k}= d11(k)*dxx+d12(k)*(dxy+dxy')+d22(k)*dyy;
    MEven{k}=dme(k)*(Mse+Mae);
    E_zero{k}=alpha*AMiA{k}+MEven{k}+R{k};
end

% define function handles for tensor product operators
A=@(x)reshape( dx*reshape(x,np,nse)*A1'+dy*reshape(x,np,nse)*A2',nso*nt,1);  % apply A
At=@(x)reshape( dx'*reshape(x,nt,nso)*A1+dy'*reshape(x,nt,nso)*A2,nse*np,1); % apply A^t
M_even=@(x)reshape((Mae+Mse)*reshape(x,np,nse)*Ide,np*nse,1); % apply M^+
M_odd=@(x) reshape((Mao+Mso)*reshape(x,nt,nso)*Ido,nt*nso,1); % apply M^-
M_oddInv=@(x) reshape(Mtoi*reshape(x,nt,nso)*Idoi,nt*nso,1);  % apply (M^-)^{-1}
Ke=@(ue)reshape( Mse*reshape(ue,np,nse)*Se,np*nse,1); % apply K^+
Ko=@(uo)reshape( Mso*reshape(uo,nt,nso)*So,nt*nso,1); % apply K^-
fprintf('%1.3f sec\n',toc)

%% Application of C^{-1}=(Mo-Ko)^{-1}
% Ci=@(z)apply_Codd_inv(z,1e-13,40,Mao,Mso,Ido,Ko,M_oddInv); % simple preconditioner
%% test performance of Ci
[V,G]=eigs(@(x)So*x,size(So,1),Ido,d_N(N_pre),'LA','IsFunctionSymmetric',1,'IsSymmetricDefinite',1);
Cipre = @(x)apply_Codd_precond(x, V, diag(G), Ido, Mao+Mso, Mtoi, Mso); % preconditioner (M-K_N)
Ci=@(z)apply_Codd_inv(z,1e-13,40,Mao,Mso,Ido,Ko,Cipre); % apply (M-K)^{-1}z with preconditioner (M-K_N)
fprintf('Setup Ci with preconditioner M-K_N, N=%g and d_N=%g in %1.3f sec\n',N_pre,d_N(N_pre),toc);

%% integrate source
Qe=reshape(L2*reshape(Qe,np,nse)*Ide,np*nse,1);         % even part
Qo=reshape(L2_const*reshape(Qo,nt,nso)*Ido,nt*nso,1);   % odd part
Qoo=At(Ci(Qo)); % odd source term part for even-parity equations

% operator E, i.e., even-parity operator is: E - Ke
E=@(x)apply_op_cellwise(R,x,np,nse)+M_even(x)+At(Ci(A(x)));

%% factorize E0
fprintf('LU(E0) in '), tic
apply_E_zero=@(x)apply_op_cellwise(E_zero,x,np,nse);
[E_zeroP,E_zeroL,E_zeroU,E_zeroQ]=compute_lu_cellwise(E_zero,nse);
apply_E_zero_inv=@(x)apply_lu_cellwise(E_zeroP,E_zeroL,E_zeroU,E_zeroQ,x,np,nse);
fprintf('%1.3f sec\n',toc)
%apply_E_zero_inv=@(x)apply_mg_cellwise(geomE0,E_zero,x,np,nse);

%% Source iteration for even-parity system
fprintf(' Construct W_N,'), tic
[DSA,CoMatLo,PConstDSA,PConstDSAt,PLinDSA,PLinDSAt,AA,dime]=get_subspace_corr_cell(d_N(N_cor),Se,Ide,So,Ido,A1,dx,A2,dy,R,MEven,Mse,Mso,Mao);
fprintf('  dim=%d in %1.2d sec.',dime,toc)
mg=@(x)multigrid(geomDSA,DSA,0*x,x,2);
pre_corr=@(x)PConstDSA(mypcg(DSA,PConstDSAt(x),1e-10,500,mg));
tic
RHS=Qe+Qoo; % right-hand side for even-parity equations
phi0=0*RHS; % initial guess
sys=@(x)E(x)-Ke(x); % E-K system of even-parity equations
P1=@(x)richardson(E,x,1e-12,innerIt,0*x,1,apply_E_zero_inv); % P_1^l
tic,[phie,iter,relres]=richardson(sys,RHS,tol,maxIt,phi0,1,@(x)preconditioner(x,P1,sys,pre_corr),'aposteriori',sys);
fprintf(' SI-EP: it=%3d |res|=%1.2d time=%3.1f dim(WN)=%d innerIt=%d\n',iter,relres,toc,d_N(N_cor),innerIt);
fprintf('\n')
