#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Thu Nov 23 12:36:18 2023

Analyse drought characteristics by myself

@author: sutanto
"""

import numpy as np
import netCDF4
from mpl_toolkits.basemap import Basemap
from matplotlib import pyplot as plt

#########################################################
# Open drought data
modeld = 'ukesm1-0-ll'
scenari = 'past'

file2read = netCDF4.Dataset('/Users/Sutanto/Documents/postdoc_WSG/D3C2_2023/Data/Merve/SSMI/sm_'+scenari+'_ssmi_'+modeld+'.nc','r')  
lon = file2read.variables['x'][:]
lat = file2read.variables['y'][:]  
SSMI = file2read.variables['SSMI'][:]  

# for future, we only calculate from 2039-2100 -> 62 years data (744)
#SSMI = SSMI[:,:,288::]

# Making binary data
SSMI_binx = np.where(SSMI >= -1, 0, SSMI)   
SSMI_bin = np.where(SSMI_binx < -1, 2, SSMI_binx)   
    

 #########################################################
# Analysis drought characteristics
# Number of events
N_Dro = np.full((len(lon),len(lat)),np.nan)

# Total duration of events
D_Dro = np.full((len(lon),len(lat)),np.nan)


for i in range(len(lon)):
    for j in range(len(lat)):
        if SSMI_bin[i,j,0] == SSMI_bin[i,j,0]:  # to filter if not nan
            dummy = SSMI_bin[i,j,:]
            dummyx = np.append([0],dummy) # add zero for on going drought
            xxx = np.diff(dummyx)
            number = np.where(xxx == 2)
            ppp = np.array(number)            
            N_Dro[i,j] = len(ppp[0,:])
            D_Dro[i,j] = np.count_nonzero(dummy)
            

# average duration                
AD_Dro = D_Dro/N_Dro

# Frequency
year = len(SSMI[0,0,:])/12
F_Dro = N_Dro/year


#################################################
# Create new netCDF file to save the compound and cascading results
file2write = netCDF4.Dataset('/Users/Sutanto/Documents/Postdoc_WSG/D3C2_2023/Data/Merve/Drought_char/Sony_droughtchar_'+scenari+'_'+modeld+'.nc', 'w', format='NETCDF4')

# Create dimensions for the netCDF file
lon_dim = file2write.createDimension('x', len(lon))
lat_dim = file2write.createDimension('y', len(lat))


# Create variables for the netCDF file
lon_var = file2write.createVariable('x', 'f4', ('x',))
lat_var = file2write.createVariable('y', 'f4', ('y',))

numDrought = file2write.createVariable('numDrought', 'f4', ('x', 'y'))
todurDrought = file2write.createVariable('todurDrought', 'f4', ('x', 'y'))
avdurDrought = file2write.createVariable('avdurDrought', 'f4', ('x', 'y'))
freDrought = file2write.createVariable('freDrought', 'f4', ('x', 'y'))


# Add the data to the variables
lon_var[:] = lon
lat_var[:] = lat

numDrought[:,:] = N_Dro
todurDrought[:,:] = D_Dro
avdurDrought[:,:] = AD_Dro
freDrought[:,:] = F_Dro

# Add some attributes to the variables
lon_var.units = 'degrees_east'
lat_var.units = 'degrees_north'


# Close the netCDF file
file2write.close()



"""
Plotting
"""

# Plotting basemap
m = Basemap(resolution='i')
# Because our lon and lat variables are 1D,
late, lone = np.meshgrid(lat, lon)

# Add Grid Lines
lat_ticks = np.arange(np.amin(lat), np.amax(lat), 10.)
lon_ticks = np.arange(np.amin(lon), np.amax(lon), 10.)

m.pcolormesh(lone,late,D_Dro, vmin=0, vmax=500, cmap='jet')


# Plotting color bar
plt.colorbar(label='Total duration (month)', shrink=0.4)
# Add Title 
plt.title('Total drought duration',fontsize= 8) 


plt.xticks(lon_ticks, ["%3.1f$\degree$"%ll for ll in lon_ticks], fontsize= 8)
plt.yticks(lat_ticks, ["%3.1f$\degree$"%ll for ll in lat_ticks], fontsize= 8)

m.drawparallels(lat_ticks, linewidth=0.4,fontsize=8,fmt=  '%3.1f' )
m.drawmeridians(lon_ticks, linewidth=0.4,fontsize=8,fmt= '%3.1f'  )

# Add Coastlines, States, and Country Boundaries
m.drawcoastlines()
m.drawstates()
m.drawcountries()


# Limiting the plot
plt.xlim(-23,53)
plt.ylim(30,80)
plt.show()