% Accompanying code to 'Local characteristics of sand wave patterns are 
% governed by underlying sand bank: A linear stability analysis'
% by Laura Portos-Amill et al. 
% last edited on 28 October 2025

% compute perturbed state eigenmodes for given along bank topographic
% wavenumber k
% uses functions my_basic_state_transformed and
% my_basicstate_transformed_anygamma

tic
% clear all

omega = 2*pi/(12*3600 + 25*60 + 12); %M2 frequency in rad/s
latitude = 52.1326*pi/180; %latitude in radians (NL default)
f = 2*7.2921e-5*sin(latitude)/omega; %coriolis parameter (dimensionless)

L_star = 5000; % length scale (m)
H_star = 25; % depth scale (m)
U_star = L_star*omega;
r = U_star/L_star/omega; %keulegen-carpenter number
W_star = H_star*U_star/L_star;

% sand bank topography
Ny = 100;
L_dom = 35000; % wavelength of sand bank in m
y = linspace(0,L_dom/L_star,Ny + 1);
y = y(1:end-1).';
dy = y(2) - y(1);
gam = 2*pi/(y(end) + dy);
H0 = 10/H_star; % scaled height of the sand bank
y_trans = y*gam;
dy_trans = dy*gam;
n_cos = 10; % power that determines how focused the sand bank is
H = 1 - H0*(((1 + cos(y_trans + pi))/2).^n_cos);
dH_dY = H0*n_cos*((1 + cos(y_trans + pi))/2).^(n_cos-1).*sin(y_trans + pi)/2;
%asymmetric bank
% A = 0.7; phi = -pi/4;
% func = atan(A*sin(y_trans + phi)./(1 + A*cos(y_trans + phi)))/asin(A);
% H = 1 - H0*(((1 + func)/2).^n_cos);
% dH_dY = -H0*n_cos*((1+func)/2).^(n_cos - 1)/2/asin(A)./...
%     (1 + (A*sin(y_trans + phi)./(1 + A*cos(y_trans + phi))).^2).*...
%     (A^2 + A*cos(y_trans + phi))./(1 + A*cos(y_trans + phi)).^2;

N = 1e4 + 1; % length of the vector in the vertical direction
dz = abs((-1)/(N-1));

P = 2; % solution computed up to M4 component
p_vec = zeros(1,1,2*P+1); p_vec(1:2*P+1) = -P:P;
teta = 30; %sand bank orientation wrt M2 tide, angle in degrees, positive anticlockwise
e = 0.1; %eccentricity, positive is anticlokwise circulation
Q = 1.0/U_star; % magnitude of the flow scaled
% depth-integrated flow in along-bank direction scales with H^2
%   because depth-averaged scales with H because friction-dominated
Qx = Q*H.*H*H_star^2/H_star^2; 
Qp_00x = zeros(length(y),1,2*P+1); 
Qp_00x(:,1,P) = Qx/2*(cosd(teta) + 1i*e*sind(teta)); % M2 forcing
Qp_00x(:,1,P + 2) = Qx/2*(cosd(teta) - 1i*e*sind(teta)); % M2 forcing
% --------------------------------------------------------------------------
%residual circulation from loder 1980 but scaled (along-bank)
Ud = 0.35; % m/s depth-averaged
phi_R = 6.5; %degr
R = 1/tand(30);
F = 2.5e-3/omega*Ud*H(1)*H_star./(H*H_star).^2;
v_mean = H(1)*H_star*Ud^2*gam*dH_dY.*H_star/L_star./...
    (2*omega^2*(H.*H_star).^2.*(F.^2 + 1).^2).*...
    (2*omega*(1 - F.^2 + 2*F(1)*F)*R*sind(phi_R) + ...
    2*omega*(F.*(2 + F*F(1)) - F(1))*R*cosd(phi_R) + ...
    f*omega*F.^2.*(2 - H(1)./H) + f*omega*(3*H(1)./H - 2));
Q_res = 0.01*H0*H_star/10*v_mean/max(abs(v_mean))/U_star.*H; 
% --------------------------------------------------------------
Qp_00x(:,1,P+1) = Q_res;
Qy = Q;
% cross-bank depth-integrated flow scales with -- because depth-averaged
%   scales with 1/H because it is continuity-driven
Qp_00y = zeros(length(y),1,2*P+1); 
Qp_00y(:,1,P) = Qy/2*(-sind(teta) + 1i*e*cosd(teta)); % M2 forcing
Qp_00y(:,1,P + 2) = Qy/2*(-sind(teta) - 1i*e*cosd(teta)); % M2 forcing

s = 0.01/(H_star*omega);
Av = 2.5e-3*H_star*U_star/(H_star^2*omega); 

dt = 2e-4; t = [0:dt:1-dt]*2*pi; % time vector
z = [0:-dz:-1]; % depth vector

% scaled slope correction factor for bed load transport
lam = 1.5*H_star/L_star;

gam_max = 2; free_Uj = 0; % 1 is imposing Ures and other U free for first order in gamma
[Up0_gam,Vp0_gam,Wp0_gam,Fp,Gp] = my_basicstate_transformed_anygamma(H,dH_dY,...
    Qp_00x,Qp_00y,f,dy_trans,gam_max,dz,free_Uj,s,Av,r);
[Ly,Lz,Lp,~] = size(Up0_gam);

%build total basic state flow
Up0 = zeros(Ly,Lz,Lp); Vp0 = zeros(Ly,Lz,Lp); Wp0 = zeros(Ly,Lz,Lp);

for i_gam = 1:gam_max + 1
    Up0 = Up0 + gam^(i_gam-1)*Up0_gam(:,:,:,i_gam);
    Vp0 = Vp0 + gam^(i_gam-1)*Vp0_gam(:,:,:,i_gam);
    Wp0 = Wp0 + gam^(i_gam-1)*Wp0_gam(:,:,:,i_gam);
end


%% compute forcings from basic state
% quantities used for flow equations

dU0_dy = (Up0(mod(1:Ly,Ly) + 1,:,:) - Up0(mod((1:Ly)-2,Ly) + 1,:,:))./(2*dy);

dU0_dz = zeros(size(Up0));
dU0_dz(:,2:end-1,:) = (Up0(:,1:end-2,:) - Up0(:,3:end,:))./(2*dz);
dU0_dz(:,1,:) = (Up0(:,3,:) + 3*Up0(:,1,:) - 4*Up0(:,2,:))./(2*dz);
dU0_dz(:,end,:) = (4*Up0(:,end-1,:) - 3*Up0(:,end,:) - Up0(:,end-2,:))./(2*dz);

dV0_dy = (Vp0(mod(1:Ly,Ly) + 1,:,:) - Vp0(mod((1:Ly)-2,Ly) + 1,:,:))./(2*dy);

dV0_dz = zeros(size(Vp0));
dV0_dz(:,2:end-1,:) = (Vp0(:,1:end-2,:) - Vp0(:,3:end,:))./(2*dz);
dV0_dz(:,1,:) = (Vp0(:,3,:) + 3*Vp0(:,1,:) - 4*Vp0(:,2,:))./(2*dz);
dV0_dz(:,end,:) = (4*Vp0(:,end-1,:) - 3*Vp0(:,end,:) - Vp0(:,end-2,:))./(2*dz);

dW0_dz = zeros(size(Wp0));
dW0_dz(:,2:end-1,:) = (Wp0(:,1:end-2,:) - Wp0(:,3:end,:))./(2*dz);
dW0_dz(:,1,:) = (Wp0(:,3,:) + 3*Wp0(:,1,:) - 4*Wp0(:,2,:))./(2*dz);
dW0_dz(:,end,:) = (4*Wp0(:,end-1,:) - 3*Wp0(:,end,:) - Wp0(:,end-2,:))./(2*dz);


d2U0_dz2 = zeros(size(Up0));
d2U0_dz2(:,2:end-1,:) = (Up0(:,1:end-2,:) - 2*Up0(:,2:end-1,:) + Up0(:,3:end,:))/dz^2;
d2U0_dz2(:,1,:) = (2*Up0(:,1,:) - 5*Up0(:,2,:) + 4*Up0(:,3,:) - Up0(:,4,:))/dz^2;
d2U0_dz2(:,end,:) = (2*Up0(:,end,:) - 5*Up0(:,end-1,:) + 4*Up0(:,end-2,:) - Up0(:,end-3,:))/dz^2;

d2V0_dz2 = zeros(size(Vp0));
d2V0_dz2(:,2:end-1,:) = (Vp0(:,1:end-2,:) - 2*Vp0(:,2:end-1,:) + Vp0(:,3:end,:))/dz^2;
d2V0_dz2(:,1,:) = (2*Vp0(:,1,:) - 5*Vp0(:,2,:) + 4*Vp0(:,3,:) - Vp0(:,4,:))/dz^2;
d2V0_dz2(:,end,:) = (2*Vp0(:,end,:) - 5*Vp0(:,end-1,:) + 4*Vp0(:,end-2,:) - Vp0(:,end-3,:))/dz^2;

% quantities used for Exner equation
%compute nonlinear terms in physical time space and then convert to fourier
U0_t_bed = zeros(length(y),3,length(t)); % last 3 z positions to compute derivatives
V0_t_bed = zeros(length(y),3,length(t)); 
dU0_dz_t_bed = zeros(length(y),3,length(t));
dV0_dz_t_bed = zeros(length(y),3,length(t));

for i_t = 1:length(t)
    for i_p = 1:2*P + 1
        U0_t_bed(:,:,i_t) = U0_t_bed(:,:,i_t) + ...
            squeeze(Up0(:,end-2:end,i_p))*exp(1i*p_vec(i_p)*t(i_t));
        V0_t_bed(:,:,i_t) = V0_t_bed(:,:,i_t) + ...
            squeeze(Vp0(:,end-2:end,i_p))*exp(1i*p_vec(i_p)*t(i_t));

        dU0_dz_t_bed(:,:,i_t) = dU0_dz_t_bed(:,:,i_t) + ...
            squeeze(dU0_dz(:,end-2:end,i_p))*exp(1i*p_vec(i_p)*t(i_t));
        dV0_dz_t_bed(:,:,i_t) = dV0_dz_t_bed(:,:,i_t) + ...
            squeeze(dV0_dz(:,end-2:end,i_p))*exp(1i*p_vec(i_p)*t(i_t));
    end
end
U0_t_bed = real(U0_t_bed);
V0_t_bed = real(V0_t_bed);
dU0_dz_t_bed = real(dU0_dz_t_bed);
dV0_dz_t_bed = real(dV0_dz_t_bed);

term1_qbx_t = (U0_t_bed.^2 + V0_t_bed.^2).^(1/4) + 0.5*(U0_t_bed.^2 + ...
    V0_t_bed.^2).^(-3/4).*U0_t_bed.^2;
term2_qbx_t = 0.5*(U0_t_bed.^2 + V0_t_bed.^2).^(-3/4).*V0_t_bed.*U0_t_bed;
term3_qbx_t = (U0_t_bed.^2 + V0_t_bed.^2).^(1/4)./H.*dU0_dz_t_bed + ...
    0.5*U0_t_bed.*(U0_t_bed.^2 + V0_t_bed.^2).^(-3/4).*(U0_t_bed./H.*...
    dU0_dz_t_bed + V0_t_bed./H.*dV0_dz_t_bed);

term1_qby_t = 0.5*(U0_t_bed.^2 + V0_t_bed.^2).^(-3/4).*U0_t_bed.*V0_t_bed + ...
    3/2*lam*gam*dH_dY.*(U0_t_bed.^2 + V0_t_bed.^2).^(-1/4).*U0_t_bed;
term2_qby_t = (U0_t_bed.^2 + V0_t_bed.^2).^(1/4) + ...
    0.5*(U0_t_bed.^2 + V0_t_bed.^2).^(-3/4).*V0_t_bed.^2 + ...
    3/2*lam*gam*dH_dY.*(U0_t_bed.^2 + V0_t_bed.^2).^(-1/4).*V0_t_bed;
term3_qby_t = (U0_t_bed.^2 + V0_t_bed.^2).^(1/4)./H.*dV0_dz_t_bed + ...
    0.5*(U0_t_bed.^2 + V0_t_bed.^2).^(-3/4).*(U0_t_bed./H.*dU0_dz_t_bed + ...
    V0_t_bed./H.*dV0_dz_t_bed).*V0_t_bed + 3/2*lam*gam*dH_dY.*...
    (U0_t_bed.^2 + V0_t_bed.^2).^(-1/4).*(U0_t_bed./H.*dU0_dz_t_bed + ...
    V0_t_bed./H.*dV0_dz_t_bed);
U0_32_t = (U0_t_bed.^2 + V0_t_bed.^2).^(3/4);

% in fourier space
term1_qby = zeros(length(y),3,2*P+1);
term2_qby = zeros(length(y),3,2*P+1);
term3_qby = zeros(length(y),3,2*P+1);
Up0_32 = zeros(length(y),3,2*P+1);
term1_qbx = zeros(length(y),1,2*P+1);
term2_qbx = zeros(length(y),1,2*P+1);
term3_qbx = zeros(length(y),1,2*P+1);

ft1 = fft(real(term1_qbx_t),length(t),3)/length(t);
ft2 = fft(real(term2_qbx_t),length(t),3)/length(t);
ft3 = fft(real(term3_qbx_t),length(t),3)/length(t);
ft4 = fft(real(term1_qby_t),length(t),3)/length(t);
ft5 = fft(real(term2_qby_t),length(t),3)/length(t);
ft6 = fft(real(term3_qby_t),length(t),3)/length(t);
ft7 = fft(real(U0_32_t),length(t),3)/length(t);

for i_p = 1:P
    term1_qbx(:,:,i_p) = ft1(:,end,end - (P - i_p));
    term1_qbx(:,:,2*P+1 - (i_p - 1)) = conj(ft1(:,end,end - (P - i_p)));

    term2_qbx(:,:,i_p) = ft2(:,end,end - (P - i_p));
    term2_qbx(:,:,2*P+1 - (i_p - 1)) = conj(ft2(:,end,end - (P - i_p)));

    term3_qbx(:,:,i_p) = ft3(:,end,end - (P - i_p));
    term3_qbx(:,:,2*P+1 - (i_p - 1)) = conj(ft3(:,end,end - (P - i_p)));

    term1_qby(:,:,i_p) = ft4(:,:,end - (P - i_p));
    term1_qby(:,:,2*P+1 - (i_p - 1)) = conj(ft4(:,:,end - (P - i_p)));

    term2_qby(:,:,i_p) = ft5(:,:,end - (P - i_p));
    term2_qby(:,:,2*P+1 - (i_p - 1)) = conj(ft5(:,:,end - (P - i_p)));

    term3_qby(:,:,i_p) = ft6(:,:,end - (P - i_p));
    term3_qby(:,:,2*P+1 - (i_p - 1)) = conj(ft6(:,:,end - (P - i_p)));

    Up0_32(:,:,i_p) = ft7(:,:,end - (P - i_p));
    Up0_32(:,:,2*P+1 - (i_p - 1)) = conj(ft7(:,:,end - (P - i_p)));
end

term1_qbx(:,:,P+1) = ft1(:,end,1);
term2_qbx(:,:,P+1) = ft2(:,end,1);
term3_qbx(:,:,P+1) = ft3(:,end,1);
term1_qby(:,:,P+1) = ft4(:,:,1);
term2_qby(:,:,P+1) = ft5(:,:,1);
term3_qby(:,:,P+1) = ft6(:,:,1);
Up0_32(:,:,P+1) = ft7(:,:,1);

% compute derivatives
% d stands for (d/dy - gam z H'/H d/dz), here == d/dy becase at seabed
d_term1_qby = (term1_qby(mod(1:Ly,Ly) + 1,end,:) - term1_qby(mod((1:Ly)-2,Ly) + 1,end,:))./(2*dy);
d_term2_qby = (term2_qby(mod(1:Ly,Ly) + 1,end,:) - term2_qby(mod((1:Ly)-2,Ly) + 1,end,:))./(2*dy);
d_term3_qby = (term3_qby(mod(1:Ly,Ly) + 1,end,:) - term3_qby(mod((1:Ly)-2,Ly) + 1,end,:))./(2*dy);
d_Up0_32 = (Up0_32(mod(1:Ly,Ly) + 1,end,:) - Up0_32(mod((1:Ly)-2,Ly) + 1,end,:))./(2*dy);


%% make coarser for solving first order

Ny_1 = 100; Nz_1 = 50 + 1;
transform_y_for1 = (Ly)/(Ny_1);
transform_z_for1 = (Lz-1)/(Nz_1-1);

dU0_dy_for1 = dU0_dy([1:transform_y_for1:end],[1:transform_z_for1:end],:); 
dU0_dz_for1 = dU0_dz([1:transform_y_for1:end],[1:transform_z_for1:end],:);
dV0_dy_for1 = dV0_dy([1:transform_y_for1:end],[1:transform_z_for1:end],:); 
dV0_dz_for1 = dV0_dz([1:transform_y_for1:end],[1:transform_z_for1:end],:);
dW0_dz_for1 = dW0_dz([1:transform_y_for1:end],[1:transform_z_for1:end],:);

d2U0_dz2_for1 = d2U0_dz2([1:transform_y_for1:end],[1:transform_z_for1:end],:);
d2V0_dz2_for1 = d2V0_dz2([1:transform_y_for1:end],[1:transform_z_for1:end],:);

Up0_for1 = Up0([1:transform_y_for1:end],[1:transform_z_for1:end],:);
Vp0_for1 = Vp0([1:transform_y_for1:end],[1:transform_z_for1:end],:);
Wp0_for1 = Wp0([1:transform_y_for1:end],[1:transform_z_for1:end],:);

dH_dY_for1 = dH_dY([1:transform_y_for1:end]); 
H_for1 = H([1:transform_y_for1:end]);
z_for1 = z([1:transform_z_for1:end]); dz_for1 = abs(z_for1(2) - z_for1(1));
y_for1 = y([1:transform_y_for1:end]); dy_for1 = y_for1(2) - y_for1(1);

term1_qbx_for1 = term1_qbx([1:transform_y_for1:end],end,:);
term2_qbx_for1 = term2_qbx([1:transform_y_for1:end],end,:);
term3_qbx_for1 = term3_qbx([1:transform_y_for1:end],end,:);

term1_qby_for1 = term1_qby([1:transform_y_for1:end],end,:);
term2_qby_for1 = term2_qby([1:transform_y_for1:end],end,:);
term3_qby_for1 = term3_qby([1:transform_y_for1:end],end,:);
d_term1_qby_for1 = d_term1_qby([1:transform_y_for1:end],end,:);
d_term2_qby_for1 = d_term2_qby([1:transform_y_for1:end],end,:);
d_term3_qby_for1 = d_term3_qby([1:transform_y_for1:end],end,:);

Up0_32_for1 = Up0_32([1:transform_y_for1:end],end,:);
d_Up0_32_for1 = d_Up0_32([1:transform_y_for1:end],end,:);

fprintf('compute basic state: %.2f seconds. \n',toc')
%% build sparse matrix defining generalised eigenvalue problem
%sparse matrix will be constructed with indices and values of non-zero
%elements
%S = sparse(i,j,v) generates a sparse matrix S from the triplets i, j, and
%   v such that S(i(k),j(k)) = v(k).

% build per row following set of equations:

% ss bc for U1
% x momentum at z = 2:Nz-1
% sb bc for U1
% ss bc for V1
% y-momentum at z = 2:Nz - 1
% sb bc for V1
% ss bc for W1
% continuity equation at z = 2:Nz 
% sb bc for W1
% Exner equation (Ny_1 evaluations)

% matrix and vector rows are aranged as z,y,p

%
tic

[Ny_1, Nz_1, Np] = size(Up0_for1);
N = Np*Ny_1*(3*Nz_1 + 1) + Ny_1; %size of matrix is NxN

i_p_vec = zeros(1,N);
i_y_vec = zeros(1,N);
i_z_vec = zeros(1,N);
p_mat = zeros(1,N);
dH_dY_vec = zeros(1,N);
H_vec = zeros(1,N);
y_vec = zeros(1,N);
z_vec = zeros(1,N);
i_z_base = zeros(1,N);

% each row in the matrix has a value of y,z,p,... assigned
i_r = 1;
for i_loop = 1:3
    for i_z = 1:Nz_1
        i_z_base_num = i_r;
        for i_y = 1:Ny_1
            for i_p = 1:Np
                i_p_vec(i_r) = i_p;
                p_mat(i_r) = p_vec(i_p);
                i_y_vec(i_r) = i_y;
                i_z_vec(i_r) = i_z;
                dH_dY_vec(i_r) = dH_dY_for1(i_y);
                H_vec(i_r) = H_for1(i_y);
                y_vec(i_r) = y_for1(i_y);
                z_vec(i_r) = z_for1(i_z);
                i_z_base(i_r) = i_z_base_num;

                i_r = i_r + 1;
            end
        end
    end
end
for i_y = 1:Ny_1
    for i_p = 1:Np
        i_p_vec(i_r) = i_p;
        p_mat(i_r) = p_vec(i_p);
        i_y_vec(i_r) = i_y;
        i_z_vec(i_r) = i_z;
        dH_dY_vec(i_r) = dH_dY_for1(i_y);
        H_vec(i_r) = H_for1(i_y);
        y_vec(i_r) = y_for1(i_y);
        z_vec(i_r) = z_for1(i_z);

        i_r = i_r + 1;
    end
end
for i_y = 1:Ny_1
    i_y_vec(i_r) = i_y;
    dH_dY_vec(i_r) = dH_dY_for1(i_y);
    H_vec(i_r) = H_for1(i_y);
    y_vec(i_r) = y_for1(i_y);
    z_vec(i_r) = z_for1(i_z);

    i_r = i_r + 1;
end

index_conv = cell(Np,1);
for i_p = 1:Np
    index_conv{i_p} = [squeeze(p_vec(max(1,(P+2) - i_p):1:min(3*P+2 - i_p,2*P+1)))'];
end


% if several kx: (also uncomment lines 785-788)
% kx_vec = [0:0.001:0.03]; % wave number in m-1
% val_mat = zeros(100,length(kx_vec));
% vec_mat = zeros(Ny_1*Np*(3*Nz_1 + 1) + Ny_1,100,length(kx_vec));

clear vec val M_sp M_right_sp

k = L_star*0.014; %scaled wavenumber
%%
val_mat = cell(N,1);
i_row_mat = cell(N,1);
i_col_mat = cell(N,1);

%built sparse matrix
parfor i_row = 1:N

    i_y = i_y_vec(i_row); i_z = i_z_vec(i_row); i_p = i_p_vec(i_row);

    if i_row <= Np*Ny_1 % seasurface bc for u
        % d/dz U1
        i_col_mat{i_row} = [i_col_mat{i_row},i_row + 2*Ny_1*Np, i_row, i_row + 1*Ny_1*Np];
        i_row_mat{i_row} = [i_row_mat{i_row},i_row*ones(1,3)];
        val_mat{i_row} = [val_mat{i_row}, [1,3,-4]/(2*dz_for1)];

    elseif i_row <= Np*Ny_1*(Nz_1 - 1) % x-momentum inner points in z

        % ip U1
        i_col_mat{i_row} = [i_col_mat{i_row}, i_row];
        val_mat{i_row} = [val_mat{i_row}, 1i*p_mat(i_row)];
        i_row_mat{i_row} = [i_row_mat{i_row},i_row];

        % U0 ik U1
        i_col_mat{i_row} = [i_col_mat{i_row}, i_row + index_conv{i_p}];
        val_mat{i_row} = [val_mat{i_row},1i*k*r*...
            squeeze(Up0_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).'];
        i_row_mat{i_row} = [i_row_mat{i_row},i_row*ones(1,length(index_conv{i_p}))];

        % V0 d/dy U1
        i_col_mat{i_row} = [i_col_mat{i_row}, ...
                (i_z_base(i_row) - 1) + mod((i_row + index_conv{i_p} + Np - ...
                (i_z_base(i_row) - 1)) - 1,Ny_1*Np) + 1,...
                (i_z_base(i_row) - 1) + mod((i_row + index_conv{i_p} - Np - ...
                (i_z_base(i_row) - 1)) - 1,Ny_1*Np) + 1];
        val_mat{i_row} = [val_mat{i_row},...
            r*[squeeze(Vp0_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).',...
            - squeeze(Vp0_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).']./(2*dy_for1)];
        i_row_mat{i_row} = [i_row_mat{i_row},i_row*ones(1,2*length(index_conv{i_p}))];

        % - gam V0 z H'/H d/dz  U1
        i_col_mat{i_row} = [i_col_mat{i_row},...
            i_row - Ny_1*Np + index_conv{i_p},...
            i_row + Ny_1*Np + index_conv{i_p}];
        val_mat{i_row} = [val_mat{i_row}, ...
            -r*gam*z_vec(i_row)*dH_dY_vec(i_row)/(H_vec(i_row)*2*dz_for1).*...
            [squeeze(Vp0_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).',...
            -squeeze(Vp0_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).']];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,2*length(index_conv{i_p}))];

        % W0/H d/dz U1
        i_col_mat{i_row} = [i_col_mat{i_row},...
            i_row - Ny_1*Np + index_conv{i_p}, ...
            i_row + Ny_1*Np + index_conv{i_p}];
        val_mat{i_row} = [val_mat{i_row}, ...
            r*1/(H_vec(i_row)*2*dz_for1).*...
            [squeeze(Wp0_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).',...
            -squeeze(Wp0_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).']];
        i_row_mat{i_row} = [i_row_mat{i_row},i_row*ones(1,2*length(index_conv{i_p}))];

        % - Av/H^2 d^2/dz^2 U1
        i_col_mat{i_row} = [i_col_mat{i_row},...
            i_row - Np*Ny_1, i_row, i_row + Np*Ny_1];
        val_mat{i_row} = [val_mat{i_row}, -Av/(H_vec(i_row)^2*dz_for1^2).*...
            [1, -2, 1]];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,3)];

       % dU0/dy V1
       i_col_mat{i_row} = [i_col_mat{i_row}, ...
           i_row + Ny_1*Np*Nz_1 + index_conv{i_p}];
       val_mat{i_row} = [val_mat{i_row},...
           r*squeeze(dU0_dy_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).'];
        i_row_mat{i_row} = [i_row_mat{i_row}, ...
            i_row*ones(1,length(index_conv{i_p}))];

        % - gam z H'/H dU0/dz V1
        i_col_mat{i_row} = [i_col_mat{i_row},...
            i_row + Ny_1*Np*Nz_1 + index_conv{i_p}];
        val_mat{i_row} = [val_mat{i_row}, ...
            -r*gam*z_vec(i_row)*dH_dY_vec(i_row)/H_vec(i_row)*...
            squeeze(dU0_dz_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).'];
        i_row_mat{i_row} = [i_row_mat{i_row}, ...
            i_row*ones(1,length(index_conv{i_p}))];

        % - f V1
        i_col_mat{i_row} = [i_col_mat{i_row}, i_row + Ny_1*Np*Nz_1];
        val_mat{i_row} = [val_mat{i_row},-f];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row];

        % 1/H dU0/dz W1
        i_col_mat{i_row} = [i_col_mat{i_row}, ...
            i_row + 2*Ny_1*Np*Nz_1 + index_conv{i_p}];
        val_mat{i_row} = [val_mat{i_row},...
            r*squeeze(dU0_dz_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).'./H_vec(i_row)];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,length(index_conv{i_p}))];

        % ik ZETA1
        i_col_mat{i_row} = [i_col_mat{i_row},...
            3*Np*Ny_1*Nz_1 + (i_y - 1)*Np + i_p];
        val_mat{i_row} = [val_mat{i_row}, 1i*k];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row];

    elseif i_row <= Np*Ny_1*Nz_1 % seabed bc for U
        % (Av/H d/dz U1 at z = -1) (- s U1 at z = -1)
        i_col_mat{i_row} = [i_col_mat{i_row},...
            i_row - Np*Ny_1, i_row - 2*Np*Ny_1, i_row, i_row];
        val_mat{i_row} = [val_mat{i_row}, Av/(H_vec(i_row)*2*dz_for1).*...
            [4, -1, -3], -s];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,3), i_row];

        % (Av/H^2 d2U0/dz2 - s/H dU0/dz) h1
        i_col_mat{i_row} = [i_col_mat{i_row},...
            Np*Ny_1*(3*Nz_1 + 1) + i_y];
        val_mat{i_row} = [val_mat{i_row}, ...
            Av/H_vec(i_row)^2*d2U0_dz2_for1(i_y,end,i_p) - ...
            s/H_vec(i_row)*dU0_dz_for1(i_y,end,i_p)];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row]

    elseif i_row <= Np*Ny_1*Nz_1 + Np*Ny_1 % seasurface bc for V1
        % d/dz V1 at z = 0
        i_col_mat{i_row} = [i_col_mat{i_row},
            i_row + 2*Ny_1*Np, i_row + Ny_1*Np, i_row];
        val_mat{i_row} = [val_mat{i_row}, 1/(2*dz_for1)*[1, -4, 3]];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,3)];

    elseif i_row <= Np*Ny_1*Nz_1 + Np*Ny_1*(Nz_1 - 1)% y-momentum

        % f U1
        i_col_mat{i_row} = [i_col_mat{i_row}, i_row - Np*Ny_1*Nz_1];
        val_mat{i_row} = [val_mat{i_row}, f];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row];

        % ip V1
        i_col_mat{i_row} = [i_col_mat{i_row}, i_row];
        val_mat{i_row} = [val_mat{i_row}, 1i*p_mat(i_row)];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row];

        % ik U0 V1
        i_col_mat{i_row} = [i_col_mat{i_row}, i_row + index_conv{i_p}];
        val_mat{i_row} = [val_mat{i_row}, ...
            r*1i*k*squeeze(Up0_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).'];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,length(index_conv{i_p}))];

        % V0 d/dy V1
        i_col_mat{i_row} = [i_col_mat{i_row}, ...
            (i_z_base(i_row) - 1) + mod((i_row + index_conv{i_p} + Np - ...
            (i_z_base(i_row) - 1)) - 1,Ny_1*Np) + 1,...
            (i_z_base(i_row) - 1) + mod((i_row + index_conv{i_p} - Np - ...
            (i_z_base(i_row) - 1)) - 1,Ny_1*Np) + 1];
        val_mat{i_row} = [val_mat{i_row}, ...
            r*[squeeze(Vp0_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).',...
            -squeeze(Vp0_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).']./(2*dy_for1)];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,2*length(index_conv{i_p}))];

        % - gam V0 z H'/H d/dz V1
        i_col_mat{i_row} = [i_col_mat{i_row}, ...
            i_row - Ny_1*Np + index_conv{i_p}, i_row + Ny_1*Np + index_conv{i_p}];
        val_mat{i_row} = [val_mat{i_row}, -gam*z_vec(i_row)*dH_dY_vec(i_row)/H_vec(i_row).*...
            r*[squeeze(Vp0_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).',...
            -squeeze(Vp0_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).']./(2*dz_for1)];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,2*length(index_conv{i_p}))];

        %  dV0/dy V1
        i_col_mat{i_row} = [i_col_mat{i_row}, i_row + index_conv{i_p}];
        val_mat{i_row} = [val_mat{i_row}, ...
            r*squeeze(dV0_dy_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).'];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,length(index_conv{i_p}))];

        % - gam z H'/H dV0/dz V1
        i_col_mat{i_row} = [i_col_mat{i_row}, i_row + index_conv{i_p}];
        val_mat{i_row} = [val_mat{i_row}, -gam*z_vec(i_row)*dH_dY_vec(i_row)/H_vec(i_row)*...
               r*squeeze(dV0_dz_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).'];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,length(index_conv{i_p}))];

        % 1/H W0 dV1/dz
        i_col_mat{i_row} = [i_col_mat{i_row}, ...
            i_row - Ny_1*Np + index_conv{i_p}, i_row + Ny_1*Np + index_conv{i_p}];
        val_mat{i_row} = [val_mat{i_row}, ...
            r*[squeeze(Wp0_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).',...
            -squeeze(Wp0_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).']./...
            (H_vec(i_row)*2*dz_for1)];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,2*length(index_conv{i_p}))];

        % -Av/H^2 d^2/dz^2 V1
        i_col_mat{i_row} = [i_col_mat{i_row}, i_row - Np*Ny_1, i_row, i_row + Np*Ny_1];
        val_mat{i_row} = [val_mat{i_row}, ...
            -Av/H_vec(i_row)^2/dz_for1^2*[1, -2, 1]];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,3)];

        % 1/H dV0/dz W1
        i_col_mat{i_row} = [i_col_mat{i_row}, i_row + Np*Ny_1*Nz_1 + index_conv{i_p}];
        val_mat{i_row} = [val_mat{i_row}, ...
            r*squeeze(dV0_dz_for1(i_y,i_z,min(i_p+P,2*P+1):-1:max(1,i_p-P))).'./H_vec(i_row)];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,length(index_conv{i_p}))];

        % d/dy ZETA1
        i_col_mat{i_row} = [i_col_mat{i_row}, ...
            3*Np*Nz_1*Ny_1 + mod(((i_y - 1)*Np + i_p + Np) - 1,Ny_1*Np) + 1,...
            3*Np*Nz_1*Ny_1 + mod(((i_y - 1)*Np + i_p - Np) - 1,Ny_1*Np) + 1];
        val_mat{i_row} = [val_mat{i_row}, [1, -1]/(2*dy_for1)];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,2)];

    elseif i_row <= 2*Np*Nz_1*Ny_1 % seabed bc for V1
        % Av/H d/dz V1 - s V1 at z = -1
        i_col_mat{i_row} = [i_col_mat{i_row}, ...
            i_row - Np*Ny_1, i_row - 2*Np*Ny_1, i_row, i_row];
        val_mat{i_row} = [val_mat{i_row}, ...
            Av/(H_vec(i_row)*2*dz_for1)*[4,-1,-3], -s];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,4)];

        % (Av/H^2 d^2U0/dz^2 - s/H dU0/dz) h1
        i_col_mat{i_row} = [i_col_mat{i_row},...
            3*Np*Ny_1*Nz_1 + Np*Ny_1 + i_y];
        val_mat{i_row} = [val_mat{i_row}, ...
            Av/H_vec(i_row)^2*d2V0_dz2_for1(i_y,end,i_p) - ...
            s/H_vec(i_row)*dV0_dz_for1(i_y,end,i_p)];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row];

    elseif i_row <= 2*Np*Ny_1*Nz_1 + Np*Ny_1 % seasurface bc for W1
        % W1 at z = 0
        i_col_mat{i_row} = [i_col_mat{i_row}, i_row];
        val_mat{i_row} = [val_mat{i_row}, 1];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row];

    elseif i_row <= 3*Np*Nz_1*Ny_1 % continuity equation (also at seabed boundary!)
        % ik U1
        i_col_mat{i_row} = [i_col_mat{i_row}, i_row - 2*Np*Ny_1*Nz_1];
        val_mat{i_row} = [val_mat{i_row}, 1i*k];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row];

        % d/dy V1
        i_col_mat{i_row} = [i_col_mat{i_row},...
            (i_z_base(i_row) - 1) - Np*Ny_1*Nz_1 + mod((i_row + Np - ...
            (i_z_base(i_row) - 1)) - 1,Ny_1*Np) + 1,...
            (i_z_base(i_row) - 1) - Np*Ny_1*Nz_1 + mod((i_row - Np - ...
            (i_z_base(i_row) - 1)) - 1,Ny_1*Np) + 1];
        val_mat{i_row} = [val_mat{i_row}, [1, -1]/(2*dy_for1)];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,2)];

        % -gam z H'/H d/dz V1
        if i_z < Nz_1 % centered differences
            i_col_mat{i_row} = [i_col_mat{i_row}, ...
                i_row - Np*Ny_1*Nz_1 + Np*Ny_1, i_row - Np*Ny_1*Nz_1 - Np*Ny_1];
            val_mat{i_row} = [val_mat{i_row}, ...
                -gam*z_vec(i_row)*dH_dY_vec(i_row)/(H_vec(i_row)*2*dz_for1)*[-1,1]];
            i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,2)];
        else % backward scheme second order
            i_col_mat{i_row} = [i_col_mat{i_row},...
                i_row - Np*Ny_1*Nz_1 - Np*Ny_1,...
                i_row - Np*Ny_1*Nz_1 - 2*Np*Ny_1, i_row - Np*Ny_1*Nz_1];
            val_mat{i_row} = [val_mat{i_row},...
                -gam*z_vec(i_row)*dH_dY_vec(i_row)/(H_vec(i_row)*2*dz_for1)*[4,-1,-3]];
            i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,3)];
        end            

        % 1/H d/dz W1
        if i_z < Nz_1
            i_col_mat{i_row} = [i_col_mat{i_row}, ...
                i_row + Np*Ny_1, i_row - Np*Ny_1];
            val_mat{i_row} = [val_mat{i_row}, 1/(H_vec(i_row)*2*dz_for1)*[-1, 1]];
            i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,2)];
        else
            i_col_mat{i_row} = [i_col_mat{i_row}, ...
                i_row - Np*Ny_1, i_row - 2*Np*Ny_1, i_row];
            val_mat{i_row} = [val_mat{i_row}, 1/(H_vec(i_row)*2*dz_for1)*[4,-1,-3]];
            i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,3)];
        end

    elseif i_row <= Np*Ny_1*(3*Nz_1 + 1) % seabed boundary condition for W1
        % gam H' V1
        i_col_mat{i_row} = [i_col_mat{i_row}, i_row - Np*Ny_1 - Np*Ny_1*Nz_1];
        val_mat{i_row} = [val_mat{i_row}, gam*dH_dY_vec(i_row)];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row];

        % W1
        i_col_mat{i_row} = [i_col_mat{i_row}, i_row - Np*Ny_1];
        val_mat{i_row} = [val_mat{i_row}, 1];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row];

        % (1/H dW0/dz - ik U0 + gam H'/H dV0/dz) h1
        i_col_mat{i_row} = [i_col_mat{i_row}, 3*Np*Ny_1*Nz_1 + Np*Ny_1 + i_y];
        val_mat{i_row} = [val_mat{i_row}, 1/H_vec(i_row)*dW0_dz_for1(i_y,end,i_p) - ...
            1i*k*Up0_for1(i_y,end,i_p) + gam*dH_dY_vec(i_row)/H_vec(i_row).*...
            dV0_dz_for1(i_y,end,i_p)];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row];

        % - V0 d/dy h1
        i_col_mat{i_row} = [i_col_mat{i_row}, ...
            Np*Ny_1*(3*Nz_1 + 1) + mod((i_y + 1) - 1,Ny_1) + 1,...
            Np*Ny_1*(3*Nz_1 + 1) + mod((i_y - 1) - 1,Ny_1) + 1];
        val_mat{i_row} = [val_mat{i_row}, ...
            -Vp0_for1(i_y,end,i_p)*[1,-1]/(2*dy_for1)];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,2)];

    else % Exner equation
        %indices needed for convolution for time-averaged
        index1 = [1:2*P+1]; index2 = [2*P + 1:-1:1] - 1;

        % dqbx/dx
        % s^(3/2) term1_qbx 1i k U1_bed
        i_col_mat{i_row} = [i_col_mat{i_row},...
            Np*Ny_1*(Nz_1 - 1) + (i_y - 1)*Np + 1 + index2];
        val_mat{i_row} = [val_mat{i_row},...
            s^(3/2)*(squeeze(term1_qbx_for1(i_y,end,index1)).')*1i*k];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,length(index2))];

        % s^(3/2) term2_qbx 1i k V1_bed
        i_col_mat{i_row} = [i_col_mat{i_row},...
            Np*Ny_1*Nz_1 + Np*Ny_1*(Nz_1 - 1) + (i_y - 1)*Np + 1 + index2];
        val_mat{i_row} = [val_mat{i_row}, ...
            s^(3/2)*squeeze(term2_qbx_for1(i_y,end,index1)).'*1i*k];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,length(index2))];

        % s^(3/2) (term3_qbx - lam*Up0_32 1i k) 1i k h1
        i_col_mat{i_row} = [i_col_mat{i_row}, Np*Ny_1*(3*Nz_1 + 1) + i_y];
        val_mat{i_row} = [val_mat{i_row}, ...
            s^(3/2)*(term3_qbx_for1(i_y,end,P+1) - ...
            lam*Up0_32_for1(i_y,end,P+1)*1i*k)*1i*k];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row];

        %dqby/dy
        % s^(3/2) d_term1 U1
        i_col_mat{i_row} = [i_col_mat{i_row}, ...
            Np*Ny_1*(Nz_1 - 1) + (i_y - 1)*Np + 1 + index2];
        val_mat{i_row} = [val_mat{i_row}, ...
            s^(3/2)*squeeze(d_term1_qby_for1(i_y,end,index1)).'];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,length(index2))];

        % s^(3/2) term1_qby d/dy U1_bed
        i_col_mat{i_row} = [i_col_mat{i_row},...
        Np*Ny_1*(Nz_1 - 1) + mod(((i_y - 1)*Np + Np + 1 + index2 - 1),Ny_1*Np) + 1,...
        Np*Ny_1*(Nz_1 - 1) + mod(((i_y - 1)*Np - Np + 1 + index2 - 1),Ny_1*Np) + 1];
        val_mat{i_row} = [val_mat{i_row}, s^(3/2)/(2*dy_for1)*...
            [squeeze(term1_qby_for1(i_y,end,index1)).',...
            -squeeze(term1_qby_for1(i_y,end,index1)).']];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,2*length(index2))];

        % s^(3/2) d_term2_qby V1
        i_col_mat{i_row} = [i_col_mat{i_row},...
            Np*Ny_1*Nz_1 + Np*Ny_1*(Nz_1 - 1) + (i_y - 1)*Np + 1 + index2];
        val_mat{i_row} = [val_mat{i_row}, ...
            s^(3/2)*squeeze(d_term2_qby_for1(i_y,end,index1)).'];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,length(index1))];

        % s^(3/2) term2 d/dy V1
        i_col_mat{i_row} = [i_col_mat{i_row},...
            Np*Nz_1*Ny_1 + Np*Ny_1*(Nz_1 - 1) + mod(((i_y - 1)*Np + Np + 1 +...
            index2 - 1),Ny_1*Np) + 1,...
            Np*Nz_1*Ny_1 + Np*Ny_1*(Nz_1 - 1) + mod(((i_y - 1)*Np - Np + 1 + ...
            index2 - 1),Ny_1*Np) + 1];
        val_mat{i_row} = [val_mat{i_row}, s^(3/2)*[squeeze(term2_qby_for1(i_y,end,index1)).',...
            -squeeze(term2_qby_for1(i_y,end,index1)).']./(2*dy_for1)];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,2*length(index1))];

        % s^(3/2) d_term3_qby h1
        i_col_mat{i_row} = [i_col_mat{i_row}, Np*Ny_1*(3*Nz_1 + 1) + i_y];
        val_mat{i_row} = [val_mat{i_row}, s^(3/2)*d_term3_qby_for1(i_y,end,P+1)];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row];

        % s^(3/2) (-lam d_Up0_32 + term3_qby) d/dy h1
        i_col_mat{i_row} = [i_col_mat{i_row}...
            Np*Ny_1*(3*Nz_1 + 1) + mod((i_y + 1) - 1,Ny_1) + 1,...
            Np*Ny_1*(3*Nz_1 + 1) + mod((i_y - 1) - 1,Ny_1) + 1];
        val_mat{i_row} = [val_mat{i_row}, s^(3/2)*(-lam*d_Up0_32_for1(i_y,end,P+1) + ...
            term3_qby_for1(i_y,end,P+1))*[1,-1]./(2*dy_for1)];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,2)];

        % -s^(3/2) lam Up0_32 d^2/dz^2 h1
        i_col_mat{i_row} = [i_col_mat{i_row},...
            Np*Ny_1*(3*Nz_1 + 1) + mod((i_y + 1) - 1,Ny_1) + 1,...
            Np*Ny_1*(3*Nz_1 + 1) + mod((i_y) - 1,Ny_1) + 1,...
            Np*Ny_1*(3*Nz_1 + 1) + mod((i_y - 1) - 1,Ny_1) + 1];
        val_mat{i_row} = [val_mat{i_row}, -s^(3/2)*lam*...
            Up0_32_for1(i_y,end,P+1)*[1,-2,1]/dy_for1^2];
        i_row_mat{i_row} = [i_row_mat{i_row}, i_row*ones(1,3)];

    end

end

% horzcat(c{:});% concatenates all arrays in cell
i_col_vec = horzcat(i_col_mat{:});
val_vec = horzcat(val_mat{:});
i_row_vec = horzcat(i_row_mat{:});

M_sp = sparse(i_row_vec,i_col_vec,val_vec);
M_right_sp = sparse(Np*Ny_1*(3*Nz_1 + 1) + [1:Ny_1],...
    Np*Ny_1*(3*Nz_1 + 1) + [1:Ny_1],-ones(1,Ny_1));
fprintf('build sparse matrix: %.2f seconds. \n',toc')

%
sum_M = sum(abs(M_sp),2);
max_sum_M = max(sum_M);
%%
tic
[vec,val] = eigs(M_sp,M_right_sp,Ny_1,max_sum_M,'maxiterations',2000,'tol',1e-16);
fprintf('compute eigenvectors: %.2f seconds. \n',toc')

% https://en.wikipedia.org/wiki/Gershgorin_circle_theorem


% vec_mat(:,:,i_k) = vec;
% val_mat(:,i_k) = diag(val);

% end

val = diag(val);
h1 = vec(Np*Ny_1*(3*Nz_1 + 1) + 1:end,:);

%% plots

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
x_vec = [0:1:1000]; % meter
alpha_star = 1.56e-5; 
rho = 0.4; % porosity
% morpohlogical time scale
T_mor = L_star*H_star*(1-rho)./(alpha_star*(1020*H_star*U_star*omega)^(3/2))/(3600*24*365); 

growth_rate = real(val)/T_mor;
migr_rate = -imag(val)/T_mor./k*L_star;
%
%% plot all modes (works if Ny = 100)
set(figure,'position',[100,100,1800,1000])
Fnt = 12;

i_mode = 1;

for i_row = 1:10
    for i_col = 1:10
        bathy = 1*h1(:,i_mode).*exp(1i*k*x_vec/L_star);

        subplot('position',[...
            0.05 + 0.06*(i_row - 1) + 0.03*(i_row - 1),...
            1-(0.1 + 0.06*(i_col - 1) + 0.03*(i_col - 1)),0.06,0.06])
        surf(y*L_star/1000,x_vec/1000,(bathy + conj(bathy)).'/...
            max(max(abs((bathy + conj(bathy)).'))),'EdgeColor','interp')
        view(2)
        ylim([0 1])
        xlim([0 L_star*y(end)/1000])
        set(gca,'FontSize',Fnt,'TickLabelInterpreter','latex'), grid on
        ax = gca;
        box on
        ax.BoxStyle = 'full';
        ax.LineWidth = 2;
        set(gca,'Ydir','reverse')

        if i_col == 10
            xlabel('$y$ (km)','interpreter','latex')
        else
            xticks('')
        end
        if i_row == 1
            ylabel('$x$ ( km)','interpreter','latex')
        else
            yticks('')
        end

        i_mode = i_mode + 1;
    end
end

colorbar('ticklabelinterpreter','latex','position',[0.95 0.09 0.005 0.87]);
%% growth and migration rate of modes
set(figure,'position',[100 100 1000 500])
subplot('position',[0.1 0.15 0.35 0.8])
plot([1:Ny_1],growth_rate,'k.'), hold on, grid on
set(gca,'FontSize',Fnt,'TickLabelInterpreter','latex'), grid on
xlabel('Mode','interpreter','latex')
ylabel('Growth rate (1/yr)','interpreter','latex')

subplot('position',[0.6 0.15 0.35 0.8])
plot([1:Ny_1],migr_rate,'k.'), grid on
set(gca,'FontSize',Fnt,'TickLabelInterpreter','latex'), grid on
xlabel('Mode','interpreter','latex')
ylabel('Migration rate (m/yr)','interpreter','latex')

%% sand bank bathymetry and cross-bank derivative
set(figure,'position',[100 100 500 500])
subplot('position',[0.15 0.6 0.8 0.3])
plot(y*L_star/1000,-H*H_star,'k','linewidth',2), grid on
ylabel('$H(y)$ (m)','interpreter','latex')
set(gca,'FontSize',Fnt,'TickLabelInterpreter','latex'),

subplot('position',[0.15 0.15 0.8 0.3])
plot(y*L_star/1000,dH_dY*H_star/L_star,'k','linewidth',2), grid on
set(gca,'FontSize',Fnt,'TickLabelInterpreter','latex'),
xlabel('$y$ (km)','interpreter','latex')
ylabel('d$H$/d$y$ (--)','interpreter','latex')

%% residual velocities at seabed
figure
plot(y*L_star/1000,Up0(:,end,3)*U_star,'k','linewidth',2), grid on,hold on
plot(y*L_star/1000,Vp0(:,end,3)*U_star,'b','linewidth',2),
set(gca,'FontSize',Fnt,'TickLabelInterpreter','latex'),
xlabel('$y$ (km)','interpreter','latex')
ylabel('Seabed residual velocity (m/s)','interpreter','latex')
legend('$U_\mathrm{b}$','$V_\mathrm{b}$','interpreter','latex')

V_dav = (sum(Vp0(:,2:end-1,:),2) + 0.5*(Vp0(:,1,:) + Vp0(:,end,:)))*U_star*dz.*H*H_star./(H*H_star);
U_dav = (sum(Up0(:,2:end-1,:),2) + 0.5*(Up0(:,1,:) + Up0(:,end,:)))*U_star*dz.*H*H_star./(H*H_star);

%% residual depth-averaged velocities
set(figure,'position',[100 100 600 800])
subplot('position',[0.15 0.5 0.8 0.4])
plot(y*L_star/1000,U_dav(:,1,3),'LineWidth',2), grid on, hold on
plot(y*L_star/1000,V_dav(:,1,3),'LineWidth',2)
set(gca,'FontSize',Fnt,'TickLabelInterpreter','latex')
ylabel('Residual depth-averaged velocity (m/s)','interpreter','latex')
legend('U (along-bank, m/s)','V (cross-bank, m/s)','interpreter','latex')
xlim([y(1) y(end)]*L_star/1000)
ylim([-0.1 0.1])

subplot('position',[0.15 0.1 0.8 0.3])
plot(y*L_star/1000,-H*H_star,'k','linewidth',2), grid on
xlabel('$y$ (km)','interpreter','latex')
ylabel('$-H$','interpreter','latex')
set(gca,'FontSize',Fnt,'TickLabelInterpreter','latex')
xlim([y(1) y(end)]*L_star/1000)

%% M2 depth-averaged velocities
set(figure,'position',[100 100 500 400])
plot(y*L_star/1000,2*abs(U_dav(:,1,2)),'linewidth',2), hold on, grid on
plot(y*L_star/1000,2*abs(V_dav(:,1,2)),'linewidth',2), hold on
plot(y*L_star/1000,sqrt((2*abs(U_dav(:,1,2))).^2 + (2*abs(V_dav(:,1,2))).^2),...
    'linewidth',2)
xlabel('$y$ (km)','interpreter','latex')
ylabel('$|$M2 depth-averaged flow$|$','interpreter','latex')
set(gca,'FontSize',Fnt,'TickLabelInterpreter','latex')
legend('$U_\mathrm{M2}$','$V_\mathrm{M2}$','$|\vec{U}_\mathrm{M2}|$','interpreter','latex')
xlim([y(1) y(end)]*L_star/1000)

Ut = zeros(length(y),length(t)); Vt = zeros(length(y),length(t));

for i_t = 1:length(t)
    for i_p = 1:2*P+1
        Ut(:,i_t) = Ut(:,i_t) + U_dav(:,1,i_p)*exp(1i*p_vec(i_p)*t(i_t));
        Vt(:,i_t) = Vt(:,i_t) + V_dav(:,1,i_p)*exp(1i*p_vec(i_p)*t(i_t));
    end
end

y_index = [10 40 51 61 91];
set(figure,'position',[100 100 1000 500])
subplot('position',[0.1 0.15 0.8 0.2])
plot(y*L_star/1000,-H*H_star,'k','LineWidth',2), grid on
xlabel('$y$ (km)','interpreter','latex')
ylabel('$-H$ (m)','interpreter','latex')
set(gca,'FontSize',Fnt,'TickLabelInterpreter','latex')
xlim([y(1) y(end)]*L_star/1000), hold on
scatter(y(y_index)*L_star/1000,-H(y_index)*H_star,50,'r')

pos_vec = 0.1 + 0.17*[0:4];
for i_plot = 1:length(y_index)
    subplot('position',[pos_vec(i_plot) 0.5 0.12 0.45])
    plot(Vt(y_index(i_plot),:),Ut(y_index(i_plot),:),'k','LineWidth',2), grid on, hold on
    scatter(Vt(y_index(i_plot),1),Ut(y_index(i_plot),1),'red')
    scatter(Vt(y_index(i_plot),1000),Ut(y_index(i_plot),1000),'blue')
    xlim([-1 1])
    ylim([-1 1])
    set(gca,'Ydir','reverse') 
    set(gca,'FontSize',Fnt,'TickLabelInterpreter','latex')
    xlabel('$V$ (m/s)','interpreter','latex')
    if i_plot == 1
        ylabel('$U$ (m/s)','interpreter','latex')
    end
end









