% Accompanying code to 'Local characteristics of sand wave patterns are 
% governed by underlying sand bank: A linear stability analysis'
% by Laura Portos-Amill et al. 
% last edited on 28 October 2025

% semi-analytical solution to the basic order flow of flow over a sloping bed
% basic state means no variations in x direction
% x direction is along-bank, y direction is cross-bank
% all variables are dimensionless, transformed

% inputs:
% H: seabed profile, such that seabed is denoted by z = -H, can be linear
% profile or any other shape
% dH_dy: y derivative of the seabed (with respect to (gamma*y))
% Qp_00x, Qp_00y: Fourier components of the vertically averaged flow in the x,y 
% direction (uniform along the y direction). Used to compute local forcing.
% In the y direction it is zero everywhere for well-posedness of the first
% order problem. Vector of 2*P + 1 components, where P is the order at
% which the solution is truncated, thus Qp_00x = [0 1 0 1 0] gives an M2
% forcing but forces to truncate at M4 component
% f: scaled coriolis parameter
% dy: scaled grid spacing
% gamma_max: maximum order in gamma for which to solve, should be >= 2
% dz: scaled vertical spacing
% free_Uj: boolean defining if Uj (j>0) has free depth-integral (=1) or
% imposed to be zero (=0)
% s: scaled slip parameter
% Av: scaled eddy viscosity (constant and uniform)
% r: keulegan-carpenter number

% outputs:
% Up,Vp,Wp velocities in x,y,z direction respectively
% local forcings Fp, Gp

% dimensions are in Y,z,p,gamma-spaces, respectively

function [Up,Vp,Wp,Fp,Gp] = my_basicstate_transformed_anygamma(H,dH_dy,Qp_00x,Qp_00y,...
    f,dy,gamma_max,dz,free_Uj,s,Av,r)
%%
% compute 0 and first order in gamma
[Up_00,Vp_00,Up_01,Vp_01,Wp_01,Fp_0,Gp_0,Fp_1,...
    Gp_1,dR_dy_plus_00,dR_dy_min_00,dR_dz_plus_00,dR_dz_min_00] = ...
    my_basicstate_transformed(H,dH_dy,Qp_00x,Qp_00y,f,dy,dz,free_Uj,s,Av,r);

[~,~,Np] = size(Qp_00x);
P = floor(Np/2); %Fourier mode at which the solution is truncated

z = [0:-dz:-1]; % depth vector in transformed coordinate system

p_vec = zeros(1,1,2*P + 1); p_vec(1:2*P+1) = -P:P;
lam_plus = H.*sqrt(1i*(p_vec + f)./Av); lam_min = H.*sqrt(1i*(p_vec - f)./Av);

Up = zeros([size(Up_00) (gamma_max + 1)]); % 0,1,2,...,gamma_max
Vp = zeros([size(Up_00) (gamma_max + 1)]);
Wp = zeros([size(Up_00) (gamma_max + 1)]);
Fp = zeros([size(Fp_0) (gamma_max + 1)]);
Gp = zeros([size(Fp_0) (gamma_max + 1)]);

Up(:,:,:,1) = Up_00; Up(:,:,:,2) = Up_01;
Vp(:,:,:,1) = Vp_00; Vp(:,:,:,2) = Vp_01;
Wp(:,:,:,2) = Wp_01;
Fp(:,:,:,1) = Fp_0; Fp(:,:,:,2) = Fp_1;
Gp(:,:,:,1) = Gp_0; Gp(:,:,:,2) = Gp_1;

%compute derivatives
dR_dy_plus = zeros([size(Up_00) (gamma_max + 1)]);
dR_dy_min = zeros([size(Up_00) (gamma_max + 1)]);
dR_dz_plus = zeros([size(Up_00) (gamma_max + 1)]);
dR_dz_min = zeros([size(Up_00) (gamma_max + 1)]);

dR_dy_plus(:,:,:,1) = dR_dy_plus_00;
dR_dy_min(:,:,:,1) = dR_dy_min_00;
dR_dz_plus(:,:,:,1) = dR_dz_plus_00;
dR_dz_min(:,:,:,1) = dR_dz_min_00;

Rp_plus = Up_01 + 1i*Vp_01; Rp_min = Up_01 - 1i*Vp_01; % R values for different gamma's are not stored

dR_dy_plus(2:end-1,:,:,2) = (Rp_plus(3:end,:,:) - Rp_plus(1:end-2,:,:))./(2*dy);
dR_dy_plus(1,:,:,2) = (4*Rp_plus(2,:,:) - 3*Rp_plus(1,:,:) - Rp_plus(3,:,:))./(2.*dy); % second order accuracy derivative at boundaries
dR_dy_plus(end,:,:,2) = (Rp_plus(end-2,:,:) + 3*Rp_plus(end,:,:) - 4*Rp_plus(end-1,:,:))./(2*dy);

dR_dy_min(2:end-1,:,:,2) = (Rp_min(3:end,:,:) - Rp_min(1:end-2,:,:))./(2*dy);
dR_dy_min(1,:,:,2) = (4*Rp_min(2,:,:) - 3*Rp_min(1,:,:) - Rp_min(3,:,:))./(2.*dy); % second order accuracy derivative at boundaries
dR_dy_min(end,:,:,2) = (Rp_min(end-2,:,:) + 3*Rp_min(end,:,:) - 4*Rp_min(end-1,:,:))./(2*dy);

dR_dz_plus(:,2:end-1,:,2) = (Rp_plus(:,1:end-2,:) - Rp_plus(:,3:end,:))./(2.*dz);
dR_dz_plus(:,1,:,2) = (Rp_plus(:,3,:) + 3*Rp_plus(:,1,:) - 4*Rp_plus(:,2,:))./(2*dz); % second order accuracy derivative at boundaries
dR_dz_plus(:,end,:,2) = (4*Rp_plus(:,end-1,:) - 3*Rp_plus(:,end,:) - Rp_plus(:,end-2,:))./(2*dz);

dR_dz_min(:,2:end-1,:,2) = (Rp_min(:,1:end-2,:) - Rp_min(:,3:end,:))./(2.*dz);
dR_dz_min(:,1,:,2) = (Rp_min(:,3,:) + 3*Rp_min(:,1,:) - 4*Rp_min(:,2,:))./(2*dz); % second order accuracy derivative at boundaries
dR_dz_min(:,end,:,2) = (4*Rp_min(:,end-1,:) - 3*Rp_min(:,end,:) - Rp_min(:,end-2,:))./(2*dz);

%%

for j = 2:gamma_max % order in gamma that I'm solving (stored in position j+1)
    
    % ----------------------- solve for Wp,j ------------------------------
    dV_dy = zeros(size(Up_00));
    dV_dy(2:end-1,:,:) = (Vp(3:end,:,:,j) - Vp(1:end-2,:,:,j))./(2.*dy);
    dV_dy(1,:,:) = (4*Vp(2,:,:,j) - 3*Vp(1,:,:,j) - Vp(3,:,:,j))./(2*dy); % second order accuracy derivative at boundaries
    dV_dy(end,:,:) = (Vp(end-2,:,:,j) + 3*Vp(end,:,:,j) - 4*Vp(end-1,:,:,j))./(2*dy);
    
    dV_dz = zeros(size(Up_00));
    dV_dz(:,2:end-1,:) = (Vp(:,1:end-2,:,j) - Vp(:,3:end,:,j))./(2.*dz);
    dV_dz(:,1,:) = (Vp(:,3,:,j) + 3*Vp(:,1,:,j) - 4*Vp(:,2,:,j))./(2*dz); % second order accuracy derivative at boundaries
    dV_dz(:,end,:) = (4*Vp(:,end-1,:,j) - 3*Vp(:,end,:,j) - Vp(:,end-2,:,j))./(2*dz);
    
    func = -H.*dV_dy + z.*dH_dy.*dV_dz;
    
    Wp(:,end,:,j+1) = -Vp(:,end,:,j).*dH_dy;
    
    for i_z = length(z)-1:-1:1
        Wp(:,i_z,:,j+1) = Wp(:,i_z+1,:,j+1) + 0.5*(func(:,i_z+1,:) + func(:,i_z,:)).*dz;
    end

    % ----------------------- solve for U,V p,j ---------------------------
    % compute forcings
    F_vy_plus = zeros(size(Up_00)); % H^2/Av*{V*dR/dy}
    F_vz_plus = zeros(size(Up_00)); % H^2/Av*H'/H*z*{V*dR/dz}
    F_wz_plus = zeros(size(Up_00)); % H^2/Av*{W/H*dR/dz}
    
    F_vy_min = zeros(size(Up_00)); % H^2/Av*{V*dR/dy}
    F_vz_min = zeros(size(Up_00)); % H^2/Av*H'/H*z*{V*dR/dz}
    F_wz_min = zeros(size(Up_00)); % H^2/Av*{W/H*dR/dz}

    for i_j = 1:j+1
        for i_p = 1:2*P + 1
            index_p1 = max(1,i_p-P):1:min(i_p+P,2*P+1);
            index_p2 = min(i_p+P,2*P+1):-1:max(1,i_p-P);

            if i_j == j + 1 % only the Fwz forcing is order j
                F_wz_plus(:,:,i_p) = F_wz_plus(:,:,i_p) + ...
                    r*H./Av.*sum(Wp(:,:,index_p1,i_j).*dR_dz_plus(:,:,index_p2,j+1-i_j+1),3);
                F_wz_min(:,:,i_p) = F_wz_min(:,:,i_p) + ...
                    r*H./Av.*sum(Wp(:,:,index_p1,i_j).*dR_dz_min(:,:,index_p2,j+1-i_j+1),3);
            else
                F_vy_plus(:,:,i_p) = F_vy_plus(:,:,i_p) + ...
                    r*H.^2./Av.*sum(Vp(:,:,index_p1,i_j).*dR_dy_plus(:,:,index_p2,j+1-i_j),3);
                F_vy_min(:,:,i_p) = F_vy_min(:,:,i_p) +...
                    r*H.^2./Av.*sum(Vp(:,:,index_p1,i_j).*dR_dy_min(:,:,index_p2,j+1-i_j),3);
            
                F_vz_plus(:,:,i_p) = F_vz_plus(:,:,i_p) + ...
                    r*H.^2./Av.*z.*dH_dy./H.*sum(Vp(:,:,index_p1,i_j).*dR_dz_plus(:,:,index_p2,j+1-i_j),3);
                F_vz_min(:,:,i_p) = F_vz_min(:,:,i_p) + ...
                    r*H.^2./Av.*z.*dH_dy./H.*sum(Vp(:,:,index_p1,i_j).*dR_dz_min(:,:,index_p2,j+1-i_j),3);
            
                F_wz_plus(:,:,i_p) = F_wz_plus(:,:,i_p) + ...
                    r*H./Av.*sum(Wp(:,:,index_p1,i_j).*dR_dz_plus(:,:,index_p2,j+1-i_j+1),3);
                F_wz_min(:,:,i_p) = F_wz_min(:,:,i_p) + ...
                    r*H./Av.*sum(Wp(:,:,index_p1,i_j).*dR_dz_min(:,:,index_p2,j+1-i_j+1),3);
            end
        end
    end

    %solve particular solution for Rp,j with Runge Kutta
    Rp_plus_part = zeros(size(Up_00)); Rp_min_part = zeros(size(Up_00));
    Dp_plus_part = zeros(size(Up_00)); Dp_min_part = zeros(size(Up_00));
    % boundary condition at z = 0, nonzero for R, zero for D
    Rp_min_part(:,1,:) = 1; Rp_plus_part(:,1,:) = 1;

    for i_z = 2:length(z)
        %plus
        k1_D_plus = lam_plus.^2.*Rp_plus_part(:,i_z-1,:) + ...
            F_vy_plus(:,i_z-1,:) - F_vz_plus(:,i_z-1,:) + F_wz_plus(:,i_z-1,:);
        k1_R_plus = Dp_plus_part(:,i_z-1,:);
    
        k2_D_plus = lam_plus.^2.*(Rp_plus_part(:,i_z-1,:) - dz.*0.5.*k1_R_plus) + ...
            0.5*(F_vy_plus(:,i_z-1,:) - F_vz_plus(:,i_z-1,:) + F_wz_plus(:,i_z-1,:) + ...
            F_vy_plus(:,i_z,:) - F_vz_plus(:,i_z,:) + F_wz_plus(:,i_z,:));
        k2_R_plus = Dp_plus_part(:,i_z-1,:) - dz.*0.5.*k1_D_plus;
    
        k3_D_plus = lam_plus.^2.*(Rp_plus_part(:,i_z-1,:) - dz.*0.5.*k2_R_plus) + ...
            0.5*(F_vy_plus(:,i_z-1,:) - F_vz_plus(:,i_z-1,:) + F_wz_plus(:,i_z-1,:) + ...
            F_vy_plus(:,i_z,:) - F_vz_plus(:,i_z,:) + F_wz_plus(:,i_z,:));
        k3_R_plus = Dp_plus_part(:,i_z-1,:) - dz.*0.5.*k2_D_plus;
    
        k4_D_plus = lam_plus.^2.*(Rp_plus_part(:,i_z-1,:) - dz.*k3_R_plus) + ...
            F_vy_plus(:,i_z,:) - F_vz_plus(:,i_z,:) + F_wz_plus(:,i_z,:);
        k4_R_plus = Dp_plus_part(:,i_z-1,:) - dz.*k3_D_plus;
    
        Rp_plus_part(:,i_z,:) = Rp_plus_part(:,i_z-1,:) - dz./6.*...
            (k1_R_plus + 2.*k2_R_plus + 2.*k3_R_plus + k4_R_plus);
        Dp_plus_part(:,i_z,:) = Dp_plus_part(:,i_z-1,:) - dz./6.*...
            (k1_D_plus + 2.*k2_D_plus + 2.*k3_D_plus + k4_D_plus);
    
    
        %min
        k1_D_min = lam_min.^2.*Rp_min_part(:,i_z-1,:) + ...
            F_vy_min(:,i_z-1,:) - F_vz_min(:,i_z-1,:) + F_wz_min(:,i_z-1,:);
        k1_R_min = Dp_min_part(:,i_z-1,:);
    
        k2_D_min = lam_min.^2.*(Rp_min_part(:,i_z-1,:) - dz.*0.5.*k1_R_min) + ...
            0.5*(F_vy_min(:,i_z-1,:) - F_vz_min(:,i_z-1,:) + F_wz_min(:,i_z-1,:) + ...
            F_vy_min(:,i_z,:) - F_vz_min(:,i_z,:) + F_wz_min(:,i_z,:));
        k2_R_min = Dp_min_part(:,i_z-1,:) - dz.*0.5.*k1_D_min;
    
        k3_D_min = lam_min.^2.*(Rp_min_part(:,i_z-1,:) - dz.*0.5.*k2_R_min) + ...
            0.5*(F_vy_min(:,i_z-1,:) - F_vz_min(:,i_z-1,:) + F_wz_min(:,i_z-1,:) + ...
            F_vy_min(:,i_z,:) - F_vz_min(:,i_z,:) + F_wz_min(:,i_z,:));
        k3_R_min = Dp_min_part(:,i_z-1,:) - dz.*0.5.*k2_D_min;
    
        k4_D_min = lam_min.^2.*(Rp_min_part(:,i_z-1,:) - dz.*k3_R_min) + ...
            F_vy_min(:,i_z,:) - F_vz_min(:,i_z,:) + F_wz_min(:,i_z,:);
        k4_R_min = Dp_min_part(:,i_z-1,:) - dz.*k3_D_min;
    
        Rp_min_part(:,i_z,:) = Rp_min_part(:,i_z-1,:) - dz./6.*...
            (k1_R_min + 2.*k2_R_min + 2.*k3_R_min + k4_R_min);
        Dp_min_part(:,i_z,:) = Dp_min_part(:,i_z-1,:) - dz./6.*...
            (k1_D_min + 2.*k2_D_min + 2.*k3_D_min + k4_D_min);
    end
    
    %find scaling constant for homog sol
    
    alpha_plus = -(s.*Rp_plus_part(:,end,:) - Av./H.*Dp_plus_part(:,end,:))./...
        (Av./H.*lam_plus.*sinh(lam_plus) + s.*cosh(lam_plus));
    alpha_min = -(s.*Rp_min_part(:,end,:) - Av./H.*Dp_min_part(:,end,:))./...
        (Av./H.*lam_min.*sinh(lam_min) + s.*cosh(lam_min));



    % find F that is used in analytical solution
    if free_Uj
    
        Gp(:,:,:,j+1) = -1./(1./(p_vec + f).*(-1 + 1./lam_plus.*sinh(lam_plus)./...
            (cosh(lam_plus) + Av.*lam_plus./(H.*s).*sinh(lam_plus))) + ...
            1./(p_vec - f).*(-1 + 1./lam_min.*sinh(lam_min)./...
            (cosh(lam_min) + Av.*lam_min./(H.*s).*sinh(lam_min)))).*...
            (dz.*(sum(Rp_plus_part(:,2:end-1,:),2) + 0.5.*(Rp_plus_part(:,1,:) + ...
            Rp_plus_part(:,end,:))) + alpha_plus./lam_plus.*sinh(lam_plus) - ...
            (dz.*(sum(Rp_min_part(:,2:end-1,:),2) + 0.5.*(Rp_min_part(:,1,:) + ...
            Rp_min_part(:,end,:))) + alpha_min./lam_min.*sinh(lam_min)));
    
        Fp_plus = 1i.*Gp(:,:,:,j+1);
    
        Fp_min = -1i.*Gp(:,:,:,j+1);
    
        Fp(:,:,:,j+1) = zeros(size(Gp(:,:,:,j+1)));
    
    
    else % zero depth-integral
    
        Fp_plus = -1i.*(p_vec + f)./(-1 + 1./lam_plus.*sinh(lam_plus)./...
            (cosh(lam_plus) + Av.*lam_plus./(H.*s).*sinh(lam_plus))).*...
            (dz.*(sum(Rp_plus_part(:,2:end-1,:),2) + 0.5.*(Rp_plus_part(:,1,:) + ...
            Rp_plus_part(:,end,:))) + alpha_plus./lam_plus.*sinh(lam_plus));
        
        Fp_min = -1i.*(p_vec - f)./(-1 + 1./lam_min.*sinh(lam_min)./...
            (cosh(lam_min) + Av.*lam_min./(H.*s).*sinh(lam_min))).*...
            (dz.*(sum(Rp_min_part(:,2:end-1,:),2) + 0.5.*(Rp_min_part(:,1,:) + ...
            Rp_min_part(:,end,:))) + alpha_min./lam_min.*sinh(lam_min));
        
        Fp(:,:,:,j+1) = (Fp_plus + Fp_min)/2;
        Gp(:,:,:,j+1) = (Fp_plus - Fp_min)/(2*1i);
    
    end

    Rp_plus = Fp_plus./(1i.*(p_vec + f)).*(-1 + cosh(lam_plus.*z)./...
        (cosh(lam_plus) + Av.*lam_plus./(H.*s).*sinh(lam_plus))) + ...
        Rp_plus_part + alpha_plus.*cosh(lam_plus.*z);
    Rp_min = Fp_min./(1i.*(p_vec - f)).*(-1 + cosh(lam_min.*z)./...
        (cosh(lam_min) + Av.*lam_min./(H.*s).*sinh(lam_min))) + ...
        Rp_min_part + alpha_min.*cosh(lam_min.*z);
    
    %compute velocities
    Up(:,:,:,j+1) = (Rp_plus + Rp_min)./2;
    Vp(:,:,:,j+1) = (Rp_plus - Rp_min)./(2*1i);

    dR_dy_plus(2:end-1,:,:,j+1) = (Rp_plus(3:end,:,:) - Rp_plus(1:end-2,:,:))./(2*dy);
    dR_dy_plus(1,:,:,j+1) = (4*Rp_plus(2,:,:) - 3*Rp_plus(1,:,:) - Rp_plus(3,:,:))./(2*dy); % second order accuracy derivative at boundaries
    dR_dy_plus(end,:,:,j+1) = (Rp_plus(end-2,:,:) + 3*Rp_plus(end,:,:) - 4*Rp_plus(end-1,:,:))./(2*dy);
    
    dR_dy_min(2:end-1,:,:,j+1) = (Rp_min(3:end,:,:) - Rp_min(1:end-2,:,:))./(2*dy);
    dR_dy_min(1,:,:,j+1) = (4*Rp_min(2,:,:) - 3*Rp_min(1,:,:) - Rp_min(3,:,:))./(2*dy); % second order accuracy derivative at boundaries
    dR_dy_min(end,:,:,j+1) = (Rp_min(end-2,:,:) + 3*Rp_min(end,:,:) - 4*Rp_min(end-1,:,:))./(2*dy);
    
    dR_dz_plus(:,2:end-1,:,j+1) = (Rp_plus(:,1:end-2,:) - Rp_plus(:,3:end,:))./(2.*dz);
    dR_dz_plus(:,1,:,j+1) = (Rp_plus(:,3,:) + 3*Rp_plus(:,1,:) - 4*Rp_plus(:,2,:))./(2*dz); % second order accuracy derivative at boundaries
    dR_dz_plus(:,end,:,j+1) = (4*Rp_plus(:,end-1) - 3*Rp_plus(:,end,:) - Rp_plus(:,end-2,:))./(2*dz);
    
    dR_dz_min(:,2:end-1,:,j+1) = (Rp_min(:,1:end-2,:) - Rp_min(:,3:end,:))./(2.*dz);
    dR_dz_min(:,1,:,j+1) = (Rp_min(:,3,:) + 3*Rp_min(:,1,:) - 4*Rp_min(:,2,:))./(2*dz); % second order accuracy derivative at boundaries
    dR_dz_min(:,end,:,j+1) = (4*Rp_min(:,end-1) - 3*Rp_min(:,end,:) - Rp_min(:,end-2,:))./(2*dz);

end

end























