% Accompanying code to 'Local characteristics of sand wave patterns are 
% governed by underlying sand bank: A linear stability analysis'
% by Laura Portos-Amill et al. 
% last edited on 28 October 2025

% analytical solution to the basic order flow of flow over a sloping bed
% basic state means no variations in x direction
% x direction is alongshore, y direction is cross-shore pointing seaward
% all variables are dimensionless, transformed

% inputs:
% H: seabed profile, such that seabed is denoted by z = -H, can be linear
% profile or any other shape
% dH_dy: y derivative of the seabed (with respect to (gamma*y))
% Qp_00x, Qp_00y: Fourier components of the vertically averaged flow in the x,y 
% direction (uniform along the y direction). Used to compute local forcing.
% In the y direction it is zero everywhere for well-posedness of the first
% order problem. Vector of 2*P + 1 components, where P is the order at
% which the solution is truncated, thus Qp_00x = [0 1 0 1 0] gives an M2
% forcing but forces to truncate at M4 component
% f: scaled coriolis parameter
% dy: scaled grid spacing
% dz: scaled vertical spacing
% free_Uj: boolean defining if Uj (j>0) has free depth-integral (=1) or
% imposed to be zero (=0)
% s: scaled slip parameter
% Av: scaled eddy viscosity (constant and uniform)
% r: keulegan-carpenter number

% outputs:
% Up_00, Vp_00: zero order (in epsilon and gamma) flow velocities in
% Fourier space
% Ap_plus, Ap_min: variables used to solve system of equations for the
% local forcing. Also used to solve first order flow in gamma
% Wp_01: vertical velocity for zero order in epsilon, first order in gamma

% dimensions are in Y,z,p-spaces, respectively

function [Up_00,Vp_00,Up_01,Vp_01,Wp_01,Fp_0,Gp_0,...
    Fp_1,Gp_1,dR_dy_plus_00,dR_dy_min_00,dR_dz_plus_00,...
    dR_dz_min_00] = my_basicstate_transformed(H,dH_dy,Qp_00x,Qp_00y,f,dy,dz,free_Uj,s,Av,r)

%% environmental/numerical parameters

[~,~,Np] = size(Qp_00x);
P = floor(Np/2); %Fourier mode at which the solution is truncated

z = [0:-dz:-1]; % depth vector in transformed coordinate system

% forcings in y direction, found after imposing depth-averaged transports
Fp_0 = zeros(length(H),1,2*P + 1);
Gp_0 = zeros(length(H),1,2*P + 1);

%rotary components
Rp_plus_00 = zeros(length(H),length(z),2*P + 1);
Rp_min_00 = zeros(length(H),length(z),2*P + 1);

Up_00 = zeros(length(H),length(z),2*P + 1);
Vp_00 = zeros(length(H),length(z),2*P + 1);

%% solve zero order in gamma
% find Ap_pm, Fp_0, Gp_0, Rp_pm, Up

p_vec = zeros(1,1,2*P + 1); p_vec(1:2*P+1) = -P:P;
lam_plus = H.*sqrt(1i*(p_vec + f)./Av); lam_min = H.*sqrt(1i*(p_vec - f)./Av);

Ap_plus = H./(2*1i*(p_vec + f)).*(-1 + 1./lam_plus.*sinh(lam_plus)./ ...
    (cosh(lam_plus) + lam_plus.*Av./(s.*H).*sinh(lam_plus)));
Ap_min = H./(2*1i*(p_vec - f)).*(-1 + 1./lam_min.*sinh(lam_min)./ ...
    (cosh(lam_min) + lam_min.*Av./(s.*H).*sinh(lam_min)));

% matrix defining the system of equations to solve for F,G
M11 = Ap_plus + Ap_min; M12 = 1i*Ap_plus - 1i*Ap_min;
M21 = Ap_plus./1i - Ap_min./1i; M22 = Ap_plus + Ap_min;

%compute forcing
Fp_0 = (Qp_00y - M22./M12.*Qp_00x)./(M21 - M11./M12.*M22);
Gp_0 = Qp_00x./M12 - M11./M12.*Fp_0; %(-M21.*Qp_00x + M11.*Qp_00y)./(M11.*M22 - M12.*M21); 

Fp_0_plus = Fp_0 + 1i*Gp_0; Fp_0_min = Fp_0 - 1i*Gp_0;

Rp_plus_00 = (Fp_0 + 1i*Gp_0)./(1i*(p_vec + f)).*(-1 + ...
    cosh(lam_plus.*z)./...
    (cosh(lam_plus) + lam_plus.*Av./(s.*H).*sinh(lam_plus)));
Rp_min_00 = (Fp_0 - 1i*Gp_0)./(1i*(p_vec - f)).*(-1 + ...
    cosh(lam_min.*z)./...
    (cosh(lam_min) + lam_min.*Av./(s.*H).*sinh(lam_min)));

%compute velocities
Up_00 = (Rp_plus_00 + Rp_min_00)./2;
Vp_00 = (Rp_plus_00 - Rp_min_00)./(2*1i);

%% ------------------ solve first order in gamma --------------------------
%% .......................... solve for W01 ...............................

Wp_01 = zeros(length(H),length(z),2*P + 1);

dAp_plus_dy = dH_dy./(2*1i*(p_vec + f)).*(-1 + ...
    (cosh(lam_plus).^2 - sinh(lam_plus).^2)./...
    (cosh(lam_plus) + Av.*lam_plus./(H.*s).*sinh(lam_plus)).^2);

dAp_min_dy = dH_dy./(2*1i*(p_vec - f)).*(-1 + ...
    (cosh(lam_min).^2 - sinh(lam_min).^2)./...
    (cosh(lam_min) + Av.*lam_min./(H.*s).*sinh(lam_min)).^2);

Wp_01(:,:,:) = -Vp_00(:,end,:).*dH_dy + ... %%%%
    H./(2*(p_vec + f)).*Fp_0_plus./Ap_plus.*dAp_plus_dy.*(z + 1 - 1./lam_plus.*...
    (sinh(lam_plus.*z) + sinh(lam_plus))./...
    (cosh(lam_plus) + Av.*lam_plus./(H*s).*sinh(lam_plus))) + ...
    Fp_0_plus./(2.*(p_vec + f)).*dH_dy.*(1./...
    (cosh(lam_plus) + Av.*lam_plus./(H.*s).*sinh(lam_plus)).*...
    (z.*cosh(lam_plus.*z) - 1./lam_plus.*sinh(lam_plus.*z) + cosh(lam_plus) - ...
    1./lam_plus.*sinh(lam_plus)) - ...
    (sinh(lam_plus) + Av.*lam_plus./(H.*s).*cosh(lam_plus))./...
    (cosh(lam_plus) + Av.*lam_plus./(H.*s).*sinh(lam_plus)).^2.*...
    (sinh(lam_plus.*z) + sinh(lam_plus))) - ...
    H./(2*(p_vec - f)).*Fp_0_min./Ap_min.*dAp_min_dy.*(z + 1 - 1./lam_min.*...
    (sinh(lam_min.*z) + sinh(lam_min))./...
    (cosh(lam_min) + Av.*lam_min./(H*s).*sinh(lam_min))) - ...
    Fp_0_min./(2.*(p_vec - f)).*dH_dy.*(1./...
    (cosh(lam_min) + Av.*lam_min./(H.*s).*sinh(lam_min)).*...
    (z.*cosh(lam_min.*z) - 1./lam_min.*sinh(lam_min.*z) + cosh(lam_min) - ...%%
    1./lam_min.*sinh(lam_min)) - ...
    (sinh(lam_min) + Av.*lam_min./(H.*s).*cosh(lam_min))./...
    (cosh(lam_min) + Av.*lam_min./(H.*s).*sinh(lam_min)).^2.*...
    (sinh(lam_min.*z) + sinh(lam_min))) + ...
    dH_dy./2.*(-Fp_0_plus./(p_vec+f).*(z.*cosh(lam_plus.*z) + cosh(lam_plus) - ...
    1./lam_plus.*(sinh(lam_plus.*z) + sinh(lam_plus)))./...
    (cosh(lam_plus) + Av.*lam_plus./(H.*s).*sinh(lam_plus)) + ...
    Fp_0_min./(p_vec-f).*(z.*cosh(lam_min.*z) + cosh(lam_min) - ...
    1./lam_min.*(sinh(lam_min.*z) + sinh(lam_min)))./...
    (cosh(lam_min) + Av.*lam_min./(H.*s).*sinh(lam_min)));

%%  .................... solve for U01, V01 ...............................
%% compute forcings

F_vy_plus_01 = zeros(size(Wp_01)); % H^2/Av*V_00*dR0/dy
F_vz_plus_01 = zeros(size(Wp_01)); % H^2/Av*V_00*z*dR0/dz*H'/H
F_wz_plus_01 = zeros(size(Wp_01)); %H^2/Av*W_01/H*dR0/dz

F_vy_min_01 = zeros(size(Wp_01)); % H^2/Av*V_00*dR0/dy
F_vz_min_01 = zeros(size(Wp_01)); % H^2/Av*V_00*z*dR0/dz*H'/H
F_wz_min_01 = zeros(size(Wp_01)); %H^2/Av*W_01/H*dR0/dz

dR_dy_plus_00 = -1./(1i.*(p_vec + f)).*Fp_0_plus./Ap_plus.*dAp_plus_dy.*...
    (-1 + cosh(lam_plus.*z)./(cosh(lam_plus) + Av.*lam_plus./(H.*s).*sinh(lam_plus))) + ...
    Fp_0_plus./(1i.*(p_vec + f)).*lam_plus./H.*dH_dy.*...
    (z.*sinh(lam_plus.*z).*(cosh(lam_plus) + Av.*lam_plus./(H.*s).*sinh(lam_plus)) - ...
    cosh(lam_plus.*z).*(sinh(lam_plus) + Av.*lam_plus./(H.*s).*cosh(lam_plus)))./...
    (cosh(lam_plus) + Av.*lam_plus./(H.*s).*sinh(lam_plus)).^2;

dR_dy_min_00 = -1./(1i.*(p_vec - f)).*Fp_0_min./Ap_min.*dAp_min_dy.*...
    (-1 + cosh(lam_min.*z)./(cosh(lam_min) + Av.*lam_min./(H.*s).*sinh(lam_min))) + ...
    Fp_0_min./(1i.*(p_vec - f)).*lam_min./H.*dH_dy.*...
    (z.*sinh(lam_min.*z).*(cosh(lam_min) + Av.*lam_min./(H.*s).*sinh(lam_min)) - ...
    cosh(lam_min.*z).*(sinh(lam_min) + Av.*lam_min./(H.*s).*cosh(lam_min)))./...
    (cosh(lam_min) + Av.*lam_min./(H.*s).*sinh(lam_min)).^2;

dR_dz_plus_00 = Fp_0_plus./(1i.*(p_vec + f)).*lam_plus.*sinh(lam_plus.*z)./...
    (cosh(lam_plus) + Av.*lam_plus./(H.*s).*sinh(lam_plus));

dR_dz_min_00 = Fp_0_min./(1i.*(p_vec - f)).*lam_min.*sinh(lam_min.*z)./...
    (cosh(lam_min) + Av.*lam_min./(H.*s).*sinh(lam_min));

for i_p = 1:2*P + 1
    index1 = max(1,i_p-P):1:min(i_p+P,2*P+1);
    index2 = min(i_p+P,2*P+1):-1:max(1,i_p-P);

    F_vy_plus_01(:,:,i_p) = r*H.^2./Av.*sum(Vp_00(:,:,index1).*dR_dy_plus_00(:,:,index2),3);
    F_vy_min_01(:,:,i_p) = r*H.^2./Av.*sum(Vp_00(:,:,index1).*dR_dy_min_00(:,:,index2),3);

    F_vz_plus_01(:,:,i_p) = r*H.^2./Av.*sum(Vp_00(:,:,index1).*z.*dH_dy./H.*dR_dz_plus_00(:,:,index2),3);
    F_vz_min_01(:,:,i_p) = r*H.^2./Av.*sum(Vp_00(:,:,index1).*z.*dH_dy./H.*dR_dz_min_00(:,:,index2),3);

    F_wz_plus_01(:,:,i_p) = r*H.^2./Av.*sum(Wp_01(:,:,index1)./H.*dR_dz_plus_00(:,:,index2),3);
    F_wz_min_01(:,:,i_p) = r*H.^2./Av.*sum(Wp_01(:,:,index1)./H.*dR_dz_min_00(:,:,index2),3);
end

%% solve U01 V01 numerical part

% vectors for particular sols for R01plus,R01min
Rp_plus_01_part = zeros(size(Up_00)); Rp_min_01_part = zeros(size(Up_00));
Dp_plus_01_part = zeros(size(Up_00)); Dp_min_01_part = zeros(size(Up_00));

% vectors to scale 
alpha_plus = zeros([length(H),1,2*P+1]); alpha_min = zeros([length(H),1,2*P+1]);

% k used in Runge-kutta
k1_R_plus = zeros([length(H),1,length(p_vec)]);
k1_D_plus = zeros([length(H),1,length(p_vec)]);
k1_R_min = zeros([length(H),1,length(p_vec)]);
k1_D_min = zeros([length(H),1,length(p_vec)]);

k2_R_plus = zeros([length(H),1,length(p_vec)]);
k2_D_plus = zeros([length(H),1,length(p_vec)]);
k2_R_min = zeros([length(H),1,length(p_vec)]);
k2_D_min = zeros([length(H),1,length(p_vec)]);

k3_R_plus = zeros([length(H),1,length(p_vec)]);
k3_D_plus = zeros([length(H),1,length(p_vec)]);
k3_R_min = zeros([length(H),1,length(p_vec)]);
k3_D_min = zeros([length(H),1,length(p_vec)]);

k4_R_plus = zeros([length(H),1,length(p_vec)]);
k4_D_plus = zeros([length(H),1,length(p_vec)]);
k4_R_min = zeros([length(H),1,length(p_vec)]);
k4_D_min = zeros([length(H),1,length(p_vec)]);

% boundary condition at z = 0, nonzero for R, zero for D
Rp_min_01_part(:,1,:) = 1; Rp_plus_01_part(:,1,:) = 1;

%solve particular solution numerically with Runge Kutta
for i_z = 2:length(z)
    %plus
    k1_D_plus = lam_plus.^2.*Rp_plus_01_part(:,i_z-1,:) + ...
        F_vy_plus_01(:,i_z-1,:) - F_vz_plus_01(:,i_z-1,:) + F_wz_plus_01(:,i_z-1,:);
    k1_R_plus = Dp_plus_01_part(:,i_z-1,:);

    k2_D_plus = lam_plus.^2.*(Rp_plus_01_part(:,i_z-1,:) - dz.*0.5.*k1_R_plus) + ...
        0.5*(F_vy_plus_01(:,i_z-1,:) - F_vz_plus_01(:,i_z-1,:) + F_wz_plus_01(:,i_z-1,:) + ...
        F_vy_plus_01(:,i_z,:) - F_vz_plus_01(:,i_z,:) + F_wz_plus_01(:,i_z,:));
    k2_R_plus = Dp_plus_01_part(:,i_z-1,:) - dz.*0.5.*k1_D_plus;

    k3_D_plus = lam_plus.^2.*(Rp_plus_01_part(:,i_z-1,:) - dz.*0.5.*k2_R_plus) + ...
        0.5*(F_vy_plus_01(:,i_z-1,:) - F_vz_plus_01(:,i_z-1,:) + F_wz_plus_01(:,i_z-1,:) + ...
        F_vy_plus_01(:,i_z,:) - F_vz_plus_01(:,i_z,:) + F_wz_plus_01(:,i_z,:));
    k3_R_plus = Dp_plus_01_part(:,i_z-1,:) - dz.*0.5.*k2_D_plus;

    k4_D_plus = lam_plus.^2.*(Rp_plus_01_part(:,i_z-1,:) - dz.*k3_R_plus) + ...
        F_vy_plus_01(:,i_z,:) - F_vz_plus_01(:,i_z,:) + F_wz_plus_01(:,i_z,:);
    k4_R_plus = Dp_plus_01_part(:,i_z-1,:) - dz.*k3_D_plus;

    Rp_plus_01_part(:,i_z,:) = Rp_plus_01_part(:,i_z-1,:) - dz./6.*...
        (k1_R_plus + 2.*k2_R_plus + 2.*k3_R_plus + k4_R_plus);
    Dp_plus_01_part(:,i_z,:) = Dp_plus_01_part(:,i_z-1,:) - dz./6.*...
        (k1_D_plus + 2.*k2_D_plus + 2.*k3_D_plus + k4_D_plus);


    %min
    k1_D_min = lam_min.^2.*Rp_min_01_part(:,i_z-1,:) + ...
        F_vy_min_01(:,i_z-1,:) - F_vz_min_01(:,i_z-1,:) + F_wz_min_01(:,i_z-1,:);
    k1_R_min = Dp_min_01_part(:,i_z-1,:);

    k2_D_min = lam_min.^2.*(Rp_min_01_part(:,i_z-1,:) - dz.*0.5.*k1_R_min) + ...
        0.5*(F_vy_min_01(:,i_z-1,:) - F_vz_min_01(:,i_z-1,:) + F_wz_min_01(:,i_z-1,:) + ...
        F_vy_min_01(:,i_z,:) - F_vz_min_01(:,i_z,:) + F_wz_min_01(:,i_z,:));
    k2_R_min = Dp_min_01_part(:,i_z-1,:) - dz.*0.5.*k1_D_min;

    k3_D_min = lam_min.^2.*(Rp_min_01_part(:,i_z-1,:) - dz.*0.5.*k2_R_min) + ...
        0.5*(F_vy_min_01(:,i_z-1,:) - F_vz_min_01(:,i_z-1,:) + F_wz_min_01(:,i_z-1,:) + ...
        F_vy_min_01(:,i_z,:) - F_vz_min_01(:,i_z,:) + F_wz_min_01(:,i_z,:));
    k3_R_min = Dp_min_01_part(:,i_z-1,:) - dz.*0.5.*k2_D_min;

    k4_D_min = lam_min.^2.*(Rp_min_01_part(:,i_z-1,:) - dz.*k3_R_min) + ...
        F_vy_min_01(:,i_z,:) - F_vz_min_01(:,i_z,:) + F_wz_min_01(:,i_z,:);
    k4_R_min = Dp_min_01_part(:,i_z-1,:) - dz.*k3_D_min;

    Rp_min_01_part(:,i_z,:) = Rp_min_01_part(:,i_z-1,:) - dz./6.*...
        (k1_R_min + 2.*k2_R_min + 2.*k3_R_min + k4_R_min);
    Dp_min_01_part(:,i_z,:) = Dp_min_01_part(:,i_z-1,:) - dz./6.*...
        (k1_D_min + 2.*k2_D_min + 2.*k3_D_min + k4_D_min);
end

%find scaling constant for homog sol

alpha_plus = -(s.*Rp_plus_01_part(:,end,:) - Av./H.*Dp_plus_01_part(:,end,:))./...
    (Av./H.*lam_plus.*sinh(lam_plus) + s.*cosh(lam_plus));
alpha_min = -(s.*Rp_min_01_part(:,end,:) - Av./H.*Dp_min_01_part(:,end,:))./...
    (Av./H.*lam_min.*sinh(lam_min) + s.*cosh(lam_min));

%% find analytical sol (depending on Fp_01)

if free_Uj % depth integral of U is free, depth-integral of V is zero

    Gp_1 = -1./(1./(p_vec + f).*(-1 + 1./lam_plus.*sinh(lam_plus)./...
        (cosh(lam_plus) + Av.*lam_plus./(H.*s).*sinh(lam_plus))) + ...
        1./(p_vec - f).*(-1 + 1./lam_min.*sinh(lam_min)./...
        (cosh(lam_min) + Av.*lam_min./(H.*s).*sinh(lam_min)))).*...
        (dz.*(sum(Rp_plus_01_part(:,2:end-1,:),2) + 0.5.*(Rp_plus_01_part(:,1,:) + ...
        Rp_plus_01_part(:,end,:))) + alpha_plus./lam_plus.*sinh(lam_plus) - ...
        (dz.*(sum(Rp_min_01_part(:,2:end-1,:),2) + 0.5.*(Rp_min_01_part(:,1,:) + ...
        Rp_min_01_part(:,end,:))) + alpha_min./lam_min.*sinh(lam_min)));
    % 
    Fp_1_plus = 1i.*Gp_1;

    Fp_1_min = -1i.*Gp_1;

    Fp_1 = zeros(size(Gp_1));

else % depth-integral of both U and V is zero

    Fp_1_plus = -1i.*(p_vec + f)./(-1 + 1./lam_plus.*sinh(lam_plus)./...
        (cosh(lam_plus) + Av.*lam_plus./(H.*s).*sinh(lam_plus))).*...
        (dz.*(sum(Rp_plus_01_part(:,2:end-1,:),2) + 0.5.*(Rp_plus_01_part(:,1,:) + ...
        Rp_plus_01_part(:,end,:))) + alpha_plus./lam_plus.*sinh(lam_plus));
    
    Fp_1_min = -1i.*(p_vec - f)./(-1 + 1./lam_min.*sinh(lam_min)./...
        (cosh(lam_min) + Av.*lam_min./(H.*s).*sinh(lam_min))).*...
        (dz.*(sum(Rp_min_01_part(:,2:end-1,:),2) + 0.5.*(Rp_min_01_part(:,1,:) + ...
        Rp_min_01_part(:,end,:))) + alpha_min./lam_min.*sinh(lam_min));
    
    Fp_1 = (Fp_1_plus + Fp_1_min)/2;
    Gp_1 = (Fp_1_plus - Fp_1_min)/(2*1i);
end


%% find full sol for Up_01, Vp_01

Rp_plus_01 = Fp_1_plus./(1i.*(p_vec + f)).*(-1 + cosh(lam_plus.*z)./...
    (cosh(lam_plus) + Av.*lam_plus./(H.*s).*sinh(lam_plus))) + ...
    Rp_plus_01_part + alpha_plus.*cosh(lam_plus.*z);
Rp_min_01 = Fp_1_min./(1i.*(p_vec - f)).*(-1 + cosh(lam_min.*z)./...
    (cosh(lam_min) + Av.*lam_min./(H.*s).*sinh(lam_min))) + ...
    Rp_min_01_part + alpha_min.*cosh(lam_min.*z);

%compute velocities
Up_01 = (Rp_plus_01 + Rp_min_01)./2;
Vp_01 = (Rp_plus_01 - Rp_min_01)./(2*1i);

end