# Disclaimer
# ApPlyWood
# © (2023) Michele Mirra

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see https://www.gnu.org/licenses/.


# Import relevant modules

import wx
import sys
import matplotlib.pyplot as plt
from matplotlib.image import imread
from math import sqrt, tan, radians
from numpy import pi, exp, linspace, argmax, diff
from matplotlib.backends.backend_wxagg import FigureCanvasWxAgg as FigureCanvas
from scipy.integrate import quad
from fpdf import FPDF
from datetime import datetime


# Define user interface

class MainWindow(wx.Frame):

    def __init__(self, parent, title):
        wx.Frame.__init__(self, parent, title=title, pos=(30,30))
        
        # Sizers
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        grid = wx.GridBagSizer(hgap=20, vgap=15)
        hSizer = wx.BoxSizer(wx.HORIZONTAL)
                
        self.statusbar = self.CreateStatusBar(1)
        self.statusbar.SetStatusText('© (2023) Michele Mirra')
        #Label on floor type
        self.properties = wx.StaticText(self, label="Properties of the existing timber diaphragm")
        font = wx.Font(14, wx.DEFAULT, wx.NORMAL, wx.BOLD)
        self.properties.SetFont(font)
        grid.Add(self.properties, pos=(0,0), span=(1,3))

        # Radio Buttons Floor type
        self.lblfloor = wx.StaticText(self, label="Type of diaphragm")
        grid.Add(self.lblfloor, pos=(1,0))
        self.rb1 = wx.RadioButton(self, label="Floor", style = wx.RB_GROUP)
        self.rb2 = wx.RadioButton(self, label="Roof", style = 0)
        self.rb1.SetValue(False)
        self.rb2.SetValue(False)
        grid.Add(self.rb1, pos=(1,1))
        grid.Add(self.rb2, pos=(1,2))
        self.Bind(wx.EVT_RADIOBUTTON, self.EvtRadioBox1, self.rb1)
        self.Bind(wx.EVT_RADIOBUTTON, self.EvtRadioBox1, self.rb2)
        
        # Direction selection radiobox
        self.lbldir = wx.StaticText(self, label="Orientation of panels with load")
        grid.Add(self.lbldir, pos=(2,0))        
        self.rb7 = wx.RadioButton(self, label="Parallel", style = wx.RB_GROUP)
        self.rb8 = wx.RadioButton(self, label="Perpendicular", style = 0)
        self.rb7.SetValue(False)
        self.rb8.SetValue(False)
        grid.Add(self.rb7, pos=(2,1))
        grid.Add(self.rb8, pos=(2,2))
        self.Bind(wx.EVT_RADIOBUTTON, self.EvtRadioBox3, self.rb7)
        self.Bind(wx.EVT_RADIOBUTTON, self.EvtRadioBox3, self.rb8)

        # Span insertion
        self.lblspan = wx.StaticText(self, label="Span L (m)")
        grid.Add(self.lblspan, pos=(3,0))
        self.editspan = wx.TextCtrl(self, value="", size=(77, 20))
        grid.Add(self.editspan, pos=(3,2), span=(1,2))
        self.Bind(wx.EVT_TEXT, self.EvtText, self.editspan)
        self.Bind(wx.EVT_CHAR, self.EvtChar, self.editspan)
        
        # Width insertion
        self.lblwidth = wx.StaticText(self, label="Width B (m)")
        grid.Add(self.lblwidth, pos=(4,0))
        self.editwidth = wx.TextCtrl(self, value="", size=(77, 20))
        grid.Add(self.editwidth, pos=(4,2), span=(1,2))
        self.Bind(wx.EVT_TEXT, self.EvtText, self.editwidth)
        self.Bind(wx.EVT_CHAR, self.EvtChar, self.editwidth)
   
        # Spacer
        grid.Add((0, 5), pos=(5,0))
        
        #Label on sheathing properties
        self.propsheathing = wx.StaticText(self, label="Properties of the existing sheathing")
        self.propsheathing.SetFont(font)
        grid.Add(self.propsheathing, pos=(6,0), span=(1,3))   
        
        # Density of sheathing insertion
        self.lblrho1 = wx.StaticText(self, label="Density \N{GREEK SMALL LETTER RHO}\N{SUBSCRIPT ONE} (kg/m\N{SUPERSCRIPT THREE})")
        grid.Add(self.lblrho1, pos=(7,0))
        self.editrho1 = wx.TextCtrl(self, value="", size=(77,20))
        grid.Add(self.editrho1, pos=(7,2), span=(1,2))
        self.Bind(wx.EVT_TEXT, self.EvtText, self.editrho1)
        self.Bind(wx.EVT_CHAR, self.EvtChar, self.editrho1)
        
        # Thickness of sheathing insertion
        self.lblt1 = wx.StaticText(self, label="Thickness t\N{SUBSCRIPT ONE} (mm)")
        grid.Add(self.lblt1, pos=(8,0))
        self.editt1 = wx.TextCtrl(self, value="", size=(77,20))
        grid.Add(self.editt1, pos=(8,2), span=(1,2))
        self.Bind(wx.EVT_TEXT, self.EvtText, self.editt1)
        self.Bind(wx.EVT_CHAR, self.EvtChar, self.editt1)    
        
        # Spacer
        grid.Add((1, 4), pos=(9,0))
        
        #Label on plywood properties
        self.propply = wx.StaticText(self, label="Properties of the plywood overlay")
        self.propply.SetFont(font)
        grid.Add(self.propply, pos=(10,0), span=(1,3))   
        
        # Density of sheathing insertion
        self.lblrho2 = wx.StaticText(self, label="Density \N{GREEK SMALL LETTER RHO}\N{SUBSCRIPT TWO} (kg/m\N{SUPERSCRIPT THREE})")
        grid.Add(self.lblrho2, pos=(11,0))
        self.editrho2 = wx.TextCtrl(self, value="", size=(77,20))
        grid.Add(self.editrho2, pos=(11,2), span=(1,2))
        self.Bind(wx.EVT_TEXT, self.EvtText, self.editrho2)
        self.Bind(wx.EVT_CHAR, self.EvtChar, self.editrho2)
        
        # Thickness of sheathing insertion
        self.lblt2 = wx.StaticText(self, label="Thickness t\N{SUBSCRIPT TWO} (mm)")
        grid.Add(self.lblt2, pos=(12,0))
        self.editt2 = wx.TextCtrl(self, value="", size=(77,20))
        grid.Add(self.editt2, pos=(12,2))
        self.Bind(wx.EVT_TEXT, self.EvtText, self.editt2)
        self.Bind(wx.EVT_CHAR, self.EvtChar, self.editt2)     

        # Panels width insertion
        self.lblw2 = wx.StaticText(self, label="Width w\N{SUBSCRIPT TWO} of the panels (mm)")
        grid.Add(self.lblw2, pos=(13,0), span=(1,2))
        self.editw2 = wx.TextCtrl(self, value="", size=(77,20))
        grid.Add(self.editw2, pos=(13,2))
        self.Bind(wx.EVT_TEXT, self.EvtText, self.editw2)
        self.Bind(wx.EVT_CHAR, self.EvtChar, self.editw2)           
        
        # Spacer
        grid.Add((1, 4), pos=(14,0))
        
        #Label on fastener properties
        self.propfasteners = wx.StaticText(self, label="Properties of the fasteners")
        self.propfasteners.SetFont(font)
        grid.Add(self.propfasteners, pos=(15,0), span=(1,3))

        # Radio Buttons fastener type
        self.lblfastener = wx.StaticText(self, label="Type of fasteners")
        grid.Add(self.lblfastener, pos=(16,0))
        self.rb3 = wx.RadioButton(self, label="Screws", style = wx.RB_GROUP)
        self.rb4 = wx.RadioButton(self, label="Anker nails", style = 0)
        self.rb3.SetValue(False)
        self.rb4.SetValue(False)
        grid.Add(self.rb3, pos=(16,1))
        grid.Add(self.rb4, pos=(16,2))
        self.Bind(wx.EVT_RADIOBUTTON, self.EvtRadioBox2, self.rb3)
        self.Bind(wx.EVT_RADIOBUTTON, self.EvtRadioBox2, self.rb4)
        
        # Radio Buttons fastener properties specification
        self.lblfastener = wx.StaticText(self, label="Specification of properties")
        grid.Add(self.lblfastener, pos=(17,0))
        self.rb5 = wx.RadioButton(self, label="Built-in", style = wx.RB_GROUP)
        self.rb6 = wx.RadioButton(self, label="User-defined", style = 0)    
        self.dummy = wx.RadioButton(self, style = 0)
        self.dummy.SetValue(True)
        self.dummy.Show(False)
        self.rb5.SetValue(False)
        self.rb6.SetValue(False)
        grid.Add(self.rb5, pos=(17,1))
        grid.Add(self.rb6, pos=(17,2))
        self.Bind(wx.EVT_RADIOBUTTON, self.EvtRadioBox2, self.rb5)
        self.Bind(wx.EVT_RADIOBUTTON, self.EvtRadioBox2, self.rb6)

        # Dropdown of built-in diameters
        self.lbld = wx.StaticText(self, label="Nominal diameter d (mm)")
        grid.Add(self.lbld, pos=(18,0))
        self.dlist = ['3.0','3.5','4.0','4.5','5.0']         
        self.editd = wx.ComboBox(self, size=(55, 20), choices=self.dlist, style=wx.CB_DROPDOWN)
        grid.Add(self.editd, pos=(18,1))
        self.editd.SetSelection(-1)
        self.editd.Enable(False)
        self.Bind(wx.EVT_COMBOBOX, self.EvtComboBox, self.editd)
        self.Bind(wx.EVT_TEXT, self.EvtText,self.editd)        
       
        #Parameters of the fasteners: nominal diameter, shank diameter, yield moment, withdrawal parameter
        
        self.editdiam = wx.TextCtrl(self, value="", size=(77,20))
        self.editdiam.SetBackgroundColour((171,171,171))
        self.editdiam.SetEditable(False)
        grid.Add(self.editdiam, pos=(18,2))
        self.Bind(wx.EVT_TEXT, self.EvtText, self.editdiam)
        self.Bind(wx.EVT_CHAR, self.EvtChar, self.editdiam)

        self.lbld1 = wx.StaticText(self, label="Shank diameter d\N{SUBSCRIPT ONE} (mm)")
        grid.Add(self.lbld1, pos=(19,0))
        self.editd1 = wx.TextCtrl(self, value="", size=(77,20))
        self.editd1.SetBackgroundColour((171,171,171))
        self.editd1.SetEditable(False)
        grid.Add(self.editd1, pos=(19,2))
        self.Bind(wx.EVT_TEXT, self.EvtText, self.editd1)
        self.Bind(wx.EVT_CHAR, self.EvtChar, self.editd1)
 
        self.lblMy = wx.StaticText(self, label="Yield moment M (Nmm)")
        grid.Add(self.lblMy, pos=(20,0))
        self.editMy = wx.TextCtrl(self, value="", size=(77,20))
        self.editMy.SetBackgroundColour((171,171,171))
        self.editMy.SetEditable(False)
        grid.Add(self.editMy, pos=(20,2))
        self.Bind(wx.EVT_TEXT, self.EvtText, self.editMy)
        self.Bind(wx.EVT_CHAR, self.EvtChar, self.editMy)
 
        self.lblfax = wx.StaticText(self, label="Withdrawal parameter f\N{LATIN SUBSCRIPT SMALL LETTER A}\N{LATIN SUBSCRIPT SMALL LETTER X} (N/mm\N{SUPERSCRIPT TWO})")
        grid.Add(self.lblfax, pos=(21,0))
        self.editfax = wx.TextCtrl(self, value="", size=(77,20))
        self.editfax.SetBackgroundColour((171,171,171))
        self.editfax.SetEditable(False)
        grid.Add(self.editfax, pos=(21,2))
        self.Bind(wx.EVT_TEXT, self.EvtText, self.editfax)
        self.Bind(wx.EVT_CHAR, self.EvtChar, self.editfax)

        # Spacing insertion
        self.lbls = wx.StaticText(self, label="Spacing s (mm)")
        grid.Add(self.lbls, pos=(22,0))
        self.edits = wx.TextCtrl(self, value="", size=(77,20))
        grid.Add(self.edits, pos=(22,2))
        self.Bind(wx.EVT_TEXT, self.EvtText, self.edits)
        self.Bind(wx.EVT_CHAR, self.EvtChar, self.edits)     

        # Distance from panel edge insertion
        self.lble = wx.StaticText(self, label="Distance e from panel edge (mm)")
        grid.Add(self.lble, pos=(23,0), span=(1,2))
        self.edite = wx.TextCtrl(self, value="", size=(77,20))
        grid.Add(self.edite, pos=(23,2))
        self.Bind(wx.EVT_TEXT, self.EvtText, self.edite)
        self.Bind(wx.EVT_CHAR, self.EvtChar, self.edite)                   

        # Calculate button
        self.buttonC =wx.Button(self, label="Calculate", size=(77,30))
        self.Bind(wx.EVT_BUTTON, self.OnClickCALC,self.buttonC)
        grid.Add(self.buttonC, pos=(24,2), span=(1,2))

        # Reset button
        self.buttonR =wx.Button(self, label="Clear", size=(77,30))
        self.Bind(wx.EVT_BUTTON, self.OnClickRESET,self.buttonR) 
        grid.Add(self.buttonR, pos=(24,0))

        #Graph location and content
        self.file = "LOGO.png"
        self.logo = imread(self.file)     
        self.figure = plt.figure(figsize=(8.37,5.78))
        self.axes = self.figure.add_subplot(111)
        self.axes.imshow(self.logo)
        self.axes.axis('off')
        self.graph = FigureCanvas(self, -1, self.figure)
        grid.Add(self.graph, pos=(1,4), span = (17,5))
        
        #Results label and output parameters
        self.results = wx.StaticText(self, label="Results")
        self.results.SetFont(font)
        grid.Add(self.results, pos=(0,6))
        
        self.lblFmax = wx.StaticText(self, label="In-plane peak force (kN)")
        grid.Add(self.lblFmax, pos=(20,4))
        self.lblFmaxval = wx.TextCtrl(self, value="", size=(77,20))
        self.lblFmaxval.SetBackgroundColour((255,241,205))
        self.lblFmaxval.SetEditable(False)
        grid.Add(self.lblFmaxval, pos=(20,5))
        
        self.lblvmax = wx.StaticText(self, label="Maximum transferred seismic shear (kN/m)")
        grid.Add(self.lblvmax, pos=(21,4))
        self.lblvmaxval = wx.TextCtrl(self, value="", size=(77,20))
        self.lblvmaxval.SetBackgroundColour((255,241,205))
        self.lblvmaxval.SetEditable(False)
        grid.Add(self.lblvmaxval, pos=(21,5))
        
        self.lbldispmax = wx.StaticText(self, label="In-plane displacement at peak force (mm)")
        grid.Add(self.lbldispmax, pos=(22,4))
        self.lbldispmaxval = wx.TextCtrl(self, value="", size=(77,20))
        self.lbldispmaxval.SetBackgroundColour((255,241,205))
        self.lbldispmaxval.SetEditable(False)
        grid.Add(self.lbldispmaxval, pos=(22,5))
        
        self.lbldriftmax = wx.StaticText(self, label="In-plane drift at peak force (%)")
        grid.Add(self.lbldriftmax, pos=(23,4))
        self.lbldriftmaxval = wx.TextCtrl(self, value="", size=(77,20))
        self.lbldriftmaxval.SetBackgroundColour((255,241,205))
        self.lbldriftmaxval.SetEditable(False)
        grid.Add(self.lbldriftmaxval, pos=(23,5))

        self.lblK0 = wx.StaticText(self, label="Initial stiffness (kN/mm)")
        grid.Add(self.lblK0, pos=(20,7))
        self.lblK0val = wx.TextCtrl(self, value="", size=(77,20))
        self.lblK0val.SetBackgroundColour((255,241,205))
        self.lblK0val.SetEditable(False)
        grid.Add(self.lblK0val, pos=(20,8))        
                
        self.lblgd0 = wx.StaticText(self, label="Initial equivalent shear stiffness (kN/m)")
        grid.Add(self.lblgd0, pos=(21,7))
        self.lblgd0val = wx.TextCtrl(self, value="", size=(77,20))
        self.lblgd0val.SetBackgroundColour((255,241,205))
        self.lblgd0val.SetEditable(False)
        grid.Add(self.lblgd0val, pos=(21,8))

        self.lblgdmax = wx.StaticText(self, label="Equivalent shear stiffness at peak force (kN/m)")
        grid.Add(self.lblgdmax, pos=(22,7))
        self.lblgdmaxval = wx.TextCtrl(self, value="", size=(77,20))
        self.lblgdmaxval.SetBackgroundColour((255,241,205))
        self.lblgdmaxval.SetEditable(False)
        grid.Add(self.lblgdmaxval, pos=(22,8))
        
        self.lblcsi = wx.StaticText(self, label="Average equivalent hysteretic damping ratio (%)")
        grid.Add(self.lblcsi, pos=(23,7))
        self.lblcsival = wx.TextCtrl(self, value="", size=(77,20))
        self.lblcsival.SetBackgroundColour((255,241,205))
        self.lblcsival.SetEditable(False)
        grid.Add(self.lblcsival, pos=(23,8))
        
        # Report button
        self.buttonG =wx.Button(self, label="Export PDF", size=(85,30))
        self.Bind(wx.EVT_BUTTON, self.OnClickGEN,self.buttonG) 
        grid.Add(self.buttonG, pos=(24,8))
        self.buttonG.Enable(False)
        
        
        #Set Sizer
                
        hSizer.Add(grid, 0, wx.ALL, 5)
        mainSizer.Add(hSizer, 0, wx.ALL, 5)
        self.SetSizerAndFit(mainSizer)
        
        #Closing Window event
        self.Bind(wx.EVT_CLOSE, self.on_close)

    
    def EvtRadioBox1(self, event):
        
        if(self.rb1.GetValue() == False):
            self.file = "Roof.png"
            self.logo = imread(self.file)     
            self.figure = plt.figure(figsize=(8.37,5.78))
            self.axes = self.figure.add_subplot(111)
            self.axes.imshow(self.logo)
            self.axes.axis('off')
            self.canvas = FigureCanvas(self.graph, -1, self.figure) 
        else:
            self.file = "Floor.png"
            self.logo = imread(self.file)     
            self.figure = plt.figure(figsize=(8.37,5.78))
            self.axes = self.figure.add_subplot(111)
            self.axes.imshow(self.logo)
            self.axes.axis('off')
            self.canvas = FigureCanvas(self.graph, -1, self.figure)
        self.rb7.SetValue(False)
        self.rb8.SetValue(False)
    
    #Definition of fastener type and properties radiobox event   
     
    def EvtRadioBox2(self, event):
        
        if(self.rb5.GetValue() == True) and (self.rb3.GetValue() == True):
            self.editd.Enable(True)
            self.editd.Set(['3.0','3.5','4.0','4.5','5.0'])
            self.editdiam.SetBackgroundColour((171,171,171))
            self.editdiam.SetEditable(False)
            self.editd1.SetBackgroundColour((171,171,171))
            self.editd1.SetEditable(False)
            self.editMy.SetBackgroundColour((171,171,171))
            self.editMy.SetEditable(False)
            self.editfax.SetBackgroundColour((171,171,171))
            self.editfax.SetEditable(False)
            self.editdiam.SetValue("")
            self.editd1.SetValue("")
            self.editMy.SetValue("")
            self.editfax.SetValue("")
        elif(self.rb5.GetValue() == True) and (self.rb4.GetValue() == True):
            self.editd.Enable(True)
            self.editd.Set(['4.0','6.0'])
            self.editdiam.SetBackgroundColour((171,171,171))
            self.editdiam.SetEditable(False)
            self.editd1.SetBackgroundColour((171,171,171))
            self.editd1.SetEditable(False)
            self.editMy.SetBackgroundColour((171,171,171))
            self.editMy.SetEditable(False)
            self.editfax.SetBackgroundColour((171,171,171))
            self.editfax.SetEditable(False)
            self.editdiam.SetValue("")
            self.editd1.SetValue("")
            self.editMy.SetValue("")
            self.editfax.SetValue("")
        elif(self.rb3.GetValue() == True) and (self.rb5.GetValue() == False) and (self.rb6.GetValue() == False):
            self.editd.SetSelection(-1)
            self.editd.Enable(False)
            self.editdiam.SetBackgroundColour((171,171,171))
            self.editdiam.SetEditable(False)
            self.editd1.SetBackgroundColour((171,171,171))
            self.editd1.SetEditable(False)
            self.editMy.SetBackgroundColour((171,171,171))
            self.editMy.SetEditable(False)
            self.editfax.SetBackgroundColour((171,171,171))
            self.editfax.SetEditable(False)
            self.editdiam.SetValue("")
            self.editd1.SetValue("")
            self.editMy.SetValue("")
            self.editfax.SetValue("")            
        elif(self.rb4.GetValue() == True) and (self.rb5.GetValue() == False) and (self.rb6.GetValue() == False):
            self.editd.SetSelection(-1)
            self.editd.Enable(False)
            self.editdiam.SetBackgroundColour((171,171,171))
            self.editdiam.SetEditable(False)
            self.editd1.SetBackgroundColour((171,171,171))
            self.editd1.SetEditable(False)
            self.editMy.SetBackgroundColour((171,171,171))
            self.editMy.SetEditable(False)
            self.editfax.SetBackgroundColour((171,171,171))
            self.editfax.SetEditable(False)
            self.editdiam.SetValue("")
            self.editd1.SetValue("")
            self.editMy.SetValue("")
            self.editfax.SetValue("")
        elif(self.rb5.GetValue() == True) and (self.rb3.GetValue() == False) and (self.rb4.GetValue() == False):
            self.editd.SetSelection(-1)
            self.editd.Enable(False)
            self.editdiam.SetBackgroundColour((171,171,171))
            self.editdiam.SetEditable(False)
            self.editd1.SetBackgroundColour((171,171,171))
            self.editd1.SetEditable(False)
            self.editMy.SetBackgroundColour((171,171,171))
            self.editMy.SetEditable(False)
            self.editfax.SetBackgroundColour((171,171,171))
            self.editfax.SetEditable(False)
            self.editdiam.SetValue("")
            self.editd1.SetValue("")
            self.editMy.SetValue("")
            self.editfax.SetValue("")            
        else:            
            self.editd.SetSelection(-1)
            self.editd.Enable(False)
            self.editdiam.SetBackgroundColour('white')
            self.editdiam.SetEditable(True)
            self.editd1.SetBackgroundColour('white')
            self.editd1.SetEditable(True)
            self.editMy.SetBackgroundColour('white')
            self.editMy.SetEditable(True)
            self.editfax.SetBackgroundColour('white')
            self.editfax.SetEditable(True)
            self.editdiam.SetValue("")
            self.editd1.SetValue("")
            self.editMy.SetValue("")
            self.editfax.SetValue("")
            
    #Definition of loading orientation radiobox event
    def EvtRadioBox3(self, event):
        if(self.rb7.GetValue() == True):
            if(self.rb1.GetValue() == True):
                self.file = "Floor_para.png"
                self.logo = imread(self.file)     
                self.figure = plt.figure(figsize=(8.37,5.78))
                self.axes = self.figure.add_subplot(111)
                self.axes.imshow(self.logo)
                self.axes.axis('off')
                self.canvas = FigureCanvas(self.graph, -1, self.figure)
            else:
                self.file = "Roof_para.png"
                self.logo = imread(self.file)     
                self.figure = plt.figure(figsize=(8.37,5.78))
                self.axes = self.figure.add_subplot(111)
                self.axes.imshow(self.logo)
                self.axes.axis('off')
                self.canvas = FigureCanvas(self.graph, -1, self.figure)
        else:
            if(self.rb1.GetValue() == True):
                self.file = "Floor_perp.png"
                self.logo = imread(self.file)     
                self.figure = plt.figure(figsize=(8.37,5.78))
                self.axes = self.figure.add_subplot(111)
                self.axes.imshow(self.logo)
                self.axes.axis('off')
                self.canvas = FigureCanvas(self.graph, -1, self.figure)
            else:
                self.file = "Roof_perp.png"
                self.logo = imread(self.file)     
                self.figure = plt.figure(figsize=(8.37,5.78))
                self.axes = self.figure.add_subplot(111)
                self.axes.imshow(self.logo)
                self.axes.axis('off')
                self.canvas = FigureCanvas(self.graph, -1, self.figure)
            
            
    #Definition of dropdown list event
       
    def EvtComboBox(self, event):
        
        if(self.rb5.GetValue() == True):
            if(self.rb3.GetValue() == True):
                if(self.editd.GetValue() == '3.0'):
                    self.editdiam.SetValue('3.0')
                    self.editd1.SetValue('2.2')
                    self.editMy.SetValue('2200')
                    self.editfax.SetValue('18.5')
                    self.editdiam.SetBackgroundColour((171,171,171))
                    self.editdiam.SetEditable(False)
                    self.editd1.SetBackgroundColour((171,171,171))
                    self.editd1.SetEditable(False)
                    self.editMy.SetBackgroundColour((171,171,171))
                    self.editMy.SetEditable(False)
                    self.editfax.SetBackgroundColour((171,171,171))
                    self.editfax.SetEditable(False)
                elif(self.editd.GetValue() == '3.5'):
                    self.editdiam.SetValue('3.5')
                    self.editd1.SetValue('2.45')
                    self.editMy.SetValue('2700')
                    self.editfax.SetValue('17.9')
                    self.editdiam.SetBackgroundColour((171,171,171))
                    self.editdiam.SetEditable(False)
                    self.editd1.SetBackgroundColour((171,171,171))
                    self.editd1.SetEditable(False)
                    self.editMy.SetBackgroundColour((171,171,171))
                    self.editMy.SetEditable(False)
                    self.editfax.SetBackgroundColour((171,171,171))
                    self.editfax.SetEditable(False)
                elif(self.editd.GetValue() == '4.0'):
                    self.editdiam.SetValue('4.0')
                    self.editd1.SetValue('2.75')
                    self.editMy.SetValue('3800')
                    self.editfax.SetValue('17.1')
                    self.editdiam.SetBackgroundColour((171,171,171))
                    self.editdiam.SetEditable(False)
                    self.editd1.SetBackgroundColour((171,171,171))
                    self.editd1.SetEditable(False)
                    self.editMy.SetBackgroundColour((171,171,171))
                    self.editMy.SetEditable(False)
                    self.editfax.SetBackgroundColour((171,171,171))
                    self.editfax.SetEditable(False)
                elif(self.editd.GetValue() == '4.5'):
                    self.editdiam.SetValue('4.5')
                    self.editd1.SetValue('3.2')
                    self.editMy.SetValue('5800')
                    self.editfax.SetValue('17.0')
                    self.editdiam.SetBackgroundColour((171,171,171))
                    self.editdiam.SetEditable(False)
                    self.editd1.SetBackgroundColour((171,171,171))
                    self.editd1.SetEditable(False)
                    self.editMy.SetBackgroundColour((171,171,171))
                    self.editMy.SetEditable(False)
                    self.editfax.SetBackgroundColour((171,171,171))
                    self.editfax.SetEditable(False)
                else:
                    self.editdiam.SetValue('5.0')
                    self.editd1.SetValue('3.65')
                    self.editMy.SetValue('8800')
                    self.editfax.SetValue('15.5')
                    self.editdiam.SetBackgroundColour((171,171,171))
                    self.editdiam.SetEditable(False)
                    self.editd1.SetBackgroundColour((171,171,171))
                    self.editd1.SetEditable(False)
                    self.editMy.SetBackgroundColour((171,171,171))
                    self.editMy.SetEditable(False)
                    self.editfax.SetBackgroundColour((171,171,171))
                    self.editfax.SetEditable(False)
            else:
                if(self.editd.GetValue() == '4.0'):
                    self.editdiam.SetValue('4.0')
                    self.editd1.SetValue('4.0')
                    self.editMy.SetValue('6680')
                    self.editfax.SetValue('6.43')
                    self.editdiam.SetBackgroundColour((171,171,171))
                    self.editdiam.SetEditable(False)
                    self.editd1.SetBackgroundColour((171,171,171))
                    self.editd1.SetEditable(False)
                    self.editMy.SetBackgroundColour((171,171,171))
                    self.editMy.SetEditable(False)
                    self.editfax.SetBackgroundColour((171,171,171))
                    self.editfax.SetEditable(False)
                else:
                    self.editdiam.SetValue('6.0')
                    self.editd1.SetValue('6.0')
                    self.editMy.SetValue('20200')
                    self.editfax.SetValue('8.37')
                    self.editdiam.SetBackgroundColour((171,171,171))
                    self.editdiam.SetEditable(False)
                    self.editd1.SetBackgroundColour((171,171,171))
                    self.editd1.SetEditable(False)
                    self.editMy.SetBackgroundColour((171,171,171))
                    self.editMy.SetEditable(False)
                    self.editfax.SetBackgroundColour((171,171,171))
                    self.editfax.SetEditable(False) 
                
    #Definition of Reset button event
         
    def OnClickRESET(self,event):
        
        self.rb1.SetValue(False)
        self.rb2.SetValue(False)
        self.rb3.SetValue(False)
        self.rb4.SetValue(False)
        self.rb5.SetValue(False)
        self.rb6.SetValue(False)
        self.rb7.SetValue(False)
        self.rb8.SetValue(False)
        self.editspan.Clear()
        self.editwidth.Clear()
        self.editrho1.Clear()
        self.editt1.Clear()
        self.editrho2.Clear()
        self.editt2.Clear()
        self.editw2.Clear()
        self.editd.SetSelection(-1)
        self.editd.Enable(False)
        self.edits.Clear()
        self.edite.Clear()
        self.lblFmaxval.Clear()
        self.lblvmaxval.Clear()
        self.lbldispmaxval.Clear()
        self.lbldriftmaxval.Clear()
        self.lblK0val.Clear()
        self.lblgd0val.Clear()
        self.lblgdmaxval.Clear()
        self.lblcsival.Clear()
        self.file = "LOGO.png"
        self.logo = imread(self.file)     
        self.figure = plt.figure(figsize=(8.37,5.78))
        self.axes = self.figure.add_subplot(111)
        self.axes.imshow(self.logo)
        self.axes.axis('off')
        self.canvas = FigureCanvas(self.graph, -1, self.figure)
        self.editdiam.SetBackgroundColour((171,171,171))
        self.editdiam.SetEditable(False)
        self.editd1.SetBackgroundColour((171,171,171))
        self.editd1.SetEditable(False)
        self.editMy.SetBackgroundColour((171,171,171))
        self.editMy.SetEditable(False)
        self.editfax.SetBackgroundColour((171,171,171))
        self.editfax.SetEditable(False)
        self.editdiam.SetValue("")
        self.editd1.SetValue("")
        self.editMy.SetValue("")
        self.editfax.SetValue("")
        self.statusbar.SetStatusText('© (2023) Michele Mirra')
        self.buttonG.Enable(False)
        
    #Definition of text typing event 
       
    def EvtText(self, event):
        self.editspan.GetValue()
        self.editwidth.GetValue()
        self.editrho1.GetValue()
        self.editt1.GetValue()
        self.editrho2.GetValue()
        self.editt2.GetValue()
        self.editw2.GetValue()
        self.edite.GetValue()
        self.edits.GetValue()
        
    #Definition of Calculate button event 
    
    def OnClickCALC(self,event):
        
        #Import parameters
        
        if(self.rb1.GetValue()==False and self.rb2.GetValue()==False):
            self.statusbar.SetStatusText('Please select diaphragm type')
            
        if(self.rb7.GetValue()==False and self.rb8.GetValue()==False):
            self.statusbar.SetStatusText('Please select orientation of panels with load')
        
        if(self.editspan.GetValue()==""):
            self.statusbar.SetStatusText('Please insert diaphragm span')
   
        if(self.editwidth.GetValue()==""):
            self.statusbar.SetStatusText('Please insert diaphragm width')

        if(self.editrho1.GetValue()==""):
            self.statusbar.SetStatusText('Please insert sheathing density')
        
        if(self.editt1.GetValue()==""):
            self.statusbar.SetStatusText('Please insert sheathing thickness')

        if(self.editrho2.GetValue()==""):
            self.statusbar.SetStatusText('Please insert plywood density')            

        if(self.editt2.GetValue()==""):
            self.statusbar.SetStatusText('Please insert plywood thickness')

        if(self.editw2.GetValue()==""):
            self.statusbar.SetStatusText('Please insert panel width')

        if(self.rb3.GetValue()==False and self.rb4.GetValue()==False):
            self.statusbar.SetStatusText('Please select fastener type')
            
        if(self.rb5.GetValue()==False and self.rb6.GetValue()==False):
            self.statusbar.SetStatusText('Please select specification of fastener properties')
            
        if(self.editdiam.GetValue()==""):
            self.statusbar.SetStatusText('Please insert nominal diameter of fasteners')            

        if(self.editd1.GetValue()==""):
            self.statusbar.SetStatusText('Please insert shank diameter of fasteners')            

        if(self.editMy.GetValue()==""):
            self.statusbar.SetStatusText('Please insert yield moment of fasteners')            

        if(self.editfax.GetValue()==""):
            self.statusbar.SetStatusText('Please insert withdrawal parameter of fasteners')

        if(self.edits.GetValue()==""):
            self.statusbar.SetStatusText('Please insert spacing of fasteners') 

        if(self.edite.GetValue()==""):
            self.statusbar.SetStatusText('Please insert distance of fasteners from panel edge')            
             
           
                
        
        L = float(self.editspan.GetValue())
        B = float(self.editwidth.GetValue())
        rho1 = float(self.editrho1.GetValue())
        t1 = float(self.editt1.GetValue())
        rho2 = float(self.editrho2.GetValue())
        t2 = float(self.editt2.GetValue())
        w2 = float(self.editw2.GetValue())
        d = float(self.editdiam.GetValue())
        d1 = float(self.editd1.GetValue())
        My = float(self.editMy.GetValue())
        fax = float(self.editfax.GetValue())
        e = float(self.edite.GetValue())
        s = float(self.edits.GetValue())
        
        #Calculate strength and displacement capacity of fasteners
        fh1 = 0.082 * rho1 * d**(-0.3)
        fh2 = 0.11 * rho2 * d**(-0.3)
        beta = fh2 / fh1
        Fax = min(fax*t1*d*(rho1/350)**0.8,fax*t2*d*(rho2/350)**0.8)        
        b1 = sqrt(2*My/(fh1*d))*sqrt(2*beta/(1+beta))
        b2 = b1/beta
        
        if(self.rb3.GetValue() == True):
            alpha = min(110/d1,45)
        else:
            alpha = 45
            
        dmax_fastener = (b1+b2)*tan(radians(alpha))
     
        if(self.rb1.GetValue() == True):
            if(self.rb7.GetValue() == True):
                dmax = dmax_fastener*(L*1000/2/w2)
            elif(self.rb8.GetValue() == True):
                dmax = dmax_fastener/(w2-e)*L*1000/4
            else:
                self.statusbar.SetStatusText('Please select orientation of panels with load')
        elif(self.rb2.GetValue() == True):
            if(self.rb7.GetValue() == True):
                dmax = dmax_fastener*(L*1000/w2)
            elif(self.rb8.GetValue() == True):
                dmax = dmax_fastener/(w2-e)*L*1000/2
            else:
                self.statusbar.SetStatusText('Please select orientation of panels with load')
        else:
            self.statusbar.SetStatusText('Please select diaphragm type')
        
        
        Fmax_fastener = (sqrt(2*beta/(1+beta))*sqrt(2*My*fh1*d)+Fax/4)/1000
        if(self.rb7.GetValue() == True):
            Fmax = 2*Fmax_fastener*B*1000/s
        else:
            Fmax = 2*Fmax_fastener*(w2-e)/s*(B*1000/w2)
        
        if(self.rb3.GetValue() == True):
            F0 = Fmax/8
        else:
            F0 = 0.4*Fmax
        
        K0_fastener = (50*d**1.7)/1000
        
        if(self.rb1.GetValue() == True):
            if(self.rb7.GetValue() == True):
                K0 = K0_fastener*2*B*1000/s/(L*1000/2/w2)
            elif(self.rb8.GetValue() == True):
                K0 = K0_fastener*2*(w2-e)**2/s*(B*1000/w2)/(L*1000/4)         
        else:
            if(self.rb7.GetValue() == True):
                K0 = K0_fastener*2*B*1000/s/(L*1000/w2)
            elif(self.rb8.GetValue() == True):
                K0 = K0_fastener*2*(w2-e)**2/s*(B*1000/w2)/(L*1000/2) 
            
        a = 2*(Fmax-F0)/dmax
        b = -(Fmax-F0)/(dmax**2)
      
        #Define backbone curve equation 
        def F(x):
            if(self.rb1.GetValue() == True):
                if(self.rb7.GetValue() == True):
                    return (F0+a*x+b*x**2)*(1-exp(-K0*x/F0))
                else:
                    return (F0+a*x+b*x**2)*(1-exp(-K0*x/F0))*(1.05+10*x/(L*1000/4))
            else:
                if(self.rb7.GetValue() == True):
                    return (F0+a*x+b*x**2)*(1-exp(-K0*x/F0))
                else:
                    return (F0+a*x+b*x**2)*(1-exp(-K0*x/F0))*(1.05+10*x/(L*1000/2))

        #Define unloading and reloading pinching curves
        
        F0_pinch = 0.1*Fmax
        
        def Pinchul(y,dispref):
            k1 = (F(dispref)-F0_pinch)/dispref
            q2 = F(dispref)-(K0)*dispref
            d3 = (-sqrt(1+(K0)**2)*F0_pinch-sqrt(1+(k1)**2)*q2)/(sqrt(1+(k1)**2)*K0+sqrt(1+(K0)**2)*k1)
            k3 = F(dispref)/(dispref-d3)
            q3 = F(dispref)-k3*dispref
            f4 = k3*(dispref/2)+q3
            k4 = (f4 - F0_pinch)/(dispref/2)
            if k4 < 0:
                k4 = 0
            p0_UL = (-dispref)*k4+F0_pinch
            return (p0_UL+F(dispref)+k4*(y+dispref))*(1-exp(-2*(K0)/(p0_UL+F(dispref))*(y+dispref)))-F(dispref)
            
        def Pinchrel(y,dispref):
            k1 = (F(dispref)-F0_pinch)/dispref
            q2 = F(dispref)-(K0)*dispref
            d3 = (-sqrt(1+(K0)**2)*F0_pinch-sqrt(1+(k1)**2)*q2)/(sqrt(1+(k1)**2)*K0+sqrt(1+(K0)**2)*k1)
            k3 = F(dispref)/(dispref-d3)
            q3 = F(dispref)-k3*dispref
            f4 = k3*(dispref/2)+q3
            k4 = (f4 - F0_pinch)/(dispref/2)
            if k4 < 0:
                k4 = 0
            p0_L = (-dispref)*k4-F0_pinch
            return (p0_L+F(dispref)+k4*(y+dispref))*(1-exp(-(1+dispref**3/dispmax**3)*k3/(p0_L+F(dispref))*(y+dispref)))-F(dispref)
        
        #Plotting graph with in-plane response of the diaphragm
        x = linspace(0, 1.5*dmax, 1000)
        dispmax=x[argmax(F(x))]
        dispset = [0.025*dispmax, 0.05*dispmax, 0.075*dispmax, 0.1*dispmax, 0.2*dispmax, 0.4*dispmax, 0.6*dispmax, 0.8*dispmax, 1.0*dispmax, 1.2*dispmax]
        
        self.figure = plt.figure(figsize=(8.37,5.78))
        self.axes = self.figure.add_subplot(111, title = 'In-plane response of the diaphragm', xlabel = 'Displacement (mm)', ylabel = 'Force (kN)')
        self.axes.plot(x, F(x), color='black')
        self.axes.plot(-x, -F(x), color='black')
        self.axes.axhline(0, color='black', linewidth=0.5) # horizontal line
        self.axes.axvline(0, color='black', linewidth=0.5) # vertical line
        csi_list = []
        
        for dispref in dispset:
           c = (Pinchul(0,dispref)+Pinchrel(0,dispref))/2
           y = linspace(-dispref, 0, 500)
           self.axes.plot(y, Pinchul(y,dispref)-c, color='black')
           self.axes.plot(-y, -Pinchul(y,dispref)+c, color='black')
           self.axes.plot(y, Pinchrel(y,dispref)-c, color='black')
           self.axes.plot(-y, -Pinchrel(y,dispref)+c, color='black')
           
           def UL(p):
               return Pinchul(p,dispref)-c
           def REL(q):
               return Pinchrel(q,dispref)+c
           
           #Calculate equivalent hysteretic damping ratio with energy loss per cycle method
           int_ul, err_ul = quad(UL, -dispref, 0)
           int_rel, err_rel = quad(REL, -dispref, 0)
           int_el = dispref*F(dispref)/2
           csi = diff([int_rel,int_ul])*2/(4*pi*int_el)
           csi_list.append(csi)
           
        csi_av = sum(csi_list)/len(csi_list)*100
        
        if(self.rb1.GetValue() == True):
            if(self.rb7.GetValue() == True):
                self.file = "Floor_para.png"
            else:
                self.file = "Floor_perp.png"
        else:
            if(self.rb7.GetValue() == True):
                self.file = "Roof_para.png"
            else:
                self.file = "Roof_perp.png"            
        self.logo = imread(self.file) 
        self.staticscheme = plt.axes([0.11, 0.54, 0.33, 0.33])      
        self.staticscheme.set_axis_off()
        self.staticscheme.imshow(self.logo, aspect="equal")
                
        self.figure.savefig('graph.png')
        
        #Show results
        self.canvas = FigureCanvas(self.graph, -1, self.figure)        
        self.lblFmaxval.SetValue('%.2f' %(max(F(x))))
        self.lblvmaxval.SetValue('%.2f' %(max(F(x))/2/B))
        self.lbldispmaxval.SetValue('%.2f' %(dispmax))
        self.lblK0val.SetValue('%.2f' %(K0))
        if(self.rb1.GetValue() == True):
            self.lbldriftmaxval.SetValue('%.2f' %(100*dispmax/(L*1000/2)))
            self.lblgd0val.SetValue('%.0f' %(K0/2*(L*1000/4)/(B)))
            self.lblgdmaxval.SetValue('%.0f' %(max(F(x))/dispmax/2*(L*1000/4)/(B)))
        else:
            self.lbldriftmaxval.SetValue('%.2f' %(100*dispmax/(L*1000)))
            self.lblgd0val.SetValue('%.0f' %(K0/2*(L*1000/2)/(B)))
            self.lblgdmaxval.SetValue('%.0f' %(max(F(x))/dispmax/2*(L*1000/2)/(B)))
        self.lblcsival.SetValue('%.2f' %(csi_av[0]))
        self.statusbar.SetStatusText('In-plane response of the diaphragm successfully determined')
        self.buttonG.Enable(True)
        
    def OnClickGEN(self,event):
        
        #Create report
        pdf = FPDF()
        pdf.set_margins(20.0, 20.0, 20.0)
        pdf.add_page()
        pdf.set_font('Arial','B',20.0)
        pdf.cell(140,30,'SUMMARY OF THE CALCULATIONS',1,align='C')
        pdf.image('LOGO2.png',x=160,w=30,h=30)
        pdf.ln(5)
        pdf.set_font('Arial','B',12.0)
        pdf.cell(170,10,'Input data:', align = 'C')
        pdf.ln(7)
        pdf.set_font('Arial','',10.0)
        pdf.cell(45,10,'Diaphragm type:')
        if(self.rb1.GetValue() == True):
            pdf.cell(40,10,'Floor')
        else:
            pdf.cell(40,10,'Roof')
        pdf.cell(55,10,'Fastener type:')
        if(self.rb3.GetValue() == True):
            pdf.cell(30,10,'Screws')
        else:
            pdf.cell(30,10,'Anker nails')        
        pdf.ln(7)
        pdf.cell(45,10,'Span (m):')
        pdf.cell(40,10,'%.1f' %(float(self.editspan.GetValue())))
        pdf.cell(55,10,'Fastener properties:')
        if(self.rb5.GetValue() == True):
            pdf.cell(30,10,'Built-in')
        else:
            pdf.cell(30,10,'User-defined')         
        pdf.ln(7)
        pdf.cell(45,10,'Width (m):')
        pdf.cell(40,10,'%.1f' %(float(self.editwidth.GetValue())))
        pdf.cell(55,10,'Nominal diameter (mm):')
        pdf.cell(30,10,'%.1f' %(float(self.editdiam.GetValue())))
        pdf.ln(7)
        pdf.cell(45,10,'Sheathing density (kg/m\N{SUPERSCRIPT THREE}):')
        pdf.cell(40,10,'%.0f' %(float(self.editrho1.GetValue())))
        pdf.cell(55,10,'Shank diameter (mm):')
        pdf.cell(30,10,'%.1f' %(float(self.editd1.GetValue())))
        pdf.ln(7)
        pdf.cell(45,10,'Sheathing thickness (mm):')
        pdf.cell(40,10,'%.1f' %(float(self.editt1.GetValue())))
        pdf.cell(55,10,'Yield moment (Nmm):')
        pdf.cell(30,10,'%.0f' %(float(self.editMy.GetValue())))
        pdf.ln(7)
        pdf.cell(45,10,'Plywood density (kg/m\N{SUPERSCRIPT THREE}):')
        pdf.cell(40,10,'%.0f' %(float(self.editrho2.GetValue())))
        pdf.cell(55,10,'Withdrawal parameter (N/mm\N{SUPERSCRIPT TWO}):')
        pdf.cell(30,10,'%.1f' %(float(self.editfax.GetValue())))
        pdf.ln(7)
        pdf.cell(45,10,'Plywood thickness (mm):')
        pdf.cell(40,10,'%.1f' %(float(self.editt2.GetValue())))
        pdf.cell(55,10,'Spacing (mm):')
        pdf.cell(30,10,'%.0f' %(float(self.edits.GetValue())))
        pdf.ln(7)
        pdf.cell(45,10,'Width of panels (mm):')
        pdf.cell(40,10,'%.0f' %(float(self.editw2.GetValue())))
        pdf.cell(55,10,'Distance from panel edge (mm):')
        pdf.cell(30,10,'%.0f' %(float(self.edite.GetValue())))
        pdf.ln(10)
        pdf.set_font('Arial','B',12.0)
        pdf.cell(170,10,'Results:', align = 'C')
        pdf.ln(7)
        pdf.set_font('Arial','',10.0)        
        pdf.image('graph.png',x=42,w=126,h=95)
        pdf.ln(5)
        pdf.cell(62,10,'In-plane peak force (kN):')
        pdf.cell(23,10,'%.2f' %(float(self.lblFmaxval.GetValue())))
        pdf.cell(62,10,'Initial stiffness (kN/mm):')
        pdf.cell(23,10,'%.2f' %(float(self.lblK0val.GetValue())))
        pdf.ln(7)        
        pdf.cell(62,10,'Transferred seismic shear (kN/m):')
        pdf.cell(23,10,'%.2f' %(float(self.lblvmaxval.GetValue())))
        pdf.cell(62,10,'Initial shear stiffness (kN/m):')
        pdf.cell(23,10,'%.0f' %(float(self.lblgd0val.GetValue())))
        pdf.ln(7)        
        pdf.cell(62,10,'Displacement at peak force (mm):')
        pdf.cell(23,10,'%.2f' %(float(self.lbldispmaxval.GetValue())))
        pdf.cell(62,10,'Shear stiffness at peak force (kN/m):')
        pdf.cell(23,10,'%.0f' %(float(self.lblgdmaxval.GetValue())))
        pdf.ln(7)   
        pdf.cell(62,10,'In-plane drift at peak force (%):')
        pdf.cell(23,10,'%.2f' %(float(self.lbldriftmaxval.GetValue())))
        pdf.cell(62,10,'Average equivalent damping ratio (%):')
        pdf.cell(23,10,'%.2f' %(float(self.lblcsival.GetValue())))
        pdf.ln(13)
        pdf.set_font('Arial','',6.0)
        pdf.cell(170,5,f'ApPlyWood © (2023) Michele Mirra. Report created on: {datetime.now():%d-%m-%Y %H:%M:%S}.', align='C')
        pdf.ln(2)
        pdf.cell(170,5, 'Disclaimer: ApPlyWood is provided by Dr.Ir. Michele Mirra under the GNU General Public License (GPLv3). The software is supplied "AS IS", without any warranties and support.', align = 'C')
        pdf.ln(2)
        pdf.cell(170,5,'The Author assumes no responsibility or liability for the use of the software and reserves the right to make changes in the software without notification.', align = 'C')
        pdf.ln(2)
        pdf.cell(170,5,'The Author also does not guarantee that such application will be suitable for the use selected by the user without further calculations and/or checks.', align='C')
        
 
        # Saving file dialogue
        
        with wx.FileDialog(self, "Save PDF file", defaultFile="Report", wildcard="PDF files (*.PDF)|*.PDF",
                   style=wx.FD_SAVE | wx.FD_OVERWRITE_PROMPT) as fileDialog:

            if fileDialog.ShowModal() == wx.ID_CANCEL:
                return     # the user changed their mind   
            pathname = fileDialog.GetPath()        
            try:
                with open(pathname, 'w'):
                    pdf.output(pathname,'F')
            except IOError:
                wx.LogError("Cannot save current data in file '%s'." % pathname)
        
    def EvtChar(self, event):
        event.Skip()
    
    def on_close(self,event):
        self.Destroy()
        sys.exit(0)

app = wx.App(False)
frame = MainWindow(None, "ApPlyWood")
frame.SetIcon(wx.Icon('Icon.ico'))
frame.Show()
app.MainLoop()