/**
 * Helper Functions used in multiple places, especially concerning sequences
 * @author Lukas Armborst, University of Twente
 * @year 2021
 */

public class Util {

    /*****************
     sequence prefixes
     *****************/
    
    /*@
    /// is seq1 a prefix of seq2?
        ensures seq1==seq2 ==> \result == true;
        ensures \result ==> |seq1|<=|seq2|;
        ensures \result && |seq1|>0 ==> isPrefix(tail(seq1), tail(seq2));
        ensures \result == (|seq1|<=|seq2| && (\forall int i; 0<=i && i<|seq1|; seq1[i]==seq2[i]));
    pure static boolean isPrefix(seq<int> seq1, seq<int> seq2)
        = |seq1| <= 0 ? true :
          (|seq2| <= 0 ? false
                       : seq1[0]==seq2[0]
                         && isPrefix(tail(seq1), tail(seq2)));

    /// is seq1 a prefix of seq2?
        ensures seq1==seq2 ==> \result == true;
        ensures \result ==> |seq1|<=|seq2|;
        ensures \result && |seq1|>0 ==> isPrefix(tail(seq1), tail(seq2));
        ensures \result == (|seq1|<=|seq2| && (\forall int i; 0<=i && i<|seq1|; seq1[i]==seq2[i]));
    pure static boolean isPrefix(seq<Node> seq1, seq<Node> seq2)
        = |seq1| <= 0 ? true :
          (|seq2| <= 0 ? false
                       : seq1[0]==seq2[0] &&
                         isPrefix(tail(seq1), tail(seq2)));

    /// if pref is a prefix of all, then it is also a prefix of any extended version of all
        requires isPrefix(pref, all);
        requires newAll == all + extension;
        ensures isPrefix(pref, newAll);
        ensures \result;
    static boolean prefixExtensionLemma(seq<Node> pref, seq<Node> all,
                                        seq<Node> extension, seq<Node> newAll)
        = |pref|>0 ==> |all|>0 && |newAll|>0
                        && prefixExtensionLemma(tail(pref), tail(all), extension, tail(newAll));

    /// isPrefix is transitive
        requires isPrefix(pref, intermed) ** isPrefix(intermed, all);
        ensures isPrefix(pref, all);
        ensures \result;
    static boolean prefixTransLemma(seq<Node> pref, seq<Node> intermed, seq<Node> all)
        = |pref|>0 ==> |intermed|>0 && |all|>0 
                       && prefixTransLemma(tail(pref), tail(intermed), tail(all));
    @*/
    
    
    /*****************
     sequence suffixes
     *****************/
    
    /*@
    /// is seq1 a suffix of seq2?
        ensures seq1==seq2 ==> \result == true;
        ensures \result ==> |seq1|<=|seq2|;
        ensures \result == (|seq1|<=|seq2| && (\forall int i; 0<=i && i<|seq1|; 
                                                  seq1[i]==seq2[i+|seq2|-|seq1|]));
    pure static boolean isSuffix(seq<Node> seq1, seq<Node> seq2)
        = |seq1| > |seq2| ? false :
          (|seq1| <= 0 ? true :
          (|seq1| < |seq2| ? isSuffix(seq1, tail(seq2))
                           : seq1[0]==seq2[0] &&
                             isSuffix(tail(seq1), tail(seq2))));

    /// is seq1 a suffix of seq2?
        ensures seq1==seq2 ==> \result == true;
        ensures \result ==> |seq1|<=|seq2|;
        ensures \result == (|seq1|<=|seq2| && (\forall int i; 0<=i && i<|seq1|; 
                                                    seq1[i]==seq2[i+|seq2|-|seq1|]));
    pure static boolean isSuffix(seq<int> seq1, seq<int> seq2)
        = |seq1| > |seq2| ? false :
          (|seq1| <= 0 ? true :
          (|seq1| < |seq2| ? isSuffix(seq1, tail(seq2))
                           : seq1[0]==seq2[0] &&
                             isSuffix(tail(seq1), tail(seq2))));

    /// isSuffix is transitive, shortening the suffix preserves relation
        requires isSuffix(suff, all);
        requires suff == taken + remainder;
        ensures isSuffix(remainder, all);
        ensures \result;
    pure static boolean suffixShortenLemma(seq<Node> suff, seq<Node> all, 
                                           seq<Node> taken, seq<Node> remainder)
        = |suff| < |all| ? suffixShortenLemma(suff, tail(all), taken, remainder) :
          (|taken| <= 0 ? suff==remainder && isSuffix(remainder, all)
                        : |suff| > 0 && |all|>0 
                          && suffixShortenLemma(tail(suff), tail(all), tail(taken), remainder));
    @*/
    
    
    /*****************
     sequence infixes
     *****************/
    
    /*@
    /// is inf an infix of all, starting at position idx?
        requires idx >= 0;
        ensures \result && |inf|>0 ==> idx <= |all|-|inf| && all[idx] == inf[0];
        ensures \result == (idx <= |all|-|inf| && (\forall int i; 0<=i && i<|inf|; 
                                                      inf[i]==all[i+idx]));
    pure static boolean isInfix(seq<int> inf, seq<int> all, int idx)
        = |all| < idx ? false :
          (idx == 0 ? isPrefix(inf, all)
                    : isInfix(inf, tail(all), idx-1));

    /// is inf an infix of all, starting at position idx?
        requires idx >= 0;
        ensures \result && |inf|>0 ==> idx <= |all|-|inf| && all[idx] == inf[0];
        ensures \result == (idx <= |all|-|inf| && (\forall int i; 0<=i && i<|inf|; 
                                                      inf[i]==all[i+idx]));
    pure static boolean isInfix(seq<Node> inf, seq<Node> all, int idx)
        = |all| < idx ? false :
          (idx == 0 ? isPrefix(inf, all)
                    : isInfix(inf, tail(all), idx-1));

    /// the suffix of an infix is an infix, too (at later index)
        requires idx >= 0;
        requires isInfix(suff, all, idx);
        requires suff == taken + remainder;
        ensures isInfix(remainder, all, idx+|taken|);
        ensures \result;
    pure static boolean infixSuffixLemma(seq<int> suff, seq<int> all, 
                                         seq<int> taken, seq<int> remainder, 
                                         int idx)
        = idx>0 ? infixSuffixLemma(suff, tail(all), taken, remainder, idx-1) 
                : (|taken| <= 0 ? suff==remainder && isInfix(remainder, all, idx)
                      : |suff| > 0 && |all|>0 
                        && infixSuffixLemma(tail(suff), tail(all), tail(taken), remainder, idx));

    /// if inf is an infix of all at position idx, then tail(inf) is an infix one index higher
        requires idx >= 0;
        requires isInfix(inf, all, idx);
        requires |inf| > 0;
        ensures \result;
        ensures isInfix(tail(inf), all, idx+1);
    pure static boolean infixTailLemma(seq<Node> inf, seq<Node> all, int idx);
        /// = idx == 0 ? isPrefix(tail(inf), tail(all)) ==> isInfix(tail(inf), all, 1)
                   /// : infixSuffixLemma(inf, tail(all), idx-1);

    /// concatenating two infixes that are back to back gives one big infix
        requires idx >= 0;
        requires isInfix(inf1, all, idx);
        requires isInfix(inf2, all, idx + |inf1|);
        ensures \result;
        ensures isInfix(inf1 + inf2, all, idx);
    pure static boolean infixAdditionLemma(seq<int> inf1, seq<int> inf2, seq<int> all, int idx);

    /// concatenating two infixes that are back to back gives one big infix
        requires idx >= 0;
        requires isInfix(inf1, all, idx);
        requires isInfix(inf2, all, idx + |inf1|);
        ensures \result;
        ensures isInfix(inf1 + inf2, all, idx);
    pure static boolean infixAdditionLemma(seq<Node> inf1, seq<Node> inf2, seq<Node> all, int idx);
        /// = |inf1| > 0 ? infixAdditionLemma(tail(inf1), inf2, all, idx+1)
                     /// : true;
    @*/


    /******************
     sortedness of ints
     ******************/
    
    /*@
    /// is the given integer sequence sorted?
    pure static boolean sorted(seq<int> s)
        = (\forall int i; 0<=i && i<|s|; (\forall int j; i<j && j<|s|; s[i]<=s[j]));

    /// the tail of a sorted sequence is sorted, too
        requires sorted(s);
        ensures sorted(tail(s));
    pure static boolean sortedTailLemma(seq<int> s) 
        = true;

    /// parts of a sorted list are sorted, 
    ///     and if the whole list is entirely smaller than another, then so are the parts
        requires sorted(all);
        requires all == front + back;
        requires smallerSeq(all, larger);
        ensures sorted(back);
        ensures smallerSeq(back, larger);
        ensures \result;
    pure static boolean sortedLemma(seq<int> all, seq<int> front, seq<int> back, seq<int> larger);

    /// are all values in "smaller" smaller than any value in "larger"?
    pure static boolean smallerSeq(seq<int> smaller, seq<int> larger)
        = (\forall int i; 0<=i && i<|smaller|;
            (\forall int j; 0<=j && j<|larger|;
                smaller[i] <= larger[j]));

    @*/


    /*****************
     sequence-to-bag
     *****************/
    
    /*@
    /// turn a sequence of Nodes into a multiset (bag)
        ensures |\result| == |s|;
        ensures (\forall int i; 0 <= i && i<|s|; (s[i] \memberof \result) > 0);
    public pure static bag<Node> toBag(seq<Node> s)
        = toBag(s, |s|);

    /// turn a sequence of Nodes up to given index into a multiset (bag)
        requires maxIdx >= 0 && maxIdx <= |s|;
        ensures |\result| == maxIdx;
        ensures (\forall int i; 0 <= i && i<maxIdx; (s[i] \memberof \result) > 0);
    public pure static bag<Node> toBag(seq<Node> s, int maxIdx)
        = toBagFrom(s, 0, maxIdx);

    /// turn a sequence of Nodes from given index up to given maxIndex into a multiset (bag)
        requires idx >= 0;
        requires maxIdx >= 0 && maxIdx <= |s|;
        ensures idx <= maxIdx ==> |\result| == maxIdx-idx;
        ensures (\forall int i; idx <= i && i<maxIdx; (s[i] \memberof \result) > 0);
    public pure static bag<Node> toBagFrom(seq<Node> s, int idx, int maxIdx)
        = idx >= maxIdx
            ? bag<Node>{}
            : bag<Node>{s[idx]} + toBagFrom(s, idx+1, maxIdx);

    /// if seq-to-bag only goes up to maxIdx, then replacing the seq with a prefix 
    ///     that is longer than maxIdx yields the same bag
        requires idx >= 0;
        requires idx <= maxIdx && maxIdx <= |pre|;
        requires toBagFrom(pre, idx, maxIdx) == asBag;
        requires isPrefix(pre, all);
        ensures toBagFrom(all, idx, maxIdx) == asBag;
        ensures \result;
    static pure boolean toBagPrefixLemma(seq<Node> pre, seq<Node> all, 
                                         int idx, int maxIdx, 
                                         bag<Node> asBag)
        = idx < maxIdx ? pre[idx] == all[idx]
                        && toBagPrefixLemma(pre, all, idx+1, maxIdx, asBag - bag<Node>{pre[idx]})
                     : asBag == bag<Node>{};

    /// increasing the maximal index of a seq-to-bag conversion adds the respective seq entry
        requires idx >= 0;
        requires idx <= maxIdx && maxIdx < |s|;
        requires toBagFrom(s, idx, maxIdx) == asBag;
        ensures toBagFrom(s, idx, maxIdx+1) == asBag + bag<Node>{s[maxIdx]};
        ensures \result;
    pure static boolean toBagMaxIdxIncrLemma(seq<Node> s, int idx, int maxIdx, bag<Node> asBag)
        = idx < maxIdx
            ? toBagMaxIdxIncrLemma(s, idx+1, maxIdx, asBag - bag<Node>{s[idx]})
            : asBag == bag<Node>{} && toBagFrom(s, idx, maxIdx+1) == bag<Node>{s[maxIdx]};

     @*/
}
