/**
 * functionality related to valid red-black trees
 * @author Lukas Armborst, University of Twente
 * @year 2021
 */

final class Tree {
    Node root;

    /*@ 
    static inline int max(int x,int y)
        = x > y ? x : y; 
    @*/
    
    
    /********************************************
     Tree structure
     ********************************************/
    /*@ 
    /// recursive permission to entire tree 
    /// this gives full access to the node, as well as to the subtrees (recursively).
    /// also does a few sanity checks
    public static resource tree_perm(Node current) 
        = current != null
                ==> Node.node_perm(current, write)
                    ** tree_perm(current.left) 
                    ** tree_perm(current.right) 
                    ** (current.dblackNull ==> current.left==null
                                                && current.right==null
                                                && !current.color
                                                && current.dblack)
                    ** (current.dblack ==> !current.color); 
    @*/


    /*@
    /// combination of the properties that make a tree a valid RB tree
        requires [read]tree_perm(current);
    public static pure boolean validTree(Node current)
        = validSubtree(current)
          && !getColor(current);

    /// combination of the properties that make a tree a sub-tree of a valid RB tree
        requires [read]tree_perm(current);
        requires sortedCritEqualLemma(current);
    public static pure boolean validSubtree(Node current)
        = blackBalanced(current) 
          && noDBlack(current) && noDoubleRed(current)
          && sorted(current)
          ;
    @*/

    /********************************************
     Black properties
     ********************************************/
    /*@ 
    /// tree height in terms of black nodes. 
        requires [read]tree_perm(current);
        ensures \result >= 1;
    public static pure int blackHeight(Node current) 
        = \unfolding [read]tree_perm(current) 
            \in (current == null ? 1
                 : (current.dblackNull ? 2
                 : (getBlacks(current) +
                    max(blackHeight(current.left),
                        blackHeight(current.right))
                )));
    @*/

    /*@
    /// get number of black markers on given node (0, 1 or 2)
        requires current != null ==> Perm(current.color, read) 
                                     ** Perm(current.dblack, read);
    public static pure int getBlacks(Node current) 
        = current == null ? 1 
            : (current.color ? 0 
            : (current.dblack ? 2 
            : 1));
    
    /// check whether given node is double-black
        requires current != null ==> Perm(current.dblack, read);
    public static pure boolean isDBlack(Node current) 
        = current != null && current.dblack;
    @*/
    
    /*@
    /// check whether black height of tree is balanced. 
        requires [read]tree_perm(current);
    public static pure boolean blackBalanced(Node current) 
        = \unfolding [read]tree_perm(current) 
            \in (current == null ? true 
                : blackHeight(current.left) == blackHeight(current.right)
                  && blackBalanced(current.left)
                  && blackBalanced(current.right));
    @*/    
    
    /*@
    /// check that given tree contains no double-black nodes
        requires [read]tree_perm(current);
    public static pure boolean noDBlack(Node current)
        = current != null
            ==> \unfolding [read]tree_perm(current) 
                    \in !isDBlack(current)
                        && noDBlack(current.left)
                        && noDBlack(current.right);
    @*/
    
    /*@
    /// check whether given tree has a double-black root and no other double-black nodes
        requires [read]tree_perm(current);
    static pure boolean dblackAtTop(Node current)
        = \unfolding [read]tree_perm(current) 
            \in isDBlack(current) && noDBlack(current.left) && noDBlack(current.right);

    /// check whether given tree has a double-black left node and no other double-black nodes
        requires [read]tree_perm(current);
    static pure boolean dblackAtLeft(Node current)
        = \unfolding [read]tree_perm(current) 
            \in current != null && !isDBlack(current) 
                && dblackAtTop(current.left) && noDBlack(current.right);

    /// check whether given tree has a double-black left-most grandchild 
    ///     and no other double-black nodes
        requires [read]tree_perm(current);
    static pure boolean dblackAtLeftLeft(Node current)
        = \unfolding [read]tree_perm(current) 
            \in current != null && !isDBlack(current) 
                && dblackAtLeft(current.left) && noDBlack(current.right);

    /// check whether given tree has a double-black right node and no other double-black nodes
        requires [read]tree_perm(current);
    static pure boolean dblackAtRight(Node current)
        = \unfolding [read]tree_perm(current) 
            \in current != null && !isDBlack(current) 
                && noDBlack(current.left) && dblackAtTop(current.right);

    /// check whether given tree has a double-black right-most grandchild 
    ///     and no other double-black nodes
        requires [read]tree_perm(current);
    static pure boolean dblackAtRightRight(Node current)
        = \unfolding [read]tree_perm(current) 
            \in current != null && !isDBlack(current) 
                && noDBlack(current.left) && dblackAtRight(current.right);
     @*/
    

    /********************************************
     Red properties
     ********************************************/
    /*@ 
    /// check that there are no double-red node pairs in the tree
        requires [read]tree_perm(current);
    public static pure boolean noDoubleRed(Node current) 
        = current != null
            ==> \unfolding [read]tree_perm(current) 
                    \in ((!getColor(current) 
                          || (!getColor(current.left) && !getColor(current.right))) 
                         && noDoubleRed(current.left) 
                         && noDoubleRed(current.right)); 
    @*/

    /*@ 
    /// check that given node and one of its children are red, otherwise no double-red
        requires [read]tree_perm(current); 
    static pure inline boolean dbRedAtTop(Node current)
        = dbRedAtLeft(current) || dbRedAtRight(current); 

    /// check that given node and its left child are red, otherwise no double-red
        requires [read]tree_perm(current);
    public static pure boolean dbRedAtLeft(Node current) 
        = current != null && \unfolding [read]tree_perm(current) 
                                \in (getColor(current) 
                                     && (getColor(current.left) && !getColor(current.right)) 
                                     && noDoubleRed(current.left) 
                                     && noDoubleRed(current.right)); 

    /// check that given node and its right child are red, otherwise no double-red
        requires [read]tree_perm(current);
    public static pure boolean dbRedAtRight(Node current) 
        = current != null && \unfolding [read]tree_perm(current) 
                                \in (getColor(current) 
                                     && (!getColor(current.left) && getColor(current.right)) 
                                     && noDoubleRed(current.left) 
                                     && noDoubleRed(current.right)); 
    @*/



    /********************************************
     Bag and Sequence properties
     ********************************************/
    /*@ 
    /// make a bag of the given tree's keys
        requires [read]tree_perm(current);
    public static pure bag<int> toBag(Node current)
        = current == null
            ? bag<int> { }
            : \unfolding [read]tree_perm(current) 
                \in (current.dblackNull ? bag<int> {}
                        : toBag(current.left) 
                          + bag<int> { current.key }
                          + toBag(current.right)); 
    @*/

    /*@ 
    /// check that b1 is a sub-bag of b2 (like subset)
    static pure boolean subbag(bag<int> b1, bag<int> b2) 
        = (\forall int i; (i \memberof b1) != 0; (i \memberof b2) >= (i \memberof b1)); 
    @*/


    /*@ 
    /// make a sequence of the given tree's nodes
        requires [read]tree_perm(current);
    public static pure seq<Node> toSeq(Node current)
        = current == null
            ? seq<Node> { }
            : \unfolding [read]tree_perm(current) 
                \in (current.dblackNull ? seq<Node> {}
                        : toSeq(current.left) 
                          + seq<Node> { current }
                          + toSeq(current.right)); 


    /// make a sequence of the given tree's keys
        requires [read]tree_perm(current);
        ensures |\result| == |toSeq(current)|;
    public static pure seq<int> toSeqKeys(Node current)
        = current == null
            ? seq<int> { }
            : \unfolding [read]tree_perm(current) 
                \in (current.dblackNull ? seq<int> {}
                        : toSeqKeys(current.left) 
                          + seq<int> { current.key }
                          + toSeqKeys(current.right)); 
    @*/



    /********************************************
     Sorting properties
     ********************************************/
    /*@ 
    /// check that all values in given bag are smaller than given value
    public static pure boolean smaller(bag<int> b, int max) 
        = (\forall int i; {: (i \memberof b) :} != 0; i <= max); 

    /// check that all values in given bag are greater than given value
    public static pure boolean larger(bag<int> b, int min) 
        = (\forall int i; {: (i \memberof b) :} != 0; i >= min); 
    
    /// check that all values in b1 are greater than all values in b2
    public static pure boolean larger(bag<int> b1, bag<int> b2) 
        = (\forall int i; {: (i \memberof b2) :} != 0; larger(b1, i)); 
    @*/

    /*@
    /// check that keys in given tree are sorted
        requires [read]tree_perm(current);
    public static pure boolean sorted(Node current) 
        = current != null
            ==> \unfolding [read]tree_perm(current) 
                    \in (smaller(toBag(current.left), current.key) 
                         && larger(toBag(current.right), current.key) 
                         && sorted(current.left) 
                         && sorted(current.right)); 

        requires [read]tree_perm(current);
    public static pure boolean sortedKeySeq(Node current) 
        = current != null 
            ==> \unfolding [read]tree_perm(current) 
                    \in (\forall int i; 0<=i && i<|toSeqKeys(current.left)|; 
                            toSeqKeys(current.left)[i] <= current.key) 
                        && (\forall int i; 0<=i && i<|toSeqKeys(current.right)|; 
                            toSeqKeys(current.right)[i] >= current.key) 
                        && sortedKeySeq(current.left) 
                        && sortedKeySeq(current.right); 
    

        requires [read]tree_perm(current);
        ensures \result;
        ensures sorted(current) == sortedKeySeq(current);
    static pure boolean sortedCritEqualLemma(Node current);
    @*/

    /*@
    /// assert that smaller/larger also holds for sub-bags
        requires bigBag == subBag + diff;
        ensures smaller(bigBag,key) ==> smaller(subBag,key);
        ensures larger(bigBag,key) ==> larger(subBag,key);
    ghost public static void subbagCompareLemma(bag<int> bigBag, bag<int> subBag, bag<int> diff,
                                         int key)
    {
        /// the critical piece to prove the relation: ensuring subbag property
        assert (\forall int i; {: (i \memberof subBag) :} != 0; (i \memberof bigBag) != 0);
    }
    @*/
        
    
    /********************************************
     Color
     ********************************************/
    /*@ 
    /// get the color of the given node
        requires [read]tree_perm(node);
    public static pure boolean getColor(Node node) 
        = node != null
            && \unfolding [read]tree_perm(node) \in (node.color); 
    @*/


    /*@
    /// get the color of the given node's left child
        requires [read]tree_perm(node);
        requires node != null;
    public static pure inline boolean getColorLeft(Node node) 
        = \unfolding tree_perm(node) \in getColor(node.left);
    
    /// get the color of the given node's right child
        requires [read]tree_perm(node);
        requires node != null;
    public static pure inline boolean getColorRight(Node node) 
        = \unfolding tree_perm(node) \in getColor(node.right);
    @*/


    /********************************************
     Manage Magic Wand
     ********************************************/

    /*@
        requires [read]tree_perm(current);
    static pure boolean subtreeFitsHole(Node current, boolean oldColor, 
                                         int oldHeight, bag<int> oldBag)
        = getColor(current) == oldColor
            && blackHeight(current) == oldHeight
            && toBag(current) == oldBag;
    
    
    /// create a magic wand for current and res, 
    ///     based on a magic wand for containingSubtree and res
        requires current != null;
        /// necessary parts to fold tree
        requires Node.node_perm(current, write);
        requires tree_perm(otherSubtree);
        requires (current.dblackNull ==> current.left==null
                                            && current.right==null
                                            && !current.color
                                            && current.dblack)
                    ** (current.dblack ==> !current.color);
        requires goLeft ? (containingSubtree == current.left && otherSubtree == current.right)
                         : (containingSubtree == current.right && otherSubtree == current.left);
        requires (tree_perm(res) ** validSubtree(res)
                        ** subtreeFitsHole(res, resColor, resHeight, resBag))
                    -* (tree_perm(containingSubtree) ** validSubtree(containingSubtree)
                        ** subtreeFitsHole(containingSubtree, subColor, subHeight, subBag));
        /// necessary parts for black balanced
        requires subHeight == blackHeight(otherSubtree) 
                    && blackBalanced(otherSubtree);
        requires oldHeight == getBlacks(current) + subHeight;
        /// necessary parts for noDBlack
        requires noDBlack(otherSubtree) ** !current.dblack;
        /// necessary parts for noDoubleRed
        requires current.color == oldColor;
        requires (!current.color || (!subColor && !getColor(otherSubtree))) 
                 && noDoubleRed(otherSubtree);
        /// necessary parts for sortedKeySeq
        requires oldBag == subBag + toBag(otherSubtree) + bag<int>{current.key};
        requires goLeft
                    ? smaller(subBag, current.key) && larger(toBag(otherSubtree), current.key)
                    : larger(subBag, current.key) && smaller(toBag(otherSubtree), current.key);
        requires sorted(otherSubtree);
        ensures (tree_perm(res) ** validSubtree(res)
                    ** subtreeFitsHole(res, resColor, resHeight, resBag))
                -* (tree_perm(current) ** validSubtree(current)
                    ** subtreeFitsHole(current, oldColor, oldHeight, oldBag));
    ghost static void makeWand(Node res, Node current, Node containingSubtree, Node otherSubtree,
                          boolean oldColor, int oldHeight, bag<int> oldBag,
                          boolean subColor, int subHeight, bag<int> subBag,
                          boolean resColor, int resHeight, bag<int> resBag,
                          boolean goLeft)
    {
        create {
            /// necessary parts to fold tree
            use Node.node_perm(current, write);
            use tree_perm(otherSubtree);
            use (current.dblackNull ==> current.left==null
                                        && current.right==null
                                        && !current.color
                                        && current.dblack)
                ** (current.dblack ==> !current.color);
            use goLeft ? (containingSubtree == current.left && otherSubtree == current.right)
                        : (containingSubtree == current.right && otherSubtree == current.left);
            use (tree_perm(res) ** validSubtree(res)
                        ** subtreeFitsHole(res, resColor, resHeight, resBag))
                -* (tree_perm(containingSubtree) ** validSubtree(containingSubtree)
                    ** subtreeFitsHole(containingSubtree, subColor, subHeight, subBag));
            apply (tree_perm(res) ** validSubtree(res)
                        ** subtreeFitsHole(res, resColor, resHeight, resBag))
                    -* (tree_perm(containingSubtree) ** validSubtree(containingSubtree)
                        ** subtreeFitsHole(containingSubtree, subColor, subHeight, subBag));
            assert goLeft
                    ? containingSubtree == current.left ** otherSubtree == current.right 
                        ** tree_perm(current.left) ** tree_perm(current.right)
                    : containingSubtree == current.right ** otherSubtree == current.left
                        ** tree_perm(current.left) ** tree_perm(current.right);
            /// necessary parts for black balanced
            use subHeight == blackHeight(otherSubtree) 
                && blackBalanced(otherSubtree);
            use oldHeight == getBlacks(current) + subHeight;
            /// necessary parts for noDBlack
            use noDBlack(otherSubtree) ** !current.dblack;
            /// necessary parts for noDoubleRed
            use current.color == oldColor;
            use (!current.color || (!subColor && !getColor(otherSubtree))) 
                && noDoubleRed(otherSubtree);
            /// necessary parts for sortedKeySeq
            use oldBag == subBag + toBag(otherSubtree) + bag<int>{current.key};
            use goLeft
                    ? smaller(subBag, current.key) && larger(toBag(otherSubtree), current.key)
                    : larger(subBag, current.key) && smaller(toBag(otherSubtree), current.key);
            use sorted(otherSubtree);
            /// join things together
            fold tree_perm(current);
            qed (tree_perm(res) ** validSubtree(res)
                    ** subtreeFitsHole(res, resColor, resHeight, resBag))
                -* (tree_perm(current) ** validSubtree(current)
                    ** subtreeFitsHole(current, oldColor, oldHeight, oldBag));
        }
    }
    @*/





    /********************************************
     Actual Code
     ********************************************/
    
    
    
    
    /*@ 
        context [1\2]tree_perm(node);
        ensures \result == (node == null || \unfolding [1\2]tree_perm(node) \in !node.color);
    @*/
    public static boolean isBlack(Node node) {
        /*@ unfold [1\2]tree_perm(node); @*/ 
        boolean res = node == null || !node.color;
        /*@ fold [1\2]tree_perm(node); @*/ 
        return res;
    }
    
    /*@ 
        context [1\2]tree_perm(node);
        ensures \result == (node != null && \unfolding [1\2]tree_perm(node) \in node.color);
    @*/
    public static boolean isRed(Node node) {
        /*@ unfold [1\2]tree_perm(node); @*/ 
        boolean res = node != null && node.color;
        /*@ fold [1\2]tree_perm(node); @*/ 
        return res;
    }

    /// removes a potential double black marker, turning the node into a simple black.
    ///     if given node is double-black null, return actual null
    /*@
        requires tree_perm(node); 
        requires node != null; 
        ensures tree_perm(\result); 
        ensures \unfolding tree_perm(\result) \in getBlacks(\result)==1; 
        ensures \old(sorted(node)) ==> sorted(\result); 
        ensures \old(blackBalanced(node)) ==> blackBalanced(\result); 
        ensures \old(noDoubleRed(node)|| dbRedAtTop(node)) ==> noDoubleRed(\result); 
        ensures \old(toBag(node)) == toBag(\result); 
        ensures \old(\unfolding tree_perm(node) \in getBlacks(node)) == 2 
                    ? \old(blackHeight(node))-1 == blackHeight(\result)
                    : (\old(\unfolding tree_perm(node) \in getBlacks(node)) == 1 
                        ? \old(blackHeight(node)) == blackHeight(\result)
                        : \old(blackHeight(node))+1 == blackHeight(\result)); 
        ensures \old(dblackAtTop(node)) ==> noDBlack(\result);
        ensures \old(noDBlack(node)) ==> noDBlack(\result);
    @*/
    public static Node makeNodeSingleBlack(Node node) {
        /*@ unfold tree_perm(node); @*/
        if (node.dblackNull) {
            /*@ fold tree_perm(node); @*/
            Node res = null;
            /*@ fold tree_perm(res); @*/
            return res;
        }
        node.dblack = false;
        node.color = false;
        /*@ fold tree_perm(node); @*/
        return node;
    }
    
    
    
    
    /********************************************
     Insert
     ********************************************/
    /// insert a new integer into the tree spanned by root
    /*@ 
        context Perm(root, 1) ** tree_perm(root); 
        context noDBlack(root) ** sorted(root) ** noDoubleRed(root) ** blackBalanced(root);
        requires Perm(Integer.MIN_VALUE, read);
        requires Integer.MIN_VALUE <= key;
    @*/
    public void insert(int key)
    {
        root = insertRec(root, key);
        /// insertRec might color root red, but root must always be black
        root = makeNodeSingleBlack(root);
    }

    /// recursively traverse tree and insert given int at proper place. 
    ///     Ensure that the resulting tree is a valid RB tree 
    ///     (except potential double-red at top, which will be fixed at higher level)
    /*@
        requires tree_perm(current); 
        requires Perm(Integer.MIN_VALUE, read);
        requires Integer.MIN_VALUE <= key;
        requires sorted(current) ** noDBlack(current) ** blackBalanced(current);
        requires noDoubleRed(current);
        ensures \result != null ** tree_perm(\result); 
        ensures sorted(\result) ** noDBlack(\result) ** blackBalanced(\result);
        ensures noDoubleRed(\result) || dbRedAtTop(\result); 
        ensures \old(blackHeight(current)) == blackHeight(\result); 
        /// insertRec actually inserted the key
        ensures \old(toBag(current)) + bag<int> { key } == toBag(\result); 
        /// color of current did not change, or it turned from black to red. The latter might 
        ///     create a double-red with parent. But then, children must be black, so no triple-red.
        ensures (\old(getColor(current)) == getColor(\result))
                || (getColor(\result) && !getColorLeft(\result) && !getColorRight(\result));
    @*/
    Node insertRec(Node current, int key)
    {
        /// leaf -> create new red node
        if (current == null) {
            Node result = new Node(key);
            /*@ fold tree_perm(result.left); @*/
            /*@ fold tree_perm(result.right); @*/
            /*@ fold tree_perm(result); @*/
            return result;
        }
    
        /// not leaf -> recursively descend to leaf
        /*@ unfold tree_perm(current); @*/
        if (key <= current.key) {
            current.left = insertRec(current.left, key);
        } else {
            current.right = insertRec(current.right, key);
        }
        
        /// fix potential double-red
        if (current.color == false) {
            if (isRed(current.left) && isRed(current.right)) {
            /*@ unfold tree_perm(current.left); @*/
            /*@ unfold tree_perm(current.right); @*/
                if (isRed(current.left.left) || isRed(current.left.right)
                        || isRed(current.right.left) || isRed(current.right.right)) {
                    /// double red easy to fix by pushing down black marker
                    current.color = true;
                    current.left.color = false;
                    current.right.color = false;
                }
                /*@ fold tree_perm(current.left); @*/
                /*@ fold tree_perm(current.right); @*/
                /*@ fold tree_perm(current); @*/
                return current;
            }
            if (isRed(current.left)) {
                /*@ unfold tree_perm(current.left); @*/
                if (isRed(current.left.right)) {
                    /// rotate double-red to outer grandchild
                    /*@ fold tree_perm(current.left); @*/
                    current.left = rotateLeft(current.left);
                    /*@ unfold tree_perm(current.left); @*/
                }
                if (isRed(current.left.left)) {
                    /// fix double red in outer child/grandchild by rotating tree
                    /*@ fold tree_perm(current.left); @*/
                    /*@ fold tree_perm(current); @*/
                    Node res = rotateRight(current);
                    return res;
                }
                /// no double-red
                /*@ fold tree_perm(current.left); @*/
                /*@ fold tree_perm(current); @*/
                return current;
            }
            if (isRed(current.right)) {
                /*@ unfold tree_perm(current.right); @*/
                if (isRed(current.right.left)) {
                    /// rotate double-red to outer grandchild
                    /*@ fold tree_perm(current.right); @*/
                    current.right = rotateRight(current.right);
                    /*@ unfold tree_perm(current.right); @*/
                }
                if (isRed(current.right.right)) {
                    /// fix double red in outer child/grandchild by rotating tree
                    /*@ fold tree_perm(current.right); @*/
                    /*@ fold tree_perm(current); @*/
                    Node res = rotateLeft(current); 
                    return res;
                }
                /// no double-red
                /*@ fold tree_perm(current.right); @*/
                /*@ fold tree_perm(current); @*/
                return current;
            }
        }
        /// no double-red in subtrees, but maybe at top
        /*@ fold tree_perm(current); @*/
        return current;
    }



    /********************************************
     Delete
     ********************************************/
    /// remove given value from tree root
    /*@ 
        context Perm(root, 1) ** tree_perm(root); 
        context sorted(root) ** noDoubleRed(root) ** noDBlack(root) ** blackBalanced(root);
    @*/
    public void deleteKey(int key)
    {
        if (root != null) {
            root = deleteRec(root, key);
        }
        if (root != null) {
            root = makeNodeSingleBlack(root);
        }
    }


    /// Take a tree with double-black at left child, and rotate and recolor 
    ///     until no double-black, or only at top
    /*@
        requires current != null; 
        requires tree_perm(current); 
        requires sorted(current) ** noDoubleRed(current);
        requires blackBalanced(current) ** dblackAtLeft(current); 
        ensures tree_perm(\result);
        ensures sorted(\result) ** noDoubleRed(\result); 
        ensures blackBalanced(\result) ** (noDBlack(\result) || dblackAtTop(\result));
        ensures \old(toBag(current)) == toBag(\result); 
        ensures !\old(getColor(current)) ==> !getColor(\result); 
        ensures \old(blackHeight(current)) == blackHeight(\result); 
    @*/
    Node fixDBlackLeft(Node current)
    {
        /*@ unfold tree_perm(current); @*/
        if (isRed(current.right)) {
            /// unfold left node to assure that it is black (this follows from it being 
            ///     double black, but this implication is folded in the resource)
            /*@ unfold tree_perm(current.left); @*/
            /*@ fold tree_perm(current.left); @*/
            /*@ fold tree_perm(current); @*/
            Node result = rotateLeft(current);
            /*@ unfold tree_perm(result); @*/
            Node newCur = result.left;
            /*@ unfold tree_perm(newCur); @*/
            /// remove double-black marker. This creates a slight imbalance in newCur
            newCur.left = makeNodeSingleBlack(newCur.left);
            /*@ unfold tree_perm(newCur.right); @*/
            if (isBlack(newCur.right.left) && isBlack(newCur.right.right)) {
                newCur.right.color = true;
                /*@ fold tree_perm(newCur.right); @*/
                newCur.color = false;
                /*@ fold tree_perm(newCur); @*/
                /*@ fold tree_perm(result); @*/
                return result;
            }
            if (isRed(newCur.right.left) && isBlack(newCur.right.right)) {
                /// rotate red node outwards
                /*@ fold tree_perm(newCur.right); @*/
                newCur.right = rotateRight(newCur.right);
                /*@ unfold tree_perm(newCur.right); @*/
            }
            if (isRed(newCur.right.right)) {
                /// rotate black newCur.right over, shifting imbalance
                /*@ fold tree_perm(newCur.right); @*/
                newCur.color = false;  /// temp. recolor newCur to avoid making double-red
                /*@ fold tree_perm(newCur); @*/
                result.left = rotateLeft(newCur);
                /*@ unfold tree_perm(result.left); @*/
                result.left.color = true;   /// undo temp. recoloring
                /// fix imbalance
                /*@ unfold tree_perm(result.left.right); @*/
                result.left.right.color = false;
                /*@ fold tree_perm(result.left.right); @*/
                /*@ fold tree_perm(result.left); @*/
                /*@ fold tree_perm(result); @*/
                return result;
            }
            /// this should be unreachable, as previous cases are exhaustive
            /*@ fold tree_perm(newCur.right); @*/
            /*@ fold tree_perm(newCur); @*/
            /*@ fold tree_perm(result); @*/
            return result;
        }
        
        /// remove double-black marker. This creates a slight imbalance in current
        current.left = makeNodeSingleBlack(current.left);
        /*@ unfold tree_perm(current.right); @*/
        if (isBlack(current.right.left)
                && isBlack(current.right.right))
        {
            current.right.color = true;
            if (current.color == true) {
                current.color = false;
            } else {
                current.dblack = true;
            }
            /*@ fold tree_perm(current.right); @*/
            /*@ fold tree_perm(current); @*/
            return current;
        }
        if (isRed(current.right.left)
            && isBlack(current.right.right))
        {
            /// rotate red node outwards
            /*@ fold tree_perm(current.right); @*/
            current.right = rotateRight(current.right);
            /*@ unfold tree_perm(current.right); @*/
        }
        if (isRed(current.right.right)) {
            /// rotate black current.right over, shifting imbalance
            /*@ fold tree_perm(current.right); @*/
            boolean curColor = current.color;
            current.color = false;  /// temp. recolor current to avoid making double-red
            /*@ fold tree_perm(current); @*/
            Node result = rotateLeft(current);
            /*@ unfold tree_perm(result); @*/
            result.color = curColor;   /// undo temp. recoloring
            /// fix imbalance
            /*@ unfold tree_perm(result.right); @*/
            result.right.color = false;
            /*@ fold tree_perm(result.right); @*/
            /*@ fold tree_perm(result); @*/
            return result;
        }
        /// this should be unreachable, as previous cases are exhaustive
        /*@ fold tree_perm(current.right); @*/
        /*@ fold tree_perm(current); @*/
        return current;
    }


    /// Take a tree with double-black at right child, and rotate and recolor 
    ///     until no double-black, or only at top
    /*@
        requires current != null; 
        requires tree_perm(current); 
        requires sorted(current) ** noDoubleRed(current); 
        requires blackBalanced(current) ** dblackAtRight(current); 
        ensures tree_perm(\result);
        ensures sorted(\result) ** noDoubleRed(\result); 
        ensures blackBalanced(\result) ** (noDBlack(\result) || dblackAtTop(\result));
        ensures \old(toBag(current)) == toBag(\result); 
        ensures !\old(getColor(current)) ==> !getColor(\result); 
        ensures \old(blackHeight(current)) == blackHeight(\result); 
    @*/
    Node fixDBlackRight(Node current)
    {
        /*@ unfold tree_perm(current); @*/
        if (isRed(current.left)) {
            /// unfold right node to assure that it is black (this follows from it being 
            ///     double black, but this implication is folded in the resource)
            /*@ unfold tree_perm(current.right); @*/
            /*@ fold tree_perm(current.right); @*/
            /*@ fold tree_perm(current); @*/
            Node result = rotateRight(current);
            /*@ unfold tree_perm(result); @*/
            Node newCur = result.right;
            /*@ unfold tree_perm(newCur); @*/
            /// remove double-black marker. This creates a slight imbalance in newCur
            newCur.right = makeNodeSingleBlack(newCur.right);
            /*@ unfold tree_perm(newCur.left); @*/
            if (isBlack(newCur.left.left) && isBlack(newCur.left.right)) {
                newCur.left.color = true;
                newCur.color = false;
                /*@ fold tree_perm(newCur.left); @*/
                /*@ fold tree_perm(newCur); @*/
                /*@ fold tree_perm(result); @*/
                return result;
            }
            if (isBlack(newCur.left.left) && isRed(newCur.left.right)) {
                /// rotate red node outwards
                /*@ fold tree_perm(newCur.left); @*/
                newCur.left = rotateLeft(newCur.left);
                /*@ unfold tree_perm(newCur.left); @*/
            }
            if (isRed(newCur.left.left)) {
                /// rotate black newCur.left over, shifting imbalance
                /*@ fold tree_perm(newCur.left); @*/
                newCur.color = false;  /// temp. recolor newCur to avoid making double-red
                /*@ fold tree_perm(newCur); @*/
                result.right = rotateRight(newCur);
                /*@ unfold tree_perm(result.right); @*/
                result.right.color = true;   /// undo temp. recoloring
                /// fix imbalance
                /*@ unfold tree_perm(result.right.left); @*/
                result.right.left.color = false;
                /*@ fold tree_perm(result.right.left); @*/
                /*@ fold tree_perm(result.right); @*/
                /*@ fold tree_perm(result); @*/
                return result;
            }
            /// this should be unreachable, as previous cases are exhaustive
            /*@ fold tree_perm(newCur.left); @*/
            /*@ fold tree_perm(newCur); @*/
            /*@ fold tree_perm(result); @*/
            return result;
        }
        /// remove double-black marker. This creates a slight imbalance in current
        current.right = makeNodeSingleBlack(current.right);
        /*@ unfold tree_perm(current.left); @*/
        if (current.left.color == false 
            && isBlack(current.left.left)
            && isBlack(current.left.right))
        {
            current.left.color = true;
            if (current.color == true) {
                current.color = false;
            } else {
                current.dblack = true;
            }
            /*@ fold tree_perm(current.left); @*/
            /*@ fold tree_perm(current); @*/
            return current;
        }
        if (isBlack(current.left.left) && isRed(current.left.right)) {
            /// rotate red node outwards
            /*@ fold tree_perm(current.left); @*/
            current.left = rotateLeft(current.left);
            /*@ unfold tree_perm(current.left); @*/
        }
        if (isRed(current.left.left)) {
            /// rotate black current.left over, shifting imbalance
            /*@ fold tree_perm(current.left); @*/
            boolean curColor = current.color;
            current.color = false;  /// temp. recolor current to avoid making double-red
            /*@ fold tree_perm(current); @*/
            Node result = rotateRight(current);
            /*@ unfold tree_perm(result); @*/
            result.color = curColor;   /// undo temp. recoloring
            /// fix imbalance
            /*@ unfold tree_perm(result.left); @*/
            result.left.color = false;
            /*@ fold tree_perm(result.left); @*/
            /*@ fold tree_perm(result); @*/
            return result;
        }
        /*@ fold tree_perm(current.left); @*/
        /*@ fold tree_perm(current); @*/
        return current;
    }
    
    /// recursive function to delete given value from given tree.
    ///     Ensures that result is still valid RB tree (except maybe double-black at top)
    /*@
        requires current != null ** tree_perm(current);
        requires sorted(current) ** noDoubleRed(current) ; 
        requires blackBalanced(current)** noDBlack(current);
        ensures tree_perm(\result);
        ensures sorted(\result) ** noDoubleRed(\result); 
        ensures blackBalanced(\result) ** (noDBlack(\result) || dblackAtTop(\result));
        ensures (key \memberof \old(toBag(current))) == 0
                        ? \old(toBag(current)) == toBag(\result)
                        : \old(toBag(current)) == toBag(\result) + bag<int> { key }; 
        ensures !\old(getColor(current)) ==> !getColor(\result); 
        ensures blackHeight(\result) == \old(blackHeight(current)); 
    @*/
    Node deleteRec(Node current, int key)
    {
        /*@ unfold tree_perm(current); @*/
        if (key < current.key) {
            if (current.left == null) {
                /// key does not exist in tree
                /*@ fold tree_perm(current); @*/
                return current;
            }
            /*@ ghost bag<int> leftBefore = toBag(current.left); @*/
            current.left = deleteRec(current.left, key);
            /*@ ghost bag<int> leftAfter = toBag(current.left); @*/
            /*@ assert subbag(leftAfter, leftBefore); @*/
            
            /// double black? fix
            /*@ unfold tree_perm(current.left); @*/
            if (current.right != null && current.left != null 
                && current.left.color == false && current.left.dblack == true) 
            {
                /*@ fold tree_perm(current.left); @*/
                /*@ fold tree_perm(current); @*/
                Node fixed = fixDBlackLeft(current);
                return fixed;
            }
            /*@ fold tree_perm(current.left); @*/
            /*@ fold tree_perm(current); @*/
            return current;
        }
        if (key > current.key) {
            if (current.right == null) {
                /// key does not exist in tree
                /*@ fold tree_perm(current); @*/
                return current;
            }
            /*@ ghost bag<int> rightBefore = toBag(current.right); @*/
            current.right = deleteRec(current.right, key);
            /*@ ghost bag<int> rightAfter = toBag(current.right); @*/
            /*@ assert subbag(rightAfter, rightBefore); @*/
            
            /// double black? fix
            /*@ unfold tree_perm(current.right); @*/
            if (current.left != null && current.right != null 
                && current.right.color == false && current.right.dblack == true)
            {
                /*@ fold tree_perm(current.right); @*/
                /*@ fold tree_perm(current); @*/
                Node fixed = fixDBlackRight(current);
                return fixed;
            }
            /*@ fold tree_perm(current.right); @*/
            /*@ fold tree_perm(current); @*/
            return current;
        }
        
        /// need to remove current
        
        if (current.left == null) {
            Node result = current.right;
            if (current.color == false) {
                if (result == null) {
                    /// both children are null, should make current null. 
                    ///     But double-black -> keep dummy around
                    current.dblack = true;
                    current.dblackNull = true;
                    /*@ fold tree_perm(current); @*/
                    return current;
                } else {
                    /// pass black marker to replacement node
                    /*@ unfold tree_perm(result); @*/
                    if (result.color == true) {
                        result.color = false;
                    } else {
                        result.dblack = true;
                    }
                    /*@ fold tree_perm(result); @*/
                }
            }
            /// replace current with right child
            return result;
        }
        if (current.right == null) {
            /// replace current with left child
            Node result = current.left;
            /*@ unfold tree_perm(result); @*/
            if (current.color == false && result != null) {
                /// pass black marker to replacement node
                if (result.color == true) {
                    result.color = false;
                } else {
                    result.dblack = true;
                }
            }
            /*@ fold tree_perm(result); @*/
            return result;
        }
        /// current must be deleted, but has two children -> find replacement
        /*@ ghost boolean rightColor = getColor(current.right); @*/
        /*@ ghost int rightHeight = blackHeight(current.right); @*/
        /*@ ghost bag<int> rightBag = toBag(current.right); @*/
        Node successor = getMin(current.right);
        /*@ unfold tree_perm(successor); @*/
        /// copy content of replacement into current node
        current.key = successor.key;
        /*@ fold tree_perm(successor); @*/
        /*@
            apply (tree_perm(successor) ** validSubtree(successor)
                    ** subtreeFitsHole(successor, getColor(successor), 
                                        blackHeight(successor), toBag(successor)))
                    -* (tree_perm(current.right) ** validSubtree(current.right)
                        ** subtreeFitsHole(current.right, rightColor, rightHeight, rightBag));
        @*/
        /// delete original version of replacing node
        current.right = deleteRec(current.right, current.key);
        /*@ ghost bag<int> rightAfter = toBag(current.right); @*/
        /*@ assert subbag(rightAfter, rightBag); @*/
        
        /// double black? fix
        /*@ unfold tree_perm(current.right); @*/
        if (current.right != null && current.right.color == false && current.right.dblack == true) {
            /*@ fold tree_perm(current.right); @*/
            /*@ fold tree_perm(current); @*/
            Node fixed = fixDBlackRight(current);
            return fixed;
        }
        /*@ fold tree_perm(current.right); @*/
        /*@ fold tree_perm(current); @*/
        return current;
    }

    
    /********************************************
     Retrieve Node
     ********************************************/

    /// return the node with the smallest key in the given tree (i.e. left-most)
    /*@
        yields boolean resColor;
        yields int resHeight;
        yields bag<int> resBag;
        requires tree_perm(current) ** validSubtree(current);
        ensures tree_perm(\result) ** validSubtree(\result);
        ensures (tree_perm(\result) ** validSubtree(\result)
                    ** subtreeFitsHole(\result, resColor, resHeight, resBag))
                -* (tree_perm(current) ** validSubtree(current)
                    ** subtreeFitsHole(current, \old(getColor(current)),
                                        \old(blackHeight(current)), \old(toBag(current))));
        ensures current == null ==> \result == null;
        ensures current != null ==> \result != null 
                                    ** (\unfolding tree_perm(\result) \in \result.left == null)
                                    ** larger(\old(toBag(current)), 
                                              \unfolding tree_perm(\result) \in \result.key)
                                    ** (\unfolding tree_perm(\result)
                                          \in \result.key \memberof \old(toBag(current))) > 0;
        ensures resColor == getColor(\result) ** resHeight == blackHeight(\result) 
                ** resBag == toBag(\result);
    @*/
    Node getMin(Node current)
    {
        Node res;
        /*@ 
        ghost boolean oldColor = getColor(current); 
        ghost int oldHeight = blackHeight(current); 
        ghost bag<int> oldBag = toBag(current); 
        @*/
        /*@ unfold tree_perm(current); @*/
        if (current == null || current.left == null) {
            /*@ fold tree_perm(current); @*/
            res = current;
            /*@
            ghost resColor = oldColor;
            ghost resHeight = oldHeight;
            ghost resBag = oldBag;
            create {
                use res == current;
                qed (tree_perm(res) ** validSubtree(res)
                        ** subtreeFitsHole(res, oldColor, oldHeight, oldBag))
                    -* (tree_perm(current) ** validSubtree(current)
                        ** subtreeFitsHole(current, \old(getColor(current)), 
                                            \old(blackHeight(current)), \old(toBag(current))));
            }
            @*/
        } else {
            /*@
            ghost boolean leftColor = getColor(current.left);
            ghost int leftHeight = blackHeight(current.left);
            ghost bag<int> leftBag = toBag(current.left);
            @*/
            res = getMin(current.left) /*@ then {resColor=resColor;
                                                 resHeight=resHeight;
                                                 resBag=resBag;} @*/;
            /*@
            ghost makeWand(res, current, current.left, current.right,
                            oldColor, oldHeight, oldBag,
                            leftColor, leftHeight, leftBag,
                            getColor(res), blackHeight(res), toBag(res),
                            true);
            @*/
        }
        return res;
    }

    /// Find node with given key in given tree
    /*@
        yields boolean resColor;
        yields int resHeight;
        yields bag<int> resBag;
        requires tree_perm(current) ** validSubtree(current);
        ensures tree_perm(\result) ** validSubtree(\result);
        ensures current == null ==> \result == null; 
        ensures \result != null ==> \unfolding tree_perm(\result) \in \result.key == key;
        ensures (tree_perm(\result) ** validSubtree(\result)
                    ** subtreeFitsHole(\result, resColor, resHeight, resBag))
                -* (tree_perm(current) ** validSubtree(current)
                    ** subtreeFitsHole(current, \old(getColor(current)), 
                                        \old(blackHeight(current)), \old(toBag(current))));
        ensures resColor == getColor(\result) ** resHeight == blackHeight(\result) 
                ** resBag == toBag(\result);
    @*/
    Node search(Node current, int key)
    {
        Node res;
        /*@ 
        ghost boolean oldColor = getColor(current);
        ghost int oldHeight = blackHeight(current); 
        ghost bag<int> oldBag = toBag(current); 
        @*/
        /*@ unfold tree_perm(current); @*/
        if (current == null || current.key == key) {
            /*@ fold tree_perm(current); @*/
            res = current;
            /*@
            ghost resColor = oldColor;
            ghost resHeight = oldHeight;
            ghost resBag = oldBag;
            create {
                use res == current;
                qed (tree_perm(res) ** validSubtree(res)
                        ** subtreeFitsHole(res, oldColor, oldHeight, oldBag))
                    -* (tree_perm(current) ** validSubtree(current)
                        ** subtreeFitsHole(current, getColor(current), 
                                            blackHeight(current), toBag(current)));
            }
            @*/
        } else if (key < current.key) {
            /*@ 
            ghost boolean leftColor = getColor(current.left);
            ghost int leftHeight = blackHeight(current.left);
            ghost bag<int> leftBag = toBag(current.left);
            @*/
            res = search(current.left, key) /*@ then {resColor=resColor;
                                                      resHeight=resHeight;
                                                      resBag=resBag;} @*/;
            /*@ 
            ghost makeWand(res, current, current.left, current.right,
                            oldColor, oldHeight, oldBag,
                            leftColor, leftHeight, leftBag,
                            getColor(res), blackHeight(res), toBag(res),
                            true);
                
            @*/
        }
        else {
            /*@ 
            ghost boolean oldColor = current.color;
            ghost boolean rightColor = getColor(current.right);
            ghost int rightHeight = blackHeight(current.right);
            ghost bag<int> rightBag = toBag(current.right);
            @*/
            res = search(current.right, key) /*@ then {resColor=resColor;
                                                       resHeight=resHeight;
                                                       resBag=resBag;} @*/;
            /*@ 
            ghost makeWand(res, current, current.right, current.left,
                            oldColor, oldHeight, oldBag,
                            rightColor, rightHeight, rightBag,
                            getColor(res), blackHeight(res), toBag(res),
                            false);                
            @*/
        }
    
        return res;
    }


    /********************************************
     Rotate Tree Structure
     ********************************************/
    /// rotate given tree to the left
    /*@ 
        requires node != null ** tree_perm(node) ** sorted(node); 
        requires \unfolding tree_perm(node) \in (node.right != null); 
        /// regarding double-red:
        ///   - new connections must not create double-red
        requires !getColorLeft(node) || !getColorRight(node);
        requires !getColor(node) || \unfolding tree_perm(node) 
                                        \in !getColorRight(node.right);
        ///   - allow only specific cases of double-red, to be able to assure result
        requires noDoubleRed(node) || dbRedAtRight(node) 
                 || (!getColor(node) && \unfolding tree_perm(node) 
                                            \in (noDoubleRed(node.left) && dbRedAtRight(node.right)));
        
        /// regarding double-black: only allow one specific case to be able to assure result
        requires noDBlack(node) || dblackAtLeft(node);
        /// require roughly balanced to assure roughly balanced
        requires \unfolding tree_perm(node) 
                    \in (blackBalanced(node.left) && blackBalanced(node.right));
        ///    - new siblings must have same height
        requires \unfolding tree_perm(node) 
                    \in (blackHeight(node.left) == \unfolding tree_perm(node.right) 
                                                      \in blackHeight(node.right.left));
        /// ensure preservation properties
        ensures \result != null ** tree_perm(\result) ** sorted(\result); 
        ensures \unfolding tree_perm(\result) \in (\result.left != null); 
        ensures \old(toBag(node)) == toBag(\result); 
        ensures \old(getColor(node)) == getColor(\result);
        ensures \old(getColorRight(node)) == getColorLeft(\result); 
        ensures \old(\unfolding tree_perm(node) \in getColorRight(node.right)) 
                    == getColorRight(\result); 
        ensures \old(blackHeight(node)) == blackHeight(\result); 
        /// fixed double-red in most cases
        ensures \old(dbRedAtRight(node)) ? dbRedAtLeft(\result) : noDoubleRed(\result);
        /// double-black status preserved
        ensures \old(noDBlack(node)) ==> noDBlack(\result);
        ensures \old(dblackAtLeft(node)) ==> dblackAtLeftLeft(\result); 
        /// still roughly balanced, height of old siblings still equal
        ensures \unfolding tree_perm(\result) 
                    \in (blackBalanced(\result.left) && blackBalanced(\result.right));
        ensures \unfolding tree_perm(\result) 
                    \in (blackHeight(\result.right) == \unfolding tree_perm(\result.left) 
                                                          \in blackHeight(\result.left.right));
    @*/
    Node rotateLeft(Node node)
    {
        /*@ unfold tree_perm(node); @*/
        Node tempNode = node.right;
        boolean tempColor = node.color;
    
        /*@ ghost bag<int> tempContent = toBag(tempNode); @*/
        /*@ unfold tree_perm(tempNode); @*/
        /*@ ghost bag<int> tempLeftContent = toBag(tempNode.left); @*/
        /*@ assert subbag(tempLeftContent, tempContent); @*/
        /*@ assert (tempNode.key \memberof tempContent) != 0; @*/
        node.right = tempNode.left;
        tempNode.left = node;
    
        node.color = tempNode.color;
        tempNode.color = tempColor;
        /*@ fold tree_perm(node); @*/
        /*@ fold tree_perm(tempNode); @*/
        return tempNode;
    }

    /// rotate given tree to the right
    /*@
        requires node != null ** tree_perm(node) ** sorted(node); 
        requires \unfolding tree_perm(node) \in (node.left != null); 
        /// regarding double-red:
        ///   - new connections must not create double-red
        requires !getColorLeft(node) || !getColorRight(node);
        requires !getColor(node) || \unfolding tree_perm(node) 
                                        \in !getColorLeft(node.left);
        ///   - allow only specific cases of double-red, to be able to assure result
        requires noDoubleRed(node) || dbRedAtLeft(node) 
                 || (!getColor(node) && \unfolding tree_perm(node) 
                                            \in (noDoubleRed(node.right) && dbRedAtLeft(node.left)));
        
        /// regarding double-black: only allow one specific case to be able to assure result
        requires noDBlack(node) || dblackAtRight(node);
        /// require roughly balanced to assure roughly balanced
        requires \unfolding tree_perm(node) 
                    \in (blackBalanced(node.left) && blackBalanced(node.right));
        ///    - new siblings must have same height
        requires \unfolding tree_perm(node) 
                    \in (blackHeight(node.right) == \unfolding tree_perm(node.left) 
                                                      \in blackHeight(node.left.right));
        /// ensure preservation properties
        ensures \result != null ** tree_perm(\result) ** sorted(\result); 
        ensures \unfolding tree_perm(\result) \in (\result.right != null); 
        ensures \old(toBag(node)) == toBag(\result); 
        ensures \old(getColor(node)) == getColor(\result);
        ensures \old(getColorLeft(node)) == getColorRight(\result); 
        ensures \old(\unfolding tree_perm(node) \in getColorLeft(node.left)) 
                    == getColorLeft(\result); 
        ensures \old(blackHeight(node)) == blackHeight(\result); 
        /// fixed double-red in most cases
        ensures \old(dbRedAtLeft(node)) ? dbRedAtRight(\result) : noDoubleRed(\result);
        /// double-black status preserved
        ensures \old(noDBlack(node)) ==> noDBlack(\result);
        ensures \old(dblackAtRight(node)) ==> dblackAtRightRight(\result); 
        /// still roughly balanced, height of old siblings still equal
        ensures \unfolding tree_perm(\result) 
                    \in (blackBalanced(\result.left) && blackBalanced(\result.right));
        ensures \unfolding tree_perm(\result) 
                    \in (blackHeight(\result.left) == \unfolding tree_perm(\result.right) 
                                                          \in blackHeight(\result.right.left));

    @*/
    Node rotateRight(Node node)
    {
        /*@ unfold tree_perm(node); @*/
        Node tempNode = node.left;
        boolean tempColor = node.color;
        /*@ ghost bag<int> tempContent = toBag(tempNode); @*/
        /*@ unfold tree_perm(tempNode); @*/
        /*@ ghost bag<int> tempRightContent = toBag(tempNode.right); @*/
        /*@ assert subbag(tempRightContent, tempContent); @*/
        /*@ assert (tempNode.key \memberof tempContent) != 0; @*/
        node.left = tempNode.right;
        tempNode.right = node;
    
        node.color = tempNode.color;
        tempNode.color = tempColor;
        /*@ fold tree_perm(node); @*/
        /*@ fold tree_perm(tempNode); @*/
        return tempNode;
    }
}
