/**
 * Iterator to traverse a NodeList
 * @author Lukas Armborst, University of Twente
 * @year 2021
 */

final class NodeListIterator {

    /// NodeList to be iterated
    NodeList current;

    /// resource encapsulating access to current
    /*@
        public inline resource consumer() 
            = Perm(current, write)
              ** NodeList.list_perm(current)
              ** sorted(this);
    @*/
    

    
    /****************************************
     Wrapper functions for contained NodeList
    *****************************************/
    
    /*@
    /// NodeListIterator is sorted if the respective NodeList is
        requires it != null ==> Perm(it.current, read) ** [read]NodeList.list_perm(it.current);
    pure public static boolean sorted(NodeListIterator it)
        = it != null ==> NodeList.sorted(it.current);

    /// sequence representation of the Iterator is that of the NodeList
        requires it != null ==> Perm(it.current, read) ** [read]NodeList.list_perm(it.current);
        ensures it == null ? |\result| == 0 : |\result| == NodeList.size(it.current);
    pure public static seq<Node> toSeq(NodeListIterator it)
        = it == null ? seq<Node>{}
                     : NodeList.toSeq(it.current);

    /// sequence of the keys in the Iterator is that of the NodeList
        requires it != null ==> Perm(it.current, read) ** [read]NodeList.list_perm(it.current);
    pure public static seq<int> toSeqKeys(NodeListIterator it)
        = it == null ? seq<int>{}
                     : NodeList.toSeqKeys(it.current);
    @*/
    
    
    
    /***************************
     iterator-specific functions
     ***************************/
    
    /// constructor, for an iterator over a given NodeList
    /*@
        requires NodeList.list_perm(list);
        requires NodeList.sorted(list);
        ensures consumer();
        ensures current == list;
        ensures toSeq(this) == \old(NodeList.toSeq(list));
        ensures toSeqKeys(this) == \old(NodeList.toSeqKeys(list));
    @*/
    public NodeListIterator(NodeList list) {
        current = list;
    }
    
    /// check if there are more elements to iterate
    /*@
        given frac p;
        context Perm(current, p) ** ([p]NodeList.list_perm(current)) ** sorted(this);
        ensures \result == (current != null);
        ensures !\result ==> |toSeq(this)| == 0;
    @*/
    public boolean hasNext() {
        return current != null;
    }
    
    /// get next node in the contained list. Note that the node will be removed from the list
    /*@
        context consumer();
        requires current != null;
        ensures Node.node_perm(\result, write);
        ensures toSeq(this) == \old(tail(toSeq(this)));
        ensures toSeqKeys(this) == \old(tail(toSeqKeys(this)));
        ensures \result == \old(\unfolding NodeList.list_perm(current) \in current.node);
        ensures \result.key == \old(toSeqKeys(this)[0]);
        ensures \result != null;
    @*/
    public Node getNext() {
        /*@ ghost seq<int> keys = NodeList.toSeqKeys(current);  @*/
        /*@ unfold NodeList.list_perm(current); @*/
        Node res = current.node;
        /*@ assert NodeList.toSeqKeys(current.next) == tail(keys); @*/
        current = current.next;
        return res;
    }
}
